/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPairGenerator;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.interfaces.ECKey;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.elasticsearch.common.ssl.DerParser;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslFileUtil;
import org.elasticsearch.common.ssl.SslUtil;
import org.elasticsearch.core.CharArrays;

public final class PemUtils {
    private static final String PKCS1_HEADER = "-----BEGIN RSA PRIVATE KEY-----";
    private static final String PKCS1_FOOTER = "-----END RSA PRIVATE KEY-----";
    private static final String OPENSSL_DSA_HEADER = "-----BEGIN DSA PRIVATE KEY-----";
    private static final String OPENSSL_DSA_FOOTER = "-----END DSA PRIVATE KEY-----";
    private static final String OPENSSL_DSA_PARAMS_HEADER = "-----BEGIN DSA PARAMETERS-----";
    private static final String OPENSSL_DSA_PARAMS_FOOTER = "-----END DSA PARAMETERS-----";
    private static final String PKCS8_HEADER = "-----BEGIN PRIVATE KEY-----";
    private static final String PKCS8_FOOTER = "-----END PRIVATE KEY-----";
    private static final String PKCS8_ENCRYPTED_HEADER = "-----BEGIN ENCRYPTED PRIVATE KEY-----";
    private static final String PKCS8_ENCRYPTED_FOOTER = "-----END ENCRYPTED PRIVATE KEY-----";
    private static final String OPENSSL_EC_HEADER = "-----BEGIN EC PRIVATE KEY-----";
    private static final String OPENSSL_EC_FOOTER = "-----END EC PRIVATE KEY-----";
    private static final String OPENSSL_EC_PARAMS_HEADER = "-----BEGIN EC PARAMETERS-----";
    private static final String OPENSSL_EC_PARAMS_FOOTER = "-----END EC PARAMETERS-----";
    private static final String HEADER = "-----BEGIN";
    private static final String PBES2_OID = "1.2.840.113549.1.5.13";
    private static final String AES_OID = "2.16.840.1.101.3.4.1";

    private PemUtils() {
        throw new IllegalStateException("Utility class should not be instantiated");
    }

    public static PrivateKey readPrivateKey(Path path, Supplier<char[]> passwordSupplier) throws IOException, GeneralSecurityException {
        try {
            PrivateKey privateKey = PemUtils.parsePrivateKey(path, passwordSupplier);
            if (privateKey == null) {
                throw new SslConfigException("could not load ssl private key file [" + path + "]");
            }
            return privateKey;
        }
        catch (AccessControlException e) {
            throw SslFileUtil.accessControlFailure("PEM private key", List.of(path), e, null);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException("PEM private key", List.of(path), e);
        }
        catch (GeneralSecurityException e) {
            throw SslFileUtil.securityException("PEM private key", List.of(path), e);
        }
    }

    static PrivateKey parsePrivateKey(Path keyPath, Supplier<char[]> passwordSupplier) throws IOException, GeneralSecurityException {
        try (BufferedReader bReader = Files.newBufferedReader(keyPath, StandardCharsets.UTF_8);){
            String line = bReader.readLine();
            while (null != line && !line.startsWith(HEADER)) {
                line = bReader.readLine();
            }
            if (null == line) {
                throw new SslConfigException("Error parsing Private Key [" + keyPath.toAbsolutePath() + "], file is empty");
            }
            if (PKCS8_ENCRYPTED_HEADER.equals(line.trim())) {
                char[] password = passwordSupplier.get();
                if (password == null) {
                    throw new SslConfigException("cannot read encrypted key [" + keyPath.toAbsolutePath() + "] without a password");
                }
                PrivateKey privateKey = PemUtils.parsePKCS8Encrypted(bReader, password);
                return privateKey;
            }
            if (PKCS8_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parsePKCS8(bReader);
                return privateKey;
            }
            if (PKCS1_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parsePKCS1Rsa(bReader, passwordSupplier);
                return privateKey;
            }
            if (OPENSSL_DSA_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parseOpenSslDsa(bReader, passwordSupplier);
                return privateKey;
            }
            if (OPENSSL_DSA_PARAMS_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parseOpenSslDsa(PemUtils.removeDsaHeaders(bReader), passwordSupplier);
                return privateKey;
            }
            if (OPENSSL_EC_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parseOpenSslEC(bReader, passwordSupplier);
                return privateKey;
            }
            if (OPENSSL_EC_PARAMS_HEADER.equals(line.trim())) {
                PrivateKey privateKey = PemUtils.parseOpenSslEC(PemUtils.removeECHeaders(bReader), passwordSupplier);
                return privateKey;
            }
            throw new SslConfigException("cannot read PEM private key [" + keyPath.toAbsolutePath() + "] because the file does not contain a supported key format");
        }
    }

    private static BufferedReader removeECHeaders(BufferedReader bReader) throws IOException {
        String line = bReader.readLine();
        while (line != null && !OPENSSL_EC_PARAMS_FOOTER.equals(line.trim())) {
            line = bReader.readLine();
        }
        if (null == line || !OPENSSL_EC_PARAMS_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, EC Parameters footer is missing");
        }
        if (!OPENSSL_EC_HEADER.equals(bReader.readLine())) {
            throw new IOException("Malformed PEM file, EC Key header is missing");
        }
        return bReader;
    }

    private static BufferedReader removeDsaHeaders(BufferedReader bReader) throws IOException {
        String line = bReader.readLine();
        while (line != null && !OPENSSL_DSA_PARAMS_FOOTER.equals(line.trim())) {
            line = bReader.readLine();
        }
        if (null == line || !OPENSSL_DSA_PARAMS_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, DSA Parameters footer is missing");
        }
        if (!OPENSSL_DSA_HEADER.equals(bReader.readLine())) {
            throw new IOException("Malformed PEM file, DSA Key header is missing");
        }
        return bReader;
    }

    private static PrivateKey parsePKCS8(BufferedReader bReader) throws IOException, GeneralSecurityException {
        StringBuilder sb = new StringBuilder();
        String line = bReader.readLine();
        while (line != null && !PKCS8_FOOTER.equals(line.trim())) {
            sb.append(line.trim());
            line = bReader.readLine();
        }
        if (null == line || !PKCS8_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, PEM footer is invalid or missing");
        }
        return PemUtils.parsePKCS8PemString(sb.toString());
    }

    public static PrivateKey parsePKCS8PemString(String pemString) throws IOException, GeneralSecurityException {
        byte[] keyBytes = Base64.getDecoder().decode(pemString);
        String keyAlgo = PemUtils.getKeyAlgorithmIdentifier(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(keyAlgo);
        return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(keyBytes));
    }

    private static PrivateKey parseOpenSslEC(BufferedReader bReader, Supplier<char[]> passwordSupplier) throws IOException, GeneralSecurityException {
        StringBuilder sb = new StringBuilder();
        String line = bReader.readLine();
        HashMap<String, String> pemHeaders = new HashMap<String, String>();
        while (line != null && !OPENSSL_EC_FOOTER.equals(line.trim())) {
            if (line.contains(":")) {
                String[] header = line.split(":");
                pemHeaders.put(header[0].trim(), header[1].trim());
            } else {
                sb.append(line.trim());
            }
            line = bReader.readLine();
        }
        if (null == line || !OPENSSL_EC_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, PEM footer is invalid or missing");
        }
        byte[] keyBytes = PemUtils.possiblyDecryptPKCS1Key(pemHeaders, sb.toString(), passwordSupplier);
        KeyFactory keyFactory = KeyFactory.getInstance("EC");
        ECPrivateKeySpec ecSpec = PemUtils.parseEcDer(keyBytes);
        return keyFactory.generatePrivate(ecSpec);
    }

    private static PrivateKey parsePKCS1Rsa(BufferedReader bReader, Supplier<char[]> passwordSupplier) throws IOException, GeneralSecurityException {
        StringBuilder sb = new StringBuilder();
        String line = bReader.readLine();
        HashMap<String, String> pemHeaders = new HashMap<String, String>();
        while (line != null && !PKCS1_FOOTER.equals(line.trim())) {
            if (line.contains(":")) {
                String[] header = line.split(":");
                pemHeaders.put(header[0].trim(), header[1].trim());
            } else {
                sb.append(line.trim());
            }
            line = bReader.readLine();
        }
        if (null == line || !PKCS1_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, PEM footer is invalid or missing");
        }
        byte[] keyBytes = PemUtils.possiblyDecryptPKCS1Key(pemHeaders, sb.toString(), passwordSupplier);
        RSAPrivateCrtKeySpec spec = PemUtils.parseRsaDer(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        return keyFactory.generatePrivate(spec);
    }

    private static PrivateKey parseOpenSslDsa(BufferedReader bReader, Supplier<char[]> passwordSupplier) throws IOException, GeneralSecurityException {
        StringBuilder sb = new StringBuilder();
        String line = bReader.readLine();
        HashMap<String, String> pemHeaders = new HashMap<String, String>();
        while (line != null && !OPENSSL_DSA_FOOTER.equals(line.trim())) {
            if (line.contains(":")) {
                String[] header = line.split(":");
                pemHeaders.put(header[0].trim(), header[1].trim());
            } else {
                sb.append(line.trim());
            }
            line = bReader.readLine();
        }
        if (null == line || !OPENSSL_DSA_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, PEM footer is invalid or missing");
        }
        byte[] keyBytes = PemUtils.possiblyDecryptPKCS1Key(pemHeaders, sb.toString(), passwordSupplier);
        DSAPrivateKeySpec spec = PemUtils.parseDsaDer(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance("DSA");
        return keyFactory.generatePrivate(spec);
    }

    private static PrivateKey parsePKCS8Encrypted(BufferedReader bReader, char[] keyPassword) throws IOException, GeneralSecurityException {
        StringBuilder sb = new StringBuilder();
        String line = bReader.readLine();
        while (line != null && !PKCS8_ENCRYPTED_FOOTER.equals(line.trim())) {
            sb.append(line.trim());
            line = bReader.readLine();
        }
        if (null == line || !PKCS8_ENCRYPTED_FOOTER.equals(line.trim())) {
            throw new IOException("Malformed PEM file, PEM footer is invalid or missing");
        }
        byte[] keyBytes = Base64.getDecoder().decode(sb.toString());
        EncryptedPrivateKeyInfo encryptedPrivateKeyInfo = PemUtils.getEncryptedPrivateKeyInfo(keyBytes);
        String algorithm = encryptedPrivateKeyInfo.getAlgName();
        if (algorithm.equals("PBES2") || algorithm.equals(PBES2_OID)) {
            algorithm = PemUtils.getPBES2Algorithm(encryptedPrivateKeyInfo);
        }
        SecretKeyFactory secretKeyFactory = SecretKeyFactory.getInstance(algorithm);
        SecretKey secretKey = secretKeyFactory.generateSecret(new PBEKeySpec(keyPassword));
        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(2, (Key)secretKey, encryptedPrivateKeyInfo.getAlgParameters());
        PKCS8EncodedKeySpec keySpec = encryptedPrivateKeyInfo.getKeySpec(cipher);
        String keyAlgo = PemUtils.getKeyAlgorithmIdentifier(keySpec.getEncoded());
        KeyFactory keyFactory = KeyFactory.getInstance(keyAlgo);
        return keyFactory.generatePrivate(keySpec);
    }

    private static EncryptedPrivateKeyInfo getEncryptedPrivateKeyInfo(byte[] keyBytes) throws IOException, GeneralSecurityException {
        try {
            return new EncryptedPrivateKeyInfo(keyBytes);
        }
        catch (IOException e) {
            DerParser parser = new DerParser(keyBytes);
            DerParser.Asn1Object rootSeq = parser.readAsn1Object(16);
            parser = rootSeq.getParser();
            DerParser.Asn1Object algSeq = parser.readAsn1Object(16);
            parser = algSeq.getParser();
            String algId = parser.readAsn1Object(6).getOid();
            if (PBES2_OID.equals(algId)) {
                DerParser.Asn1Object algData = parser.readAsn1Object(16);
                parser = algData.getParser();
                DerParser.Asn1Object ignoreKdf = parser.readAsn1Object(16);
                DerParser.Asn1Object cryptSeq = parser.readAsn1Object(16);
                String encryptionId = (parser = cryptSeq.getParser()).readAsn1Object(6).getOid();
                if (!encryptionId.startsWith(AES_OID)) {
                    String name = PemUtils.getAlgorithmNameFromOid(encryptionId);
                    throw new GeneralSecurityException("PKCS#8 Private Key is encrypted with unsupported PBES2 algorithm [" + encryptionId + "]" + (String)(name == null ? "" : " (" + name + ")"), e);
                }
            }
            throw e;
        }
    }

    private static String getPBES2Algorithm(EncryptedPrivateKeyInfo encryptedPrivateKeyInfo) {
        AlgorithmParameters algParameters = encryptedPrivateKeyInfo.getAlgParameters();
        if (algParameters != null) {
            return algParameters.toString();
        }
        return encryptedPrivateKeyInfo.getAlgName();
    }

    private static byte[] possiblyDecryptPKCS1Key(Map<String, String> pemHeaders, String keyContents, Supplier<char[]> passwordSupplier) throws GeneralSecurityException, IOException {
        byte[] keyBytes = Base64.getDecoder().decode(keyContents);
        String procType = pemHeaders.get("Proc-Type");
        if ("4,ENCRYPTED".equals(procType)) {
            String encryptionParameters = pemHeaders.get("DEK-Info");
            if (null == encryptionParameters) {
                throw new IOException("Malformed PEM File, DEK-Info header is missing");
            }
            char[] password = passwordSupplier.get();
            if (password == null) {
                throw new IOException("cannot read encrypted key without a password");
            }
            Cipher cipher = PemUtils.getCipherFromParameters(encryptionParameters, password);
            byte[] decryptedKeyBytes = cipher.doFinal(keyBytes);
            return decryptedKeyBytes;
        }
        return keyBytes;
    }

    private static Cipher getCipherFromParameters(String dekHeaderValue, char[] password) throws GeneralSecurityException, IOException {
        SecretKeySpec encryptionKey;
        byte[] iv;
        String padding = "PKCS5Padding";
        String[] valueTokens = dekHeaderValue.split(",");
        if (valueTokens.length != 2) {
            throw new IOException("Malformed PEM file, DEK-Info PEM header is invalid");
        }
        String algorithm = valueTokens[0];
        String ivString = valueTokens[1];
        try {
            iv = PemUtils.hexStringToByteArray(ivString);
        }
        catch (IllegalArgumentException e) {
            throw new IOException("Malformed PEM file, DEK-Info IV is invalid", e);
        }
        if ("DES-CBC".equals(algorithm)) {
            key = PemUtils.generateOpenSslKey(password, iv, 8);
            encryptionKey = new SecretKeySpec(key, "DES");
        } else if ("DES-EDE3-CBC".equals(algorithm)) {
            key = PemUtils.generateOpenSslKey(password, iv, 24);
            encryptionKey = new SecretKeySpec(key, "DESede");
        } else if ("AES-128-CBC".equals(algorithm)) {
            key = PemUtils.generateOpenSslKey(password, iv, 16);
            encryptionKey = new SecretKeySpec(key, "AES");
        } else if ("AES-192-CBC".equals(algorithm)) {
            key = PemUtils.generateOpenSslKey(password, iv, 24);
            encryptionKey = new SecretKeySpec(key, "AES");
        } else if ("AES-256-CBC".equals(algorithm)) {
            key = PemUtils.generateOpenSslKey(password, iv, 32);
            encryptionKey = new SecretKeySpec(key, "AES");
        } else {
            throw new GeneralSecurityException("Private Key encrypted with unsupported algorithm [" + algorithm + "]");
        }
        String transformation = encryptionKey.getAlgorithm() + "/CBC/PKCS5Padding";
        Cipher cipher = Cipher.getInstance(transformation);
        cipher.init(2, (Key)encryptionKey, new IvParameterSpec(iv));
        return cipher;
    }

    private static byte[] generateOpenSslKey(char[] password, byte[] salt, int keyLength) {
        int bytesToCopy;
        byte[] passwordBytes = CharArrays.toUtf8Bytes((char[])password);
        MessageDigest md5 = SslUtil.messageDigest("md5");
        byte[] key = new byte[keyLength];
        for (int copied = 0; copied < keyLength; copied += bytesToCopy) {
            int remaining = keyLength - copied;
            md5.update(passwordBytes, 0, passwordBytes.length);
            md5.update(salt, 0, 8);
            byte[] tempDigest = md5.digest();
            bytesToCopy = remaining > 16 ? 16 : remaining;
            System.arraycopy(tempDigest, 0, key, copied, bytesToCopy);
            if (remaining == 0) break;
            md5.update(tempDigest, 0, 16);
        }
        Arrays.fill(passwordBytes, (byte)0);
        return key;
    }

    private static byte[] hexStringToByteArray(String hexString) {
        int len = hexString.length();
        if (len % 2 == 0) {
            byte[] data = new byte[len / 2];
            for (int i = 0; i < len; i += 2) {
                int k = Character.digit(hexString.charAt(i), 16);
                int l = Character.digit(hexString.charAt(i + 1), 16);
                if (k == -1 || l == -1) {
                    throw new IllegalStateException("String [" + hexString + "] is not hexadecimal");
                }
                data[i / 2] = (byte)((k << 4) + l);
            }
            return data;
        }
        throw new IllegalStateException("Hexadecimal string [" + hexString + "] has odd length and cannot be converted to a byte array");
    }

    private static ECPrivateKeySpec parseEcDer(byte[] keyBytes) throws IOException, GeneralSecurityException {
        DerParser parser = new DerParser(keyBytes);
        DerParser.Asn1Object sequence = parser.readAsn1Object();
        parser = sequence.getParser();
        parser.readAsn1Object().getInteger();
        String keyHex = parser.readAsn1Object().getString();
        BigInteger privateKeyInt = new BigInteger(keyHex, 16);
        DerParser.Asn1Object choice = parser.readAsn1Object();
        parser = choice.getParser();
        String namedCurve = PemUtils.getEcCurveNameFromOid(parser.readAsn1Object().getOid());
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("EC");
        ECGenParameterSpec algorithmParameterSpec = new ECGenParameterSpec(namedCurve);
        keyPairGenerator.initialize(algorithmParameterSpec);
        ECParameterSpec parameterSpec = ((ECKey)((Object)keyPairGenerator.generateKeyPair().getPrivate())).getParams();
        return new ECPrivateKeySpec(privateKeyInt, parameterSpec);
    }

    private static RSAPrivateCrtKeySpec parseRsaDer(byte[] keyBytes) throws IOException {
        DerParser parser = new DerParser(keyBytes);
        DerParser.Asn1Object sequence = parser.readAsn1Object();
        parser = sequence.getParser();
        parser.readAsn1Object().getInteger();
        BigInteger modulus = parser.readAsn1Object().getInteger();
        BigInteger publicExponent = parser.readAsn1Object().getInteger();
        BigInteger privateExponent = parser.readAsn1Object().getInteger();
        BigInteger prime1 = parser.readAsn1Object().getInteger();
        BigInteger prime2 = parser.readAsn1Object().getInteger();
        BigInteger exponent1 = parser.readAsn1Object().getInteger();
        BigInteger exponent2 = parser.readAsn1Object().getInteger();
        BigInteger coefficient = parser.readAsn1Object().getInteger();
        return new RSAPrivateCrtKeySpec(modulus, publicExponent, privateExponent, prime1, prime2, exponent1, exponent2, coefficient);
    }

    private static DSAPrivateKeySpec parseDsaDer(byte[] keyBytes) throws IOException {
        DerParser parser = new DerParser(keyBytes);
        DerParser.Asn1Object sequence = parser.readAsn1Object();
        parser = sequence.getParser();
        parser.readAsn1Object().getInteger();
        BigInteger p = parser.readAsn1Object().getInteger();
        BigInteger q = parser.readAsn1Object().getInteger();
        BigInteger g = parser.readAsn1Object().getInteger();
        parser.readAsn1Object().getInteger();
        BigInteger x = parser.readAsn1Object().getInteger();
        return new DSAPrivateKeySpec(x, p, q, g);
    }

    private static String getKeyAlgorithmIdentifier(byte[] keyBytes) throws IOException, GeneralSecurityException {
        String oidString;
        DerParser parser = new DerParser(keyBytes);
        DerParser.Asn1Object sequence = parser.readAsn1Object();
        parser = sequence.getParser();
        parser.readAsn1Object().getInteger();
        DerParser.Asn1Object algSequence = parser.readAsn1Object();
        parser = algSequence.getParser();
        return switch (oidString = parser.readAsn1Object().getOid()) {
            case "1.2.840.10040.4.1" -> "DSA";
            case "1.2.840.113549.1.1.1" -> "RSA";
            case "1.2.840.10045.2.1" -> "EC";
            default -> throw new GeneralSecurityException("Error parsing key algorithm identifier. Algorithm with OID [" + oidString + "] is not supported");
        };
    }

    public static List<Certificate> readCertificates(Collection<Path> certPaths) throws CertificateException, IOException {
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        ArrayList<Certificate> certificates = new ArrayList<Certificate>(certPaths.size());
        for (Path path : certPaths) {
            InputStream input = Files.newInputStream(path, new OpenOption[0]);
            try {
                Collection<? extends Certificate> parsed = certFactory.generateCertificates(input);
                if (parsed.isEmpty()) {
                    throw new SslConfigException("failed to parse any certificates from [" + path.toAbsolutePath() + "]");
                }
                certificates.addAll(parsed);
            }
            finally {
                if (input == null) continue;
                input.close();
            }
        }
        return certificates;
    }

    private static String getAlgorithmNameFromOid(String oidString) throws GeneralSecurityException {
        return switch (oidString) {
            case "1.2.840.10040.4.1" -> "DSA";
            case "1.2.840.113549.1.1.1" -> "RSA";
            case "1.2.840.10045.2.1" -> "EC";
            case "1.3.14.3.2.7" -> "DES-CBC";
            case "2.16.840.1.101.3.4.1.1" -> "AES-128_ECB";
            case "2.16.840.1.101.3.4.1.2" -> "AES-128_CBC";
            case "2.16.840.1.101.3.4.1.3" -> "AES-128_OFB";
            case "2.16.840.1.101.3.4.1.4" -> "AES-128_CFB";
            case "2.16.840.1.101.3.4.1.6" -> "AES-128_GCM";
            case "2.16.840.1.101.3.4.1.21" -> "AES-192_ECB";
            case "2.16.840.1.101.3.4.1.22" -> "AES-192_CBC";
            case "2.16.840.1.101.3.4.1.23" -> "AES-192_OFB";
            case "2.16.840.1.101.3.4.1.24" -> "AES-192_CFB";
            case "2.16.840.1.101.3.4.1.26" -> "AES-192_GCM";
            case "2.16.840.1.101.3.4.1.41" -> "AES-256_ECB";
            case "2.16.840.1.101.3.4.1.42" -> "AES-256_CBC";
            case "2.16.840.1.101.3.4.1.43" -> "AES-256_OFB";
            case "2.16.840.1.101.3.4.1.44" -> "AES-256_CFB";
            case "2.16.840.1.101.3.4.1.46" -> "AES-256_GCM";
            case "2.16.840.1.101.3.4.1.5" -> "AESWrap-128";
            case "2.16.840.1.101.3.4.1.25" -> "AESWrap-192";
            case "2.16.840.1.101.3.4.1.45" -> "AESWrap-256";
            default -> null;
        };
    }

    private static String getEcCurveNameFromOid(String oidString) throws GeneralSecurityException {
        return switch (oidString) {
            case "1.2.840.10045.3.1" -> "secp192r1";
            case "1.3.132.0.1" -> "sect163k1";
            case "1.3.132.0.15" -> "sect163r2";
            case "1.3.132.0.33" -> "secp224r1";
            case "1.3.132.0.26" -> "sect233k1";
            case "1.3.132.0.27" -> "sect233r1";
            case "1.2.840.10045.3.1.7" -> "secp256r1";
            case "1.3.132.0.16" -> "sect283k1";
            case "1.3.132.0.17" -> "sect283r1";
            case "1.3.132.0.34" -> "secp384r1";
            case "1.3.132.0.36" -> "sect409k1";
            case "1.3.132.0.37" -> "sect409r1";
            case "1.3.132.0.35" -> "secp521r1";
            case "1.3.132.0.38" -> "sect571k1";
            case "1.3.132.0.39" -> "sect571r1";
            default -> throw new GeneralSecurityException("Error parsing EC named curve identifier. Named curve with OID: " + oidString + " is not supported");
        };
    }
}

