/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.lease;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.lease.Lease;
import org.terracotta.lease.LeaseAcquirer;
import org.terracotta.lease.LeaseException;
import org.terracotta.lease.LeaseInternal;
import org.terracotta.lease.LeaseMaintainer;
import org.terracotta.lease.LeaseReconnectListener;
import org.terracotta.lease.LeaseReconnectingException;
import org.terracotta.lease.NullLease;
import org.terracotta.lease.TimeSource;
import org.terracotta.lease.TimeSourceProvider;

class LeaseMaintainerImpl
implements LeaseMaintainer,
LeaseReconnectListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(LeaseMaintainerImpl.class);
    private static final long MAXIMUM_WAIT_LENGTH = TimeUnit.MILLISECONDS.convert(5L, TimeUnit.MINUTES);
    private static final long RETRY_MILLIS_DURING_RECONNECT = 200L;
    private final LeaseAcquirer leaseAcquirer;
    private final TimeSource timeSource;
    private final AtomicReference<LeaseInternal> currentLease;
    private final CountDownLatch hasLease;

    LeaseMaintainerImpl(LeaseAcquirer leaseAcquirer) {
        this.leaseAcquirer = leaseAcquirer;
        this.timeSource = TimeSourceProvider.getTimeSource();
        this.currentLease = new AtomicReference<NullLease>(new NullLease());
        this.hasLease = new CountDownLatch(1);
    }

    @Override
    public Lease getCurrentLease() {
        return this.currentLease.get();
    }

    @Override
    public void waitForLease() throws InterruptedException {
        this.hasLease.await();
    }

    @Override
    public boolean waitForLease(long timeout, TimeUnit timeUnit) throws InterruptedException {
        return this.hasLease.await(timeout, timeUnit);
    }

    @Override
    public void close() throws IOException {
        this.leaseAcquirer.close();
    }

    long refreshLease() throws LeaseException, InterruptedException {
        try {
            LOGGER.debug("Refreshing lease");
            do {
                LeaseInternal lease = this.currentLease.get();
                long leaseRequestStartNanos = this.timeSource.nanoTime();
                long leaseLengthMillis = this.leaseAcquirer.acquireLease();
                long leaseRequestEndNanos = this.timeSource.nanoTime();
                boolean updated = this.updateLease(lease, leaseRequestStartNanos, leaseRequestEndNanos, leaseLengthMillis);
                LOGGER.debug("Upodated lease {}", (Object)updated);
                if (!updated) continue;
                this.hasLease.countDown();
                long waitTime = this.calculateWaitLength(leaseRequestStartNanos, leaseRequestEndNanos, leaseLengthMillis);
                LOGGER.debug("lease wait time {}", (Object)waitTime);
                return waitTime;
            } while (!Thread.interrupted());
            throw new InterruptedException();
        }
        catch (LeaseReconnectingException e) {
            LOGGER.debug(e.getMessage());
            return 200L;
        }
    }

    @Override
    public void reconnecting() {
        this.currentLease.set(new NullLease());
    }

    @Override
    public void reconnected() {
    }

    private long calculateWaitLength(long leaseRequestStartNanos, long leaseRequestEndNanos, long leaseLengthMillis) {
        long leaseAcquisitionNanos = leaseRequestEndNanos - leaseRequestStartNanos;
        long leaseAcquisitionMillis = TimeUnit.MILLISECONDS.convert(leaseAcquisitionNanos, TimeUnit.NANOSECONDS);
        long waitLength = leaseLengthMillis / 3L - leaseAcquisitionMillis;
        return Math.max(0L, Math.min(MAXIMUM_WAIT_LENGTH, waitLength));
    }

    private boolean updateLease(LeaseInternal lease, long leaseRequestStartNanos, long leaseRequestEndNanos, long leaseLengthMillis) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("updateLease: leaseRequestStartNanos: " + leaseRequestStartNanos + " leaseRequestEndNanos: " + leaseRequestEndNanos + " leaseLengthMillis: " + leaseLengthMillis);
        }
        long leaseStart = leaseRequestEndNanos;
        long leaseExpiry = leaseRequestStartNanos + TimeUnit.MILLISECONDS.toNanos(leaseLengthMillis);
        if (leaseExpiry - leaseStart < 0L) {
            LOGGER.warn("Received new lease but it expires before it starts.");
        }
        LeaseInternal updatedLease = lease.extend(this.timeSource, leaseStart, leaseExpiry);
        boolean updated = this.currentLease.compareAndSet(lease, updatedLease);
        this.logLease(updated, lease, updatedLease);
        return updated;
    }

    private void logLease(boolean updated, Lease previousLease, Lease newLease) {
        boolean gap;
        if (!updated) {
            LOGGER.info("Received new lease, but could not use it because another lease was more recent, perhaps due to reconnect.");
            return;
        }
        boolean bl = gap = !newLease.isValidAndContiguous(previousLease) && !(previousLease instanceof NullLease);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Lease updated. Last lease: " + previousLease + " latest lease: " + newLease);
        }
        if (gap) {
            LOGGER.warn("A gap in leases occurred. nanoTime: " + this.timeSource.nanoTime());
        }
    }

    @Override
    public void destroy() throws IOException {
        throw new UnsupportedOperationException();
    }
}

