/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import org.eclipse.collections.api.CharIterable;
import org.eclipse.collections.api.LazyCharIterable;
import org.eclipse.collections.api.bag.ImmutableBag;
import org.eclipse.collections.api.bag.primitive.CharBag;
import org.eclipse.collections.api.bag.primitive.ImmutableCharBag;
import org.eclipse.collections.api.bag.primitive.MutableCharBag;
import org.eclipse.collections.api.block.function.primitive.CharToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.CharPredicate;
import org.eclipse.collections.api.block.procedure.primitive.CharIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.CharProcedure;
import org.eclipse.collections.api.iterator.CharIterator;
import org.eclipse.collections.api.list.primitive.MutableCharList;
import org.eclipse.collections.api.set.primitive.MutableCharSet;
import org.eclipse.collections.impl.bag.mutable.primitive.CharHashBag;
import org.eclipse.collections.impl.block.procedure.checked.primitive.CheckedCharIntProcedure;
import org.eclipse.collections.impl.iterator.UnmodifiableCharIterator;

/**
 * ImmutableCharHashBag is the non-modifiable equivalent of {@link CharHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableCharHashBag implements ImmutableCharBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableCharBag delegate;

    private ImmutableCharHashBag(char[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = CharHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use CharBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableCharHashBag newBagWith(char... elements)
    {
        return new ImmutableCharHashBag(elements);
    }

    public ImmutableCharBag newWith(char element)
    {
        return CharHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableCharBag newWithout(char element)
    {
        CharHashBag hashBag = CharHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableCharBag newWithAll(CharIterable elements)
    {
        CharHashBag bag = CharHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableCharBag newWithoutAll(CharIterable elements)
    {
        CharHashBag bag = CharHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(char value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(CharIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(char... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(CharProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(CharProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableCharBag select(CharPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableCharBag reject(CharPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(CharToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableCharList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(char item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(CharIntProcedure charIntProcedure)
    {
        this.delegate.forEachWithOccurrences(charIntProcedure);
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(CharPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public char min()
    {
        return this.delegate.min();
    }

    public char max()
    {
        return this.delegate.max();
    }

    public char maxIfEmpty(char defaultValue)
    {
        return this.delegate.maxIfEmpty(defaultValue);
    }

    public char minIfEmpty(char defaultValue)
    {
        return this.delegate.minIfEmpty(defaultValue);
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public char[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableCharList toSortedList()
    {
        return this.delegate.toSortedList();
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        return this.delegate.injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableCharSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableCharBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableCharBag toImmutable()
    {
        return this;
    }

    public LazyCharIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public char[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public CharIterator charIterator()
    {
        return new UnmodifiableCharIterator(this.delegate.charIterator());
    }

    private Object writeReplace()
    {
        return new ImmutableCharBagSerializationProxy(this);
    }

    protected static class ImmutableCharBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private CharBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableCharBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableCharBagSerializationProxy(CharBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedCharIntProcedure()
                {
                    @Override
                    public void safeValue(char item, int count) throws IOException
                    {
                        out.writeChar(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableCharBag deserializedBag = new CharHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readChar(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
