/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.StringUtil;

public class Asn1DerDecoder {
    public static final String EC = "EC";
    public static final String RSA = "RSA";
    public static final String DSA = "DSA";
    public static final String DH = "DH";
    public static final String ECv2 = "EC.v2";
    private static final int MAX_DEFAULT_LENGTH = 65536;
    private static final int TAG_SEQUENCE = 48;
    private static final int MAX_OID_LENGTH = 32;
    private static final int TAG_OID = 6;
    private static final int TAG_INTEGER = 2;
    private static final int TAG_OCTET_STRING = 4;
    private static final int TAG_BIT_STRING = 3;
    private static final int TAG_CONTEXT_0_SPECIFIC = 160;
    private static final int TAG_CONTEXT_1_SPECIFIC = 161;
    private static final byte[] OID_RSA_PUBLIC_KEY = new byte[]{42, -122, 72, -122, -9, 13, 1, 1, 1};
    private static final byte[] OID_DH_PUBLIC_KEY = new byte[]{42, -122, 72, -122, -9, 13, 1, 3, 1};
    private static final byte[] OID_DSA_PUBLIC_KEY = new byte[]{42, -122, 72, -50, 56, 4, 1};
    private static final byte[] OID_EC_PUBLIC_KEY = new byte[]{42, -122, 72, -50, 61, 2, 1};
    private static final EntityDefinition SEQUENCE = new EntityDefinition(48, 65536, "SEQUENCE");
    private static final OidEntityDefinition OID = new OidEntityDefinition();
    private static final IntegerEntityDefinition INTEGER = new IntegerEntityDefinition();
    private static final EntityDefinition BIT_STRING = new EntityDefinition(3, 65536, "BIT STRING");
    private static final EntityDefinition OCTET_STRING = new EntityDefinition(4, 65536, "OCTET STRING");
    private static final EntityDefinition CONTEXT_SPECIFIC_0 = new EntityDefinition(160, 65536, "CONTEXT SPECIFIC 0");
    private static final EntityDefinition CONTEXT_SPECIFIC_1 = new EntityDefinition(161, 65536, "CONTEXT SPECIFIC 1");

    public static byte[] readSequenceEntity(DatagramReader reader) {
        return SEQUENCE.readEntity(reader);
    }

    public static byte[] readSequenceValue(DatagramReader reader) {
        return SEQUENCE.readValue(reader);
    }

    public static byte[] readOidValue(DatagramReader reader) {
        return OID.readValue(reader);
    }

    public static String readOidString(DatagramReader reader) {
        byte[] oid = OID.readValue(reader);
        return OID.toString(oid);
    }

    public static String readSubjectPublicKeyAlgorithm(byte[] data) {
        DatagramReader reader = new DatagramReader(data, false);
        reader = SEQUENCE.createRangeReader(reader, false);
        reader = SEQUENCE.createRangeReader(reader, false);
        byte[] value = Asn1DerDecoder.readOidValue(reader);
        String algorithm = null;
        if (Arrays.equals(value, OID_EC_PUBLIC_KEY)) {
            algorithm = EC;
        } else if (Arrays.equals(value, OID_RSA_PUBLIC_KEY)) {
            algorithm = RSA;
        } else if (Arrays.equals(value, OID_DSA_PUBLIC_KEY)) {
            algorithm = DSA;
        } else if (Arrays.equals(value, OID_DH_PUBLIC_KEY)) {
            algorithm = DH;
        }
        return algorithm;
    }

    public static PublicKey readSubjectPublicKey(byte[] data) throws GeneralSecurityException {
        PublicKey publicKey = null;
        String algorithm = Asn1DerDecoder.readSubjectPublicKeyAlgorithm(data);
        if (algorithm != null) {
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(data);
            publicKey = KeyFactory.getInstance(algorithm).generatePublic(publicKeySpec);
        }
        return publicKey;
    }

    public static String readPrivateKeyAlgorithm(byte[] data) {
        String algorithm = null;
        DatagramReader reader = new DatagramReader(data, false);
        byte[] readValue = INTEGER.readValue(reader = SEQUENCE.createRangeReader(reader, false));
        int version = INTEGER.toInteger(readValue);
        if (version == 0) {
            byte[] value = Asn1DerDecoder.readOidValue(reader = SEQUENCE.createRangeReader(reader, false));
            if (Arrays.equals(value, OID_EC_PUBLIC_KEY)) {
                algorithm = EC;
            } else if (Arrays.equals(value, OID_RSA_PUBLIC_KEY)) {
                algorithm = RSA;
            } else if (Arrays.equals(value, OID_DSA_PUBLIC_KEY)) {
                algorithm = DSA;
            } else if (Arrays.equals(value, OID_DH_PUBLIC_KEY)) {
                algorithm = DH;
            }
        } else if (version == 1) {
            OCTET_STRING.createRangeReader(reader, false);
            byte[] oid = Asn1DerDecoder.readOidValue(CONTEXT_SPECIFIC_0.createRangeReader(reader, false));
            String oidAsString = "0x" + StringUtil.byteArray2Hex(oid);
            try {
                oidAsString = OID.toString(oid);
                try {
                    ECParameterSpec ecParameterSpec = Asn1DerDecoder.getECParameterSpec(oidAsString);
                    if (ecParameterSpec != null) {
                        algorithm = ECv2;
                    }
                }
                catch (GeneralSecurityException generalSecurityException) {}
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
            if (algorithm == null) {
                throw new IllegalArgumentException("OID " + oidAsString + " not supported!");
            }
        } else {
            throw new IllegalArgumentException("Version 0x" + StringUtil.byteArray2Hex(data) + " not supported!");
        }
        return algorithm;
    }

    public static Keys readPrivateKey(byte[] data) throws GeneralSecurityException {
        Keys keys = null;
        String algorithm = Asn1DerDecoder.readPrivateKeyAlgorithm(data);
        if (algorithm != null) {
            if (algorithm.equals(ECv2)) {
                keys = Asn1DerDecoder.readEcPrivateKeyV2(data);
            } else {
                PKCS8EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(data);
                keys = new Keys();
                keys.privateKey = KeyFactory.getInstance(algorithm).generatePrivate(privateKeySpec);
            }
        }
        return keys;
    }

    public static Keys readEcPrivateKeyV2(byte[] data) throws GeneralSecurityException {
        Keys keys = null;
        DatagramReader reader = new DatagramReader(data, false);
        byte[] readValue = INTEGER.readValue(reader = SEQUENCE.createRangeReader(reader, false));
        if (readValue.length == 1 && readValue[0] == 1) {
            byte[] privateKeyValue = OCTET_STRING.readValue(reader);
            byte[] oid = Asn1DerDecoder.readOidValue(CONTEXT_SPECIFIC_0.createRangeReader(reader, false));
            try {
                ECParameterSpec ecParameterSpec = Asn1DerDecoder.getECParameterSpec(OID.toString(oid));
                int keySize = (ecParameterSpec.getCurve().getField().getFieldSize() + 8 - 1) / 8;
                if (privateKeyValue.length != keySize) {
                    throw new GeneralSecurityException("private key size " + privateKeyValue.length + " doesn't macth " + keySize);
                }
                ECPrivateKeySpec privateKeySpec = new ECPrivateKeySpec(new BigInteger(1, privateKeyValue), ecParameterSpec);
                keys = new Keys();
                keys.privateKey = KeyFactory.getInstance(EC).generatePrivate(privateKeySpec);
                DatagramReader value = CONTEXT_SPECIFIC_1.createRangeReader(reader, false);
                value = BIT_STRING.createRangeReader(value, false);
                int unusedBits = value.read(8);
                if (unusedBits == 0) {
                    keys.publicKey = Asn1DerDecoder.readEcPublicKey(value, ecParameterSpec);
                }
            }
            catch (IllegalArgumentException e) {
                throw new GeneralSecurityException(e.getMessage(), e);
            }
            catch (GeneralSecurityException e) {
                throw e;
            }
        }
        return keys;
    }

    public static ECPublicKey readEcPublicKey(DatagramReader reader, ECParameterSpec ecParameterSpec) throws GeneralSecurityException {
        int keySize = (ecParameterSpec.getCurve().getField().getFieldSize() + 8 - 1) / 8;
        int compress = reader.read(8);
        int left = reader.bitsLeft() / 8;
        if (compress == 4 && left % 2 == 0 && (left /= 2) == keySize) {
            BigInteger x = new BigInteger(1, reader.readBytes(left));
            BigInteger y = new BigInteger(1, reader.readBytes(left));
            ECPublicKeySpec publicKeySpec = new ECPublicKeySpec(new ECPoint(x, y), ecParameterSpec);
            return (ECPublicKey)KeyFactory.getInstance(EC).generatePublic(publicKeySpec);
        }
        return null;
    }

    public static boolean equalKeyAlgorithmSynonyms(String keyAlgorithm1, String keyAlgorithm2) {
        if (!keyAlgorithm1.equals(keyAlgorithm2)) {
            if (keyAlgorithm1.equals(DH) && keyAlgorithm2.equals("DiffieHellman")) {
                return true;
            }
            return keyAlgorithm2.equals(DH) && keyAlgorithm1.equals("DiffieHellman");
        }
        return true;
    }

    public static ECParameterSpec getECParameterSpec(String oid) throws GeneralSecurityException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(EC);
        keyPairGenerator.initialize(new ECGenParameterSpec(oid));
        ECPublicKey apub = (ECPublicKey)keyPairGenerator.generateKeyPair().getPublic();
        return apub.getParams();
    }

    private static class IntegerEntityDefinition
    extends EntityDefinition {
        public IntegerEntityDefinition() {
            super(2, 65536, "INTEGER");
        }

        public int toInteger(byte[] integerByteArray) {
            int sign;
            if (integerByteArray == null) {
                throw new NullPointerException("INTEGER byte array must not be null!");
            }
            if (integerByteArray.length == 0) {
                throw new IllegalArgumentException("INTEGER byte array must not be empty!");
            }
            if (integerByteArray.length > 4) {
                throw new IllegalArgumentException("INTEGER byte array " + integerByteArray.length + " bytes is too large for int (max. 4 bytes)!");
            }
            int result = sign = integerByteArray[0];
            for (int index = 1; index < integerByteArray.length; ++index) {
                result <<= 8;
                result |= integerByteArray[index] & 0xFF;
            }
            if (sign >= 0 ^ result >= 0) {
                throw new IllegalArgumentException("INTEGER byte array value overflow!");
            }
            return result;
        }
    }

    private static class OidEntityDefinition
    extends EntityDefinition {
        public OidEntityDefinition() {
            super(6, 32, "OID");
        }

        public String toString(byte[] oid) {
            StringBuilder result = new StringBuilder();
            int value = oid[0] & 0xFF;
            result.append(value / 40).append(".").append(value % 40);
            for (int index = 1; index < oid.length; ++index) {
                byte bValue = oid[index];
                if (bValue < 0) {
                    value = bValue & 0x7F;
                    if (++index == oid.length) {
                        throw new IllegalArgumentException("Invalid OID 0x" + StringUtil.byteArray2Hex(oid));
                    }
                    value <<= 7;
                    result.append(".").append(value |= oid[index] & 0x7F);
                    continue;
                }
                result.append(".").append(bValue);
            }
            return result.toString();
        }
    }

    private static class EntityDefinition {
        private static final int HEADER_LENGTH = 2;
        private final int expectedTag;
        private final int maxLength;
        private final String description;

        public EntityDefinition(int expectedTag, int maxLength, String description) {
            this.expectedTag = expectedTag;
            this.maxLength = maxLength;
            this.description = description;
        }

        public byte[] readEntity(DatagramReader reader) {
            return this.read(reader, true);
        }

        public byte[] readValue(DatagramReader reader) {
            return this.read(reader, false);
        }

        public byte[] read(DatagramReader reader, boolean entity) {
            int length = this.readLength(reader, entity);
            return reader.readBytes(length);
        }

        public DatagramReader createRangeReader(DatagramReader reader, boolean entity) {
            int length = this.readLength(reader, entity);
            return reader.createRangeReader(length);
        }

        public int readLength(DatagramReader reader, boolean entity) {
            int tag;
            int leftBytes = reader.bitsLeft() / 8;
            if (leftBytes < 2) {
                throw new IllegalArgumentException(String.format("Not enough bytes for %s! Required %d, available %d.", this.description, 2, leftBytes));
            }
            if (entity) {
                reader.mark();
            }
            if ((tag = reader.read(8)) != this.expectedTag) {
                throw new IllegalArgumentException(String.format("No %s, found %02x instead of %02x!", this.description, tag, this.expectedTag));
            }
            int length = reader.read(8);
            int entityLength = length + 2;
            if (length > 127) {
                if ((length &= 0x7F) > 4) {
                    throw new IllegalArgumentException(String.format("%s length-size %d too long!", this.description, length));
                }
                leftBytes = reader.bitsLeft() / 8;
                if (length > leftBytes) {
                    throw new IllegalArgumentException(String.format("%s length %d exceeds available bytes %d!", this.description, length, leftBytes));
                }
                byte[] lengthBytes = reader.readBytes(length);
                length = 0;
                for (int index = 0; index < lengthBytes.length; ++index) {
                    length <<= 8;
                    length += lengthBytes[index] & 0xFF;
                }
                entityLength = length + 2 + lengthBytes.length;
            }
            if (length > this.maxLength) {
                throw new IllegalArgumentException(String.format("%s lenght %d too large! (supported maxium %d)", this.description, length, this.maxLength));
            }
            leftBytes = reader.bitsLeft() / 8;
            if (length > leftBytes) {
                throw new IllegalArgumentException(String.format("%s lengh %d exceeds available bytes %d!", this.description, length, leftBytes));
            }
            if (entity) {
                reader.reset();
                length = entityLength;
            }
            return length;
        }
    }

    public static class Keys {
        private PrivateKey privateKey;
        private PublicKey publicKey;

        public Keys() {
        }

        public Keys(PrivateKey privateKey, PublicKey publicKey) {
            this.privateKey = privateKey;
            this.publicKey = publicKey;
        }

        public void add(Keys keys) {
            if (keys.privateKey != null) {
                this.privateKey = keys.privateKey;
            }
            if (keys.publicKey != null) {
                this.publicKey = keys.publicKey;
            }
        }

        public PrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public void setPrivateKey(PrivateKey privateKey) {
            this.privateKey = privateKey;
        }

        public PublicKey getPublicKey() {
            return this.publicKey;
        }

        public void setPublicKey(PublicKey publicKey) {
            this.publicKey = publicKey;
        }
    }
}

