/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.EndpointContext;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UdpEndpointContext;
import org.eclipse.californium.elements.exception.EndpointMismatchException;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.ClockUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UDPConnector
implements Connector {
    public static final Logger LOGGER = LoggerFactory.getLogger(UDPConnector.class.getName());
    public static final int UNDEFINED = 0;
    static final ThreadGroup ELEMENTS_THREAD_GROUP = new ThreadGroup("Californium/Elements");
    protected final InetSocketAddress localAddr;
    private final List<Thread> receiverThreads = new LinkedList<Thread>();
    private final List<Thread> senderThreads = new LinkedList<Thread>();
    private final BlockingQueue<RawData> outgoing;
    protected volatile boolean running;
    private volatile DatagramSocket socket;
    private volatile InetSocketAddress effectiveAddr;
    private volatile EndpointContextMatcher endpointContextMatcher;
    private volatile RawDataChannel receiver;
    private int receiveBufferSize = 0;
    private int sendBufferSize = 0;
    private int senderCount = 1;
    private int receiverCount = 1;
    private int receiverPacketSize = 2048;

    public UDPConnector() {
        this(null);
    }

    public UDPConnector(InetSocketAddress address) {
        this.localAddr = address == null ? new InetSocketAddress(0) : address;
        this.running = false;
        this.effectiveAddr = this.localAddr;
        this.outgoing = new LinkedBlockingQueue<RawData>();
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.running) {
            return;
        }
        this.init(new DatagramSocket(this.localAddr.getPort(), this.localAddr.getAddress()));
    }

    protected void init(DatagramSocket socket) throws IOException {
        int i;
        this.socket = socket;
        this.effectiveAddr = (InetSocketAddress)socket.getLocalSocketAddress();
        if (this.receiveBufferSize != 0) {
            socket.setReceiveBufferSize(this.receiveBufferSize);
        }
        this.receiveBufferSize = socket.getReceiveBufferSize();
        if (this.sendBufferSize != 0) {
            socket.setSendBufferSize(this.sendBufferSize);
        }
        this.sendBufferSize = socket.getSendBufferSize();
        this.running = true;
        LOGGER.info("UDPConnector starts up {} sender threads and {} receiver threads", (Object)this.senderCount, (Object)this.receiverCount);
        for (i = 0; i < this.receiverCount; ++i) {
            this.receiverThreads.add(new Receiver("UDP-Receiver-" + this.localAddr + "[" + i + "]"));
        }
        for (i = 0; i < this.senderCount; ++i) {
            this.senderThreads.add(new Sender("UDP-Sender-" + this.localAddr + "[" + i + "]"));
        }
        for (Thread t : this.receiverThreads) {
            t.start();
        }
        for (Thread t : this.senderThreads) {
            t.start();
        }
        LOGGER.info("UDPConnector listening on {}, recv buf = {}, send buf = {}, recv packet size = {}", this.effectiveAddr, this.receiveBufferSize, this.sendBufferSize, this.receiverPacketSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void stop() {
        ArrayList pending = new ArrayList(this.outgoing.size());
        UDPConnector uDPConnector = this;
        synchronized (uDPConnector) {
            if (!this.running) {
                return;
            }
            this.running = false;
            for (Thread t : this.senderThreads) {
                t.interrupt();
            }
            for (Thread t : this.receiverThreads) {
                t.interrupt();
            }
            this.outgoing.drainTo(pending);
            if (this.socket != null) {
                this.socket.close();
                this.socket = null;
            }
            for (Thread t : this.senderThreads) {
                t.interrupt();
                try {
                    t.join(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            this.senderThreads.clear();
            for (Thread t : this.receiverThreads) {
                t.interrupt();
                try {
                    t.join(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            this.receiverThreads.clear();
            LOGGER.info("UDPConnector on [{}] has stopped.", (Object)this.effectiveAddr);
        }
        for (RawData data : pending) {
            this.notifyMsgAsInterrupted(data);
        }
    }

    @Override
    public void destroy() {
        this.stop();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void send(RawData msg) {
        boolean running;
        if (msg == null) {
            throw new NullPointerException("Message must not be null");
        }
        UDPConnector uDPConnector = this;
        synchronized (uDPConnector) {
            running = this.running;
            if (running) {
                this.outgoing.add(msg);
            }
        }
        if (!running) {
            this.notifyMsgAsInterrupted(msg);
        }
    }

    @Override
    public void setRawDataReceiver(RawDataChannel receiver) {
        this.receiver = receiver;
    }

    @Override
    public void setEndpointContextMatcher(EndpointContextMatcher matcher) {
        this.endpointContextMatcher = matcher;
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.effectiveAddr;
    }

    private void notifyMsgAsInterrupted(RawData msg) {
        msg.onError(new InterruptedIOException("Connector is not running."));
    }

    public void setReceiveBufferSize(int size) {
        this.receiveBufferSize = size;
    }

    public int getReceiveBufferSize() {
        return this.receiveBufferSize;
    }

    public void setSendBufferSize(int size) {
        this.sendBufferSize = size;
    }

    public int getSendBufferSize() {
        return this.sendBufferSize;
    }

    public void setReceiverThreadCount(int count) {
        this.receiverCount = count;
    }

    public int getReceiverThreadCount() {
        return this.receiverCount;
    }

    public void setSenderThreadCount(int count) {
        this.senderCount = count;
    }

    public int getSenderThreadCount() {
        return this.senderCount;
    }

    public void setReceiverPacketSize(int size) {
        this.receiverPacketSize = size;
    }

    public int getReceiverPacketSize() {
        return this.receiverPacketSize;
    }

    @Override
    public String getProtocol() {
        return "UDP";
    }

    public String toString() {
        return this.getProtocol() + "-" + StringUtil.toString(this.getAddress());
    }

    static {
        ELEMENTS_THREAD_GROUP.setDaemon(false);
    }

    private class Sender
    extends NetworkStageThread {
        private DatagramPacket datagram;

        private Sender(String name) {
            super(name);
            this.datagram = new DatagramPacket(Bytes.EMPTY, 0);
        }

        @Override
        protected void work() throws InterruptedException {
            RawData raw = (RawData)UDPConnector.this.outgoing.take();
            EndpointContext destination = raw.getEndpointContext();
            InetSocketAddress destinationAddress = destination.getPeerAddress();
            UdpEndpointContext connectionContext = new UdpEndpointContext(destinationAddress);
            EndpointContextMatcher endpointMatcher = UDPConnector.this.endpointContextMatcher;
            if (endpointMatcher != null && !endpointMatcher.isToBeSent(destination, connectionContext)) {
                LOGGER.warn("UDPConnector ({}) drops {} bytes to {}:{}", UDPConnector.this.effectiveAddr, this.datagram.getLength(), destinationAddress.getAddress(), destinationAddress.getPort());
                raw.onError(new EndpointMismatchException());
                return;
            }
            this.datagram.setData(raw.getBytes());
            this.datagram.setSocketAddress(destinationAddress);
            DatagramSocket currentSocket = UDPConnector.this.socket;
            if (currentSocket != null) {
                try {
                    raw.onContextEstablished(connectionContext);
                    currentSocket.send(this.datagram);
                    raw.onSent();
                }
                catch (IOException ex) {
                    raw.onError(ex);
                }
                LOGGER.debug("UDPConnector ({}) sent {} bytes to {}:{}", this, this.datagram.getLength(), this.datagram.getAddress(), this.datagram.getPort());
            } else {
                raw.onError(new IOException("socket already closed!"));
            }
        }
    }

    private class Receiver
    extends NetworkStageThread {
        private DatagramPacket datagram;
        private int size;

        private Receiver(String name) {
            super(name);
            this.size = UDPConnector.this.receiverPacketSize + 1;
            this.datagram = new DatagramPacket(new byte[this.size], this.size);
        }

        @Override
        protected void work() throws IOException {
            this.datagram.setLength(this.size);
            DatagramSocket currentSocket = UDPConnector.this.socket;
            if (currentSocket != null) {
                currentSocket.receive(this.datagram);
                if (this.datagram.getLength() >= this.size) {
                    LOGGER.debug("UDPConnector ({}) received truncated UDP datagram from {}:{}. Maximum size allowed {}. Discarding ...", UDPConnector.this.effectiveAddr, this.datagram.getAddress(), this.datagram.getPort(), this.size - 1);
                } else {
                    long timestamp = ClockUtil.nanoRealtime();
                    LOGGER.debug("UDPConnector ({}) received {} bytes from {}:{}", UDPConnector.this.effectiveAddr, this.datagram.getLength(), this.datagram.getAddress(), this.datagram.getPort());
                    byte[] bytes = Arrays.copyOfRange(this.datagram.getData(), this.datagram.getOffset(), this.datagram.getLength());
                    RawData msg = RawData.inbound(bytes, new UdpEndpointContext(new InetSocketAddress(this.datagram.getAddress(), this.datagram.getPort())), false, timestamp);
                    UDPConnector.this.receiver.receiveData(msg);
                }
            }
        }
    }

    private abstract class NetworkStageThread
    extends Thread {
        protected NetworkStageThread(String name) {
            super(ELEMENTS_THREAD_GROUP, name);
            this.setDaemon(true);
        }

        @Override
        public void run() {
            LOGGER.debug("Starting network stage thread [{}]", (Object)this.getName());
            while (UDPConnector.this.running) {
                try {
                    this.work();
                    if (UDPConnector.this.running) continue;
                    LOGGER.debug("Network stage thread [{}] was stopped successfully", (Object)this.getName());
                    break;
                }
                catch (InterruptedIOException t) {
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (InterruptedException t) {
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (IOException t) {
                    if (UDPConnector.this.running) {
                        LOGGER.error("Exception in network stage thread [{}]:", (Object)this.getName(), (Object)t);
                        continue;
                    }
                    LOGGER.trace("Network stage thread [{}] was stopped successfully at:", (Object)this.getName(), (Object)t);
                }
                catch (Throwable t) {
                    LOGGER.error("Exception in network stage thread [{}]:", (Object)this.getName(), (Object)t);
                }
            }
        }

        protected abstract void work() throws Exception;
    }
}

