/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.cli;

import java.io.IOException;
import java.io.PrintStream;
import java.net.InetSocketAddress;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import javax.crypto.SecretKey;
import org.eclipse.californium.cli.CliConnectorFactory;
import org.eclipse.californium.cli.ClientBaseConfig;
import org.eclipse.californium.cli.ConnectorConfig;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.network.CoapEndpoint;
import org.eclipse.californium.core.network.Endpoint;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.interceptors.MessageInterceptor;
import org.eclipse.californium.core.network.interceptors.MessageTracer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.UDPConnector;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.DTLSConnector;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.CertificateType;
import org.eclipse.californium.scandium.dtls.ConnectionId;
import org.eclipse.californium.scandium.dtls.ConnectionIdGenerator;
import org.eclipse.californium.scandium.dtls.PskPublicInformation;
import org.eclipse.californium.scandium.dtls.PskSecretResult;
import org.eclipse.californium.scandium.dtls.PskSecretResultHandler;
import org.eclipse.californium.scandium.dtls.SingleNodeConnectionIdGenerator;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.pskstore.AdvancedPskStore;
import org.eclipse.californium.scandium.dtls.x509.StaticNewAdvancedCertificateVerifier;
import org.eclipse.californium.scandium.util.SecretUtil;
import org.eclipse.californium.scandium.util.ServerNames;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;

public class ClientInitializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientInitializer.class);
    public static final String KEY_DTLS_RETRANSMISSION_TIMEOUT = "DTLS_RETRANSMISSION_TIMEOUT";
    public static final String KEY_DTLS_RETRANSMISSION_MAX = "DTLS_RETRANSMISSION_MAX";
    private static final String DEFAULT_TCP_MODULE = "org.eclipse.californium.cli.tcp.netty.Initialize";
    private static final List<String> loadErrors = new ArrayList<String>();
    private static final Map<String, CliConnectorFactory> connectorFactories = new ConcurrentHashMap<String, CliConnectorFactory>();

    public static CliConnectorFactory registerConnectorFactory(String protocol, CliConnectorFactory factory) {
        return connectorFactories.put(protocol, factory);
    }

    public static CliConnectorFactory unregisterConnectorFactory(String protocol) {
        return connectorFactories.remove(protocol);
    }

    public static void init(String[] args, ClientBaseConfig config) throws IOException {
        ClientInitializer.init(args, config, true);
    }

    public static void init(String[] args, ClientBaseConfig config, boolean createEndpoint) throws IOException {
        CommandLine cmd = new CommandLine((Object)config);
        config.register(cmd);
        try {
            CommandLine.ParseResult result = cmd.parseArgs(args);
            if (result.isVersionHelpRequested()) {
                String version = StringUtil.CALIFORNIUM_VERSION == null ? "" : StringUtil.CALIFORNIUM_VERSION;
                System.out.println("\nCalifornium (Cf) " + cmd.getCommandName() + " " + version);
                cmd.printVersionHelp(System.out);
                System.out.println();
            }
            config.defaults();
            if (config.helpRequested) {
                cmd.usage(System.out);
                if (config.authHelpRequested) {
                    System.out.println();
                    System.out.println("   --auth: values");
                    ClientInitializer.print("      ", 60, Arrays.asList(ConnectorConfig.AuthenticationMode.values()), System.out);
                }
                if (config.cipherHelpRequested) {
                    ArrayList<CipherSuite> list = new ArrayList<CipherSuite>();
                    for (CipherSuite cipherSuite : CipherSuite.values()) {
                        if (!cipherSuite.isSupported() || CipherSuite.TLS_NULL_WITH_NULL_NULL.equals((Object)cipherSuite)) continue;
                        list.add(cipherSuite);
                    }
                    System.out.println();
                    System.out.println("   --cipher: values");
                    ClientInitializer.print("      ", 60, list, System.out);
                }
                return;
            }
        }
        catch (CommandLine.ParameterException ex) {
            ex.printStackTrace();
            System.err.println(ex.getMessage());
            System.err.println();
            cmd.usage(System.err);
            System.exit(-1);
        }
        if (createEndpoint) {
            CoapEndpoint coapEndpoint = ClientInitializer.createEndpoint(config, null);
            coapEndpoint.start();
            LOGGER.info("endpoint started at {}", (Object)coapEndpoint.getAddress());
            EndpointManager.getEndpointManager().setDefaultEndpoint((Endpoint)coapEndpoint);
        }
    }

    public static CoapEndpoint createEndpoint(ClientBaseConfig clientConfig, ExecutorService executor) {
        String scheme = CoAP.getSchemeFromUri((String)clientConfig.uri);
        if (scheme != null) {
            String protocol = CoAP.getProtocolForScheme((String)scheme);
            if (protocol != null) {
                CliConnectorFactory factory = connectorFactories.get(protocol);
                if (factory != null) {
                    CoapEndpoint.Builder builder = new CoapEndpoint.Builder();
                    Connector connector = factory.create(clientConfig, executor);
                    if (connector instanceof UDPConnector) {
                        builder.setConnectorWithAutoConfiguration((UDPConnector)connector);
                    } else {
                        builder.setConnector(connector);
                    }
                    builder.setNetworkConfig(clientConfig.networkConfig);
                    CoapEndpoint endpoint = builder.build();
                    if (clientConfig.verbose) {
                        endpoint.addInterceptor((MessageInterceptor)new MessageTracer());
                    }
                    return endpoint;
                }
                if (CoAP.isTcpProtocol((String)protocol) && loadErrors.contains(DEFAULT_TCP_MODULE)) {
                    throw new IllegalArgumentException(protocol + " is not supported! Tcp-module not found!");
                }
                throw new IllegalArgumentException(protocol + " is not supported!");
            }
            throw new IllegalArgumentException(scheme + " is not supported!");
        }
        throw new IllegalArgumentException("Missing scheme in " + clientConfig.uri);
    }

    public static void print(String tab, int width, List<?> values, PrintStream out) {
        StringBuilder line = new StringBuilder();
        line.append(tab);
        for (Object value : values) {
            String name = value.toString();
            if (line.length() + name.length() > width) {
                out.println(line);
                line.setLength(tab.length());
            }
            line.append(name).append(" ");
        }
        out.println(line);
    }

    static {
        connectorFactories.put("UDP", new UdpConnectorFactory());
        connectorFactories.put("DTLS", new DtlsConnectorFactory());
        String factories = StringUtil.getConfiguration((String)"CONNECTOR_FACTORIES");
        if (factories == null) {
            factories = DEFAULT_TCP_MODULE;
        }
        if (!factories.isEmpty()) {
            String[] initializers;
            for (String initializer : initializers = factories.split("#")) {
                try {
                    Class.forName(initializer);
                }
                catch (ClassNotFoundException e) {
                    loadErrors.add(initializer);
                }
            }
        }
    }

    public static class PlugPskStore
    implements AdvancedPskStore {
        private final PskPublicInformation identity;
        private final SecretKey secret;

        public PlugPskStore(String id, byte[] secret) {
            this.identity = new PskPublicInformation(id);
            this.secret = secret == null ? ConnectorConfig.PSK_SECRET : SecretUtil.create((byte[])secret, (String)"PSK");
            LOGGER.trace("DTLS-PSK-Identity: {}", (Object)this.identity);
        }

        public PlugPskStore(String id) {
            this.identity = new PskPublicInformation("cali." + id);
            this.secret = null;
            LOGGER.trace("DTLS-PSK-Identity: {} ({} random bytes)", (Object)this.identity, (Object)(id.length() / 2));
        }

        public boolean hasEcdhePskSupported() {
            return true;
        }

        public PskSecretResult requestPskSecretResult(ConnectionId cid, ServerNames serverName, PskPublicInformation identity, String hmacAlgorithm, SecretKey otherSecret, byte[] seed) {
            SecretKey secret = null;
            if (this.identity.equals((Object)identity)) {
                secret = this.secret == null && identity.getPublicInfoAsString().startsWith("cali.") ? SecretUtil.create((SecretKey)ConnectorConfig.PSK_SECRET) : SecretUtil.create((SecretKey)this.secret);
            }
            return new PskSecretResult(cid, this.identity, secret);
        }

        public PskPublicInformation getIdentity(InetSocketAddress peerAddress, ServerNames virtualHost) {
            return this.identity;
        }

        public void setResultHandler(PskSecretResultHandler resultHandler) {
        }
    }

    public static class DtlsConnectorFactory
    implements CliConnectorFactory {
        public static DtlsConnectorConfig.Builder createDtlsConfig(ClientBaseConfig clientConfig) {
            NetworkConfig config = clientConfig.networkConfig;
            int maxPeers = config.getInt("MAX_ACTIVE_PEERS");
            int staleTimeout = config.getInt("MAX_PEER_INACTIVITY_PERIOD");
            int senderThreads = config.getInt("NETWORK_STAGE_SENDER_THREAD_COUNT");
            int receiverThreads = config.getInt("NETWORK_STAGE_RECEIVER_THREAD_COUNT");
            int retransmissionTimeout = config.getInt("ACK_TIMEOUT");
            Integer healthStatusInterval = config.getInt("HEALTH_STATUS_INTERVAL");
            Integer cidLength = config.getOptInteger("DTLS_CONNECTION_ID_LENGTH");
            Integer recvBufferSize = config.getOptInteger("UDP_CONNECTOR_RECEIVE_BUFFER");
            Integer sendBufferSize = config.getOptInteger("UDP_CONNECTOR_SEND_BUFFER");
            Integer dtlsRetransmissionTimeout = config.getOptInteger(ClientInitializer.KEY_DTLS_RETRANSMISSION_TIMEOUT);
            Integer dtlsRetransmissionMax = config.getOptInteger(ClientInitializer.KEY_DTLS_RETRANSMISSION_MAX);
            if (dtlsRetransmissionTimeout != null) {
                retransmissionTimeout = dtlsRetransmissionTimeout;
            }
            int localPort = clientConfig.localPort == null ? 0 : clientConfig.localPort;
            Integer recordSizeLimit = clientConfig.recordSizeLimit;
            Integer mtu = clientConfig.mtu;
            if (clientConfig.cidLength != null) {
                cidLength = clientConfig.cidLength;
            }
            DtlsConnectorConfig.Builder dtlsConfig = new DtlsConnectorConfig.Builder();
            dtlsConfig.setClientOnly();
            boolean psk = false;
            ArrayList<CipherSuite.KeyExchangeAlgorithm> keyExchangeAlgorithms = new ArrayList<CipherSuite.KeyExchangeAlgorithm>();
            ArrayList<CertificateType> certificateTypes = new ArrayList<CertificateType>();
            for (ConnectorConfig.AuthenticationMode auth : clientConfig.authenticationModes) {
                switch (auth) {
                    case NONE: {
                        break;
                    }
                    case PSK: {
                        psk = true;
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.PSK);
                        break;
                    }
                    case RPK: {
                        certificateTypes.add(CertificateType.RAW_PUBLIC_KEY);
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.EC_DIFFIE_HELLMAN);
                        dtlsConfig.setAdvancedCertificateVerifier(StaticNewAdvancedCertificateVerifier.builder().setTrustAllRPKs().build());
                        break;
                    }
                    case X509: {
                        certificateTypes.add(CertificateType.X_509);
                        dtlsConfig.setAdvancedCertificateVerifier(StaticNewAdvancedCertificateVerifier.builder().setTrustedCertificates(clientConfig.trust.trusts).build());
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.EC_DIFFIE_HELLMAN);
                        break;
                    }
                    case ECDHE_PSK: {
                        psk = true;
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.ECDHE_PSK);
                    }
                }
            }
            if (clientConfig.authentication != null && clientConfig.authentication.credentials != null) {
                if (certificateTypes.contains(CertificateType.X_509)) {
                    dtlsConfig.setIdentity(clientConfig.authentication.credentials.getPrivateKey(), (Certificate[])clientConfig.authentication.credentials.getCertificateChain(), certificateTypes);
                } else if (certificateTypes.contains(CertificateType.RAW_PUBLIC_KEY)) {
                    dtlsConfig.setIdentity(clientConfig.authentication.credentials.getPrivateKey(), clientConfig.authentication.credentials.getPubicKey());
                }
            }
            if (psk) {
                if (clientConfig.identity != null) {
                    dtlsConfig.setAdvancedPskStore((AdvancedPskStore)new PlugPskStore(clientConfig.identity, clientConfig.secretKey));
                } else {
                    byte[] rid = new byte[8];
                    SecureRandom random = new SecureRandom();
                    random.nextBytes(rid);
                    dtlsConfig.setAdvancedPskStore((AdvancedPskStore)new PlugPskStore(StringUtil.byteArray2Hex((byte[])rid)));
                }
            }
            if (!keyExchangeAlgorithms.isEmpty() && (clientConfig.cipherSuites == null || clientConfig.cipherSuites.isEmpty())) {
                clientConfig.cipherSuites = CipherSuite.getCipherSuitesByKeyExchangeAlgorithm((boolean)false, (boolean)true, keyExchangeAlgorithms);
            }
            if (clientConfig.cipherSuites != null && !clientConfig.cipherSuites.isEmpty()) {
                dtlsConfig.setRecommendedCipherSuitesOnly(false);
                dtlsConfig.setSupportedCipherSuites(clientConfig.cipherSuites);
                if (clientConfig.verbose) {
                    System.out.println("cipher suites:");
                    ClientInitializer.print("   ", 50, clientConfig.cipherSuites, System.out);
                }
            }
            if (cidLength != null) {
                dtlsConfig.setConnectionIdGenerator((ConnectionIdGenerator)new SingleNodeConnectionIdGenerator(cidLength.intValue()));
            }
            dtlsConfig.setSocketReceiveBufferSize(recvBufferSize);
            dtlsConfig.setSocketSendBufferSize(sendBufferSize);
            dtlsConfig.setRetransmissionTimeout(retransmissionTimeout);
            if (dtlsRetransmissionMax != null) {
                dtlsConfig.setMaxRetransmissions(dtlsRetransmissionMax.intValue());
            }
            dtlsConfig.setMaxConnections(maxPeers);
            dtlsConfig.setConnectionThreadCount(senderThreads);
            dtlsConfig.setReceiverThreadCount(receiverThreads);
            dtlsConfig.setStaleConnectionThreshold((long)staleTimeout);
            dtlsConfig.setAddress(new InetSocketAddress(localPort));
            dtlsConfig.setHealthStatusInterval(healthStatusInterval);
            dtlsConfig.setRecordSizeLimit(recordSizeLimit);
            if (mtu != null) {
                dtlsConfig.setMaxTransmissionUnit(mtu.intValue());
            }
            return dtlsConfig;
        }

        @Override
        public Connector create(ClientBaseConfig clientConfig, ExecutorService executor) {
            DtlsConnectorConfig.Builder dtlsConfig = DtlsConnectorFactory.createDtlsConfig(clientConfig);
            DTLSConnector dtlsConnector = new DTLSConnector(dtlsConfig.build());
            if (executor != null) {
                dtlsConnector.setExecutor(executor);
            }
            return dtlsConnector;
        }
    }

    public static class UdpConnectorFactory
    implements CliConnectorFactory {
        @Override
        public Connector create(ClientBaseConfig clientConfig, ExecutorService executor) {
            int localPort = clientConfig.localPort == null ? 0 : clientConfig.localPort;
            return new UDPConnector(new InetSocketAddress(localPort));
        }
    }
}

