/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.storage.rdbms;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.sql.DataSource;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.dspace.core.Context;
import org.dspace.discovery.IndexingService;
import org.dspace.discovery.SearchServiceException;
import org.dspace.services.ConfigurationService;
import org.dspace.services.factory.DSpaceServicesFactory;
import org.dspace.storage.rdbms.FlywayUpgradeUtils;
import org.dspace.workflow.factory.WorkflowServiceFactory;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.FlywayException;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.api.MigrationVersion;
import org.flywaydb.core.api.callback.Callback;
import org.flywaydb.core.api.configuration.FluentConfiguration;
import org.flywaydb.core.internal.info.MigrationInfoDumper;
import org.flywaydb.core.internal.license.VersionPrinter;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.SingleConnectionDataSource;

public class DatabaseUtils {
    private static final Logger log = LogManager.getLogger(DatabaseUtils.class);
    private static final String reindexDiscoveryFilePath = DSpaceServicesFactory.getInstance().getConfigurationService().getProperty("dspace.dir") + File.separator + "solr" + File.separator + "search" + File.separator + "conf" + File.separator + "reindex.flag";
    public static final String DBMS_POSTGRES = "postgres";
    public static final String DBMS_H2 = "h2";
    public static final String FLYWAY_TABLE = "schema_version";

    private DatabaseUtils() {
    }

    public static void main(String[] argv) {
        if (argv.length < 1) {
            System.out.println("\nDatabase action argument is missing.");
            System.out.println("Valid actions: 'test', 'info', 'migrate', 'repair', 'skip', 'validate', 'update-sequences' or 'clean'");
            System.out.println("\nOr, type 'database help' for more information.\n");
            System.exit(1);
        }
        try {
            DataSource dataSource = DatabaseUtils.getDataSource();
            FluentConfiguration flywayConfiguration = DatabaseUtils.setupFlyway(dataSource);
            Flyway flyway = flywayConfiguration.load();
            FlywayUpgradeUtils.upgradeFlywayTable(flyway, dataSource.getConnection());
            switch (argv[0].toLowerCase(Locale.ENGLISH)) {
                case "test": {
                    System.out.println("\nAttempting to connect to database");
                    try (Connection connection = dataSource.getConnection();){
                        System.out.println("Connected successfully!");
                        DatabaseUtils.printDBInfo(connection);
                        break;
                    }
                    catch (SQLException sqle) {
                        System.err.println("\nError running 'test': ");
                        System.err.println(" - " + sqle);
                        System.err.println("\nPlease see the DSpace documentation for assistance.\n");
                        sqle.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "info": 
                case "status": {
                    try (Connection connection = dataSource.getConnection();){
                        DatabaseUtils.printDBInfo(connection);
                        System.out.println("\n" + MigrationInfoDumper.dumpToAsciiTable((MigrationInfo[])flyway.info().all()));
                        if (!DatabaseUtils.tableExists(connection, flyway.getConfiguration().getTable(), true)) {
                            System.out.println("\nNOTE: This database is NOT yet initialized for auto-migrations (via Flyway).");
                            String dbVersion = DatabaseUtils.determineDBVersion(connection);
                            if (dbVersion != null) {
                                System.out.println("\nYour database looks to be compatible with DSpace version " + dbVersion);
                                System.out.println("All upgrades *after* version " + dbVersion + " will be run during the next migration.");
                                System.out.println("\nIf you'd like to upgrade now, simply run 'dspace database migrate'.");
                            }
                        }
                        break;
                    }
                    catch (SQLException e) {
                        System.err.println("Info exception:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "migrate": {
                    try (Connection connection = dataSource.getConnection();){
                        System.out.println("\nDatabase URL: " + connection.getMetaData().getURL());
                        if (argv.length == 2) {
                            if (argv[1].equalsIgnoreCase("ignored")) {
                                System.out.println("Migrating database to latest version AND running previously \"Ignored\" migrations... (Check logs for details)");
                                DatabaseUtils.updateDatabase(dataSource, connection, null, true);
                            } else if (argv[1].equalsIgnoreCase("force")) {
                                DatabaseUtils.updateDatabase(dataSource, connection, null, false, true);
                            } else {
                                String migrationVersion = argv[1];
                                BufferedReader input = new BufferedReader(new InputStreamReader(System.in, StandardCharsets.UTF_8));
                                System.out.println("You've specified to migrate your database ONLY to version " + migrationVersion + " ...");
                                System.out.println("\nWARNING: In this mode, we DISABLE all callbacks, which means that you will need to manually update registries and manually run a reindex. This is because you are attempting to use an OLD version (" + migrationVersion + ") Database with a newer DSpace API. NEVER do this in a PRODUCTION scenario. The resulting database is only useful for migration testing.\n");
                                System.out.print("Are you SURE you only want to migrate your database to version " + migrationVersion + "? [y/n]: ");
                                String choiceString = input.readLine();
                                input.close();
                                if (choiceString.equalsIgnoreCase("y")) {
                                    System.out.println("Migrating database ONLY to version " + migrationVersion + " ... (Check logs for details)");
                                    DatabaseUtils.updateDatabase(dataSource, connection, migrationVersion, false);
                                } else {
                                    System.out.println("No action performed.");
                                }
                            }
                        } else {
                            System.out.println("Migrating database to latest version... (Check dspace logs for details)");
                            DatabaseUtils.updateDatabase(dataSource, connection);
                        }
                        System.out.println("Done.");
                        System.exit(0);
                        break;
                    }
                    catch (SQLException e) {
                        System.err.println("Migration exception:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "repair": {
                    try (Connection connection = dataSource.getConnection();){
                        System.out.println("\nDatabase URL: " + connection.getMetaData().getURL());
                        System.out.println("Attempting to repair any previously failed migrations (or mismatched checksums) via FlywayDB... (Check dspace logs for details)");
                        flyway.repair();
                        System.out.println("Done.");
                        System.exit(0);
                        break;
                    }
                    catch (SQLException | FlywayException e) {
                        System.err.println("Repair exception:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "skip": {
                    try {
                        if (argv.length == 2) {
                            String migrationVersion = argv[1];
                            BufferedReader input = new BufferedReader(new InputStreamReader(System.in, StandardCharsets.UTF_8));
                            System.out.println("You've specified to SKIP the migration with version='" + migrationVersion + "' ...");
                            System.out.print("\nWARNING: You should only skip migrations which are no longer required or have become obsolete. Skipping a REQUIRED migration may result in DSpace failing to startup or function properly. Are you sure you want to SKIP the migration with version '" + migrationVersion + "'? [y/n]: ");
                            String choiceString = input.readLine();
                            input.close();
                            if (choiceString.equalsIgnoreCase("y")) {
                                System.out.println("Attempting to skip migration with version " + migrationVersion + " ... (Check logs for details)");
                                DatabaseUtils.skipMigration(dataSource, migrationVersion);
                            }
                            break;
                        }
                        System.out.println("The 'skip' command REQUIRES a version to be specified. Only that single migration will be skipped. For the list of migration versions use the 'info' command.");
                    }
                    catch (IOException e) {
                        System.err.println("Exception when attempting to skip migration:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                    }
                    break;
                }
                case "validate": {
                    try (Connection connection = dataSource.getConnection();){
                        System.out.println("\nDatabase URL: " + connection.getMetaData().getURL());
                        System.out.println("Attempting to validate database status (and migration checksums) via FlywayDB...");
                        flyway.validate();
                        System.out.println("No errors thrown. Validation succeeded. (Check dspace logs for more details)");
                        System.exit(0);
                        break;
                    }
                    catch (SQLException | FlywayException e) {
                        System.err.println("Validation exception:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "clean": {
                    if (flyway.getConfiguration().isCleanDisabled()) {
                        System.out.println("\nWARNING: 'clean' command is currently disabled, as it is dangerous to run in Production scenarios!");
                        System.out.println("\nIn order to run a 'clean' you first must enable it in your DSpace config by specifying 'db.cleanDisabled=false'.\n");
                        System.exit(1);
                    }
                    try (Connection connection = dataSource.getConnection();){
                        BufferedReader input = new BufferedReader(new InputStreamReader(System.in, StandardCharsets.UTF_8));
                        System.out.println("\nDatabase URL: " + connection.getMetaData().getURL());
                        System.out.println("\nWARNING: ALL DATA AND TABLES IN YOUR DATABASE WILL BE PERMANENTLY DELETED.\n");
                        System.out.println("There is NO turning back from this action. Backup your DB before continuing.");
                        System.out.print("Do you want to PERMANENTLY DELETE everything from your database? [y/n]: ");
                        String choiceString = input.readLine();
                        input.close();
                        if (choiceString.equalsIgnoreCase("y")) {
                            System.out.println("Scrubbing database clean... (Check dspace logs for details)");
                            DatabaseUtils.cleanDatabase(flyway, dataSource);
                            System.out.println("Done.");
                            System.exit(0);
                        } else {
                            System.out.println("No action performed.");
                        }
                        break;
                    }
                    catch (SQLException e) {
                        System.err.println("Clean exception:");
                        e.printStackTrace(System.err);
                        System.exit(1);
                        break;
                    }
                }
                case "update-sequences": {
                    try (Connection connection = dataSource.getConnection();){
                        String dbType = DatabaseUtils.getDbType(connection);
                        String sqlfile = "org/dspace/storage/rdbms/sqlmigration/" + dbType + "/update-sequences.sql";
                        InputStream sqlstream = DatabaseUtils.class.getClassLoader().getResourceAsStream(sqlfile);
                        if (sqlstream != null) {
                            String s = IOUtils.toString((InputStream)sqlstream, (Charset)StandardCharsets.UTF_8);
                            if (!s.isEmpty()) {
                                System.out.println("Running " + sqlfile);
                                connection.createStatement().execute(s);
                                System.out.println("update-sequences complete");
                            } else {
                                System.err.println(sqlfile + " contains no SQL to execute");
                            }
                        } else {
                            System.err.println(sqlfile + " not found");
                        }
                        break;
                    }
                }
                default: {
                    System.out.println("\nUsage: database [action]");
                    System.out.println("Valid actions: 'test', 'info', 'migrate', 'repair', 'skip', 'validate', 'update-sequences' or 'clean'");
                    System.out.println(" - test             = Performs a test connection to database to validate connection settings");
                    System.out.println(" - info / status    = Describe basic info/status about database, including validating the compatibility of this database");
                    System.out.println(" - migrate          = Migrate the database to the latest version");
                    System.out.println(" - repair           = Attempt to repair any previously failed database migrations or checksum mismatches (via Flyway repair)");
                    System.out.println(" - skip [version]   = Skip a single, pending or ignored migration, ensuring it never runs.");
                    System.out.println(" - validate         = Validate current database's migration status (via Flyway validate), validating all migration checksums.");
                    System.out.println(" - update-sequences = Update database sequences after running AIP ingest.");
                    System.out.println(" - clean            = DESTROY all data and tables in database (WARNING there is no going back!). Requires 'db.cleanDisabled=false' setting in config.");
                    System.out.println("");
                    System.exit(0);
                }
            }
        }
        catch (Exception e) {
            System.err.println("Caught exception:");
            e.printStackTrace(System.err);
            System.exit(1);
        }
    }

    private static void printDBInfo(Connection connection) throws SQLException {
        DatabaseMetaData meta = connection.getMetaData();
        String dbType = DatabaseUtils.getDbType(connection);
        System.out.println("\nDatabase Type: " + dbType);
        if (!dbType.equals(DBMS_POSTGRES) && !dbType.equals(DBMS_H2)) {
            System.err.println("====================================");
            System.err.println("ERROR: Database type " + dbType + " is UNSUPPORTED!");
            System.err.println("=====================================");
        }
        System.out.println("Database URL: " + meta.getURL());
        System.out.println("Database Schema: " + DatabaseUtils.getSchemaName(connection));
        System.out.println("Database Username: " + meta.getUserName());
        System.out.println("Database Software: " + meta.getDatabaseProductName() + " version " + meta.getDatabaseProductVersion());
        System.out.println("Database Driver: " + meta.getDriverName() + " version " + meta.getDriverVersion());
        System.out.println("FlywayDB Version: " + VersionPrinter.getVersion());
    }

    private static synchronized FluentConfiguration setupFlyway(DataSource datasource) {
        ConfigurationService config = DSpaceServicesFactory.getInstance().getConfigurationService();
        FluentConfiguration flywayConfiguration = Flyway.configure();
        try (Connection connection = datasource.getConnection();){
            flywayConfiguration.dataSource(datasource);
            flywayConfiguration.encoding("UTF-8");
            flywayConfiguration.cleanDisabled(config.getBooleanProperty("db.cleanDisabled", true));
            String dbType = DatabaseUtils.getDbType(connection);
            connection.close();
            ArrayList<Object> scriptLocations = new ArrayList<Object>();
            String etcDirPath = config.getProperty("dspace.dir") + "/etc/" + dbType;
            File etcDir = new File(etcDirPath);
            if (etcDir.exists() && !dbType.equals(DBMS_H2)) {
                scriptLocations.add("filesystem:" + etcDirPath);
            }
            scriptLocations.add("classpath:org/dspace/storage/rdbms/sqlmigration/" + dbType);
            scriptLocations.add("classpath:org/dspace/storage/rdbms/migration");
            List<String> workflowFlywayMigrationLocations = WorkflowServiceFactory.getInstance().getWorkflowService().getFlywayMigrationLocations();
            scriptLocations.addAll(workflowFlywayMigrationLocations);
            log.info("Loading Flyway DB migrations from: " + StringUtils.join(scriptLocations, (String)", "));
            flywayConfiguration.locations(scriptLocations.toArray(new String[scriptLocations.size()]));
            flywayConfiguration.ignoreMigrationPatterns(new String[]{"*:ignored"});
            List flywayCallbacks = DSpaceServicesFactory.getInstance().getServiceManager().getServicesByType(Callback.class);
            flywayConfiguration.callbacks(flywayCallbacks.toArray(new Callback[flywayCallbacks.size()]));
            flywayConfiguration.table(FLYWAY_TABLE);
        }
        catch (SQLException e) {
            log.error("Unable to setup Flyway against DSpace database", (Throwable)e);
        }
        return flywayConfiguration;
    }

    public static synchronized void updateDatabase() throws SQLException {
        DataSource dataSource = DatabaseUtils.getDataSource();
        if (null == dataSource) {
            throw new SQLException("The DataSource is a null reference -- cannot continue.");
        }
        try (Connection connection = dataSource.getConnection();){
            DatabaseUtils.updateDatabase(dataSource, connection);
        }
    }

    protected static synchronized void updateDatabase(DataSource datasource, Connection connection) throws SQLException {
        DatabaseUtils.updateDatabase(datasource, connection, null, false);
    }

    protected static synchronized void updateDatabase(DataSource datasource, Connection connection, String targetVersion, boolean outOfOrder) throws SQLException {
        DatabaseUtils.updateDatabase(datasource, connection, targetVersion, outOfOrder, false);
    }

    protected static synchronized void updateDatabase(DataSource datasource, Connection connection, String targetVersion, boolean outOfOrder, boolean forceMigrate) throws SQLException {
        if (null == datasource) {
            throw new SQLException("The datasource is a null reference -- cannot continue.");
        }
        boolean reindexAfterUpdate = DSpaceServicesFactory.getInstance().getConfigurationService().getBooleanProperty("discovery.autoReindex", true);
        try {
            Flyway flyway;
            FluentConfiguration flywayConfiguration = DatabaseUtils.setupFlyway(datasource);
            flywayConfiguration.outOfOrder(outOfOrder);
            if (!StringUtils.isBlank((CharSequence)targetVersion)) {
                flywayConfiguration.target(targetVersion);
                flywayConfiguration.callbacks(new Callback[0]);
                reindexAfterUpdate = false;
            }
            if (!DatabaseUtils.tableExists(connection, flywayConfiguration.getTable(), true)) {
                String dspaceVersion = DatabaseUtils.determineDBVersion(connection);
                if (dspaceVersion != null) {
                    flywayConfiguration.baselineVersion(dspaceVersion);
                    flywayConfiguration.baselineDescription("Initializing from DSpace " + dspaceVersion + " database schema");
                }
                flyway = flywayConfiguration.load();
                flyway.baseline();
            } else {
                flyway = flywayConfiguration.load();
                FlywayUpgradeUtils.upgradeFlywayTable(flyway, connection);
            }
            MigrationInfo[] pending = flyway.info().pending();
            if (pending != null && pending.length > 0) {
                log.info("Pending DSpace database schema migrations:");
                for (MigrationInfo info : pending) {
                    log.info("\t" + info.getVersion() + " " + info.getDescription() + " " + info.getType() + " " + info.getState());
                }
                flyway.migrate();
                DatabaseUtils.setReindexDiscovery(reindexAfterUpdate);
            } else if (forceMigrate) {
                log.info("DSpace database schema is up to date, but 'force' was specified. Running migrate command to trigger callbacks.");
                flyway.migrate();
            } else {
                log.info("DSpace database schema is up to date");
            }
        }
        catch (FlywayException fe) {
            throw new SQLException("Flyway migration error occurred", fe);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static synchronized void skipMigration(DataSource dataSource, String skipVersion) throws SQLException {
        if (null == dataSource) {
            throw new SQLException("The datasource is a null reference -- cannot continue.");
        }
        try (Connection connection = dataSource.getConnection();){
            FluentConfiguration flywayConfiguration = DatabaseUtils.setupFlyway(dataSource);
            flywayConfiguration.outOfOrder(true);
            Flyway flyway = flywayConfiguration.load();
            boolean foundMigration = false;
            for (MigrationInfo migration : flyway.info().pending()) {
                if (!migration.getVersion().equals((Object)MigrationVersion.fromVersion((String)skipVersion))) continue;
                foundMigration = true;
                System.out.println("Found migration matching version='" + skipVersion + "'. Changing state to 'Success' in order to skip it.");
                PreparedStatement statement = null;
                try {
                    String INSERT_SQL = "INSERT INTO schema_version (installed_rank, version, description, type, script, checksum, installed_by, execution_time, success) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    statement = connection.prepareStatement(INSERT_SQL);
                    statement.setInt(1, DatabaseUtils.getNextFlywayInstalledRank(flyway));
                    statement.setString(2, migration.getVersion().getVersion());
                    statement.setString(3, migration.getDescription());
                    statement.setString(4, migration.getType().toString());
                    statement.setString(5, migration.getScript());
                    statement.setInt(6, migration.getChecksum());
                    statement.setString(7, DatabaseUtils.getDBUserName(connection));
                    statement.setInt(8, 0);
                    statement.setBoolean(9, true);
                    statement.executeUpdate();
                }
                finally {
                    if (statement != null && !statement.isClosed()) {
                        statement.close();
                    }
                }
            }
            if (!foundMigration) {
                System.err.println("Could not find migration to skip! No 'Pending' or 'Ignored' migrations match version='" + skipVersion + "'");
            }
        }
        catch (FlywayException fe) {
            throw new SQLException("Flyway error occurred", fe);
        }
    }

    private static synchronized void cleanDatabase(Flyway flyway, DataSource dataSource) throws SQLException {
        try {
            flyway.clean();
        }
        catch (FlywayException fe) {
            throw new SQLException("Flyway clean error occurred", fe);
        }
    }

    private static String determineDBVersion(Connection connection) throws SQLException {
        if (!DatabaseUtils.tableExists(connection, "Item")) {
            return null;
        }
        if (DatabaseUtils.tableColumnExists(connection, "metadatavalue", "resource_id")) {
            return "5.0.2014.09.26";
        }
        if (DatabaseUtils.tableColumnExists(connection, "requestitem", "request_message")) {
            return "5.0.2014.08.08";
        }
        if (DatabaseUtils.tableExists(connection, "Webapp")) {
            return "4.0";
        }
        if (DatabaseUtils.tableExists(connection, "versionitem")) {
            return "3.0";
        }
        if (DatabaseUtils.tableColumnExists(connection, "bundle2bitstream", "bitstream_order")) {
            return "1.8";
        }
        if (!DatabaseUtils.sequenceExists(connection, "dctyperegistry_seq")) {
            return "1.7";
        }
        if (DatabaseUtils.tableExists(connection, "harvested_collection")) {
            return "1.6";
        }
        if (DatabaseUtils.tableExists(connection, "collection_item_count")) {
            return "1.5";
        }
        if (DatabaseUtils.tableExists(connection, "Group2Group")) {
            return "1.4";
        }
        if (DatabaseUtils.tableExists(connection, "epersongroup2workspaceitem")) {
            return "1.3";
        }
        if (DatabaseUtils.tableExists(connection, "Community2Community")) {
            return "1.2";
        }
        if (DatabaseUtils.tableExists(connection, "Community")) {
            return "1.1";
        }
        throw new SQLException("CANNOT AUTOUPGRADE DSPACE DATABASE, AS IT DOES NOT LOOK TO BE A VALID DSPACE DATABASE.");
    }

    public static boolean tableExists(Connection connection, String tableName) {
        return DatabaseUtils.tableExists(connection, tableName, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean tableExists(Connection connection, String tableName, boolean caseSensitive) {
        boolean exists = false;
        ResultSet results = null;
        try {
            String schema = DatabaseUtils.getSchemaName(connection);
            DatabaseMetaData meta = connection.getMetaData();
            if (!caseSensitive) {
                schema = DatabaseUtils.canonicalize(connection, schema);
                tableName = DatabaseUtils.canonicalize(connection, tableName);
            }
            if ((results = meta.getTables(null, schema, tableName, null)) != null && results.next()) {
                exists = true;
            }
        }
        catch (SQLException e) {
            log.error("Error attempting to determine if table " + tableName + " exists", (Throwable)e);
        }
        finally {
            try {
                if (results != null && !results.isClosed()) {
                    results.close();
                }
            }
            catch (SQLException sQLException) {}
        }
        return exists;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean tableColumnExists(Connection connection, String tableName, String columnName) {
        boolean exists = false;
        ResultSet results = null;
        try {
            String schema = DatabaseUtils.getSchemaName(connection);
            schema = DatabaseUtils.canonicalize(connection, schema);
            tableName = DatabaseUtils.canonicalize(connection, tableName);
            columnName = DatabaseUtils.canonicalize(connection, columnName);
            DatabaseMetaData meta = connection.getMetaData();
            results = meta.getColumns(null, schema, tableName, columnName);
            if (results != null && results.next()) {
                exists = true;
            }
        }
        catch (SQLException e) {
            log.error("Error attempting to determine if column " + columnName + " exists", (Throwable)e);
        }
        finally {
            try {
                if (results != null && !results.isClosed()) {
                    results.close();
                }
            }
            catch (SQLException sQLException) {}
        }
        return exists;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean sequenceExists(Connection connection, String sequenceName) {
        boolean exists = false;
        Statement statement = null;
        ResultSet results = null;
        boolean schemaFilter = false;
        try {
            String schema = DatabaseUtils.getSchemaName(connection);
            schema = DatabaseUtils.canonicalize(connection, schema);
            sequenceName = DatabaseUtils.canonicalize(connection, sequenceName);
            String dbtype = DatabaseUtils.getDbType(connection);
            String sequenceSQL = null;
            switch (dbtype) {
                case "postgres": {
                    if (schema == null) {
                        schema = "public";
                    }
                    sequenceSQL = "SELECT COUNT(1) FROM pg_class, pg_namespace WHERE pg_class.relnamespace=pg_namespace.oid AND pg_class.relkind='S' AND pg_class.relname=? AND pg_namespace.nspname=?";
                    schemaFilter = true;
                    break;
                }
                case "h2": {
                    sequenceSQL = "SELECT COUNT(1) FROM INFORMATION_SCHEMA.SEQUENCES WHERE SEQUENCE_NAME = ?";
                    break;
                }
                default: {
                    throw new SQLException("DBMS " + dbtype + " is unsupported.");
                }
            }
            if (sequenceSQL != null) {
                statement = connection.prepareStatement(sequenceSQL);
                statement.setString(1, sequenceName);
                if (schemaFilter) {
                    statement.setString(2, schema);
                }
                if ((results = statement.executeQuery()) != null && results.next() && results.getInt(1) > 0) {
                    exists = true;
                }
            }
        }
        catch (SQLException e) {
            log.error("Error attempting to determine if sequence " + sequenceName + " exists", (Throwable)e);
        }
        finally {
            try {
                if (statement != null && !statement.isClosed()) {
                    statement.close();
                }
                if (results != null && !results.isClosed()) {
                    results.close();
                }
            }
            catch (SQLException sQLException) {}
        }
        return exists;
    }

    public static void executeSql(Connection connection, String sqlToExecute) throws SQLException {
        try {
            new JdbcTemplate((DataSource)new SingleConnectionDataSource(connection, true)).execute(sqlToExecute);
        }
        catch (DataAccessException dae) {
            throw new SQLException("Flyway executeSql() error occurred", dae);
        }
    }

    public static String getSchemaName(Connection connection) throws SQLException {
        String schema = null;
        try {
            schema = connection.getSchema();
        }
        catch (AbstractMethodError | Exception throwable) {
            // empty catch block
        }
        if (StringUtils.isBlank((CharSequence)schema)) {
            schema = DatabaseUtils.canonicalize(connection, DSpaceServicesFactory.getInstance().getConfigurationService().getProperty("db.schema"));
        }
        if (StringUtils.isBlank((CharSequence)schema)) {
            String dbType = DatabaseUtils.getDbType(connection);
            schema = dbType.equals(DBMS_POSTGRES) ? "public" : null;
        }
        return schema;
    }

    public static String getDBUserName(Connection connection) throws SQLException {
        String username = null;
        try {
            username = connection.getMetaData().getUserName();
        }
        catch (AbstractMethodError | Exception throwable) {
            // empty catch block
        }
        if (StringUtils.isBlank((CharSequence)username)) {
            username = DatabaseUtils.canonicalize(connection, DSpaceServicesFactory.getInstance().getConfigurationService().getProperty("db.username"));
        }
        return username;
    }

    public static String canonicalize(Connection connection, String dbIdentifier) throws SQLException {
        if (dbIdentifier == null) {
            return null;
        }
        DatabaseMetaData meta = connection.getMetaData();
        if (meta.storesLowerCaseIdentifiers()) {
            return StringUtils.lowerCase((String)dbIdentifier);
        }
        if (meta.storesUpperCaseIdentifiers()) {
            return StringUtils.upperCase((String)dbIdentifier);
        }
        return dbIdentifier;
    }

    public static synchronized void setReindexDiscovery(boolean reindex) {
        boolean deleted;
        File reindexFlag = new File(reindexDiscoveryFilePath);
        if (reindex) {
            try {
                if (!reindexFlag.exists()) {
                    reindexFlag.createNewFile();
                    reindexFlag.setWritable(true, false);
                }
            }
            catch (IOException io) {
                log.error("Unable to create Discovery reindex flag file " + reindexFlag.getAbsolutePath() + ". You may need to reindex manually.", (Throwable)io);
            }
        } else if (reindexFlag.exists() && !(deleted = reindexFlag.delete())) {
            log.error("Unable to delete Discovery reindex flag file " + reindexFlag.getAbsolutePath() + ". You may need to delete it manually.");
        }
    }

    public static synchronized boolean getReindexDiscovery() {
        boolean autoReindex = DSpaceServicesFactory.getInstance().getConfigurationService().getBooleanProperty("discovery.autoReindex", true);
        return autoReindex && new File(reindexDiscoveryFilePath).exists();
    }

    public static synchronized void checkReindexDiscovery(IndexingService indexer) {
        if (DatabaseUtils.getReindexDiscovery()) {
            ReindexerThread go = new ReindexerThread(indexer);
            go.start();
        }
    }

    public static String getDbType(Connection connection) throws SQLException {
        DatabaseMetaData meta = connection.getMetaData();
        String prodName = meta.getDatabaseProductName();
        String dbms_lc = prodName.toLowerCase(Locale.ROOT);
        if (dbms_lc.contains("postgresql")) {
            return DBMS_POSTGRES;
        }
        if (dbms_lc.contains(DBMS_H2)) {
            return DBMS_H2;
        }
        return dbms_lc;
    }

    protected static DataSource getDataSource() {
        DataSource dataSource = (DataSource)DSpaceServicesFactory.getInstance().getServiceManager().getServiceByName("dataSource", DataSource.class);
        if (null == dataSource) {
            log.error("The service manager could not find the DataSource.");
        }
        return dataSource;
    }

    public static String getCurrentFlywayState(Connection connection) throws SQLException {
        PreparedStatement statement = connection.prepareStatement("SELECT \"version\" FROM \"schema_version\" ORDER BY \"version\" desc");
        ResultSet resultSet = statement.executeQuery();
        resultSet.next();
        return resultSet.getString("version");
    }

    public static Double getCurrentFlywayDSpaceState(Connection connection) throws SQLException {
        String flywayState = DatabaseUtils.getCurrentFlywayState(connection);
        Matcher matcher = Pattern.compile("^([0-9]*\\.[0-9]*)(\\.)?.*").matcher(flywayState);
        if (matcher.matches()) {
            return Double.parseDouble(matcher.group(1));
        }
        return null;
    }

    private static int getNextFlywayInstalledRank(Flyway flyway) throws FlywayException {
        Object[] appliedMigrations = flyway.info().applied();
        if (ArrayUtils.isEmpty((Object[])appliedMigrations)) {
            throw new FlywayException("Cannot determine next 'installed_rank' as no applied migrations exist");
        }
        return appliedMigrations[appliedMigrations.length - 1].getInstalledRank() + 1;
    }

    private static class ReindexerThread
    extends Thread {
        private final IndexingService indexer;

        ReindexerThread(IndexingService is) {
            this.indexer = is;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            IndexingService indexingService = this.indexer;
            synchronized (indexingService) {
                if (DatabaseUtils.getReindexDiscovery()) {
                    Context context = null;
                    try {
                        context = new Context();
                        context.setMode(Context.Mode.READ_ONLY);
                        context.turnOffAuthorisationSystem();
                        log.info("Post database migration, reindexing all content in Discovery search and browse engine");
                        this.indexer.deleteIndex();
                        this.indexer.createIndex(context);
                        this.indexer.buildSpellCheck();
                        log.info("Reindexing is complete");
                    }
                    catch (SearchServiceException sse) {
                        log.warn("Unable to reindex content in Discovery search and browse engine. You may need to reindex manually.", (Throwable)sse);
                    }
                    catch (IOException | SQLException e) {
                        log.error("Error attempting to reindex all contents for search/browse", (Throwable)e);
                    }
                    finally {
                        DatabaseUtils.setReindexDiscovery(false);
                        if (context != null && context.isValid()) {
                            context.abort();
                        }
                    }
                }
            }
        }
    }
}

