/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.File;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.db.RocksDatabaseException;
import org.apache.hadoop.hdds.utils.db.StringCodec;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.managed.ManagedCheckpoint;
import org.apache.hadoop.hdds.utils.db.managed.ManagedColumnFamilyOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedCompactRangeOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedDBOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedFlushOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedIngestExternalFileOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedReadOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksDB;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksObjectUtils;
import org.apache.hadoop.hdds.utils.db.managed.ManagedTransactionLogIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteBatch;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.apache.ozone.rocksdiff.RocksDiffUtils;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.UncheckedAutoCloseable;
import org.rocksdb.Checkpoint;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.CompactRangeOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.FlushOptions;
import org.rocksdb.Holder;
import org.rocksdb.IngestExternalFileOptions;
import org.rocksdb.KeyMayExist;
import org.rocksdb.LiveFileMetaData;
import org.rocksdb.Options;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.TransactionLogIterator;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RocksDatabase
implements Closeable {
    static final Logger LOG = LoggerFactory.getLogger(RocksDatabase.class);
    public static final String ESTIMATE_NUM_KEYS = "rocksdb.estimate-num-keys";
    private static final ManagedReadOptions DEFAULT_READ_OPTION;
    private final String name;
    private final Throwable creationStackTrace = new Throwable("Object creation stack trace");
    private final ManagedRocksDB db;
    private final ManagedDBOptions dbOptions;
    private final ManagedWriteOptions writeOptions;
    private final List<ColumnFamilyDescriptor> descriptors;
    private final Map<String, ColumnFamily> columnFamilies;
    private final Supplier<Map<Integer, String>> columnFamilyNames;
    private final AtomicBoolean isClosed = new AtomicBoolean();
    private final AtomicLong counter = new AtomicLong();

    static String bytes2String(byte[] bytes) {
        return StringCodec.get().fromPersistedFormat(bytes);
    }

    static String bytes2String(ByteBuffer bytes) {
        return StringCodec.get().decode(bytes);
    }

    static RocksDatabaseException toRocksDatabaseException(Object name, String op, RocksDBException e) {
        return new RocksDatabaseException(name + ": Failed to " + op, (Exception)((Object)e));
    }

    private static List<TableConfig> getExtraColumnFamilies(File file, Set<TableConfig> families) throws RocksDBException {
        Set existingFamilyNames = families.stream().map(TableConfig::getName).collect(Collectors.toSet());
        List<TableConfig> columnFamilies = RocksDatabase.listColumnFamiliesEmptyOptions(file.getAbsolutePath()).stream().map(TableConfig::toName).filter(familyName -> !existingFamilyNames.contains(familyName)).map(familyName -> TableConfig.newTableConfig(file.toPath(), familyName)).collect(Collectors.toList());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found column families in DB {}: {}", (Object)file, columnFamilies);
        }
        return columnFamilies;
    }

    public static List<byte[]> listColumnFamiliesEmptyOptions(String path) throws RocksDBException {
        try (ManagedOptions emptyOptions = new ManagedOptions();){
            List list = RocksDB.listColumnFamilies((Options)emptyOptions, (String)path);
            return list;
        }
    }

    static RocksDatabase open(File dbFile, ManagedDBOptions dbOptions, ManagedWriteOptions writeOptions, Set<TableConfig> families, boolean readOnly) throws RocksDatabaseException {
        List<ColumnFamilyDescriptor> descriptors = null;
        ManagedRocksDB db = null;
        HashMap<String, ColumnFamily> columnFamilies = new HashMap<String, ColumnFamily>();
        try {
            List<TableConfig> extra = RocksDatabase.getExtraColumnFamilies(dbFile, families);
            descriptors = Stream.concat(families.stream(), extra.stream()).map(TableConfig::getDescriptor).collect(Collectors.toList());
            ArrayList<ColumnFamilyHandle> handles = new ArrayList<ColumnFamilyHandle>();
            db = readOnly ? ManagedRocksDB.openReadOnly((ManagedDBOptions)dbOptions, (String)dbFile.getAbsolutePath(), descriptors, handles) : ManagedRocksDB.open((DBOptions)dbOptions, (String)dbFile.getAbsolutePath(), descriptors, handles);
            return new RocksDatabase(dbFile, db, dbOptions, writeOptions, descriptors, handles);
        }
        catch (RocksDBException e) {
            RocksDatabase.close(columnFamilies, db, descriptors, writeOptions, dbOptions);
            throw RocksDatabase.toRocksDatabaseException(RocksDatabase.class, "open " + dbFile, e);
        }
    }

    private static void close(final ColumnFamilyDescriptor d) {
        ManagedColumnFamilyOptions options = (ManagedColumnFamilyOptions)d.getOptions();
        if (options.isReused()) {
            return;
        }
        RocksDatabase.runWithTryCatch(() -> ManagedColumnFamilyOptions.closeDeeply((ColumnFamilyOptions)options), new Object(){

            public String toString() {
                return d.getClass() + ":" + RocksDatabase.bytes2String(d.getName());
            }
        });
    }

    private static void close(Map<String, ColumnFamily> columnFamilies, ManagedRocksDB db, List<ColumnFamilyDescriptor> descriptors, ManagedWriteOptions writeOptions, ManagedDBOptions dbOptions) {
        if (columnFamilies != null) {
            for (ColumnFamily f : columnFamilies.values()) {
                RocksDatabase.runWithTryCatch(() -> f.getHandle().close(), f);
            }
        }
        if (db != null) {
            RocksDatabase.runWithTryCatch(() -> db.close(), "db");
        }
        if (descriptors != null) {
            descriptors.forEach(RocksDatabase::close);
        }
        if (writeOptions != null) {
            RocksDatabase.runWithTryCatch(() -> ((ManagedWriteOptions)writeOptions).close(), "writeOptions");
        }
        if (dbOptions != null) {
            RocksDatabase.runWithTryCatch(() -> ((ManagedDBOptions)dbOptions).close(), "dbOptions");
        }
    }

    private static void runWithTryCatch(Runnable runnable, Object name) {
        try {
            runnable.run();
        }
        catch (Throwable t) {
            LOG.error("Failed to close " + name, t);
        }
    }

    public boolean isClosed() {
        return this.isClosed.get();
    }

    private RocksDatabase(File dbFile, ManagedRocksDB db, ManagedDBOptions dbOptions, ManagedWriteOptions writeOptions, List<ColumnFamilyDescriptor> descriptors, List<ColumnFamilyHandle> handles) throws RocksDBException {
        this.name = this.getClass().getSimpleName() + "[" + dbFile + "]";
        this.db = db;
        this.dbOptions = dbOptions;
        this.writeOptions = writeOptions;
        this.descriptors = descriptors;
        this.columnFamilies = this.toColumnFamilyMap(handles);
        this.columnFamilyNames = MemoizedSupplier.valueOf(() -> RocksDatabase.toColumnFamilyNameMap(this.columnFamilies.values()));
    }

    private Map<String, ColumnFamily> toColumnFamilyMap(List<ColumnFamilyHandle> handles) throws RocksDBException {
        HashMap<String, ColumnFamily> map = new HashMap<String, ColumnFamily>();
        for (ColumnFamilyHandle h : handles) {
            ColumnFamily f = new ColumnFamily(h);
            map.put(f.getName(), f);
        }
        return Collections.unmodifiableMap(map);
    }

    private static Map<Integer, String> toColumnFamilyNameMap(Collection<ColumnFamily> families) {
        return Collections.unmodifiableMap(families.stream().collect(Collectors.toMap(f -> f.getHandle().getID(), ColumnFamily::getName)));
    }

    Map<Integer, String> getColumnFamilyNames() {
        return this.columnFamilyNames.get();
    }

    @Override
    public void close() {
        this.close(true);
    }

    private void close(boolean isSync) {
        if (this.isClosed.compareAndSet(false, true)) {
            ((RocksDB)this.db.get()).cancelAllBackgroundWork(true);
            this.dbOptions.listeners().forEach(listener -> listener.close());
            if (isSync) {
                this.waitAndClose();
                return;
            }
            new Thread(() -> this.waitAndClose(), "DBCloser-" + this.name).start();
        }
    }

    private void waitAndClose() {
        while (!this.counter.compareAndSet(0L, Long.MIN_VALUE)) {
            try {
                Thread.currentThread();
                Thread.sleep(1L);
            }
            catch (InterruptedException e) {
                RocksDatabase.close(this.columnFamilies, this.db, this.descriptors, this.writeOptions, this.dbOptions);
                Thread.currentThread().interrupt();
                return;
            }
        }
        RocksDatabase.close(this.columnFamilies, this.db, this.descriptors, this.writeOptions, this.dbOptions);
    }

    private void closeOnError(RocksDBException e) {
        if (this.shouldClose(e)) {
            this.close(false);
        }
    }

    private boolean shouldClose(RocksDBException e) {
        switch (e.getStatus().getCode()) {
            case Corruption: 
            case IOError: {
                return true;
            }
        }
        return false;
    }

    private UncheckedAutoCloseable acquire() throws RocksDatabaseException {
        if (this.isClosed()) {
            throw new RocksDatabaseException("Rocks Database is closed");
        }
        if (this.counter.getAndIncrement() < 0L) {
            this.counter.getAndDecrement();
            throw new RocksDatabaseException("Rocks Database is closed");
        }
        return this.counter::getAndDecrement;
    }

    public void ingestExternalFile(ColumnFamily family, List<String> files, ManagedIngestExternalFileOptions ingestOptions) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).ingestExternalFile(family.getHandle(), files, (IngestExternalFileOptions)ingestOptions);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            String msg = "Failed to ingest external files " + files.stream().collect(Collectors.joining(", ")) + " of " + family.getName();
            throw RocksDatabase.toRocksDatabaseException(this, msg, e);
        }
    }

    public void put(ColumnFamily family, byte[] key, byte[] value) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).put(family.getHandle(), (WriteOptions)this.writeOptions, key, value);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "put " + RocksDatabase.bytes2String(key), e);
        }
    }

    public void put(ColumnFamily family, ByteBuffer key, ByteBuffer value) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).put(family.getHandle(), (WriteOptions)this.writeOptions, key, value);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "put " + RocksDatabase.bytes2String(key), e);
        }
    }

    public void flush() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();
             ManagedFlushOptions options = new ManagedFlushOptions();){
            options.setWaitForFlush(true);
            ((RocksDB)this.db.get()).flush((FlushOptions)options);
            for (ColumnFamily columnFamily : this.getExtraColumnFamilies()) {
                ((RocksDB)this.db.get()).flush((FlushOptions)options, columnFamily.handle);
            }
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "flush", e);
        }
    }

    public void flush(String cfName) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ColumnFamilyHandle handle = this.getColumnFamilyHandle(cfName);
            try (ManagedFlushOptions options = new ManagedFlushOptions();){
                options.setWaitForFlush(true);
                if (handle != null) {
                    ((RocksDB)this.db.get()).flush((FlushOptions)options, handle);
                } else {
                    LOG.error("Provided column family doesn't exist. Calling flush on null columnFamily");
                    this.flush();
                }
            }
            catch (RocksDBException e) {
                this.closeOnError(e);
                throw RocksDatabase.toRocksDatabaseException(this, "flush", e);
            }
        }
    }

    public void flushWal(boolean sync) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).flushWal(sync);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "flushWal with sync=" + sync, e);
        }
    }

    public void compactRange() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).compactRange();
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "compactRange", e);
        }
    }

    public void compactRangeDefault(ManagedCompactRangeOptions options) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).compactRange(null, null, null, (CompactRangeOptions)options);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "compactRange", e);
        }
    }

    public void compactDB(ManagedCompactRangeOptions options) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            this.compactRangeDefault(options);
            for (ColumnFamily columnFamily : this.getExtraColumnFamilies()) {
                this.compactRange(columnFamily, null, null, options);
            }
        }
    }

    public int getLiveFilesMetaDataSize() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            int n = ((RocksDB)this.db.get()).getLiveFilesMetaData().size();
            return n;
        }
    }

    public void compactRange(String cfName) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ColumnFamilyHandle handle = this.getColumnFamilyHandle(cfName);
            try {
                if (handle != null) {
                    ((RocksDB)this.db.get()).compactRange(handle);
                } else {
                    LOG.error("Provided column family doesn't exist. Calling compactRange on null columnFamily");
                    ((RocksDB)this.db.get()).compactRange();
                }
            }
            catch (RocksDBException e) {
                this.closeOnError(e);
                throw RocksDatabase.toRocksDatabaseException(this, "compactRange", e);
            }
        }
    }

    private ColumnFamilyHandle getColumnFamilyHandle(String columnFamilyName) {
        ColumnFamily columnFamily = this.getColumnFamily(columnFamilyName);
        return columnFamily != null ? columnFamily.getHandle() : null;
    }

    public void compactRange(ColumnFamily family, byte[] begin, byte[] end, ManagedCompactRangeOptions options) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).compactRange(family.getHandle(), begin, end, (CompactRangeOptions)options);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "compactRange", e);
        }
    }

    public List<LiveFileMetaData> getLiveFilesMetaData() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            List list = ((RocksDB)this.db.get()).getLiveFilesMetaData();
            return list;
        }
    }

    RocksCheckpoint createCheckpoint() {
        return new RocksCheckpoint();
    }

    Supplier<byte[]> keyMayExist(ColumnFamily family, byte[] key) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            Holder out = new Holder();
            Supplier<byte[]> supplier = ((RocksDB)this.db.get()).keyMayExist(family.getHandle(), key, out) ? () -> ((Holder)out).getValue() : null;
            return supplier;
        }
    }

    Supplier<Integer> keyMayExist(ColumnFamily family, ByteBuffer key, ByteBuffer out) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            KeyMayExist result = ((RocksDB)this.db.get()).keyMayExist(family.getHandle(), key, out);
            switch (result.exists) {
                case kNotExist: {
                    Supplier<Integer> supplier = null;
                    return supplier;
                }
                case kExistsWithValue: {
                    Supplier<Integer> supplier = () -> result.valueLength;
                    return supplier;
                }
                case kExistsWithoutValue: {
                    Supplier<Integer> supplier = () -> null;
                    return supplier;
                }
            }
            throw new IllegalStateException("Unexpected KeyMayExistEnum case " + result.exists);
        }
    }

    public ColumnFamily getColumnFamily(String key) {
        return this.columnFamilies.get(key);
    }

    public Collection<ColumnFamily> getExtraColumnFamilies() {
        return Collections.unmodifiableCollection(this.columnFamilies.values());
    }

    byte[] get(ColumnFamily family, byte[] key) throws RocksDatabaseException {
        byte[] byArray;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                byArray = ((RocksDB)this.db.get()).get(family.getHandle(), key);
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    String message = "get " + RocksDatabase.bytes2String(key) + " from " + family;
                    throw RocksDatabase.toRocksDatabaseException(this, message, e);
                }
            }
            ignored.close();
        }
        return byArray;
    }

    Integer get(ColumnFamily family, ByteBuffer key, ByteBuffer outValue) throws RocksDatabaseException {
        Integer n;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                int size = ((RocksDB)this.db.get()).get(family.getHandle(), (ReadOptions)DEFAULT_READ_OPTION, key, outValue);
                LOG.trace("get: size={}, remaining={}", (Object)size, (Object)outValue.asReadOnlyBuffer().remaining());
                Integer n2 = n = size == -1 ? null : Integer.valueOf(size);
                if (ignored == null) break block8;
            }
            catch (Throwable size) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable) {
                            size.addSuppressed(throwable);
                        }
                    }
                    throw size;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    String message = "get " + RocksDatabase.bytes2String(key) + " from " + family;
                    throw RocksDatabase.toRocksDatabaseException(this, message, e);
                }
            }
            ignored.close();
        }
        return n;
    }

    public long estimateNumKeys() throws RocksDatabaseException {
        return this.getLongProperty(ESTIMATE_NUM_KEYS);
    }

    public long estimateNumKeys(ColumnFamily family) throws RocksDatabaseException {
        return this.getLongProperty(family, ESTIMATE_NUM_KEYS);
    }

    private long getLongProperty(String key) throws RocksDatabaseException {
        long l;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                l = ((RocksDB)this.db.get()).getLongProperty(key);
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    throw RocksDatabase.toRocksDatabaseException(this, "getLongProperty " + key, e);
                }
            }
            ignored.close();
        }
        return l;
    }

    private long getLongProperty(ColumnFamily family, String key) throws RocksDatabaseException {
        long l;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                l = ((RocksDB)this.db.get()).getLongProperty(family.getHandle(), key);
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    String message = "getLongProperty " + key + " from " + family;
                    throw RocksDatabase.toRocksDatabaseException(this, message, e);
                }
            }
            ignored.close();
        }
        return l;
    }

    public String getProperty(String key) throws RocksDatabaseException {
        String string;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                string = ((RocksDB)this.db.get()).getProperty(key);
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    throw RocksDatabase.toRocksDatabaseException(this, "getProperty " + key, e);
                }
            }
            ignored.close();
        }
        return string;
    }

    public String getProperty(ColumnFamily family, String key) throws RocksDatabaseException {
        String string;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                string = ((RocksDB)this.db.get()).getProperty(family.getHandle(), key);
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    throw RocksDatabase.toRocksDatabaseException(this, "getProperty " + key + " from " + family, e);
                }
            }
            ignored.close();
        }
        return string;
    }

    public ManagedTransactionLogIterator getUpdatesSince(long sequenceNumber) throws RocksDatabaseException {
        ManagedTransactionLogIterator managedTransactionLogIterator;
        block8: {
            UncheckedAutoCloseable ignored = this.acquire();
            try {
                managedTransactionLogIterator = ManagedTransactionLogIterator.managed((TransactionLogIterator)((RocksDB)this.db.get()).getUpdatesSince(sequenceNumber));
                if (ignored == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (RocksDBException e) {
                    this.closeOnError(e);
                    throw RocksDatabase.toRocksDatabaseException(this, "getUpdatesSince " + sequenceNumber, e);
                }
            }
            ignored.close();
        }
        return managedTransactionLogIterator;
    }

    public long getLatestSequenceNumber() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            long l = ((RocksDB)this.db.get()).getLatestSequenceNumber();
            return l;
        }
    }

    public ManagedRocksIterator newIterator(ColumnFamily family) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ManagedRocksIterator managedRocksIterator = ManagedRocksIterator.managed((RocksIterator)((RocksDB)this.db.get()).newIterator(family.getHandle()));
            return managedRocksIterator;
        }
    }

    public ManagedRocksIterator newIterator(ColumnFamily family, boolean fillCache) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ManagedRocksIterator managedRocksIterator;
            try (ManagedReadOptions readOptions = new ManagedReadOptions();){
                readOptions.setFillCache(fillCache);
                managedRocksIterator = ManagedRocksIterator.managed((RocksIterator)((RocksDB)this.db.get()).newIterator(family.getHandle(), (ReadOptions)readOptions));
            }
            return managedRocksIterator;
        }
    }

    public void batchWrite(ManagedWriteBatch writeBatch, ManagedWriteOptions options) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).write((WriteOptions)options, (WriteBatch)writeBatch);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            throw RocksDatabase.toRocksDatabaseException(this, "batchWrite", e);
        }
    }

    public void batchWrite(ManagedWriteBatch writeBatch) throws RocksDatabaseException {
        this.batchWrite(writeBatch, this.writeOptions);
    }

    public void delete(ColumnFamily family, byte[] key) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).delete(family.getHandle(), key);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            String message = "delete " + RocksDatabase.bytes2String(key) + " from " + family;
            throw RocksDatabase.toRocksDatabaseException(this, message, e);
        }
    }

    public void delete(ColumnFamily family, ByteBuffer key) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).delete(family.getHandle(), (WriteOptions)this.writeOptions, key);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            String message = "delete " + RocksDatabase.bytes2String(key) + " from " + family;
            throw RocksDatabase.toRocksDatabaseException(this, message, e);
        }
    }

    public void deleteRange(ColumnFamily family, byte[] beginKey, byte[] endKey) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            ((RocksDB)this.db.get()).deleteRange(family.getHandle(), beginKey, endKey);
        }
        catch (RocksDBException e) {
            this.closeOnError(e);
            String message = "delete range " + RocksDatabase.bytes2String(beginKey) + " to " + RocksDatabase.bytes2String(endKey) + " from " + family;
            throw RocksDatabase.toRocksDatabaseException(this, message, e);
        }
    }

    public String toString() {
        return this.name;
    }

    @VisibleForTesting
    public List<LiveFileMetaData> getSstFileList() throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            List list = ((RocksDB)this.db.get()).getLiveFilesMetaData();
            return list;
        }
    }

    private int getLastLevel() throws RocksDatabaseException {
        return this.getSstFileList().stream().max(Comparator.comparing(LiveFileMetaData::level)).get().level();
    }

    public void deleteFilesNotMatchingPrefix(Map<String, String> prefixPairs) throws RocksDatabaseException {
        try (UncheckedAutoCloseable ignored = this.acquire();){
            for (LiveFileMetaData liveFileMetaData : this.getSstFileList()) {
                String lastDbKey;
                String firstDbKey;
                String prefixForColumnFamily;
                boolean isKeyWithPrefixPresent;
                String sstFileColumnFamily = StringUtils.bytes2String((byte[])liveFileMetaData.columnFamilyName());
                int lastLevel = this.getLastLevel();
                if (!prefixPairs.containsKey(sstFileColumnFamily) || liveFileMetaData.level() != lastLevel || lastLevel == 0 || (isKeyWithPrefixPresent = RocksDiffUtils.isKeyWithPrefixPresent((String)(prefixForColumnFamily = prefixPairs.get(sstFileColumnFamily)), (String)(firstDbKey = StringUtils.bytes2String((byte[])liveFileMetaData.smallestKey())), (String)(lastDbKey = StringUtils.bytes2String((byte[])liveFileMetaData.largestKey()))))) continue;
                LOG.info("Deleting sst file: {} with start key: {} and end key: {} corresponding to column family {} from db: {}. Prefix for the column family: {}.", new Object[]{liveFileMetaData.fileName(), firstDbKey, lastDbKey, StringUtils.bytes2String((byte[])liveFileMetaData.columnFamilyName()), ((RocksDB)this.db.get()).getName(), prefixForColumnFamily});
                this.db.deleteFile(liveFileMetaData);
            }
        }
    }

    protected void finalize() throws Throwable {
        if (!this.isClosed()) {
            LOG.warn("RocksDatabase {} is not closed properly.", (Object)this.name, (Object)this.creationStackTrace);
        }
        super.finalize();
    }

    public ManagedRocksDB getManagedRocksDb() {
        return this.db;
    }

    static {
        ManagedRocksObjectUtils.loadRocksDBLibrary();
        DEFAULT_READ_OPTION = new ManagedReadOptions();
    }

    public final class ColumnFamily {
        private final byte[] nameBytes;
        private final String name;
        private final ColumnFamilyHandle handle;

        private ColumnFamily(ColumnFamilyHandle handle) throws RocksDBException {
            this.nameBytes = handle.getName();
            this.name = RocksDatabase.bytes2String(this.nameBytes);
            this.handle = handle;
            LOG.debug("new ColumnFamily for {}", (Object)this.name);
        }

        public String getName() {
            return this.name;
        }

        public String getName(StringCodec codec) {
            return codec.fromPersistedFormat(this.nameBytes);
        }

        @VisibleForTesting
        public ColumnFamilyHandle getHandle() {
            return this.handle;
        }

        public void batchDelete(ManagedWriteBatch writeBatch, byte[] key) throws RocksDatabaseException {
            try (UncheckedAutoCloseable ignored = this.acquire();){
                writeBatch.delete(this.getHandle(), key);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toRocksDatabaseException(this, "batchDelete key " + RocksDatabase.bytes2String(key), e);
            }
        }

        public void batchDeleteRange(ManagedWriteBatch writeBatch, byte[] beginKey, byte[] endKey) throws RocksDatabaseException {
            try (UncheckedAutoCloseable ignored = this.acquire();){
                writeBatch.deleteRange(this.getHandle(), beginKey, endKey);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toRocksDatabaseException(this, "batchDeleteRange key " + RocksDatabase.bytes2String(beginKey) + " - " + RocksDatabase.bytes2String(endKey), e);
            }
        }

        public void batchPut(ManagedWriteBatch writeBatch, byte[] key, byte[] value) throws RocksDatabaseException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("batchPut array key {}", (Object)RocksDatabase.bytes2String(key));
                LOG.debug("batchPut array value {}", (Object)RocksDatabase.bytes2String(value));
            }
            try (UncheckedAutoCloseable ignored = this.acquire();){
                writeBatch.put(this.getHandle(), key, value);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toRocksDatabaseException(this, "batchPut key " + RocksDatabase.bytes2String(key), e);
            }
        }

        public void batchPut(ManagedWriteBatch writeBatch, ByteBuffer key, ByteBuffer value) throws RocksDatabaseException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("batchPut buffer key {}", (Object)RocksDatabase.bytes2String(key.duplicate()));
                LOG.debug("batchPut buffer value size {}", (Object)value.remaining());
            }
            try (UncheckedAutoCloseable ignored = this.acquire();){
                writeBatch.put(this.getHandle(), key.duplicate(), value);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toRocksDatabaseException(this, "batchPut ByteBuffer key " + RocksDatabase.bytes2String(key), e);
            }
        }

        private UncheckedAutoCloseable acquire() throws RocksDatabaseException {
            if (RocksDatabase.this.isClosed.get()) {
                throw new RocksDatabaseException("Rocks Database is closed");
            }
            if (RocksDatabase.this.counter.getAndIncrement() < 0L) {
                RocksDatabase.this.counter.getAndDecrement();
                throw new RocksDatabaseException("Rocks Database is closed");
            }
            return RocksDatabase.this.counter::getAndDecrement;
        }

        public String toString() {
            return "ColumnFamily-" + this.getName();
        }
    }

    final class RocksCheckpoint
    implements Closeable {
        private final ManagedCheckpoint checkpoint;

        private RocksCheckpoint() {
            this.checkpoint = ManagedCheckpoint.create((ManagedRocksDB)RocksDatabase.this.db);
        }

        public void createCheckpoint(Path path) throws RocksDatabaseException {
            try (UncheckedAutoCloseable ignored = RocksDatabase.this.acquire();){
                ((Checkpoint)this.checkpoint.get()).createCheckpoint(path.toString());
            }
            catch (RocksDBException e) {
                RocksDatabase.this.closeOnError(e);
                throw RocksDatabase.toRocksDatabaseException(this, "createCheckpoint " + path, e);
            }
        }

        public long getLatestSequenceNumber() throws RocksDatabaseException {
            return RocksDatabase.this.getLatestSequenceNumber();
        }

        @Override
        public void close() throws RocksDatabaseException {
            this.checkpoint.close();
        }
    }
}

