/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.github;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.PrivateKey;
import java.time.Instant;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.nifi.github.GitHubAuthenticationType;
import org.apache.nifi.github.StandardPrivateKeyReader;
import org.apache.nifi.registry.flow.FlowRegistryException;
import org.apache.nifi.registry.flow.git.client.GitCommit;
import org.apache.nifi.registry.flow.git.client.GitCreateContentRequest;
import org.apache.nifi.registry.flow.git.client.GitRepositoryClient;
import org.kohsuke.github.GHCommit;
import org.kohsuke.github.GHContent;
import org.kohsuke.github.GHContentUpdateResponse;
import org.kohsuke.github.GHMyself;
import org.kohsuke.github.GHPermissionType;
import org.kohsuke.github.GHRef;
import org.kohsuke.github.GHRepository;
import org.kohsuke.github.GHUser;
import org.kohsuke.github.GitHub;
import org.kohsuke.github.GitHubBuilder;
import org.kohsuke.github.authorization.AppInstallationAuthorizationProvider;
import org.kohsuke.github.authorization.AuthorizationProvider;
import org.kohsuke.github.extras.authorization.JWTTokenProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GitHubRepositoryClient
implements GitRepositoryClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(GitHubRepositoryClient.class);
    private static final String REPOSITORY_CONTENTS_PERMISSION = "contents";
    private static final String WRITE_ACCESS = "write";
    private static final String BRANCH_REF_PATTERN = "refs/heads/%s";
    private static final int COMMIT_PAGE_SIZE = 50;
    private final String repoOwner;
    private final String repoName;
    private final String repoPath;
    private final GitHub gitHub;
    private final GHRepository repository;
    private final GitHubAuthenticationType authenticationType;
    private final boolean canRead;
    private final boolean canWrite;

    private GitHubRepositoryClient(Builder builder) throws IOException, FlowRegistryException {
        String apiUrl = Objects.requireNonNull(builder.apiUrl, "API URL is required");
        GitHubBuilder gitHubBuilder = new GitHubBuilder().withEndpoint(apiUrl);
        this.repoPath = builder.repoPath;
        this.repoOwner = Objects.requireNonNull(builder.repoOwner, "Repository Owner is required");
        this.repoName = Objects.requireNonNull(builder.repoName, "Repository Name is required");
        this.authenticationType = Objects.requireNonNull(builder.authenticationType, "Authentication Type is required");
        LinkedHashMap<String, String> appPermissions = new LinkedHashMap<String, String>();
        switch (this.authenticationType) {
            case PERSONAL_ACCESS_TOKEN: {
                gitHubBuilder.withOAuthToken(builder.personalAccessToken);
                break;
            }
            case APP_INSTALLATION: {
                gitHubBuilder.withAuthorizationProvider(this.getAppInstallationAuthorizationProvider(builder, appPermissions));
            }
        }
        this.gitHub = gitHubBuilder.build();
        String fullRepoName = this.repoOwner + "/" + this.repoName;
        try {
            this.repository = this.gitHub.getRepository(fullRepoName);
        }
        catch (FileNotFoundException fnf) {
            throw new FlowRegistryException("Repository [" + fullRepoName + "] not found");
        }
        if (this.gitHub.isAnonymous()) {
            this.canRead = true;
            this.canWrite = false;
        } else if (GitHubAuthenticationType.APP_INSTALLATION == this.authenticationType) {
            this.canRead = appPermissions.containsKey(REPOSITORY_CONTENTS_PERMISSION);
            String repositoryContentsPermissions = (String)appPermissions.get(REPOSITORY_CONTENTS_PERMISSION);
            this.canWrite = WRITE_ACCESS.equals(repositoryContentsPermissions);
        } else {
            GHMyself currentUser = this.gitHub.getMyself();
            this.canRead = this.repository.hasPermission((GHUser)currentUser, GHPermissionType.READ);
            this.canWrite = this.repository.hasPermission((GHUser)currentUser, GHPermissionType.WRITE);
        }
    }

    public String getRepoOwner() {
        return this.repoOwner;
    }

    public String getRepoName() {
        return this.repoName;
    }

    public boolean hasReadPermission() {
        return this.canRead;
    }

    public boolean hasWritePermission() {
        return this.canWrite;
    }

    public String createContent(GitCreateContentRequest request) throws IOException, FlowRegistryException {
        String branch = request.getBranch();
        String resolvedPath = this.getResolvedPath(request.getPath());
        LOGGER.debug("Creating content at path [{}] on branch [{}] in repo [{}] ", new Object[]{resolvedPath, branch, this.repository.getName()});
        return this.execute(() -> {
            try {
                GHContentUpdateResponse response = this.repository.createContent().branch(branch).path(resolvedPath).content(request.getContent()).message(request.getMessage()).sha(request.getExistingContentSha()).commit();
                return response.getCommit().getSha();
            }
            catch (FileNotFoundException fnf) {
                this.throwPathOrBranchNotFound(fnf, resolvedPath, branch);
                return null;
            }
        });
    }

    public Set<String> getBranches() throws IOException, FlowRegistryException {
        LOGGER.debug("Getting branches for repo [{}]", (Object)this.repository.getName());
        return this.execute(() -> this.repository.getBranches().keySet());
    }

    public InputStream getContentFromBranch(String path, String branch) throws IOException, FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        String branchRef = BRANCH_REF_PATTERN.formatted(branch);
        LOGGER.debug("Getting content for [{}] from branch [{}] in repo [{}] ", new Object[]{resolvedPath, branch, this.repository.getName()});
        return this.execute(() -> {
            try {
                GHContent ghContent = this.repository.getFileContent(resolvedPath, branchRef);
                return this.repository.readBlob(ghContent.getSha());
            }
            catch (FileNotFoundException fnf) {
                this.throwPathOrBranchNotFound(fnf, resolvedPath, branchRef);
                return null;
            }
        });
    }

    public InputStream getContentFromCommit(String path, String commitSha) throws IOException, FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        LOGGER.debug("Getting content for [{}] from commit [{}] in repo [{}] ", new Object[]{resolvedPath, commitSha, this.repository.getName()});
        return this.execute(() -> {
            try {
                GHContent ghContent = this.repository.getFileContent(resolvedPath, commitSha);
                return this.repository.readBlob(ghContent.getSha());
            }
            catch (FileNotFoundException fnf) {
                throw new FlowRegistryException("Path [" + resolvedPath + "] or Commit [" + commitSha + "] not found", (Throwable)fnf);
            }
        });
    }

    public List<GitCommit> getCommits(String path, String branch) throws IOException, FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        String branchRef = BRANCH_REF_PATTERN.formatted(branch);
        LOGGER.debug("Getting commits for [{}] from branch [{}] in repo [{}]", new Object[]{resolvedPath, branch, this.repository.getName()});
        return this.execute(() -> {
            try {
                GHRef branchGhRef = this.repository.getRef(branchRef);
                List ghCommits = this.repository.queryCommits().path(resolvedPath).from(branchGhRef.getObject().getSha()).pageSize(50).list().toList();
                ArrayList<GitCommit> commits = new ArrayList<GitCommit>();
                for (GHCommit ghCommit : ghCommits) {
                    commits.add(this.toGitCommit(ghCommit));
                }
                return commits;
            }
            catch (FileNotFoundException fnf) {
                this.throwPathOrBranchNotFound(fnf, resolvedPath, branchRef);
                return null;
            }
        });
    }

    public Set<String> getTopLevelDirectoryNames(String branch) throws IOException, FlowRegistryException {
        return this.getDirectoryItems("", branch, GHContent::isDirectory);
    }

    public Set<String> getFileNames(String directory, String branch) throws IOException, FlowRegistryException {
        return this.getDirectoryItems(directory, branch, GHContent::isFile);
    }

    private Set<String> getDirectoryItems(String directory, String branch, Predicate<GHContent> filter) throws IOException, FlowRegistryException {
        String resolvedDirectory = this.getResolvedPath(directory);
        String branchRef = BRANCH_REF_PATTERN.formatted(branch);
        LOGGER.debug("Getting directory items for [{}] from branch [{}] in repo [{}] ", new Object[]{resolvedDirectory, branch, this.repository.getName()});
        return this.execute(() -> {
            try {
                return this.repository.getDirectoryContent(resolvedDirectory, branchRef).stream().filter(filter).map(GHContent::getName).collect(Collectors.toSet());
            }
            catch (FileNotFoundException fnf) {
                this.throwPathOrBranchNotFound(fnf, resolvedDirectory, branchRef);
                return null;
            }
        });
    }

    public Optional<String> getContentSha(String path, String branch) throws IOException, FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        String branchRef = BRANCH_REF_PATTERN.formatted(branch);
        LOGGER.debug("Getting content SHA for [{}] from branch [{}] in repo [{}] ", new Object[]{resolvedPath, branch, this.repository.getName()});
        return this.execute(() -> {
            try {
                GHContent ghContent = this.repository.getFileContent(resolvedPath, branchRef);
                return Optional.of(ghContent.getSha());
            }
            catch (FileNotFoundException e) {
                LOGGER.warn("Unable to get content SHA for [{}] from branch [{}] because content does not exist", (Object)resolvedPath, (Object)branch);
                return Optional.empty();
            }
        });
    }

    public InputStream deleteContent(String filePath, String commitMessage, String branch) throws FlowRegistryException, IOException {
        String resolvedPath = this.getResolvedPath(filePath);
        LOGGER.debug("Deleting file [{}] in repo [{}] on branch [{}]", new Object[]{resolvedPath, this.repository.getName(), branch});
        return this.execute(() -> {
            try {
                GHContent ghContent = this.repository.getFileContent(resolvedPath);
                ghContent.delete(commitMessage, branch);
                return ghContent.read();
            }
            catch (FileNotFoundException fnf) {
                this.throwPathOrBranchNotFound(fnf, resolvedPath, branch);
                return null;
            }
        });
    }

    private String getResolvedPath(String path) {
        return this.repoPath == null ? path : this.repoPath + "/" + path;
    }

    private void throwPathOrBranchNotFound(FileNotFoundException fileNotFoundException, String path, String branch) throws FlowRegistryException {
        throw new FlowRegistryException("Path [" + path + "] or Branch [" + branch + "] not found", (Throwable)fileNotFoundException);
    }

    private GitCommit toGitCommit(GHCommit ghCommit) throws IOException {
        GHCommit.ShortInfo shortInfo = ghCommit.getCommitShortInfo();
        return new GitCommit(ghCommit.getSHA1(), ghCommit.getAuthor().getLogin(), shortInfo.getMessage(), Instant.ofEpochMilli(shortInfo.getCommitDate().getTime()));
    }

    private <T> T execute(GHRequest<T> action) throws FlowRegistryException, IOException {
        try {
            return action.execute();
        }
        catch (IOException | FlowRegistryException e) {
            throw e;
        }
        catch (Exception e) {
            throw new FlowRegistryException(e.getMessage(), (Throwable)e);
        }
    }

    private AuthorizationProvider getAppInstallationAuthorizationProvider(Builder builder, Map<String, String> appPermissions) throws FlowRegistryException {
        AuthorizationProvider appAuthorizationProvider = this.getAppAuthorizationProvider(builder.appId, builder.appPrivateKey);
        return new AppInstallationAuthorizationProvider(gitHubApp -> {
            appPermissions.putAll(gitHubApp.getPermissions());
            return gitHubApp.getInstallationByRepository(builder.repoOwner, builder.repoName);
        }, appAuthorizationProvider);
    }

    private AuthorizationProvider getAppAuthorizationProvider(String appId, String appPrivateKey) throws FlowRegistryException {
        try {
            StandardPrivateKeyReader privateKeyReader = new StandardPrivateKeyReader();
            PrivateKey privateKey = privateKeyReader.readPrivateKey(appPrivateKey);
            return new JWTTokenProvider(appId, privateKey);
        }
        catch (Exception e) {
            throw new FlowRegistryException("Failed to build Authorization Provider from App ID and App Private Key", (Throwable)e);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String apiUrl;
        private GitHubAuthenticationType authenticationType;
        private String personalAccessToken;
        private String repoOwner;
        private String repoName;
        private String repoPath;
        private String appPrivateKey;
        private String appId;

        public Builder apiUrl(String apiUrl) {
            this.apiUrl = apiUrl;
            return this;
        }

        public Builder authenticationType(GitHubAuthenticationType authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        public Builder personalAccessToken(String personalAccessToken) {
            this.personalAccessToken = personalAccessToken;
            return this;
        }

        public Builder repoOwner(String repoOwner) {
            this.repoOwner = repoOwner;
            return this;
        }

        public Builder repoName(String repoName) {
            this.repoName = repoName;
            return this;
        }

        public Builder repoPath(String repoPath) {
            this.repoPath = repoPath;
            return this;
        }

        public Builder appId(String appId) {
            this.appId = appId;
            return this;
        }

        public Builder appPrivateKey(String appPrivateKey) {
            this.appPrivateKey = appPrivateKey;
            return this;
        }

        public GitHubRepositoryClient build() throws IOException, FlowRegistryException {
            return new GitHubRepositoryClient(this);
        }
    }

    private static interface GHRequest<T> {
        public T execute() throws IOException, FlowRegistryException;
    }
}

