/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.rest.service;

import com.google.common.collect.Maps;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.NavigableSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentMap;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.util.Pair;
import org.apache.kylin.metadata.badquery.BadQueryHistoryManager;
import org.apache.kylin.rest.request.SQLRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BadQueryDetector
extends Thread {
    public static final int ONE_MB = 0x100000;
    private static final Logger logger = LoggerFactory.getLogger(BadQueryDetector.class);
    private final ConcurrentMap<Thread, Entry> runningQueries = Maps.newConcurrentMap();
    private final long detectionInterval;
    private final int alertMB;
    private final int alertRunningSec;
    private KylinConfig kylinConfig;
    private ArrayList<Notifier> notifiers = new ArrayList();

    public BadQueryDetector() {
        super("BadQueryDetector");
        this.setDaemon(true);
        this.kylinConfig = KylinConfig.getInstanceFromEnv();
        this.detectionInterval = (long)this.kylinConfig.getBadQueryDefaultDetectIntervalSeconds() * 1000L;
        this.alertMB = 100;
        this.alertRunningSec = this.kylinConfig.getBadQueryDefaultAlertingSeconds();
        this.initNotifiers();
    }

    public BadQueryDetector(long detectionInterval, int alertMB, int alertRunningSec) {
        super("BadQueryDetector");
        this.setDaemon(true);
        this.detectionInterval = detectionInterval;
        this.alertMB = alertMB;
        this.alertRunningSec = alertRunningSec;
        this.kylinConfig = KylinConfig.getInstanceFromEnv();
        this.initNotifiers();
    }

    public static long getSystemAvailBytes() {
        Runtime runtime = Runtime.getRuntime();
        long totalMemory = runtime.totalMemory();
        long freeMemory = runtime.freeMemory();
        long maxMemory = runtime.maxMemory();
        long usedMemory = totalMemory - freeMemory;
        long availableMemory = maxMemory - usedMemory;
        return availableMemory;
    }

    public static int getSystemAvailMB() {
        return (int)(BadQueryDetector.getSystemAvailBytes() / 0x100000L);
    }

    private void initNotifiers() {
        this.notifiers.add(new LoggerNotifier());
        if (this.kylinConfig.getBadQueryPersistentEnabled()) {
            this.notifiers.add(new PersistenceNotifier());
        }
    }

    public void registerNotifier(Notifier notifier) {
        this.notifiers.add(notifier);
    }

    private void notify(String adj, float runningSec, long startTime, String project, String sql, String user, Thread t) {
        for (Notifier notifier : this.notifiers) {
            try {
                notifier.badQueryFound(adj, runningSec, startTime, project, sql, user, t);
            }
            catch (Exception e) {
                logger.error("", (Throwable)e);
            }
        }
    }

    public void queryStart(Thread thread, SQLRequest sqlRequest, String user) {
        this.runningQueries.put(thread, new Entry(sqlRequest, user, thread));
    }

    public void queryEnd(Thread thread) {
        this.runningQueries.remove(thread);
    }

    @Override
    public void run() {
        while (true) {
            try {
                Thread.sleep(this.detectionInterval);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return;
            }
            try {
                this.detectBadQuery();
                continue;
            }
            catch (Exception ex) {
                logger.error("", (Throwable)ex);
                continue;
            }
            break;
        }
    }

    private void detectBadQuery() {
        long now = System.currentTimeMillis();
        ArrayList entries = new ArrayList(this.runningQueries.values());
        Collections.sort(entries);
        for (Entry e : entries) {
            float runningSec = (float)(now - e.startTime) / 1000.0f;
            if (!(runningSec >= (float)this.alertRunningSec)) break;
            this.notify("Slow", runningSec, e.startTime, e.sqlRequest.getProject(), e.sqlRequest.getSql(), e.user, e.thread);
            this.dumpStackTrace(e.thread);
        }
        if (BadQueryDetector.getSystemAvailMB() < this.alertMB) {
            logger.info("System free memory less than " + this.alertMB + " MB. " + entries.size() + " queries running.");
        }
    }

    private void dumpStackTrace(Thread t) {
        int maxStackTraceDepth = this.kylinConfig.getBadQueryStackTraceDepth();
        int current = 0;
        StackTraceElement[] stackTrace = t.getStackTrace();
        StringBuilder buf = new StringBuilder("Problematic thread 0x" + Long.toHexString(t.getId()));
        buf.append("\n");
        for (StackTraceElement e : stackTrace) {
            if (++current > maxStackTraceDepth) break;
            buf.append("\t").append("at ").append(e.toString()).append("\n");
        }
        logger.info(buf.toString());
    }

    private class Entry
    implements Comparable<Entry> {
        final SQLRequest sqlRequest;
        final long startTime;
        final Thread thread;
        final String user;

        Entry(SQLRequest sqlRequest, String user, Thread thread) {
            this.sqlRequest = sqlRequest;
            this.startTime = System.currentTimeMillis();
            this.thread = thread;
            this.user = user;
        }

        @Override
        public int compareTo(Entry o) {
            return (int)(this.startTime - o.startTime);
        }
    }

    private class PersistenceNotifier
    implements Notifier {
        BadQueryHistoryManager badQueryManager;
        String serverHostname;
        NavigableSet<Pair<Long, String>> cacheQueue;

        public PersistenceNotifier() {
            this.badQueryManager = BadQueryHistoryManager.getInstance((KylinConfig)BadQueryDetector.this.kylinConfig);
            this.cacheQueue = new TreeSet<Pair<Long, String>>(new Comparator<Pair<Long, String>>(){

                @Override
                public int compare(Pair<Long, String> o1, Pair<Long, String> o2) {
                    if (o1.equals(o2)) {
                        return 0;
                    }
                    if (((Long)o1.getFirst()).equals(o2.getFirst())) {
                        return ((String)o2.getSecond()).compareTo((String)o2.getSecond());
                    }
                    return (int)((Long)o1.getFirst() - (Long)o2.getFirst());
                }
            });
            try {
                this.serverHostname = InetAddress.getLocalHost().getHostName();
            }
            catch (UnknownHostException e) {
                this.serverHostname = "Unknow";
                logger.warn("Error in get current hostname.", (Throwable)e);
            }
        }

        @Override
        public void badQueryFound(String adj, float runningSec, long startTime, String project, String sql, String user, Thread t) {
            try {
                long cachingSeconds = (long)(BadQueryDetector.this.kylinConfig.getBadQueryDefaultAlertingSeconds() + 1) * 30L;
                Pair sqlPair = new Pair((Object)startTime, (Object)sql);
                if (!this.cacheQueue.contains(sqlPair)) {
                    this.badQueryManager.addEntryToProject(sql, startTime, adj, runningSec, this.serverHostname, t.getName(), user, project);
                    this.cacheQueue.add((Pair<Long, String>)sqlPair);
                    while (!(this.cacheQueue.isEmpty() || System.currentTimeMillis() - (Long)((Pair)this.cacheQueue.first()).getFirst() <= cachingSeconds * 1000L && this.cacheQueue.size() <= BadQueryDetector.this.kylinConfig.getBadQueryHistoryNum() * 3)) {
                        this.cacheQueue.pollFirst();
                    }
                } else {
                    this.badQueryManager.updateEntryToProject(sql, startTime, adj, runningSec, this.serverHostname, t.getName(), user, project);
                }
            }
            catch (IOException e) {
                logger.error("Error in bad query persistence.", (Throwable)e);
            }
        }
    }

    private class LoggerNotifier
    implements Notifier {
        private LoggerNotifier() {
        }

        @Override
        public void badQueryFound(String adj, float runningSec, long startTime, String project, String sql, String user, Thread t) {
            logger.info("{} query has been running {} seconds (project:{}, thread: 0x{}, user:{}) -- {}", new Object[]{adj, Float.valueOf(runningSec), project, Long.toHexString(t.getId()), user, sql});
        }
    }

    public static interface Notifier {
        public void badQueryFound(String var1, float var2, long var3, String var5, String var6, String var7, Thread var8);
    }
}

