/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.util.concurrent.ListenableFutureTask;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.nio.charset.Charset;
import java.sql.Timestamp;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.jackrabbit.core.data.DataRecord;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.apache.jackrabbit.oak.commons.FileIOUtils;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.BlobGarbageCollector;
import org.apache.jackrabbit.oak.plugins.blob.BlobReferenceRetriever;
import org.apache.jackrabbit.oak.plugins.blob.BlobTrackingStore;
import org.apache.jackrabbit.oak.plugins.blob.GarbageCollectionRepoStats;
import org.apache.jackrabbit.oak.plugins.blob.GarbageCollectorFileState;
import org.apache.jackrabbit.oak.plugins.blob.ReferenceCollector;
import org.apache.jackrabbit.oak.plugins.blob.SharedDataStore;
import org.apache.jackrabbit.oak.plugins.blob.datastore.BlobTracker;
import org.apache.jackrabbit.oak.plugins.blob.datastore.SharedDataStoreUtils;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.blob.GarbageCollectableBlobStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MarkSweepGarbageCollector
implements BlobGarbageCollector {
    public static final Logger LOG = LoggerFactory.getLogger(MarkSweepGarbageCollector.class);
    public static final String TEMP_DIR = StandardSystemProperty.JAVA_IO_TMPDIR.value();
    public static final int DEFAULT_BATCH_COUNT = 1024;
    public static final String DELIM = ",";
    private static final Function<String, String> transformer = new Function<String, String>(){

        @Nullable
        public String apply(@Nullable String input) {
            if (input != null) {
                return input.split(MarkSweepGarbageCollector.DELIM)[0];
            }
            return "";
        }
    };
    private final long maxLastModifiedInterval;
    private final GarbageCollectableBlobStore blobStore;
    private final BlobReferenceRetriever marker;
    private final Executor executor;
    private final int batchCount;
    private final String repoId;
    private final String root;

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, String root, int batchCount, long maxLastModifiedInterval, @Nullable String repositoryId) throws IOException {
        this.executor = executor;
        this.blobStore = blobStore;
        this.marker = marker;
        this.batchCount = batchCount;
        this.maxLastModifiedInterval = maxLastModifiedInterval;
        this.repoId = repositoryId;
        this.root = root;
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, @Nullable String repositoryId) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 1024, TimeUnit.HOURS.toMillis(24L), repositoryId);
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, long maxLastModifiedInterval, @Nullable String repositoryId) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 1024, maxLastModifiedInterval, repositoryId);
    }

    @Override
    public void collectGarbage(boolean markOnly) throws Exception {
        this.markAndSweep(markOnly, false);
    }

    @Override
    public void collectGarbage(boolean markOnly, boolean forceBlobRetrieve) throws Exception {
        this.markAndSweep(markOnly, forceBlobRetrieve);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<GarbageCollectionRepoStats> getStats() throws Exception {
        ArrayList stats = Lists.newArrayList();
        if (SharedDataStoreUtils.isShared((BlobStore)this.blobStore)) {
            List<DataRecord> refFiles = ((SharedDataStore)this.blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
            ImmutableMap references = Maps.uniqueIndex(refFiles, (Function)new Function<DataRecord, String>(){

                public String apply(DataRecord input) {
                    return SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getIdFromName(input.getIdentifier().toString());
                }
            });
            List<DataRecord> markerFiles = ((SharedDataStore)this.blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
            ImmutableMap markers = Maps.uniqueIndex(markerFiles, (Function)new Function<DataRecord, String>(){

                public String apply(DataRecord input) {
                    return SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getIdFromName(input.getIdentifier().toString());
                }
            });
            List<DataRecord> repoFiles = ((SharedDataStore)this.blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REPOSITORY.getType());
            for (DataRecord repoRec : repoFiles) {
                String id = SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getIdFromName(repoRec.getIdentifier().toString());
                GarbageCollectionRepoStats stat = new GarbageCollectionRepoStats();
                stat.setRepositoryId(id);
                if (id != null && id.equals(this.repoId)) {
                    stat.setLocal(true);
                }
                if (references.containsKey(id)) {
                    DataRecord refRec = (DataRecord)references.get(id);
                    stat.setEndTime(refRec.getLastModified());
                    stat.setLength(refRec.getLength());
                    if (markers.containsKey(id)) {
                        stat.setStartTime(((DataRecord)markers.get(id)).getLastModified());
                    }
                    LineNumberReader reader = null;
                    try {
                        reader = new LineNumberReader(new InputStreamReader(refRec.getStream()));
                        while (reader.readLine() != null) {
                        }
                        stat.setNumLines(reader.getLineNumber());
                    }
                    catch (Throwable throwable) {
                        Closeables.close(reader, (boolean)true);
                        throw throwable;
                    }
                    Closeables.close((Closeable)reader, (boolean)true);
                }
                stats.add(stat);
            }
        }
        return stats;
    }

    protected void markAndSweep(boolean markOnly, boolean forceBlobRetrieve) throws Exception {
        boolean threw = true;
        GarbageCollectorFileState fs = new GarbageCollectorFileState(this.root);
        try {
            Stopwatch sw = Stopwatch.createStarted();
            LOG.info("Starting Blob garbage collection with markOnly [{}]", (Object)markOnly);
            long markStart = System.currentTimeMillis();
            this.mark(fs);
            if (!markOnly) {
                long deleteCount = this.sweep(fs, markStart, forceBlobRetrieve);
                threw = false;
                long maxTime = this.getLastMaxModifiedTime(markStart) > 0L ? this.getLastMaxModifiedTime(markStart) : markStart;
                sw.stop();
                LOG.info("Blob garbage collection completed in {} ({} ms). Number of blobs deleted [{}] with max modification time of [{}]", new Object[]{sw.toString(), sw.elapsed(TimeUnit.MILLISECONDS), deleteCount, MarkSweepGarbageCollector.timestampToString(maxTime)});
            }
        }
        catch (Exception e) {
            LOG.error("Blob garbage collection error", (Throwable)e);
            throw e;
        }
        finally {
            if (!LOG.isTraceEnabled()) {
                Closeables.close((Closeable)fs, (boolean)threw);
            }
        }
    }

    protected void mark(GarbageCollectorFileState fs) throws IOException, DataStoreException {
        LOG.debug("Starting mark phase of the garbage collector");
        GarbageCollectionType.get(this.blobStore).addMarkedStartMarker(this.blobStore, this.repoId);
        this.iterateNodeTree(fs, false);
        GarbageCollectionType.get(this.blobStore).addMarked(this.blobStore, fs, this.repoId);
        LOG.debug("Ending mark phase of the garbage collector");
    }

    private void difference(GarbageCollectorFileState fs) throws IOException {
        LOG.debug("Starting difference phase of the garbage collector");
        FileIOUtils.FileLineDifferenceIterator iter = new FileIOUtils.FileLineDifferenceIterator(fs.getMarkedRefs(), fs.getAvailableRefs(), transformer);
        int candidates = FileIOUtils.writeStrings((Iterator)iter, (File)fs.getGcCandidates(), (boolean)true);
        LOG.debug("Found candidates - " + candidates);
        LOG.debug("Ending difference phase of the garbage collector");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected long sweep(GarbageCollectorFileState fs, long markStart, boolean forceBlobRetrieve) throws Exception {
        long earliestRefAvailTime;
        try {
            earliestRefAvailTime = GarbageCollectionType.get(this.blobStore).mergeAllMarkedReferences(this.blobStore, fs);
            LOG.debug("Earliest reference available for timestamp [{}]", (Object)earliestRefAvailTime);
            earliestRefAvailTime = earliestRefAvailTime < markStart ? earliestRefAvailTime : markStart;
        }
        catch (Exception e) {
            return 0L;
        }
        new BlobIdRetriever(fs, forceBlobRetrieve).call();
        this.difference(fs);
        long count = 0L;
        long deleted = 0L;
        long lastMaxModifiedTime = this.getLastMaxModifiedTime(earliestRefAvailTime);
        LOG.debug("Starting sweep phase of the garbage collector");
        LOG.debug("Sweeping blobs with modified time > than the configured max deleted time ({}). ", (Object)MarkSweepGarbageCollector.timestampToString(lastMaxModifiedTime));
        BufferedWriter removesWriter = null;
        LineIterator iterator = null;
        try {
            removesWriter = Files.newWriter((File)fs.getGarbage(), (Charset)Charsets.UTF_8);
            ArrayDeque<String> removesQueue = new ArrayDeque<String>();
            iterator = FileUtils.lineIterator((File)fs.getGcCandidates(), (String)Charsets.UTF_8.name());
            UnmodifiableIterator partitions = Iterators.partition((Iterator)iterator, (int)this.getBatchCount());
            while (partitions.hasNext()) {
                List ids = (List)partitions.next();
                count += (long)ids.size();
                deleted += BlobCollectionType.get(this.blobStore).sweepInternal(this.blobStore, ids, removesQueue, lastMaxModifiedTime);
                MarkSweepGarbageCollector.saveBatchToFile(Lists.newArrayList(removesQueue), removesWriter);
                removesQueue.clear();
            }
        }
        catch (Throwable throwable) {
            LineIterator.closeQuietly(iterator);
            IOUtils.closeQuietly((Closeable)removesWriter);
            throw throwable;
        }
        LineIterator.closeQuietly((LineIterator)iterator);
        IOUtils.closeQuietly((Closeable)removesWriter);
        BlobCollectionType.get(this.blobStore).handleRemoves(this.blobStore, fs.getGarbage());
        if (count != deleted) {
            LOG.warn("Deleted only [{}] blobs entries from the [{}] candidates identified. This may happen if blob modified time is > than the max deleted time ({})", new Object[]{deleted, count, MarkSweepGarbageCollector.timestampToString(lastMaxModifiedTime)});
        }
        GarbageCollectionType.get(this.blobStore).removeAllMarkedReferences(this.blobStore);
        LOG.debug("Ending sweep phase of the garbage collector");
        return deleted;
    }

    private int getBatchCount() {
        return this.batchCount;
    }

    private long getLastMaxModifiedTime(long maxModified) {
        return this.maxLastModifiedInterval > 0L ? (maxModified <= 0L ? System.currentTimeMillis() : maxModified) - this.maxLastModifiedInterval : 0L;
    }

    static void saveBatchToFile(List<String> ids, BufferedWriter writer) throws IOException {
        for (String id : ids) {
            FileIOUtils.writeAsLine((BufferedWriter)writer, (String)id, (boolean)true);
        }
        writer.flush();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void iterateNodeTree(GarbageCollectorFileState fs, final boolean logPath) throws IOException {
        final BufferedWriter writer = Files.newWriter((File)fs.getMarkedRefs(), (Charset)Charsets.UTF_8);
        final AtomicInteger count = new AtomicInteger();
        try {
            this.marker.collectReferences(new ReferenceCollector(){
                private final boolean debugMode = LOG.isTraceEnabled();

                @Override
                public void addReference(String blobId, final String nodeId) {
                    if (this.debugMode) {
                        LOG.trace("BlobId : {}, NodeId : {}", (Object)blobId, (Object)nodeId);
                    }
                    try {
                        Iterator idIter = MarkSweepGarbageCollector.this.blobStore.resolveChunks(blobId);
                        final Joiner delimJoiner = Joiner.on((String)MarkSweepGarbageCollector.DELIM).skipNulls();
                        UnmodifiableIterator partitions = Iterators.partition((Iterator)idIter, (int)MarkSweepGarbageCollector.this.getBatchCount());
                        while (partitions.hasNext()) {
                            List idBatch = Lists.transform((List)((List)partitions.next()), (Function)new Function<String, String>(){

                                @Nullable
                                public String apply(@Nullable String id) {
                                    if (logPath) {
                                        return delimJoiner.join((Object)id, (Object)nodeId, new Object[0]);
                                    }
                                    return id;
                                }
                            });
                            if (this.debugMode) {
                                LOG.trace("chunkIds : {}", (Object)idBatch);
                            }
                            count.getAndAdd(idBatch.size());
                            MarkSweepGarbageCollector.saveBatchToFile(idBatch, writer);
                        }
                        if (count.get() % MarkSweepGarbageCollector.this.getBatchCount() == 0) {
                            LOG.info("Collected ({}) blob references", (Object)count.get());
                        }
                    }
                    catch (Exception e) {
                        throw new RuntimeException("Error in retrieving references", e);
                    }
                }
            });
            LOG.info("Number of valid blob references marked under mark phase of Blob garbage collection [{}]", (Object)count.get());
            FileIOUtils.sort((File)fs.getMarkedRefs(), (Comparator)new Comparator<String>(){

                @Override
                public int compare(String s1, String s2) {
                    return s1.split(MarkSweepGarbageCollector.DELIM)[0].compareTo(s2.split(MarkSweepGarbageCollector.DELIM)[0]);
                }
            });
        }
        finally {
            IOUtils.closeQuietly((Closeable)writer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long checkConsistency() throws Exception {
        boolean threw = true;
        GarbageCollectorFileState fs = new GarbageCollectorFileState(this.root);
        long candidates = 0L;
        try {
            LOG.info("Starting blob consistency check");
            ListenableFutureTask blobIdRetriever = ListenableFutureTask.create((Callable)new BlobIdRetriever(fs, true));
            this.executor.execute((Runnable)blobIdRetriever);
            this.iterateNodeTree(fs, true);
            try {
                blobIdRetriever.get();
            }
            catch (ExecutionException e) {
                LOG.warn("Error occurred while fetching all the blobIds from the BlobStore");
                threw = false;
                throw e;
            }
            LOG.trace("Starting difference phase of the consistency check");
            FileIOUtils.FileLineDifferenceIterator iter = new FileIOUtils.FileLineDifferenceIterator(fs.getAvailableRefs(), fs.getMarkedRefs(), transformer);
            candidates = FileIOUtils.writeStrings((Iterator)iter, (File)fs.getGcCandidates(), (boolean)true);
            LOG.trace("Ending difference phase of the consistency check");
            LOG.info("Consistency check found [{}] missing blobs", (Object)candidates);
            if (candidates > 0L) {
                LOG.warn("Consistency check failure in the the blob store : {}, check missing candidates in file {}", (Object)this.blobStore, (Object)fs.getGcCandidates().getAbsolutePath());
            }
        }
        finally {
            if (!LOG.isTraceEnabled() && candidates == 0L) {
                Closeables.close((Closeable)fs, (boolean)threw);
            }
        }
        return candidates;
    }

    private static String timestampToString(long timestamp) {
        return (new Timestamp(timestamp) + "00").substring(0, 23);
    }

    private static enum BlobCollectionType {
        TRACKER{

            @Override
            long sweepInternal(GarbageCollectableBlobStore blobStore, List<String> ids, ArrayDeque<String> exceptionQueue, long maxModified) {
                long totalDeleted = 0L;
                LOG.trace("Blob ids to be deleted {}", ids);
                for (String id : ids) {
                    try {
                        long deleted = blobStore.countDeleteChunks((List)Lists.newArrayList((Object[])new String[]{id}), maxModified);
                        if (deleted != 1L) {
                            LOG.debug("Blob [{}] not deleted", (Object)id);
                        } else {
                            exceptionQueue.add(id);
                        }
                        totalDeleted += deleted;
                    }
                    catch (Exception e) {
                        LOG.warn("Error occurred while deleting blob with id [{}]", (Object)id, (Object)e);
                    }
                }
                return totalDeleted;
            }

            @Override
            void retrieve(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, int batchCount) throws Exception {
                ((BlobTrackingStore)blobStore).getTracker().get(fs.getAvailableRefs().getAbsolutePath());
            }

            @Override
            void handleRemoves(GarbageCollectableBlobStore blobStore, File removedIds) throws IOException {
                ((BlobTrackingStore)blobStore).getTracker().remove(removedIds);
            }

            @Override
            void track(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) {
                try {
                    File f = File.createTempFile("blobiddownload", null);
                    FileUtils.copyFile((File)fs.getAvailableRefs(), (File)f);
                    ((BlobTrackingStore)blobStore).getTracker().add(f);
                }
                catch (IOException e) {
                    LOG.warn("Unable to track blob ids locally");
                }
            }
        }
        ,
        DEFAULT;


        long sweepInternal(GarbageCollectableBlobStore blobStore, List<String> ids, ArrayDeque<String> exceptionQueue, long maxModified) {
            long deleted = 0L;
            try {
                LOG.trace("Blob ids to be deleted {}", ids);
                deleted = blobStore.countDeleteChunks(ids, maxModified);
                if (deleted != (long)ids.size()) {
                    LOG.debug("Some [{}] blobs were not deleted from the batch : [{}]", (Object)((long)ids.size() - deleted), ids);
                }
                exceptionQueue.addAll(ids);
            }
            catch (Exception e) {
                LOG.warn("Error occurred while deleting blob with ids [{}]", ids, (Object)e);
            }
            return deleted;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void retrieve(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, int batchCount) throws Exception {
            LOG.debug("Starting retrieve of all blobs");
            int blobsCount = 0;
            Iterator idsIter = null;
            try {
                idsIter = blobStore.getAllChunkIds(0L);
                blobsCount = FileIOUtils.writeStrings((Iterator)idsIter, (File)fs.getAvailableRefs(), (boolean)true, (Logger)LOG, (String)"Retrieved blobs - ");
                FileIOUtils.sort((File)fs.getAvailableRefs());
                LOG.info("Number of blobs present in BlobStore : [{}] ", (Object)blobsCount);
            }
            finally {
                if (idsIter instanceof Closeable) {
                    try {
                        Closeables.close((Closeable)((Closeable)((Object)idsIter)), (boolean)false);
                    }
                    catch (Exception e) {
                        LOG.debug("Error closing iterator");
                    }
                }
            }
        }

        void handleRemoves(GarbageCollectableBlobStore blobStore, File removedIds) throws IOException {
            FileUtils.forceDelete((File)removedIds);
        }

        void track(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) {
        }

        public static BlobCollectionType get(GarbageCollectableBlobStore blobStore) {
            BlobTracker tracker;
            if (blobStore instanceof BlobTrackingStore && (tracker = ((BlobTrackingStore)blobStore).getTracker()) != null) {
                return TRACKER;
            }
            return DEFAULT;
        }
    }

    static enum GarbageCollectionType {
        SHARED{

            @Override
            void removeAllMarkedReferences(GarbageCollectableBlobStore blobStore) {
                ((SharedDataStore)blobStore).deleteAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
                ((SharedDataStore)blobStore).deleteAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
            }

            @Override
            long mergeAllMarkedReferences(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) throws IOException, DataStoreException {
                List<DataRecord> refFiles = ((SharedDataStore)blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
                List<DataRecord> repoFiles = ((SharedDataStore)blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REPOSITORY.getType());
                Set<String> unAvailRepos = SharedDataStoreUtils.refsNotAvailableFromRepos(repoFiles, refFiles);
                if (unAvailRepos.isEmpty()) {
                    ArrayList files = Lists.newArrayList();
                    for (DataRecord refFile : refFiles) {
                        File file = FileIOUtils.copy((InputStream)refFile.getStream());
                        files.add(file);
                    }
                    FileIOUtils.merge((List)files, (File)fs.getMarkedRefs());
                    List<DataRecord> markerFiles = ((SharedDataStore)blobStore).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
                    long earliestMarker = SharedDataStoreUtils.getEarliestRecord(markerFiles).getLastModified();
                    LOG.trace("Earliest marker timestamp {}", (Object)earliestMarker);
                    long earliestRef = SharedDataStoreUtils.getEarliestRecord(refFiles).getLastModified();
                    LOG.trace("Earliest ref timestamp {}", (Object)earliestRef);
                    return earliestMarker < earliestRef ? earliestMarker : earliestRef;
                }
                LOG.error("Not all repositories have marked references available : {}", unAvailRepos);
                throw new IOException("Not all repositories have marked references available");
            }

            @Override
            void addMarked(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, String repoId) throws DataStoreException, IOException {
                ((SharedDataStore)blobStore).addMetadataRecord(fs.getMarkedRefs(), SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getNameFromId(repoId));
            }

            @Override
            public void addMarkedStartMarker(GarbageCollectableBlobStore blobStore, String repoId) {
                try {
                    ((SharedDataStore)blobStore).addMetadataRecord(new ByteArrayInputStream(new byte[0]), SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getNameFromId(repoId));
                }
                catch (DataStoreException e) {
                    LOG.debug("Error creating marked time marker for repo : {}", (Object)repoId);
                }
            }
        }
        ,
        DEFAULT;


        void removeAllMarkedReferences(GarbageCollectableBlobStore blobStore) {
        }

        void addMarked(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, String repoId) throws DataStoreException, IOException {
        }

        long mergeAllMarkedReferences(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) throws IOException, DataStoreException {
            if (!fs.getMarkedRefs().exists() || fs.getMarkedRefs().length() == 0L) {
                throw new IOException("Marked references not available");
            }
            return fs.getMarkedRefs().lastModified();
        }

        public static GarbageCollectionType get(GarbageCollectableBlobStore blobStore) {
            if (SharedDataStoreUtils.isShared((BlobStore)blobStore)) {
                return SHARED;
            }
            return DEFAULT;
        }

        public void addMarkedStartMarker(GarbageCollectableBlobStore blobStore, String repoId) {
        }
    }

    private class BlobIdRetriever
    implements Callable<Integer> {
        private final GarbageCollectorFileState fs;
        private final boolean forceRetrieve;

        public BlobIdRetriever(GarbageCollectorFileState fs, boolean forceBlobRetrieve) {
            this.fs = fs;
            this.forceRetrieve = forceBlobRetrieve;
        }

        @Override
        public Integer call() throws Exception {
            if (!this.forceRetrieve) {
                BlobCollectionType.get(MarkSweepGarbageCollector.this.blobStore).retrieve(MarkSweepGarbageCollector.this.blobStore, this.fs, MarkSweepGarbageCollector.this.getBatchCount());
                LOG.info("Length of blob ids file retrieved from tracker {}", (Object)this.fs.getAvailableRefs().length());
            }
            if (this.fs.getAvailableRefs().length() <= 0L) {
                BlobCollectionType.DEFAULT.retrieve(MarkSweepGarbageCollector.this.blobStore, this.fs, MarkSweepGarbageCollector.this.getBatchCount());
                LOG.info("Length of blob ids file retrieved {}", (Object)this.fs.getAvailableRefs().length());
                BlobCollectionType.get(MarkSweepGarbageCollector.this.blobStore).track(MarkSweepGarbageCollector.this.blobStore, this.fs);
            }
            return 0;
        }
    }
}

