/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.schemaengine.schemaregion.logfile;

import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.util.NoSuchElementException;
import javax.validation.constraints.NotNull;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.apache.iotdb.db.schemaengine.schemaregion.logfile.IDeserializer;
import org.apache.iotdb.tsfile.utils.ReadWriteIOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaLogReader<T>
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SchemaLogReader.class);
    private static final String FILE_CORRUPTED_MSG_TEMPLATE = "File {} is corrupted. The uncorrupted size is {}.";
    private final File logFile;
    private final RecordableInputStream inputStream;
    private final IDeserializer<T> deserializer;
    private T nextSchemaPlan;
    private long currentIndex = 0L;
    private boolean isFileCorrupted = false;

    public SchemaLogReader(String schemaDir, String logFileName, IDeserializer<T> deserializer) throws IOException {
        this.logFile = SystemFileFactory.INSTANCE.getFile(schemaDir + File.separator + logFileName);
        this.inputStream = new RecordableInputStream(new BufferedInputStream(new FileInputStream(this.logFile)));
        this.deserializer = deserializer;
    }

    public SchemaLogReader(String logFilePath, IDeserializer<T> deserializer) throws IOException {
        this.logFile = SystemFileFactory.INSTANCE.getFile(logFilePath);
        this.inputStream = new RecordableInputStream(new BufferedInputStream(new FileInputStream(this.logFile)));
        this.deserializer = deserializer;
    }

    public boolean hasNext() {
        if (this.isFileCorrupted()) {
            return false;
        }
        if (this.nextSchemaPlan == null) {
            this.readNext();
            return this.nextSchemaPlan != null;
        }
        return true;
    }

    public T next() {
        if (!this.hasNext()) {
            throw new NoSuchElementException();
        }
        T result = this.nextSchemaPlan;
        this.nextSchemaPlan = null;
        return result;
    }

    private void readNext() {
        this.currentIndex = this.inputStream.getReadBytes();
        try {
            this.nextSchemaPlan = this.deserializer.deserialize(this.inputStream);
        }
        catch (EOFException e) {
            this.nextSchemaPlan = null;
            this.truncateBrokenLogs();
        }
        catch (IOException e) {
            this.nextSchemaPlan = null;
            this.isFileCorrupted = true;
            LOGGER.error(FILE_CORRUPTED_MSG_TEMPLATE, new Object[]{this.logFile.getPath(), this.currentIndex, e});
        }
        catch (Exception e) {
            this.nextSchemaPlan = null;
            try {
                if (this.inputStream.available() > 0) {
                    this.isFileCorrupted = true;
                    LOGGER.error(FILE_CORRUPTED_MSG_TEMPLATE, new Object[]{this.logFile.getPath(), this.currentIndex, e});
                } else {
                    LOGGER.warn(e.getMessage(), (Throwable)e);
                    this.truncateBrokenLogs();
                }
            }
            catch (IOException ex) {
                this.isFileCorrupted = true;
                LOGGER.error(FILE_CORRUPTED_MSG_TEMPLATE, new Object[]{this.logFile.getPath(), this.currentIndex, e});
            }
        }
    }

    @Override
    public void close() throws IOException {
        this.inputStream.close();
        this.currentIndex = 0L;
    }

    public boolean isFileCorrupted() {
        return this.isFileCorrupted;
    }

    private void truncateBrokenLogs() {
        try (FileOutputStream outputStream = new FileOutputStream(this.logFile, true);
             FileChannel channel = outputStream.getChannel();){
            if (this.currentIndex != channel.size()) {
                LOGGER.warn("The end of log file {} is corrupted. Start truncate it. The unbroken size is {}. The file size is {}.", new Object[]{this.logFile.getName(), this.currentIndex, channel.size()});
                channel.truncate(this.currentIndex);
                channel.force(true);
            }
            this.isFileCorrupted = false;
        }
        catch (IOException e) {
            this.isFileCorrupted = true;
            LOGGER.error("Fail to truncate log file to size {}", (Object)this.currentIndex, (Object)e);
        }
    }

    public void skip(long n) throws IOException {
        ReadWriteIOUtils.skip((InputStream)this.inputStream, (long)n);
    }

    private static class RecordableInputStream
    extends InputStream {
        private final InputStream inputStream;
        private long readBytes = 0L;
        private long mark;

        public RecordableInputStream(InputStream inputStream) {
            this.inputStream = inputStream;
        }

        @Override
        public int read() throws IOException {
            int result = this.inputStream.read();
            ++this.readBytes;
            return result;
        }

        @Override
        public int read(@NotNull byte[] b) throws IOException {
            int num = this.inputStream.read(b);
            if (num < 0) {
                return num;
            }
            this.readBytes += (long)num;
            return num;
        }

        @Override
        public int read(@NotNull byte[] b, int off, int len) throws IOException {
            int num = this.inputStream.read(b, off, len);
            if (num < 0) {
                return num;
            }
            this.readBytes += (long)num;
            return num;
        }

        @Override
        public long skip(long n) throws IOException {
            long num = this.inputStream.skip(n);
            this.readBytes += num;
            return num;
        }

        @Override
        public synchronized void mark(int readlimit) {
            this.mark = this.readBytes;
            this.inputStream.mark(readlimit);
        }

        @Override
        public boolean markSupported() {
            return this.inputStream.markSupported();
        }

        @Override
        public int available() throws IOException {
            return this.inputStream.available();
        }

        @Override
        public synchronized void reset() throws IOException {
            this.inputStream.reset();
            this.readBytes = this.mark;
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
            this.readBytes = 0L;
        }

        public long getReadBytes() {
            return this.readBytes;
        }
    }
}

