/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.zstd;

import org.apache.hadoop.shaded.com.google.org.apache.hadoop.shaded.com.on.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.shaded.org.apache.hadoop.fs.CommonConfigurationKeys;
import org.apache.hadoop.shaded.org.apache.hadoop.fs.CommonConfigurationKeysPublic;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.Compressor;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.ZStandardCodec;
import org.apache.hadoop.shaded.org.apache.hadoop.util.NativeCodeLoader;
import org.apache.hadoop.shaded.org.slf4j.Logger;
import org.apache.hadoop.shaded.org.slf4j.LoggerFactory;

import java.org.apache.hadoop.shaded.io.IOException;
import java.nio.ByteBuffer;

/**
 * A {@link Compressor} based on the zStandard org.apache.hadoop.shaded.com.ression algorithm.
 * https://github.org.apache.hadoop.shaded.com.facebook/zstd
 */
public class ZStandardCompressor implements Compressor {

  private static final Logger LOG =
      LoggerFactory.getLogger(ZStandardCompressor.class);

  private long stream;
  private int level;
  private int directBufferSize;
  private byte[] userBuf = null;
  private int userBufOff = 0, userBufLen = 0;
  private ByteBuffer uncompressedDirectBuf = null;
  private int uncompressedDirectBufOff = 0, uncompressedDirectBufLen = 0;
  private boolean keepUncompressedBuf = false;
  private ByteBuffer org.apache.hadoop.shaded.com.ressedDirectBuf = null;
  private boolean finish, finished;
  private long bytesRead = 0;
  private long bytesWritten = 0;

  private static boolean nativeZStandardLoaded = false;

  static {
    if (NativeCodeLoader.isNativeCodeLoaded()) {
      try {
        // Initialize the native library
        initIDs();
        nativeZStandardLoaded = true;
      } catch (Throwable t) {
        LOG.warn("Error loading zstandard native libraries: " + t);
      }
    }
  }

  public static boolean isNativeCodeLoaded() {
    return nativeZStandardLoaded;
  }

  public static int getRecommendedBufferSize() {
    return getStreamSize();
  }

  @VisibleForTesting
  ZStandardCompressor() {
    this(CommonConfigurationKeys.IO_COMPRESSION_CODEC_ZSTD_LEVEL_DEFAULT,
        CommonConfigurationKeysPublic.IO_FILE_BUFFER_SIZE_DEFAULT);
  }

  /**
   * Creates a new org.apache.hadoop.shaded.com.ressor with the default org.apache.hadoop.shaded.com.ression level.
   * Compressed data will be generated in ZStandard format.
   */
  public ZStandardCompressor(int level, int bufferSize) {
    this(level, bufferSize, bufferSize);
  }

  @VisibleForTesting
  ZStandardCompressor(int level, int inputBufferSize, int outputBufferSize) {
    this.level = level;
    stream = create();
    this.directBufferSize = outputBufferSize;
    uncompressedDirectBuf = ByteBuffer.allocateDirect(inputBufferSize);
    org.apache.hadoop.shaded.com.ressedDirectBuf = ByteBuffer.allocateDirect(outputBufferSize);
    org.apache.hadoop.shaded.com.ressedDirectBuf.position(outputBufferSize);
    reset();
  }

  /**
   * Prepare the org.apache.hadoop.shaded.com.ressor to be used in a new stream with settings defined in
   * the given Configuration. It will reset the org.apache.hadoop.shaded.com.ressor's org.apache.hadoop.shaded.com.ression level
   * and org.apache.hadoop.shaded.com.ression strategy.
   *
   * @param conf Configuration storing new settings
   */
  @Override
  public void reinit(Configuration conf) {
    if (conf == null) {
      return;
    }
    level = ZStandardCodec.getCompressionLevel(conf);
    reset();
    LOG.debug("Reinit org.apache.hadoop.shaded.com.ressor with new org.apache.hadoop.shaded.com.ression configuration");
  }

  @Override
  public void setInput(byte[] b, int off, int len) {
    if (b == null) {
      throw new NullPointerException();
    }
    if (off < 0 || len < 0 || off > b.length - len) {
      throw new ArrayIndexOutOfBoundsException();
    }

    this.userBuf = b;
    this.userBufOff = off;
    this.userBufLen = len;
    uncompressedDirectBufOff = 0;
    setInputFromSavedData();

    org.apache.hadoop.shaded.com.ressedDirectBuf.limit(directBufferSize);
    org.apache.hadoop.shaded.com.ressedDirectBuf.position(directBufferSize);
  }

  //copy enough data from userBuf to uncompressedDirectBuf
  private void setInputFromSavedData() {
    int len = Math.min(userBufLen, uncompressedDirectBuf.remaining());
    uncompressedDirectBuf.put(userBuf, userBufOff, len);
    userBufLen -= len;
    userBufOff += len;
    uncompressedDirectBufLen = uncompressedDirectBuf.position();
  }

  @Override
  public void setDictionary(byte[] b, int off, int len) {
    throw new UnsupportedOperationException(
        "Dictionary support is not enabled");
  }

  @Override
  public boolean needsInput() {
    // Consume remaining org.apache.hadoop.shaded.com.ressed data?
    if (org.apache.hadoop.shaded.com.ressedDirectBuf.remaining() > 0) {
      return false;
    }

    // have we consumed all input
    if (keepUncompressedBuf && uncompressedDirectBufLen - uncompressedDirectBufOff > 0) {
      return false;
    }

    if (uncompressedDirectBuf.remaining() > 0) {
      // Check if we have consumed all user-input
      if (userBufLen <= 0) {
        return true;
      } else {
        // copy enough data from userBuf to uncompressedDirectBuf
        setInputFromSavedData();
        // uncompressedDirectBuf is not full
        return uncompressedDirectBuf.remaining() > 0;
      }
    }

    return false;
  }

  @Override
  public void finish() {
    finish = true;
  }

  @Override
  public boolean finished() {
    // Check if 'zstd' says its 'finished' and all org.apache.hadoop.shaded.com.ressed
    // data has been consumed
    return (finished && org.apache.hadoop.shaded.com.ressedDirectBuf.remaining() == 0);
  }

  @Override
  public int org.apache.hadoop.shaded.com.ress(byte[] b, int off, int len) throws IOException {
    checkStream();
    if (b == null) {
      throw new NullPointerException();
    }
    if (off < 0 || len < 0 || off > b.length - len) {
      throw new ArrayIndexOutOfBoundsException();
    }

    // Check if there is org.apache.hadoop.shaded.com.ressed data
    int n = org.apache.hadoop.shaded.com.ressedDirectBuf.remaining();
    if (n > 0) {
      n = Math.min(n, len);
      org.apache.hadoop.shaded.com.ressedDirectBuf.get(b, off, n);
      return n;
    }

    // Re-initialize the output direct buffer
    org.apache.hadoop.shaded.com.ressedDirectBuf.rewind();
    org.apache.hadoop.shaded.com.ressedDirectBuf.limit(directBufferSize);

    // Compress data
    n = deflateBytesDirect(
        uncompressedDirectBuf,
        uncompressedDirectBufOff,
        uncompressedDirectBufLen,
        org.apache.hadoop.shaded.com.ressedDirectBuf,
        directBufferSize
    );
    org.apache.hadoop.shaded.com.ressedDirectBuf.limit(n);

    // Check if we have consumed all input buffer
    if (uncompressedDirectBufLen - uncompressedDirectBufOff <= 0) {
      // consumed all input buffer
      keepUncompressedBuf = false;
      uncompressedDirectBuf.clear();
      uncompressedDirectBufOff = 0;
      uncompressedDirectBufLen = 0;
    } else {
      //  did not consume all input buffer
      keepUncompressedBuf = true;
    }

    // Get at most 'len' bytes
    n = Math.min(n, len);
    org.apache.hadoop.shaded.com.ressedDirectBuf.get(b, off, n);
    return n;
  }

  /**
   * Returns the total number of org.apache.hadoop.shaded.com.ressed bytes output so far.
   *
   * @return the total (non-negative) number of org.apache.hadoop.shaded.com.ressed bytes output so far
   */
  @Override
  public long getBytesWritten() {
    checkStream();
    return bytesWritten;
  }

  /**
   * <p>Returns the total number of uncompressed bytes input so far.</p>
   *
   * @return the total (non-negative) number of uncompressed bytes input so far
   */
  @Override
  public long getBytesRead() {
    checkStream();
    return bytesRead;
  }

  @Override
  public void reset() {
    checkStream();
    init(level, stream);
    finish = false;
    finished = false;
    bytesRead = 0;
    bytesWritten = 0;
    uncompressedDirectBuf.rewind();
    uncompressedDirectBufOff = 0;
    uncompressedDirectBufLen = 0;
    keepUncompressedBuf = false;
    org.apache.hadoop.shaded.com.ressedDirectBuf.limit(directBufferSize);
    org.apache.hadoop.shaded.com.ressedDirectBuf.position(directBufferSize);
    userBufOff = 0;
    userBufLen = 0;
  }

  @Override
  public void end() {
    if (stream != 0) {
      end(stream);
      stream = 0;
    }
  }

  private void checkStream() {
    if (stream == 0) {
      throw new NullPointerException();
    }
  }

  private native static long create();
  private native static void init(int level, long stream);
  private native int deflateBytesDirect(ByteBuffer src, int srcOffset,
      int srcLen, ByteBuffer dst, int dstLen);
  private native static int getStreamSize();
  private native static void end(long strm);
  private native static void initIDs();
  public native static String getLibraryName();
}
