/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.math.BigInteger;
import java.time.Duration;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.flink.util.Preconditions;

public class TimeUtils {
    private static final Map<String, ChronoUnit> LABEL_TO_UNIT_MAP = Collections.unmodifiableMap(TimeUtils.initMap());
    private static final BigInteger NANOS_PER_SECOND = BigInteger.valueOf(1000000000L);

    public static Duration parseDuration(String text) {
        BigInteger value;
        char current;
        int pos;
        Preconditions.checkNotNull(text);
        String trimmed = text.trim();
        Preconditions.checkArgument(!trimmed.isEmpty(), "argument is an empty- or whitespace-only string");
        int len = trimmed.length();
        for (pos = 0; pos < len && (current = trimmed.charAt(pos)) >= '0' && current <= '9'; ++pos) {
        }
        String number = trimmed.substring(0, pos);
        String unitLabel = trimmed.substring(pos).trim().toLowerCase(Locale.US);
        if (number.isEmpty()) {
            try {
                Duration parsedDuration = Duration.parse(trimmed);
                if (parsedDuration.isNegative()) {
                    throw new NumberFormatException("negative duration is not supported");
                }
                return parsedDuration;
            }
            catch (DateTimeParseException e) {
                throw new NumberFormatException("text does not start with a number, and is not a valid ISO-8601 duration format: " + trimmed);
            }
        }
        try {
            value = new BigInteger(number);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("The value '" + number + "' cannot be represented as an integer number.", e);
        }
        ChronoUnit unit = unitLabel.isEmpty() ? ChronoUnit.MILLIS : LABEL_TO_UNIT_MAP.get(unitLabel);
        if (unit == null) {
            throw new IllegalArgumentException("Time interval unit label '" + unitLabel + "' does not match any of the recognized units: " + TimeUnit.getAllUnits());
        }
        try {
            return TimeUtils.convertBigIntToDuration(value, unit);
        }
        catch (ArithmeticException e) {
            throw new IllegalArgumentException("The value '" + number + "' cannot be represented as Duration (numeric overflow).", e);
        }
    }

    private static Duration convertBigIntToDuration(BigInteger value, ChronoUnit unit) {
        BigInteger nanos = value.multiply(BigInteger.valueOf(unit.getDuration().toNanos()));
        BigInteger[] dividedAndRemainder = nanos.divideAndRemainder(NANOS_PER_SECOND);
        return Duration.ofSeconds(dividedAndRemainder[0].longValueExact()).plusNanos(dividedAndRemainder[1].longValueExact());
    }

    private static Map<String, ChronoUnit> initMap() {
        HashMap<String, ChronoUnit> labelToUnit = new HashMap<String, ChronoUnit>();
        for (TimeUnit timeUnit : TimeUnit.values()) {
            for (String label : timeUnit.getLabels()) {
                labelToUnit.put(label, timeUnit.getUnit());
            }
        }
        return labelToUnit;
    }

    public static String getStringInMillis(Duration duration) {
        return duration.toMillis() + TimeUnit.MILLISECONDS.labels.get(0);
    }

    public static String formatWithHighestUnit(Duration duration) {
        BigInteger nanos = TimeUtils.toNanos(duration);
        TimeUnit highestIntegerUnit = TimeUtils.getHighestIntegerUnit(nanos);
        return String.format("%s %s", nanos.divide(highestIntegerUnit.getUnitAsNanos()), highestIntegerUnit.getLabels().get(0));
    }

    private static BigInteger toNanos(Duration duration) {
        long tempSeconds = duration.getSeconds();
        long tempNanos = duration.getNano();
        if (tempSeconds < 0L) {
            ++tempSeconds;
            tempNanos -= NANOS_PER_SECOND.longValue();
        }
        return BigInteger.valueOf(tempSeconds).multiply(NANOS_PER_SECOND).add(BigInteger.valueOf(tempNanos));
    }

    private static TimeUnit getHighestIntegerUnit(BigInteger nanos) {
        if (nanos.compareTo(BigInteger.ZERO) == 0) {
            return TimeUnit.MILLISECONDS;
        }
        List<TimeUnit> orderedUnits = Arrays.asList(TimeUnit.NANOSECONDS, TimeUnit.MICROSECONDS, TimeUnit.MILLISECONDS, TimeUnit.SECONDS, TimeUnit.MINUTES, TimeUnit.HOURS, TimeUnit.DAYS);
        TimeUnit highestIntegerUnit = null;
        for (TimeUnit timeUnit : orderedUnits) {
            if (nanos.remainder(timeUnit.getUnitAsNanos()).compareTo(BigInteger.ZERO) != 0) break;
            highestIntegerUnit = timeUnit;
        }
        return Preconditions.checkNotNull(highestIntegerUnit, "Should find a highestIntegerUnit.");
    }

    private static enum TimeUnit {
        DAYS(ChronoUnit.DAYS, TimeUnit.singular("d"), TimeUnit.plural("day")),
        HOURS(ChronoUnit.HOURS, TimeUnit.singular("h"), TimeUnit.plural("hour")),
        MINUTES(ChronoUnit.MINUTES, TimeUnit.singular("min"), TimeUnit.singular("m"), TimeUnit.plural("minute")),
        SECONDS(ChronoUnit.SECONDS, TimeUnit.singular("s"), TimeUnit.plural("sec"), TimeUnit.plural("second")),
        MILLISECONDS(ChronoUnit.MILLIS, TimeUnit.singular("ms"), TimeUnit.plural("milli"), TimeUnit.plural("millisecond")),
        MICROSECONDS(ChronoUnit.MICROS, TimeUnit.singular("\u00b5s"), TimeUnit.plural("micro"), TimeUnit.plural("microsecond")),
        NANOSECONDS(ChronoUnit.NANOS, TimeUnit.singular("ns"), TimeUnit.plural("nano"), TimeUnit.plural("nanosecond"));

        private static final String PLURAL_SUFFIX = "s";
        private final List<String> labels;
        private final ChronoUnit unit;
        private final BigInteger unitAsNanos;

        private TimeUnit(ChronoUnit unit, String[] ... labels) {
            this.unit = unit;
            this.unitAsNanos = BigInteger.valueOf(unit.getDuration().toNanos());
            this.labels = Arrays.stream(labels).flatMap(Arrays::stream).collect(Collectors.toList());
        }

        private static String[] singular(String label) {
            return new String[]{label};
        }

        private static String[] plural(String label) {
            return new String[]{label, label + PLURAL_SUFFIX};
        }

        public List<String> getLabels() {
            return this.labels;
        }

        public ChronoUnit getUnit() {
            return this.unit;
        }

        public BigInteger getUnitAsNanos() {
            return this.unitAsNanos;
        }

        public static String getAllUnits() {
            return Arrays.stream(TimeUnit.values()).map(TimeUnit::createTimeUnitString).collect(Collectors.joining(", "));
        }

        private static String createTimeUnitString(TimeUnit timeUnit) {
            return timeUnit.name() + ": (" + String.join((CharSequence)" | ", timeUnit.getLabels()) + ")";
        }
    }
}

