/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.netty;

import java.io.IOException;
import java.net.BindException;
import java.net.InetSocketAddress;
import java.util.Iterator;
import java.util.concurrent.ThreadFactory;
import java.util.function.Function;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.runtime.io.network.netty.NettyBufferPool;
import org.apache.flink.runtime.io.network.netty.NettyConfig;
import org.apache.flink.runtime.io.network.netty.NettyProtocol;
import org.apache.flink.runtime.io.network.netty.SSLHandlerFactory;
import org.apache.flink.shaded.guava32.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.flink.shaded.netty4.io.netty.bootstrap.ServerBootstrap;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelFuture;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelHandler;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelInitializer;
import org.apache.flink.shaded.netty4.io.netty.channel.ChannelOption;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.Epoll;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.EpollEventLoopGroup;
import org.apache.flink.shaded.netty4.io.netty.channel.epoll.EpollServerSocketChannel;
import org.apache.flink.shaded.netty4.io.netty.channel.nio.NioEventLoopGroup;
import org.apache.flink.shaded.netty4.io.netty.channel.socket.SocketChannel;
import org.apache.flink.shaded.netty4.io.netty.channel.socket.nio.NioServerSocketChannel;
import org.apache.flink.shaded.netty4.io.netty.channel.unix.Errors;
import org.apache.flink.util.FatalExitExceptionHandler;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NettyServer {
    private static final ThreadFactoryBuilder THREAD_FACTORY_BUILDER = new ThreadFactoryBuilder().setDaemon(true).setUncaughtExceptionHandler(FatalExitExceptionHandler.INSTANCE);
    private static final Logger LOG = LoggerFactory.getLogger(NettyServer.class);
    private final NettyConfig config;
    private ServerBootstrap bootstrap;
    private ChannelFuture bindFuture;
    private InetSocketAddress localAddress;

    NettyServer(NettyConfig config) {
        this.config = Preconditions.checkNotNull(config);
        this.localAddress = null;
    }

    int init(NettyProtocol protocol, NettyBufferPool nettyBufferPool) throws IOException {
        return this.init(nettyBufferPool, (SSLHandlerFactory sslHandlerFactory) -> new ServerChannelInitializer(protocol, (SSLHandlerFactory)sslHandlerFactory));
    }

    int init(NettyBufferPool nettyBufferPool, Function<SSLHandlerFactory, ServerChannelInitializer> channelInitializer) throws IOException {
        SSLHandlerFactory sslHandlerFactory;
        int receiveAndSendBufferSize;
        Preconditions.checkState(this.bootstrap == null, "Netty server has already been initialized.");
        long start = System.nanoTime();
        this.bootstrap = new ServerBootstrap();
        if (Epoll.isAvailable()) {
            this.initEpollBootstrap();
            LOG.info("Transport type 'auto': using EPOLL.");
        } else {
            this.initNioBootstrap();
            LOG.info("Transport type 'auto': using NIO.");
        }
        this.bootstrap.option(ChannelOption.ALLOCATOR, nettyBufferPool);
        this.bootstrap.childOption(ChannelOption.ALLOCATOR, nettyBufferPool);
        if (this.config.getServerConnectBacklog() > 0) {
            this.bootstrap.option(ChannelOption.SO_BACKLOG, this.config.getServerConnectBacklog());
        }
        if ((receiveAndSendBufferSize = this.config.getSendAndReceiveBufferSize()) > 0) {
            this.bootstrap.childOption(ChannelOption.SO_SNDBUF, receiveAndSendBufferSize);
            this.bootstrap.childOption(ChannelOption.SO_RCVBUF, receiveAndSendBufferSize);
        }
        try {
            sslHandlerFactory = this.config.createServerSSLEngineFactory();
        }
        catch (Exception e) {
            throw new IOException("Failed to initialize SSL Context for the Netty Server", e);
        }
        this.bootstrap.childHandler(channelInitializer.apply(sslHandlerFactory));
        LOG.debug("Trying to initialize Netty server on address: {} and port range {}", (Object)this.config.getServerAddress(), (Object)this.config.getServerPortRange());
        Iterator<Integer> portsIterator = this.config.getServerPortRange().getPortsIterator();
        while (portsIterator.hasNext() && this.bindFuture == null) {
            Integer port = portsIterator.next();
            LOG.debug("Trying to bind Netty server to port: {}", (Object)port);
            this.bootstrap.localAddress(this.config.getServerAddress(), (int)port);
            try {
                this.bindFuture = this.bootstrap.bind().syncUninterruptibly();
            }
            catch (Exception e) {
                if (NettyServer.isBindFailure(e)) {
                    LOG.debug("Failed to bind Netty server", (Throwable)e);
                    continue;
                }
                throw e;
            }
        }
        if (this.bindFuture == null) {
            throw new BindException("Could not start rest endpoint on any port in port range " + this.config.getServerPortRange());
        }
        this.localAddress = (InetSocketAddress)this.bindFuture.channel().localAddress();
        long duration = (System.nanoTime() - start) / 1000000L;
        LOG.info("Successful initialization (took {} ms). Listening on SocketAddress {}.", (Object)duration, (Object)this.localAddress);
        return this.localAddress.getPort();
    }

    NettyConfig getConfig() {
        return this.config;
    }

    ServerBootstrap getBootstrap() {
        return this.bootstrap;
    }

    Integer getListeningPort() {
        return this.localAddress == null ? null : Integer.valueOf(this.localAddress.getPort());
    }

    void shutdown() {
        long start = System.nanoTime();
        if (this.bindFuture != null) {
            this.bindFuture.channel().close().awaitUninterruptibly();
            this.bindFuture = null;
        }
        if (this.bootstrap != null) {
            if (this.bootstrap.config().group() != null) {
                this.bootstrap.config().group().shutdownGracefully();
            }
            this.bootstrap = null;
        }
        long duration = (System.nanoTime() - start) / 1000000L;
        LOG.info("Successful shutdown (took {} ms).", (Object)duration);
    }

    private void initNioBootstrap() {
        String name = "Flink Netty Server (" + this.config.getServerPortRange() + ")";
        NioEventLoopGroup nioGroup = new NioEventLoopGroup(this.config.getServerNumThreads(), NettyServer.getNamedThreadFactory(name));
        this.bootstrap.group(nioGroup).channel(NioServerSocketChannel.class);
    }

    private void initEpollBootstrap() {
        String name = "Flink Netty Server (" + this.config.getServerPortRange() + ")";
        EpollEventLoopGroup epollGroup = new EpollEventLoopGroup(this.config.getServerNumThreads(), NettyServer.getNamedThreadFactory(name));
        this.bootstrap.group(epollGroup).channel(EpollServerSocketChannel.class);
    }

    public static ThreadFactory getNamedThreadFactory(String name) {
        return THREAD_FACTORY_BUILDER.setNameFormat(name + " Thread %d").build();
    }

    @VisibleForTesting
    static boolean isBindFailure(Throwable t) {
        return t instanceof BindException || t instanceof Errors.NativeIoException && t.getMessage() != null && t.getMessage().matches("^bind\\(.*\\) failed:.*") || t.getCause() != null && NettyServer.isBindFailure(t.getCause());
    }

    @VisibleForTesting
    static class ServerChannelInitializer
    extends ChannelInitializer<SocketChannel> {
        private final NettyProtocol protocol;
        private final SSLHandlerFactory sslHandlerFactory;

        public ServerChannelInitializer(NettyProtocol protocol, SSLHandlerFactory sslHandlerFactory) {
            this.protocol = protocol;
            this.sslHandlerFactory = sslHandlerFactory;
        }

        @Override
        public void initChannel(SocketChannel channel) throws Exception {
            if (this.sslHandlerFactory != null) {
                channel.pipeline().addLast("ssl", (ChannelHandler)this.sslHandlerFactory.createNettySSLHandler(channel.alloc()));
            }
            channel.pipeline().addLast(this.protocol.getServerChannelHandlers());
        }
    }
}

