/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.flink.annotation.Internal;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.OperatingSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class CompressionUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CompressionUtils.class);

    public static void extractFile(String srcFilePath, String targetDirPath, String originalFileName) throws IOException {
        if (CompressionUtils.hasOneOfSuffixes(originalFileName, ".zip", ".jar")) {
            CompressionUtils.extractZipFileWithPermissions(srcFilePath, targetDirPath);
        } else if (CompressionUtils.hasOneOfSuffixes(originalFileName, ".tar", ".tar.gz", ".tgz")) {
            CompressionUtils.extractTarFile(srcFilePath, targetDirPath);
        } else {
            LOG.warn("Only zip, jar, tar, tgz and tar.gz suffixes are supported, found {}. Trying to extract it as zip file.", (Object)originalFileName);
            CompressionUtils.extractZipFileWithPermissions(srcFilePath, targetDirPath);
        }
    }

    public static void extractTarFile(String inFilePath, String targetDirPath) throws IOException {
        File targetDir = new File(targetDirPath);
        if (!targetDir.mkdirs() && !targetDir.isDirectory()) {
            throw new IOException("Mkdirs failed to create " + targetDir);
        }
        boolean gzipped = inFilePath.endsWith("gz");
        if (CompressionUtils.isUnix()) {
            CompressionUtils.extractTarFileUsingTar(inFilePath, targetDirPath, gzipped);
        } else {
            CompressionUtils.extractTarFileUsingJava(inFilePath, targetDirPath, gzipped);
        }
    }

    private static void extractTarFileUsingTar(String inFilePath, String targetDirPath, boolean gzipped) throws IOException {
        inFilePath = CompressionUtils.makeSecureShellPath(inFilePath);
        targetDirPath = CompressionUtils.makeSecureShellPath(targetDirPath);
        String untarCommand = gzipped ? String.format("gzip -dc '%s' | (cd '%s' && tar -xf -)", inFilePath, targetDirPath) : String.format("cd '%s' && tar -xf '%s'", targetDirPath, inFilePath);
        Process process = new ProcessBuilder("bash", "-c", untarCommand).start();
        int exitCode = 0;
        try {
            exitCode = process.waitFor();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted when untarring file " + inFilePath);
        }
        if (exitCode != 0) {
            throw new IOException("Error untarring file " + inFilePath + ". Tar process exited with exit code " + exitCode);
        }
    }

    private static void extractTarFileUsingJava(String inFilePath, String targetDirPath, boolean gzipped) throws IOException {
        try (InputStream fi = Files.newInputStream(Paths.get(inFilePath, new String[0]), new OpenOption[0]);
             BufferedInputStream bi = new BufferedInputStream(fi);
             TarArchiveInputStream tai = new TarArchiveInputStream((InputStream)(gzipped ? new GzipCompressorInputStream((InputStream)bi) : bi));){
            TarArchiveEntry entry;
            File targetDir = new File(targetDirPath);
            while ((entry = tai.getNextTarEntry()) != null) {
                CompressionUtils.unpackEntry(tai, entry, targetDir);
            }
        }
    }

    private static void unpackEntry(TarArchiveInputStream tis, TarArchiveEntry entry, File targetDir) throws IOException {
        String targetDirPath = targetDir.getCanonicalPath() + File.separator;
        File outputFile = new File(targetDir, entry.getName());
        if (!outputFile.getCanonicalPath().startsWith(targetDirPath)) {
            throw new IOException("expanding " + entry.getName() + " would create entry outside of " + targetDir);
        }
        if (entry.isDirectory()) {
            if (!outputFile.mkdirs() && !outputFile.isDirectory()) {
                throw new IOException("Failed to create directory " + outputFile);
            }
            for (TarArchiveEntry e : entry.getDirectoryEntries()) {
                CompressionUtils.unpackEntry(tis, e, outputFile);
            }
            return;
        }
        if (entry.isSymbolicLink()) {
            Files.createSymbolicLink(Paths.get(new File(targetDir, entry.getName()).getCanonicalPath(), new String[0]), Paths.get(entry.getLinkName(), new String[0]), new FileAttribute[0]);
            return;
        }
        if (!outputFile.getParentFile().exists() && !outputFile.getParentFile().mkdirs()) {
            throw new IOException("Mkdirs failed to create tar internal dir " + targetDir);
        }
        try (OutputStream o = Files.newOutputStream(Paths.get(outputFile.getCanonicalPath(), new String[0]), new OpenOption[0]);){
            IOUtils.copyBytes((InputStream)tis, o, false);
        }
    }

    private static String makeSecureShellPath(String filePath) {
        return filePath.replace("'", "'\\''");
    }

    public static void extractZipFileWithPermissions(String zipFilePath, String targetPath) throws IOException {
        try (ZipFile zipFile = new ZipFile(zipFilePath);){
            Enumeration entries = zipFile.getEntries();
            boolean isUnix = CompressionUtils.isUnix();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            String canonicalTargetPath = new File(targetPath).getCanonicalPath() + File.separator;
            while (entries.hasMoreElements()) {
                int mode;
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                File outputFile = new File(canonicalTargetPath, entry.getName());
                if (!outputFile.getCanonicalPath().startsWith(canonicalTargetPath)) {
                    throw new IOException("Expand " + entry.getName() + " would create a file outside of " + targetPath);
                }
                if (entry.isDirectory()) {
                    if (!outputFile.exists() && !outputFile.mkdirs()) {
                        throw new IOException("Create dir: " + outputFile.getAbsolutePath() + " failed!");
                    }
                } else {
                    File parentDir = outputFile.getParentFile();
                    if (!parentDir.exists() && !parentDir.mkdirs()) {
                        throw new IOException("Create dir: " + outputFile.getAbsolutePath() + " failed!");
                    }
                    if (entry.isUnixSymlink()) {
                        baos.reset();
                        IOUtils.copyBytes(zipFile.getInputStream(entry), baos);
                        Files.createSymbolicLink(outputFile.toPath(), new File(parentDir, baos.toString()).toPath(), new FileAttribute[0]);
                    } else if (outputFile.createNewFile()) {
                        FileOutputStream output = new FileOutputStream(outputFile);
                        IOUtils.copyBytes(zipFile.getInputStream(entry), output);
                    } else {
                        throw new IOException("Create file: " + outputFile.getAbsolutePath() + " failed!");
                    }
                }
                if (!isUnix || (mode = entry.getUnixMode()) == 0) continue;
                Path outputPath = Paths.get(outputFile.toURI());
                HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>();
                CompressionUtils.addIfBitSet(mode, 8, permissions, PosixFilePermission.OWNER_READ);
                CompressionUtils.addIfBitSet(mode, 7, permissions, PosixFilePermission.OWNER_WRITE);
                CompressionUtils.addIfBitSet(mode, 6, permissions, PosixFilePermission.OWNER_EXECUTE);
                CompressionUtils.addIfBitSet(mode, 5, permissions, PosixFilePermission.GROUP_READ);
                CompressionUtils.addIfBitSet(mode, 4, permissions, PosixFilePermission.GROUP_WRITE);
                CompressionUtils.addIfBitSet(mode, 3, permissions, PosixFilePermission.GROUP_EXECUTE);
                CompressionUtils.addIfBitSet(mode, 2, permissions, PosixFilePermission.OTHERS_READ);
                CompressionUtils.addIfBitSet(mode, 1, permissions, PosixFilePermission.OTHERS_WRITE);
                CompressionUtils.addIfBitSet(mode, 0, permissions, PosixFilePermission.OTHERS_EXECUTE);
                try {
                    Files.setPosixFilePermissions(outputPath, permissions);
                }
                catch (NoSuchFileException noSuchFileException) {}
            }
        }
    }

    private static boolean isUnix() {
        return OperatingSystem.isLinux() || OperatingSystem.isMac() || OperatingSystem.isFreeBSD() || OperatingSystem.isSolaris();
    }

    private static void addIfBitSet(int mode, int pos, Set<PosixFilePermission> posixFilePermissions, PosixFilePermission posixFilePermissionToAdd) {
        if (((long)mode & 1L << pos) != 0L) {
            posixFilePermissions.add(posixFilePermissionToAdd);
        }
    }

    private static boolean hasOneOfSuffixes(String filePath, String ... suffixes) {
        String lowercaseFilePath = filePath.toLowerCase();
        for (String suffix : suffixes) {
            if (!lowercaseFilePath.endsWith(suffix)) continue;
            return true;
        }
        return false;
    }
}

