/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.metrics.util;

import com.sun.management.OperatingSystemMXBean;
import java.lang.management.ClassLoadingMXBean;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;
import java.lang.management.ThreadMXBean;
import java.time.Duration;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.MetricOptions;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.Meter;
import org.apache.flink.metrics.MeterView;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.memory.MemoryManager;
import org.apache.flink.runtime.metrics.MetricRegistry;
import org.apache.flink.runtime.metrics.groups.AbstractMetricGroup;
import org.apache.flink.runtime.metrics.groups.ProcessMetricGroup;
import org.apache.flink.runtime.metrics.groups.TaskManagerMetricGroup;
import org.apache.flink.runtime.metrics.util.SystemResourcesMetricsInitializer;
import org.apache.flink.runtime.rpc.RpcService;
import org.apache.flink.runtime.rpc.RpcSystem;
import org.apache.flink.runtime.taskexecutor.slot.SlotNotFoundException;
import org.apache.flink.runtime.taskexecutor.slot.TaskSlotTable;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricUtils {
    private static final Logger LOG = LoggerFactory.getLogger(MetricUtils.class);
    private static final String METRIC_GROUP_STATUS_NAME = "Status";
    private static final String METRICS_ACTOR_SYSTEM_NAME = "flink-metrics";
    static final String METRIC_GROUP_HEAP_NAME = "Heap";
    static final String METRIC_GROUP_NONHEAP_NAME = "NonHeap";
    static final String METRIC_GROUP_METASPACE_NAME = "Metaspace";
    @VisibleForTesting
    static final String METRIC_GROUP_FLINK = "Flink";
    @VisibleForTesting
    static final String METRIC_GROUP_MEMORY = "Memory";
    @VisibleForTesting
    static final String METRIC_GROUP_MANAGED_MEMORY = "Managed";
    private static final String WRITER_SUFFIX = ": Writer";
    private static final String COMMITTER_SUFFIX = ": Committer";

    private MetricUtils() {
    }

    public static ProcessMetricGroup instantiateProcessMetricGroup(MetricRegistry metricRegistry, String hostname, Optional<Duration> systemResourceProbeInterval) {
        ProcessMetricGroup processMetricGroup = ProcessMetricGroup.create(metricRegistry, hostname);
        MetricUtils.createAndInitializeStatusMetricGroup(processMetricGroup);
        systemResourceProbeInterval.ifPresent(interval -> SystemResourcesMetricsInitializer.instantiateSystemMetrics(processMetricGroup, interval));
        return processMetricGroup;
    }

    public static Tuple2<TaskManagerMetricGroup, MetricGroup> instantiateTaskManagerMetricGroup(MetricRegistry metricRegistry, String hostName, ResourceID resourceID, Optional<Duration> systemResourceProbeInterval) {
        TaskManagerMetricGroup taskManagerMetricGroup = TaskManagerMetricGroup.createTaskManagerMetricGroup(metricRegistry, hostName, resourceID);
        MetricGroup statusGroup = MetricUtils.createAndInitializeStatusMetricGroup(taskManagerMetricGroup);
        if (systemResourceProbeInterval.isPresent()) {
            SystemResourcesMetricsInitializer.instantiateSystemMetrics(taskManagerMetricGroup, systemResourceProbeInterval.get());
        }
        return Tuple2.of((Object)taskManagerMetricGroup, (Object)statusGroup);
    }

    private static MetricGroup createAndInitializeStatusMetricGroup(AbstractMetricGroup<?> parentMetricGroup) {
        MetricGroup statusGroup = parentMetricGroup.addGroup(METRIC_GROUP_STATUS_NAME);
        MetricUtils.instantiateStatusMetrics(statusGroup);
        return statusGroup;
    }

    public static void instantiateStatusMetrics(MetricGroup metricGroup) {
        MetricGroup jvm = metricGroup.addGroup("JVM");
        MetricUtils.instantiateClassLoaderMetrics(jvm.addGroup("ClassLoader"));
        MetricUtils.instantiateGarbageCollectorMetrics(jvm.addGroup("GarbageCollector"), ManagementFactory.getGarbageCollectorMXBeans());
        MetricUtils.instantiateMemoryMetrics(jvm.addGroup(METRIC_GROUP_MEMORY));
        MetricUtils.instantiateThreadMetrics(jvm.addGroup("Threads"));
        MetricUtils.instantiateCPUMetrics(jvm.addGroup("CPU"));
    }

    public static void instantiateFlinkMemoryMetricGroup(MetricGroup parentMetricGroup, TaskSlotTable<?> taskSlotTable, Supplier<Long> managedMemoryTotalSupplier) {
        Preconditions.checkNotNull((Object)parentMetricGroup);
        Preconditions.checkNotNull(taskSlotTable);
        Preconditions.checkNotNull(managedMemoryTotalSupplier);
        MetricGroup flinkMemoryMetricGroup = parentMetricGroup.addGroup(METRIC_GROUP_FLINK).addGroup(METRIC_GROUP_MEMORY);
        MetricUtils.instantiateManagedMemoryMetrics(flinkMemoryMetricGroup, taskSlotTable, managedMemoryTotalSupplier);
    }

    private static void instantiateManagedMemoryMetrics(MetricGroup metricGroup, TaskSlotTable<?> taskSlotTable, Supplier<Long> managedMemoryTotalSupplier) {
        MetricGroup managedMemoryMetricGroup = metricGroup.addGroup(METRIC_GROUP_MANAGED_MEMORY);
        managedMemoryMetricGroup.gauge("Used", () -> MetricUtils.getUsedManagedMemory(taskSlotTable));
        managedMemoryMetricGroup.gauge("Total", managedMemoryTotalSupplier::get);
    }

    private static long getUsedManagedMemory(TaskSlotTable<?> taskSlotTable) {
        Set<AllocationID> activeTaskAllocationIds = taskSlotTable.getActiveTaskSlotAllocationIds();
        long usedMemory = 0L;
        for (AllocationID allocationID : activeTaskAllocationIds) {
            try {
                MemoryManager taskSlotMemoryManager = taskSlotTable.getTaskMemoryManager(allocationID);
                usedMemory += taskSlotMemoryManager.getMemorySize() - taskSlotMemoryManager.availableMemory();
            }
            catch (SlotNotFoundException e) {
                LOG.debug("The task slot {} is not present anymore and will be ignored in calculating the amount of used memory.", (Object)allocationID);
            }
        }
        return usedMemory;
    }

    public static RpcService startRemoteMetricsRpcService(Configuration configuration, String externalAddress, @Nullable String bindAddress, RpcSystem rpcSystem) throws Exception {
        String portRange = (String)configuration.get(MetricOptions.QUERY_SERVICE_PORT);
        RpcSystem.RpcServiceBuilder rpcServiceBuilder = rpcSystem.remoteServiceBuilder(configuration, externalAddress, portRange);
        if (bindAddress != null) {
            rpcServiceBuilder.withBindAddress(bindAddress);
        }
        return MetricUtils.startMetricRpcService(configuration, rpcServiceBuilder);
    }

    public static RpcService startLocalMetricsRpcService(Configuration configuration, RpcSystem rpcSystem) throws Exception {
        return MetricUtils.startMetricRpcService(configuration, rpcSystem.localServiceBuilder(configuration));
    }

    private static RpcService startMetricRpcService(Configuration configuration, RpcSystem.RpcServiceBuilder rpcServiceBuilder) throws Exception {
        int threadPriority = (Integer)configuration.get(MetricOptions.QUERY_SERVICE_THREAD_PRIORITY);
        return rpcServiceBuilder.withComponentName(METRICS_ACTOR_SYSTEM_NAME).withExecutorConfiguration(new RpcSystem.FixedThreadPoolExecutorConfiguration(1, 1, threadPriority)).createAndStart();
    }

    private static void instantiateClassLoaderMetrics(MetricGroup metrics) {
        ClassLoadingMXBean mxBean = ManagementFactory.getClassLoadingMXBean();
        metrics.gauge("ClassesLoaded", mxBean::getTotalLoadedClassCount);
        metrics.gauge("ClassesUnloaded", mxBean::getUnloadedClassCount);
    }

    @VisibleForTesting
    static void instantiateGarbageCollectorMetrics(MetricGroup metrics, List<GarbageCollectorMXBean> garbageCollectors) {
        for (GarbageCollectorMXBean garbageCollector : garbageCollectors) {
            MetricGroup gcGroup = metrics.addGroup(garbageCollector.getName());
            gcGroup.gauge("Count", garbageCollector::getCollectionCount);
            Gauge timeGauge = gcGroup.gauge("Time", garbageCollector::getCollectionTime);
            gcGroup.meter("TimeMsPerSecond", (Meter)new MeterView(timeGauge));
        }
        Gauge totalGcTime = () -> garbageCollectors.stream().mapToLong(GarbageCollectorMXBean::getCollectionTime).sum();
        Gauge totalGcCount = () -> garbageCollectors.stream().mapToLong(GarbageCollectorMXBean::getCollectionCount).sum();
        MetricGroup allGroup = metrics.addGroup("All");
        allGroup.gauge("Count", totalGcCount);
        Gauge totalTime = allGroup.gauge("Time", totalGcTime);
        allGroup.meter("TimeMsPerSecond", (Meter)new MeterView(totalTime));
    }

    private static void instantiateMemoryMetrics(MetricGroup metrics) {
        MetricUtils.instantiateHeapMemoryMetrics(metrics.addGroup(METRIC_GROUP_HEAP_NAME));
        MetricUtils.instantiateNonHeapMemoryMetrics(metrics.addGroup(METRIC_GROUP_NONHEAP_NAME));
        MetricUtils.instantiateMetaspaceMemoryMetrics(metrics);
        MBeanServer con = ManagementFactory.getPlatformMBeanServer();
        String directBufferPoolName = "java.nio:type=BufferPool,name=direct";
        try {
            ObjectName directObjectName = new ObjectName("java.nio:type=BufferPool,name=direct");
            MetricGroup direct = metrics.addGroup("Direct");
            direct.gauge("Count", new AttributeGauge<Long>(con, directObjectName, "Count", -1L));
            direct.gauge("MemoryUsed", new AttributeGauge<Long>(con, directObjectName, "MemoryUsed", -1L));
            direct.gauge("TotalCapacity", new AttributeGauge<Long>(con, directObjectName, "TotalCapacity", -1L));
        }
        catch (MalformedObjectNameException e) {
            LOG.warn("Could not create object name {}.", (Object)"java.nio:type=BufferPool,name=direct", (Object)e);
        }
        String mappedBufferPoolName = "java.nio:type=BufferPool,name=mapped";
        try {
            ObjectName mappedObjectName = new ObjectName("java.nio:type=BufferPool,name=mapped");
            MetricGroup mapped = metrics.addGroup("Mapped");
            mapped.gauge("Count", new AttributeGauge<Long>(con, mappedObjectName, "Count", -1L));
            mapped.gauge("MemoryUsed", new AttributeGauge<Long>(con, mappedObjectName, "MemoryUsed", -1L));
            mapped.gauge("TotalCapacity", new AttributeGauge<Long>(con, mappedObjectName, "TotalCapacity", -1L));
        }
        catch (MalformedObjectNameException e) {
            LOG.warn("Could not create object name {}.", (Object)"java.nio:type=BufferPool,name=mapped", (Object)e);
        }
    }

    @VisibleForTesting
    static void instantiateHeapMemoryMetrics(MetricGroup metricGroup) {
        MetricUtils.instantiateMemoryUsageMetrics(metricGroup, () -> ManagementFactory.getMemoryMXBean().getHeapMemoryUsage());
    }

    @VisibleForTesting
    static void instantiateNonHeapMemoryMetrics(MetricGroup metricGroup) {
        MetricUtils.instantiateMemoryUsageMetrics(metricGroup, () -> ManagementFactory.getMemoryMXBean().getNonHeapMemoryUsage());
    }

    @VisibleForTesting
    static void instantiateMetaspaceMemoryMetrics(MetricGroup parentMetricGroup) {
        List memoryPoolMXBeans = ManagementFactory.getMemoryPoolMXBeans().stream().filter(bean -> METRIC_GROUP_METASPACE_NAME.equals(bean.getName())).collect(Collectors.toList());
        if (memoryPoolMXBeans.isEmpty()) {
            LOG.info("The '{}' metrics will not be exposed because no pool named 'Metaspace' could be found. This might be caused by the used JVM.", (Object)METRIC_GROUP_METASPACE_NAME);
            return;
        }
        MetricGroup metricGroup = parentMetricGroup.addGroup(METRIC_GROUP_METASPACE_NAME);
        Iterator beanIterator = memoryPoolMXBeans.iterator();
        MemoryPoolMXBean firstPool = (MemoryPoolMXBean)beanIterator.next();
        MetricUtils.instantiateMemoryUsageMetrics(metricGroup, firstPool::getUsage);
        if (beanIterator.hasNext()) {
            LOG.debug("More than one memory pool named 'Metaspace' is present. Only the first pool was used for instantiating the '{}' metrics.", (Object)METRIC_GROUP_METASPACE_NAME);
        }
    }

    private static void instantiateMemoryUsageMetrics(MetricGroup metricGroup, Supplier<MemoryUsage> memoryUsageSupplier) {
        metricGroup.gauge("Used", () -> ((MemoryUsage)memoryUsageSupplier.get()).getUsed());
        metricGroup.gauge("Committed", () -> ((MemoryUsage)memoryUsageSupplier.get()).getCommitted());
        metricGroup.gauge("Max", () -> ((MemoryUsage)memoryUsageSupplier.get()).getMax());
    }

    private static void instantiateThreadMetrics(MetricGroup metrics) {
        ThreadMXBean mxBean = ManagementFactory.getThreadMXBean();
        metrics.gauge("Count", mxBean::getThreadCount);
    }

    private static void instantiateCPUMetrics(MetricGroup metrics) {
        try {
            OperatingSystemMXBean mxBean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
            metrics.gauge("Load", mxBean::getProcessCpuLoad);
            metrics.gauge("Time", mxBean::getProcessCpuTime);
        }
        catch (Exception e) {
            LOG.warn("Cannot access com.sun.management.OperatingSystemMXBean.getProcessCpuLoad() - CPU load metrics will not be available.", (Throwable)e);
        }
    }

    public static String truncateOperatorName(String operatorName) {
        if (operatorName != null && operatorName.length() > 80) {
            LOG.warn("The operator name {} exceeded the {} characters length limit and was truncated.", (Object)operatorName, (Object)80);
            if (operatorName.endsWith(WRITER_SUFFIX)) {
                return operatorName.substring(0, Math.max(0, 80 - WRITER_SUFFIX.length())) + WRITER_SUFFIX;
            }
            if (operatorName.endsWith(COMMITTER_SUFFIX)) {
                return operatorName.substring(0, Math.max(0, 80 - COMMITTER_SUFFIX.length())) + COMMITTER_SUFFIX;
            }
            return operatorName.substring(0, 80);
        }
        return operatorName;
    }

    private static final class AttributeGauge<T>
    implements Gauge<T> {
        private final MBeanServer server;
        private final ObjectName objectName;
        private final String attributeName;
        private final T errorValue;

        private AttributeGauge(MBeanServer server, ObjectName objectName, String attributeName, T errorValue) {
            this.server = (MBeanServer)Preconditions.checkNotNull((Object)server);
            this.objectName = (ObjectName)Preconditions.checkNotNull((Object)objectName);
            this.attributeName = (String)Preconditions.checkNotNull((Object)attributeName);
            this.errorValue = errorValue;
        }

        public T getValue() {
            try {
                return (T)this.server.getAttribute(this.objectName, this.attributeName);
            }
            catch (AttributeNotFoundException | InstanceNotFoundException | MBeanException | ReflectionException e) {
                LOG.warn("Could not read attribute {}.", (Object)this.attributeName, (Object)e);
                return this.errorValue;
            }
        }
    }
}

