/*****************************************************************
 *   Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 ****************************************************************/
/* ====================================================================
 *
 * Copyright(c) 2003, Andriy Shapochka
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of the ASHWOOD nor the
 *    names of its contributors may be used to endorse or
 *    promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ====================================================================
 *
 * This software consists of voluntary contributions made by
 * individuals on behalf of the ASHWOOD Project and was originally
 * created by Andriy Shapochka.
 *
 */
package org.apache.cayenne.ashwood.graph;

import java.util.Iterator;
import java.util.function.Predicate;
import java.util.stream.StreamSupport;

/**
 * @since 3.1
 */
public class FilterIteration<E, V> implements DigraphIteration<E, V> {

    private DigraphIteration<E, V> digraph;
    private Predicate<E> acceptVertex;
    private Predicate<V> acceptArc;

    public FilterIteration(DigraphIteration<E, V> digraph, Predicate<E> acceptVertex,
            Predicate<V> acceptArc) {
        this.digraph = digraph;
        this.acceptVertex = acceptVertex;
        this.acceptArc = acceptArc;
    }

    public Iterator<E> vertexIterator() {
        Iterable<E> iterable = () -> digraph.vertexIterator();
        return StreamSupport.stream(iterable.spliterator(), false).filter(acceptVertex).iterator();
    }

    public ArcIterator<E, V> arcIterator() {
        return new FilterArcIterator<>(
                digraph.arcIterator(),
                acceptVertex,
                acceptVertex,
                acceptArc);
    }

    public ArcIterator<E, V> outgoingIterator(E vertex) {
        if (!acceptVertex.test(vertex)) {
            return EmptyIterator.instance();
        }
        return new FilterArcIterator<>(
                digraph.outgoingIterator(vertex),
                v -> true,
                acceptVertex,
                acceptArc);
    }

    public ArcIterator<E, V> incomingIterator(E vertex) {
        if (!acceptVertex.test(vertex)) {
            return EmptyIterator.instance();
        }
        return new FilterArcIterator<>(
                digraph.incomingIterator(vertex),
                acceptVertex,
                v -> true,
                acceptArc);
    }
}
