/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.gcp.bigtable;

import com.google.bigtable.v1.Mutation;
import com.google.bigtable.v1.Row;
import com.google.bigtable.v1.RowFilter;
import com.google.bigtable.v1.SampleRowKeysResponse;
import com.google.cloud.bigtable.config.BigtableOptions;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.protobuf.ByteString;
import com.google.protobuf.Empty;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentLinkedQueue;
import javax.annotation.Nullable;
import org.apache.beam.sdk.annotations.Experimental;
import org.apache.beam.sdk.coders.Coder;
import org.apache.beam.sdk.coders.VarLongCoder;
import org.apache.beam.sdk.coders.protobuf.ProtoCoder;
import org.apache.beam.sdk.io.BoundedSource;
import org.apache.beam.sdk.io.Sink;
import org.apache.beam.sdk.io.gcp.bigtable.BigtableService;
import org.apache.beam.sdk.io.gcp.bigtable.BigtableServiceImpl;
import org.apache.beam.sdk.io.range.ByteKey;
import org.apache.beam.sdk.io.range.ByteKeyRange;
import org.apache.beam.sdk.io.range.ByteKeyRangeTracker;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.transforms.PTransform;
import org.apache.beam.sdk.transforms.display.DisplayData;
import org.apache.beam.sdk.util.ReleaseInfo;
import org.apache.beam.sdk.values.KV;
import org.apache.beam.sdk.values.PBegin;
import org.apache.beam.sdk.values.PCollection;
import org.apache.beam.sdk.values.PDone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Experimental
public class BigtableIO {
    private static final Logger logger = LoggerFactory.getLogger(BigtableIO.class);

    @Experimental
    public static Read read() {
        return new Read(null, "", null, null);
    }

    @Experimental
    public static Write write() {
        return new Write(null, "", null);
    }

    private BigtableIO() {
    }

    private static String getUserAgent() {
        String javaVersion = System.getProperty("java.specification.version");
        ReleaseInfo info = ReleaseInfo.getReleaseInfo();
        return String.format("%s/%s (%s); %s", info.getName(), info.getVersion(), javaVersion, "0.2.3");
    }

    static class BigtableWriteException
    extends IOException {
        public BigtableWriteException(KV<ByteString, Iterable<Mutation>> record, Throwable cause) {
            super(String.format("Error mutating row %s with mutations %s", ((ByteString)record.getKey()).toStringUtf8(), record.getValue()), cause);
        }
    }

    private static class BigtableWriter
    extends Sink.Writer<KV<ByteString, Iterable<Mutation>>, Long> {
        private final BigtableWriteOperation writeOperation;
        private final Sink sink;
        private BigtableService.Writer bigtableWriter;
        private long recordsWritten;
        private final ConcurrentLinkedQueue<BigtableWriteException> failures;

        public BigtableWriter(BigtableWriteOperation writeOperation) {
            this.writeOperation = writeOperation;
            this.sink = writeOperation.getSink();
            this.failures = new ConcurrentLinkedQueue();
        }

        public void open(String uId) throws Exception {
            this.bigtableWriter = this.sink.getBigtableService().openForWriting(this.sink.getTableId());
            this.recordsWritten = 0L;
        }

        private void checkForFailures() throws IOException {
            int i;
            if (this.failures.isEmpty()) {
                return;
            }
            StringBuilder logEntry = new StringBuilder();
            for (i = 0; i < 10 && !this.failures.isEmpty(); ++i) {
                BigtableWriteException exc = (BigtableWriteException)this.failures.remove();
                logEntry.append("\n").append(exc.getMessage());
                if (exc.getCause() == null) continue;
                logEntry.append(": ").append(exc.getCause().getMessage());
            }
            String message = String.format("At least %d errors occurred writing to Bigtable. First %d errors: %s", i + this.failures.size(), i, logEntry.toString());
            logger.error(message);
            throw new IOException(message);
        }

        public void write(KV<ByteString, Iterable<Mutation>> rowMutations) throws Exception {
            this.checkForFailures();
            Futures.addCallback(this.bigtableWriter.writeRecord(rowMutations), (FutureCallback)new WriteExceptionCallback(rowMutations));
            ++this.recordsWritten;
        }

        public Long close() throws Exception {
            this.bigtableWriter.close();
            this.bigtableWriter = null;
            this.checkForFailures();
            logger.info("Wrote {} records", (Object)this.recordsWritten);
            return this.recordsWritten;
        }

        public Sink.WriteOperation<KV<ByteString, Iterable<Mutation>>, Long> getWriteOperation() {
            return this.writeOperation;
        }

        private class WriteExceptionCallback
        implements FutureCallback<Empty> {
            private final KV<ByteString, Iterable<Mutation>> value;

            public WriteExceptionCallback(KV<ByteString, Iterable<Mutation>> value) {
                this.value = value;
            }

            public void onFailure(Throwable cause) {
                BigtableWriter.this.failures.add(new BigtableWriteException(this.value, cause));
            }

            public void onSuccess(Empty produced) {
            }
        }
    }

    private static class BigtableWriteOperation
    extends Sink.WriteOperation<KV<ByteString, Iterable<Mutation>>, Long> {
        private final Sink sink;

        public BigtableWriteOperation(Sink sink) {
            this.sink = sink;
        }

        public Sink.Writer<KV<ByteString, Iterable<Mutation>>, Long> createWriter(PipelineOptions options) throws Exception {
            return new BigtableWriter(this);
        }

        public void initialize(PipelineOptions options) {
        }

        public void finalize(Iterable<Long> writerResults, PipelineOptions options) {
            long count = 0L;
            for (Long value : writerResults) {
                value = value + count;
            }
            logger.debug("Wrote {} elements to BigtableIO.Sink {}", (Object)this.sink);
        }

        public Sink getSink() {
            return this.sink;
        }

        public Coder<Long> getWriterResultCoder() {
            return VarLongCoder.of();
        }
    }

    private static class Sink
    extends org.apache.beam.sdk.io.Sink<KV<ByteString, Iterable<Mutation>>> {
        private final String tableId;
        private final BigtableService bigtableService;

        public Sink(String tableId, BigtableService bigtableService) {
            this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            this.bigtableService = (BigtableService)Preconditions.checkNotNull((Object)bigtableService, (Object)"bigtableService");
        }

        public String getTableId() {
            return this.tableId;
        }

        public BigtableService getBigtableService() {
            return this.bigtableService;
        }

        public String toString() {
            return MoreObjects.toStringHelper(Sink.class).add("bigtableService", (Object)this.bigtableService).add("tableId", (Object)this.tableId).toString();
        }

        public Sink.WriteOperation<KV<ByteString, Iterable<Mutation>>, Long> createWriteOperation(PipelineOptions options) {
            return new BigtableWriteOperation(this);
        }

        public void validate(PipelineOptions options) {
        }
    }

    private static class BigtableReader
    extends BoundedSource.BoundedReader<Row> {
        private BigtableSource source;
        private BigtableService service;
        private BigtableService.Reader reader;
        private final ByteKeyRangeTracker rangeTracker;
        private long recordsReturned;

        public BigtableReader(BigtableSource source, BigtableService service) {
            this.source = source;
            this.service = service;
            this.rangeTracker = ByteKeyRangeTracker.of((ByteKeyRange)source.getRange());
        }

        public boolean start() throws IOException {
            boolean hasRecord;
            this.reader = this.service.createReader(this.getCurrentSource());
            boolean bl = hasRecord = this.reader.start() && this.rangeTracker.tryReturnRecordAt(true, ByteKey.of((ByteString)this.reader.getCurrentRow().getKey()));
            if (hasRecord) {
                ++this.recordsReturned;
            }
            return hasRecord;
        }

        public synchronized BigtableSource getCurrentSource() {
            return this.source;
        }

        public boolean advance() throws IOException {
            boolean hasRecord;
            boolean bl = hasRecord = this.reader.advance() && this.rangeTracker.tryReturnRecordAt(true, ByteKey.of((ByteString)this.reader.getCurrentRow().getKey()));
            if (hasRecord) {
                ++this.recordsReturned;
            }
            return hasRecord;
        }

        public Row getCurrent() throws NoSuchElementException {
            return this.reader.getCurrentRow();
        }

        public void close() throws IOException {
            logger.info("Closing reader after reading {} records.", (Object)this.recordsReturned);
            if (this.reader != null) {
                this.reader.close();
                this.reader = null;
            }
        }

        public final Double getFractionConsumed() {
            return this.rangeTracker.getFractionConsumed();
        }

        public final synchronized BigtableSource splitAtFraction(double fraction) {
            ByteKey splitKey;
            try {
                splitKey = this.source.getRange().interpolateKey(fraction);
            }
            catch (IllegalArgumentException e) {
                logger.info("%s: Failed to interpolate key for fraction %s.", (Object)this.source.getRange(), (Object)fraction);
                return null;
            }
            logger.debug("Proposing to split {} at fraction {} (key {})", new Object[]{this.rangeTracker, fraction, splitKey});
            if (!this.rangeTracker.trySplitAtPosition(splitKey)) {
                return null;
            }
            BigtableSource primary = this.source.withEndKey(splitKey);
            BigtableSource residual = this.source.withStartKey(splitKey);
            this.source = primary;
            return residual;
        }
    }

    static class BigtableSource
    extends BoundedSource<Row> {
        private final BigtableService service;
        @Nullable
        private final String tableId;
        @Nullable
        private final RowFilter filter;
        private final ByteKeyRange range;
        @Nullable
        private Long estimatedSizeBytes;
        @Nullable
        private transient List<SampleRowKeysResponse> sampleRowKeys;

        public BigtableSource(BigtableService service, String tableId, @Nullable RowFilter filter, ByteKeyRange range, Long estimatedSizeBytes) {
            this.service = service;
            this.tableId = tableId;
            this.filter = filter;
            this.range = range;
            this.estimatedSizeBytes = estimatedSizeBytes;
        }

        public String toString() {
            return MoreObjects.toStringHelper(BigtableSource.class).add("tableId", (Object)this.tableId).add("filter", (Object)this.filter).add("range", (Object)this.range).add("estimatedSizeBytes", (Object)this.estimatedSizeBytes).toString();
        }

        protected BigtableSource withStartKey(ByteKey startKey) {
            Preconditions.checkNotNull((Object)startKey, (Object)"startKey");
            return new BigtableSource(this.service, this.tableId, this.filter, this.range.withStartKey(startKey), this.estimatedSizeBytes);
        }

        protected BigtableSource withEndKey(ByteKey endKey) {
            Preconditions.checkNotNull((Object)endKey, (Object)"endKey");
            return new BigtableSource(this.service, this.tableId, this.filter, this.range.withEndKey(endKey), this.estimatedSizeBytes);
        }

        protected BigtableSource withEstimatedSizeBytes(Long estimatedSizeBytes) {
            Preconditions.checkNotNull((Object)estimatedSizeBytes, (Object)"estimatedSizeBytes");
            return new BigtableSource(this.service, this.tableId, this.filter, this.range, estimatedSizeBytes);
        }

        private List<SampleRowKeysResponse> getSampleRowKeys() throws IOException {
            return this.service.getSampleRowKeys(this);
        }

        public List<BigtableSource> splitIntoBundles(long desiredBundleSizeBytes, PipelineOptions options) throws Exception {
            long maximumNumberOfSplits = 4000L;
            long sizeEstimate = this.getEstimatedSizeBytes(options);
            desiredBundleSizeBytes = Math.max(sizeEstimate / maximumNumberOfSplits, desiredBundleSizeBytes);
            return this.splitIntoBundlesBasedOnSamples(desiredBundleSizeBytes, this.getSampleRowKeys());
        }

        private List<BigtableSource> splitIntoBundlesBasedOnSamples(long desiredBundleSizeBytes, List<SampleRowKeysResponse> sampleRowKeys) {
            if (sampleRowKeys.isEmpty()) {
                logger.info("Not splitting source {} because no sample row keys are available.", (Object)this);
                return Collections.singletonList(this);
            }
            logger.info("About to split into bundles of size {} with sampleRowKeys length {} first element {}", new Object[]{desiredBundleSizeBytes, sampleRowKeys.size(), sampleRowKeys.get(0)});
            ByteKey lastEndKey = ByteKey.EMPTY;
            long lastOffset = 0L;
            ImmutableList.Builder splits = ImmutableList.builder();
            for (SampleRowKeysResponse response : sampleRowKeys) {
                ByteKey splitEndKey;
                ByteKey responseEndKey = ByteKey.of((ByteString)response.getRowKey());
                long responseOffset = response.getOffsetBytes();
                Preconditions.checkState((responseOffset >= lastOffset ? 1 : 0) != 0, (String)"Expected response byte offset %s to come after the last offset %s", (Object[])new Object[]{responseOffset, lastOffset});
                if (!this.range.overlaps(ByteKeyRange.of((ByteKey)lastEndKey, (ByteKey)responseEndKey)).booleanValue()) {
                    lastOffset = responseOffset;
                    lastEndKey = responseEndKey;
                    continue;
                }
                ByteKey splitStartKey = lastEndKey;
                if (splitStartKey.compareTo(this.range.getStartKey()) < 0) {
                    splitStartKey = this.range.getStartKey();
                }
                if (!this.range.containsKey(splitEndKey = responseEndKey).booleanValue()) {
                    splitEndKey = this.range.getEndKey();
                }
                long sampleSizeBytes = responseOffset - lastOffset;
                List<BigtableSource> subSplits = this.splitKeyRangeIntoBundleSizedSubranges(sampleSizeBytes, desiredBundleSizeBytes, ByteKeyRange.of((ByteKey)splitStartKey, (ByteKey)splitEndKey));
                splits.addAll(subSplits);
                lastEndKey = responseEndKey;
                lastOffset = responseOffset;
            }
            if (!lastEndKey.isEmpty() && (this.range.getEndKey().isEmpty() || lastEndKey.compareTo(this.range.getEndKey()) < 0)) {
                splits.add((Object)this.withStartKey(lastEndKey).withEndKey(this.range.getEndKey()));
            }
            ImmutableList ret = splits.build();
            logger.info("Generated {} splits. First split: {}", (Object)ret.size(), ret.get(0));
            return ret;
        }

        public long getEstimatedSizeBytes(PipelineOptions options) throws IOException {
            if (this.estimatedSizeBytes == null) {
                this.estimatedSizeBytes = this.getEstimatedSizeBytesBasedOnSamples(this.getSampleRowKeys());
            }
            return this.estimatedSizeBytes;
        }

        private long getEstimatedSizeBytesBasedOnSamples(List<SampleRowKeysResponse> samples) {
            long estimatedSizeBytes = 0L;
            long lastOffset = 0L;
            ByteKey currentStartKey = ByteKey.EMPTY;
            for (SampleRowKeysResponse response : samples) {
                ByteKey currentEndKey = ByteKey.of((ByteString)response.getRowKey());
                long currentOffset = response.getOffsetBytes();
                if (!currentStartKey.isEmpty() && currentStartKey.equals((Object)currentEndKey)) {
                    lastOffset = currentOffset;
                    continue;
                }
                if (this.range.overlaps(ByteKeyRange.of((ByteKey)currentStartKey, (ByteKey)currentEndKey)).booleanValue()) {
                    estimatedSizeBytes += currentOffset - lastOffset;
                }
                currentStartKey = currentEndKey;
                lastOffset = currentOffset;
            }
            return estimatedSizeBytes;
        }

        public boolean producesSortedKeys(PipelineOptions options) throws Exception {
            return true;
        }

        public BoundedSource.BoundedReader<Row> createReader(PipelineOptions options) throws IOException {
            return new BigtableReader(this, this.service);
        }

        public void validate() {
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"tableId cannot be empty");
        }

        public Coder<Row> getDefaultOutputCoder() {
            return ProtoCoder.of(Row.class);
        }

        private List<BigtableSource> splitKeyRangeIntoBundleSizedSubranges(long sampleSizeBytes, long desiredBundleSizeBytes, ByteKeyRange range) {
            logger.debug("Subsplit for sampleSizeBytes {} and desiredBundleSizeBytes {}", (Object)sampleSizeBytes, (Object)desiredBundleSizeBytes);
            if (sampleSizeBytes <= desiredBundleSizeBytes) {
                return Collections.singletonList(this.withStartKey(range.getStartKey()).withEndKey(range.getEndKey()));
            }
            Preconditions.checkArgument((sampleSizeBytes > 0L ? 1 : 0) != 0, (String)"Sample size %s bytes must be greater than 0.", (Object[])new Object[]{sampleSizeBytes});
            Preconditions.checkArgument((desiredBundleSizeBytes > 0L ? 1 : 0) != 0, (String)"Desired bundle size %s bytes must be greater than 0.", (Object[])new Object[]{desiredBundleSizeBytes});
            int splitCount = (int)Math.ceil((double)sampleSizeBytes / (double)desiredBundleSizeBytes);
            List splitKeys = range.split(splitCount);
            ImmutableList.Builder splits = ImmutableList.builder();
            Iterator keys = splitKeys.iterator();
            ByteKey prev = (ByteKey)keys.next();
            while (keys.hasNext()) {
                ByteKey next = (ByteKey)keys.next();
                splits.add((Object)this.withStartKey(prev).withEndKey(next).withEstimatedSizeBytes(sampleSizeBytes / (long)splitCount));
                prev = next;
            }
            return splits.build();
        }

        public ByteKeyRange getRange() {
            return this.range;
        }

        public RowFilter getRowFilter() {
            return this.filter;
        }

        public String getTableId() {
            return this.tableId;
        }
    }

    @Experimental
    public static class Write
    extends PTransform<PCollection<KV<ByteString, Iterable<Mutation>>>, PDone> {
        @Nullable
        private final BigtableOptions options;
        private final String tableId;
        @Nullable
        private final BigtableService bigtableService;

        private Write(@Nullable BigtableOptions options, String tableId, @Nullable BigtableService bigtableService) {
            this.options = options;
            this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            this.bigtableService = bigtableService;
        }

        public Write withBigtableOptions(BigtableOptions options) {
            Preconditions.checkNotNull((Object)options, (Object)"options");
            return this.withBigtableOptions(options.toBuilder());
        }

        public Write withBigtableOptions(BigtableOptions.Builder optionsBuilder) {
            Preconditions.checkNotNull((Object)optionsBuilder, (Object)"optionsBuilder");
            BigtableOptions.Builder clonedBuilder = optionsBuilder.build().toBuilder();
            BigtableOptions optionsWithAgent = clonedBuilder.setUserAgent(BigtableIO.getUserAgent()).build();
            return new Write(optionsWithAgent, this.tableId, this.bigtableService);
        }

        public Write withTableId(String tableId) {
            Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            return new Write(this.options, tableId, this.bigtableService);
        }

        public BigtableOptions getBigtableOptions() {
            return this.options;
        }

        public String getTableId() {
            return this.tableId;
        }

        public PDone apply(PCollection<KV<ByteString, Iterable<Mutation>>> input) {
            Sink sink = new Sink(this.tableId, this.getBigtableService());
            return (PDone)input.apply((PTransform)org.apache.beam.sdk.io.Write.to((org.apache.beam.sdk.io.Sink)sink));
        }

        public void validate(PCollection<KV<ByteString, Iterable<Mutation>>> input) {
            Preconditions.checkArgument((this.options != null ? 1 : 0) != 0, (Object)"BigtableOptions not specified");
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"Table ID not specified");
            try {
                Preconditions.checkArgument((boolean)this.getBigtableService().tableExists(this.tableId), (String)"Table %s does not exist", (Object[])new Object[]{this.tableId});
            }
            catch (IOException e) {
                logger.warn("Error checking whether table {} exists; proceeding.", (Object)this.tableId, (Object)e);
            }
        }

        Write withBigtableService(BigtableService bigtableService) {
            Preconditions.checkNotNull((Object)bigtableService, (Object)"bigtableService");
            return new Write(this.options, this.tableId, bigtableService);
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.add(DisplayData.item((String)"tableId", (String)this.tableId).withLabel("Table ID"));
            if (this.options != null) {
                builder.add(DisplayData.item((String)"bigtableOptions", (String)this.options.toString()).withLabel("Bigtable Options"));
            }
        }

        public String toString() {
            return MoreObjects.toStringHelper(Write.class).add("options", (Object)this.options).add("tableId", (Object)this.tableId).toString();
        }

        private BigtableService getBigtableService() {
            if (this.bigtableService != null) {
                return this.bigtableService;
            }
            return new BigtableServiceImpl(this.options);
        }
    }

    @Experimental
    public static class Read
    extends PTransform<PBegin, PCollection<Row>> {
        @Nullable
        private final BigtableOptions options;
        private final String tableId;
        @Nullable
        private final RowFilter filter;
        @Nullable
        private final BigtableService bigtableService;

        public Read withBigtableOptions(BigtableOptions options) {
            Preconditions.checkNotNull((Object)options, (Object)"options");
            return this.withBigtableOptions(options.toBuilder());
        }

        public Read withBigtableOptions(BigtableOptions.Builder optionsBuilder) {
            Preconditions.checkNotNull((Object)optionsBuilder, (Object)"optionsBuilder");
            BigtableOptions.Builder clonedBuilder = optionsBuilder.build().toBuilder();
            BigtableOptions optionsWithAgent = clonedBuilder.setUserAgent(BigtableIO.getUserAgent()).build();
            return new Read(optionsWithAgent, this.tableId, this.filter, this.bigtableService);
        }

        public Read withRowFilter(RowFilter filter) {
            Preconditions.checkNotNull((Object)filter, (Object)"filter");
            return new Read(this.options, this.tableId, filter, this.bigtableService);
        }

        public Read withTableId(String tableId) {
            Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            return new Read(this.options, tableId, this.filter, this.bigtableService);
        }

        public BigtableOptions getBigtableOptions() {
            return this.options;
        }

        public String getTableId() {
            return this.tableId;
        }

        public PCollection<Row> apply(PBegin input) {
            BigtableSource source = new BigtableSource(this.getBigtableService(), this.tableId, this.filter, ByteKeyRange.ALL_KEYS, null);
            return (PCollection)input.getPipeline().apply((PTransform)org.apache.beam.sdk.io.Read.from((BoundedSource)source));
        }

        public void validate(PBegin input) {
            Preconditions.checkArgument((this.options != null ? 1 : 0) != 0, (Object)"BigtableOptions not specified");
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"Table ID not specified");
            try {
                Preconditions.checkArgument((boolean)this.getBigtableService().tableExists(this.tableId), (String)"Table %s does not exist", (Object[])new Object[]{this.tableId});
            }
            catch (IOException e) {
                logger.warn("Error checking whether table {} exists; proceeding.", (Object)this.tableId, (Object)e);
            }
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.add(DisplayData.item((String)"tableId", (String)this.tableId).withLinkUrl("Table ID"));
            if (this.options != null) {
                builder.add(DisplayData.item((String)"bigtableOptions", (String)this.options.toString()).withLabel("Bigtable Options"));
            }
            if (this.filter != null) {
                builder.add(DisplayData.item((String)"rowFilter", (String)this.filter.toString()).withLabel("Table Row Filter"));
            }
        }

        public String toString() {
            return MoreObjects.toStringHelper(Read.class).add("options", (Object)this.options).add("tableId", (Object)this.tableId).add("filter", (Object)this.filter).toString();
        }

        private Read(@Nullable BigtableOptions options, String tableId, @Nullable RowFilter filter, @Nullable BigtableService bigtableService) {
            this.options = options;
            this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            this.filter = filter;
            this.bigtableService = bigtableService;
        }

        Read withBigtableService(BigtableService bigtableService) {
            Preconditions.checkNotNull((Object)bigtableService, (Object)"bigtableService");
            return new Read(this.options, this.tableId, this.filter, bigtableService);
        }

        private BigtableService getBigtableService() {
            if (this.bigtableService != null) {
                return this.bigtableService;
            }
            return new BigtableServiceImpl(this.options);
        }
    }
}

