/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.extensions.sql.zetasql;

import com.google.common.collect.ImmutableList;
import com.google.protobuf.ByteString;
import com.google.zetasql.ArrayType;
import com.google.zetasql.CivilTimeEncoder;
import com.google.zetasql.StructType;
import com.google.zetasql.Type;
import com.google.zetasql.TypeFactory;
import com.google.zetasql.Value;
import com.google.zetasql.ZetaSQLType;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.beam.sdk.annotations.Internal;
import org.apache.beam.sdk.schemas.Schema;
import org.apache.beam.sdk.schemas.logicaltypes.SqlTypes;
import org.apache.beam.sdk.values.Row;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.math.LongMath;
import org.joda.time.Instant;

@Internal
public final class ZetaSqlBeamTranslationUtils {
    private static final long MICROS_PER_MILLI = 1000L;

    private ZetaSqlBeamTranslationUtils() {
    }

    public static Type beamFieldTypeToZetaSqlType(Schema.FieldType fieldType) {
        switch (fieldType.getTypeName()) {
            case INT64: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_INT64);
            }
            case DOUBLE: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_DOUBLE);
            }
            case BOOLEAN: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_BOOL);
            }
            case STRING: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_STRING);
            }
            case BYTES: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_BYTES);
            }
            case DATETIME: {
                return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_TIMESTAMP);
            }
            case ARRAY: {
                return ZetaSqlBeamTranslationUtils.beamElementFieldTypeToZetaSqlArrayType(fieldType.getCollectionElementType());
            }
            case ROW: {
                return ZetaSqlBeamTranslationUtils.beamSchemaToZetaSqlStructType(fieldType.getRowSchema());
            }
            case LOGICAL_TYPE: {
                return ZetaSqlBeamTranslationUtils.beamLogicalTypeToZetaSqlType(fieldType.getLogicalType().getIdentifier());
            }
        }
        throw new UnsupportedOperationException("Unknown Beam fieldType: " + fieldType.getTypeName());
    }

    private static ArrayType beamElementFieldTypeToZetaSqlArrayType(Schema.FieldType elementFieldType) {
        return TypeFactory.createArrayType((Type)ZetaSqlBeamTranslationUtils.beamFieldTypeToZetaSqlType(elementFieldType));
    }

    public static StructType beamSchemaToZetaSqlStructType(Schema schema) {
        return TypeFactory.createStructType((Collection)schema.getFields().stream().map(ZetaSqlBeamTranslationUtils::beamFieldToZetaSqlStructField).collect(Collectors.toList()));
    }

    private static StructType.StructField beamFieldToZetaSqlStructField(Schema.Field field) {
        return new StructType.StructField(field.getName(), ZetaSqlBeamTranslationUtils.beamFieldTypeToZetaSqlType(field.getType()));
    }

    private static Type beamLogicalTypeToZetaSqlType(String identifier) {
        if (SqlTypes.DATE.getIdentifier().equals(identifier)) {
            return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_DATE);
        }
        if (SqlTypes.TIME.getIdentifier().equals(identifier)) {
            return TypeFactory.createSimpleType((ZetaSQLType.TypeKind)ZetaSQLType.TypeKind.TYPE_TIME);
        }
        throw new UnsupportedOperationException("Unknown Beam logical type: " + identifier);
    }

    public static Value javaObjectToZetaSqlValue(Object object, Schema.FieldType fieldType) {
        if (object == null) {
            return Value.createNullValue((Type)ZetaSqlBeamTranslationUtils.beamFieldTypeToZetaSqlType(fieldType));
        }
        switch (fieldType.getTypeName()) {
            case INT64: {
                return Value.createInt64Value((long)((Long)object));
            }
            case DOUBLE: {
                return Value.createDoubleValue((double)((Double)object));
            }
            case BOOLEAN: {
                return Value.createBoolValue((boolean)((Boolean)object));
            }
            case STRING: {
                return Value.createStringValue((String)((String)object));
            }
            case BYTES: {
                return Value.createBytesValue((ByteString)ByteString.copyFrom((byte[])((byte[])object)));
            }
            case DATETIME: {
                return ZetaSqlBeamTranslationUtils.jodaInstantToZetaSqlTimestampValue((Instant)object);
            }
            case ARRAY: {
                return ZetaSqlBeamTranslationUtils.javaListToZetaSqlArrayValue((List)object, fieldType.getCollectionElementType());
            }
            case ROW: {
                return ZetaSqlBeamTranslationUtils.beamRowToZetaSqlStructValue((Row)object, fieldType.getRowSchema());
            }
            case LOGICAL_TYPE: {
                return ZetaSqlBeamTranslationUtils.beamLogicalObjectToZetaSqlValue(object, fieldType.getLogicalType().getIdentifier());
            }
        }
        throw new UnsupportedOperationException("Unknown Beam fieldType: " + fieldType.getTypeName());
    }

    private static Value jodaInstantToZetaSqlTimestampValue(Instant instant) {
        return Value.createTimestampValueFromUnixMicros((long)LongMath.checkedMultiply((long)instant.getMillis(), (long)1000L));
    }

    private static Value javaListToZetaSqlArrayValue(List<Object> elements, Schema.FieldType elementType) {
        List values = elements.stream().map(e -> ZetaSqlBeamTranslationUtils.javaObjectToZetaSqlValue(e, elementType)).collect(Collectors.toList());
        return Value.createArrayValue((ArrayType)ZetaSqlBeamTranslationUtils.beamElementFieldTypeToZetaSqlArrayType(elementType), values);
    }

    public static Value beamRowToZetaSqlStructValue(Row row, Schema schema) {
        ArrayList<Value> values = new ArrayList<Value>(row.getFieldCount());
        for (int i = 0; i < row.getFieldCount(); ++i) {
            values.add(ZetaSqlBeamTranslationUtils.javaObjectToZetaSqlValue(row.getBaseValue(i, Object.class), schema.getField(i).getType()));
        }
        return Value.createStructValue((StructType)ZetaSqlBeamTranslationUtils.beamSchemaToZetaSqlStructType(schema), values);
    }

    private static Value beamLogicalObjectToZetaSqlValue(Object object, String identifier) {
        if (SqlTypes.DATE.getIdentifier().equals(identifier)) {
            if (object instanceof Long) {
                return Value.createDateValue((int)((Long)object).intValue());
            }
            return Value.createDateValue((int)((int)((LocalDate)object).toEpochDay()));
        }
        if (SqlTypes.TIME.getIdentifier().equals(identifier)) {
            if (object instanceof Long) {
                return Value.createTimeValue((long)CivilTimeEncoder.encodePacked64TimeNanos((LocalTime)LocalTime.ofNanoOfDay((Long)object)));
            }
            return Value.createTimeValue((long)CivilTimeEncoder.encodePacked64TimeNanos((LocalTime)((LocalTime)object)));
        }
        throw new UnsupportedOperationException("Unknown Beam logical type: " + identifier);
    }

    public static Schema.FieldType zetaSqlTypeToBeamFieldType(Type type) {
        switch (type.getKind()) {
            case TYPE_INT64: {
                return Schema.FieldType.INT64.withNullable(true);
            }
            case TYPE_DOUBLE: {
                return Schema.FieldType.DOUBLE.withNullable(true);
            }
            case TYPE_BOOL: {
                return Schema.FieldType.BOOLEAN.withNullable(true);
            }
            case TYPE_STRING: {
                return Schema.FieldType.STRING.withNullable(true);
            }
            case TYPE_BYTES: {
                return Schema.FieldType.BYTES.withNullable(true);
            }
            case TYPE_DATE: {
                return Schema.FieldType.logicalType((Schema.LogicalType)SqlTypes.DATE).withNullable(true);
            }
            case TYPE_TIME: {
                return Schema.FieldType.logicalType((Schema.LogicalType)SqlTypes.TIME).withNullable(true);
            }
            case TYPE_TIMESTAMP: {
                return Schema.FieldType.DATETIME.withNullable(true);
            }
            case TYPE_ARRAY: {
                return ZetaSqlBeamTranslationUtils.zetaSqlElementTypeToBeamArrayType(type.asArray().getElementType());
            }
            case TYPE_STRUCT: {
                return ZetaSqlBeamTranslationUtils.zetaSqlStructTypeToBeamRowType(type.asStruct());
            }
        }
        throw new UnsupportedOperationException("Unknown ZetaSQL type: " + type.getKind());
    }

    private static Schema.FieldType zetaSqlElementTypeToBeamArrayType(Type elementType) {
        return Schema.FieldType.array((Schema.FieldType)ZetaSqlBeamTranslationUtils.zetaSqlTypeToBeamFieldType(elementType)).withNullable(true);
    }

    private static Schema.FieldType zetaSqlStructTypeToBeamRowType(StructType structType) {
        return Schema.FieldType.row((Schema)((Schema)structType.getFieldList().stream().map(ZetaSqlBeamTranslationUtils::zetaSqlStructFieldToBeamField).collect(Schema.toSchema()))).withNullable(true);
    }

    private static Schema.Field zetaSqlStructFieldToBeamField(StructType.StructField structField) {
        return Schema.Field.of((String)structField.getName(), (Schema.FieldType)ZetaSqlBeamTranslationUtils.zetaSqlTypeToBeamFieldType(structField.getType()));
    }

    public static Object zetaSqlValueToJavaObject(Value value, boolean verifyValues) {
        return ZetaSqlBeamTranslationUtils.zetaSqlValueToJavaObject(value, ZetaSqlBeamTranslationUtils.zetaSqlTypeToBeamFieldType(value.getType()), verifyValues);
    }

    public static Object zetaSqlValueToJavaObject(Value value, Schema.FieldType fieldType, boolean verifyValues) {
        if (value.isNull()) {
            return null;
        }
        switch (fieldType.getTypeName()) {
            case INT64: {
                return value.getInt64Value();
            }
            case DOUBLE: {
                if (value.getType().getKind().equals((Object)ZetaSQLType.TypeKind.TYPE_INT64)) {
                    return (double)value.getInt64Value();
                }
                return value.getDoubleValue();
            }
            case BOOLEAN: {
                return value.getBoolValue();
            }
            case STRING: {
                return value.getStringValue();
            }
            case BYTES: {
                return value.getBytesValue().toByteArray();
            }
            case DATETIME: {
                return ZetaSqlBeamTranslationUtils.zetaSqlTimestampValueToJodaInstant(value);
            }
            case ARRAY: {
                return ZetaSqlBeamTranslationUtils.zetaSqlArrayValueToJavaList(value, fieldType.getCollectionElementType(), verifyValues);
            }
            case ROW: {
                return ZetaSqlBeamTranslationUtils.zetaSqlStructValueToBeamRow(value, fieldType.getRowSchema(), verifyValues);
            }
            case LOGICAL_TYPE: {
                return ZetaSqlBeamTranslationUtils.zetaSqlValueToBeamLogicalObject(value, fieldType.getLogicalType().getIdentifier());
            }
        }
        throw new UnsupportedOperationException("Unknown Beam fieldType: " + fieldType.getTypeName());
    }

    private static Instant zetaSqlTimestampValueToJodaInstant(Value timestampValue) {
        long millis = timestampValue.getTimestampUnixMicros() / 1000L;
        return Instant.ofEpochMilli((long)millis);
    }

    private static List<Object> zetaSqlArrayValueToJavaList(Value arrayValue, Schema.FieldType elementType, boolean verifyValues) {
        return arrayValue.getElementList().stream().map(e -> ZetaSqlBeamTranslationUtils.zetaSqlValueToJavaObject(e, elementType, verifyValues)).collect(Collectors.toList());
    }

    public static Row zetaSqlStructValueToBeamRow(Value structValue, Schema schema, boolean verifyValues) {
        ArrayList<Object> objects = new ArrayList<Object>(schema.getFieldCount());
        ImmutableList values = structValue.getFieldList();
        for (int i = 0; i < values.size(); ++i) {
            objects.add(ZetaSqlBeamTranslationUtils.zetaSqlValueToJavaObject((Value)values.get(i), schema.getField(i).getType(), verifyValues));
        }
        Row row = verifyValues ? Row.withSchema((Schema)schema).addValues(objects).build() : Row.withSchema((Schema)schema).attachValues(objects);
        return row;
    }

    private static Object zetaSqlValueToBeamLogicalObject(Value value, String identifier) {
        if (SqlTypes.DATE.getIdentifier().equals(identifier)) {
            return LocalDate.ofEpochDay(value.getDateValue());
        }
        if (SqlTypes.TIME.getIdentifier().equals(identifier)) {
            return CivilTimeEncoder.decodePacked64TimeNanosAsJavaTime((long)value.getTimeValue());
        }
        throw new UnsupportedOperationException("Unknown Beam logical type: " + identifier);
    }
}

