/**
 * BSD-style license; for more info see http://pmd.sourceforge.net/license.html
 */
package net.sourceforge.pmd.lang.java.symboltable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import net.sourceforge.pmd.lang.ast.Node;
import net.sourceforge.pmd.lang.java.ast.ASTAllocationExpression;
import net.sourceforge.pmd.lang.java.ast.ASTArgumentList;
import net.sourceforge.pmd.lang.java.ast.ASTClassOrInterfaceType;
import net.sourceforge.pmd.lang.java.ast.ASTFormalParameter;
import net.sourceforge.pmd.lang.java.ast.ASTLiteral;
import net.sourceforge.pmd.lang.java.ast.ASTName;
import net.sourceforge.pmd.lang.java.ast.ASTPrimarySuffix;
import net.sourceforge.pmd.lang.symboltable.NameDeclaration;
import net.sourceforge.pmd.lang.symboltable.NameOccurrence;
import net.sourceforge.pmd.lang.symboltable.Scope;

/**
 * This scope represents one Java class.
 * It can have variable declarations, method declarations and inner class declarations.
 */
public class ClassScope extends AbstractJavaScope {

    // FIXME - this breaks given sufficiently nested code
    private static ThreadLocal<Integer> anonymousInnerClassCounter = new ThreadLocal<Integer>() {
        protected Integer initialValue() { return Integer.valueOf(1); }
    };

    private String className;

    public ClassScope(String className) {
        this.className = className;
        anonymousInnerClassCounter.set(Integer.valueOf(1));
    }

    /**
     * This is only for anonymous inner classes
     * <p/>
     * FIXME - should have name like Foo$1, not Anonymous$1
     * to get this working right, the parent scope needs
     * to be passed in when instantiating a ClassScope
     */
    public ClassScope() {
        //this.className = getParent().getEnclosingClassScope().getClassName() + "$" + String.valueOf(anonymousInnerClassCounter);
        int v = anonymousInnerClassCounter.get().intValue();
        this.className = "Anonymous$" + v;
        anonymousInnerClassCounter.set(v + 1);
    }

    public Map<ClassNameDeclaration, List<NameOccurrence>> getClassDeclarations() {
        return getDeclarations(ClassNameDeclaration.class);
    }

    public Map<MethodNameDeclaration, List<NameOccurrence>> getMethodDeclarations() {
        return getDeclarations(MethodNameDeclaration.class);
    }

    public Map<VariableNameDeclaration, List<NameOccurrence>> getVariableDeclarations() {
        return getDeclarations(VariableNameDeclaration.class);
    }

    public NameDeclaration addNameOccurrence(NameOccurrence occurrence) {
        JavaNameOccurrence javaOccurrence = (JavaNameOccurrence)occurrence;
        NameDeclaration decl = findVariableHere(javaOccurrence);
        if (decl != null && (javaOccurrence.isMethodOrConstructorInvocation() || javaOccurrence.isMethodReference())) {
            List<NameOccurrence> nameOccurrences = getMethodDeclarations().get(decl);
            if (nameOccurrences == null) {
                // TODO may be a class name: Foo.this.super();
            } else {
                nameOccurrences.add(javaOccurrence);
                Node n = javaOccurrence.getLocation();
                if (n instanceof ASTName) {
                    ((ASTName) n).setNameDeclaration(decl);
                } // TODO what to do with PrimarySuffix case?
            }

        } else if (decl != null && !javaOccurrence.isThisOrSuper()) {
            List<NameOccurrence> nameOccurrences = getVariableDeclarations().get(decl);
            if (nameOccurrences == null) {
                // TODO may be a class name

                // search inner classes
                for (ClassNameDeclaration innerClass : getClassDeclarations().keySet()) {
                    Scope innerClassScope = innerClass.getScope();
                    if (innerClassScope.contains(javaOccurrence)) {
                        innerClassScope.addNameOccurrence(javaOccurrence);
                    }
                }
            } else {
                nameOccurrences.add(javaOccurrence);
                Node n = javaOccurrence.getLocation();
                if (n instanceof ASTName) {
                    ((ASTName) n).setNameDeclaration(decl);
                } // TODO what to do with PrimarySuffix case?
            }
        }
        return decl;
    }

    public String getClassName() {
        return this.className;
    }

    protected NameDeclaration findVariableHere(JavaNameOccurrence occurrence) {
        Map<MethodNameDeclaration, List<NameOccurrence>> methodDeclarations = getMethodDeclarations();
        Map<VariableNameDeclaration, List<NameOccurrence>> variableDeclarations = getVariableDeclarations();
        if (occurrence.isThisOrSuper() ||
                (occurrence.getImage() != null && occurrence.getImage().equals(className))) {
            if (variableDeclarations.isEmpty() && methodDeclarations.isEmpty()) {
                // this could happen if you do this:
                // public class Foo {
                //  private String x = super.toString();
                // }
                return null;
            }
            // return any name declaration, since all we really want is to get the scope
            // for example, if there's a
            // public class Foo {
            //  private static final int X = 2;
            //  private int y = Foo.X;
            // }
            // we'll look up Foo just to get a handle to the class scope
            // and then we'll look up X.
            if (!variableDeclarations.isEmpty()) {
                return variableDeclarations.keySet().iterator().next();
            }
            return methodDeclarations.keySet().iterator().next();
        }

        if (occurrence.isMethodOrConstructorInvocation()) {
            for (MethodNameDeclaration mnd: methodDeclarations.keySet()) {
                if (mnd.getImage().equals(occurrence.getImage())) {
                    List<TypedNameDeclaration> parameterTypes = determineParameterTypes(mnd);
                    List<TypedNameDeclaration> argumentTypes = determineArgumentTypes(occurrence, parameterTypes);

                    if (!mnd.isVarargs()
                            && occurrence.getArgumentCount() == mnd.getParameterCount()
                            && parameterTypes.equals(argumentTypes)) {
                        return mnd;
                    } else if (mnd.isVarargs()) {
                        int varArgIndex = parameterTypes.size() - 1;
                        TypedNameDeclaration varArgType = parameterTypes.get(varArgIndex);
                        if (parameterTypes.subList(0, varArgIndex).equals(argumentTypes.subList(0, varArgIndex))) {
                            boolean sameType = true;
                            for (int i = varArgIndex; i < argumentTypes.size(); i++) {
                                if (!varArgType.equals(argumentTypes.get(i))) {
                                    sameType = false;
                                    break;
                                }
                            }
                            if (sameType) {
                                return mnd;
                            }
                        }
                    }
                }
            }
            return null;
        }
        if (occurrence.isMethodReference()) {
            for (MethodNameDeclaration mnd: methodDeclarations.keySet()) {
                if (mnd.getImage().equals(occurrence.getImage())) {
                    return mnd;
                }
            }
            return null;
        }

        List<String> images = new ArrayList<String>();
        if (occurrence.getImage() != null) {
            images.add(occurrence.getImage());
            if (occurrence.getImage().startsWith(className)) {
                images.add(clipClassName(occurrence.getImage()));
            }
        }
        ImageFinderFunction finder = new ImageFinderFunction(images);
        Applier.apply(finder, variableDeclarations.keySet().iterator());
        NameDeclaration result = finder.getDecl();

        // search inner classes
        Map<ClassNameDeclaration, List<NameOccurrence>> classDeclarations = getClassDeclarations();
        if (result == null && !classDeclarations.isEmpty()) {
            for (ClassNameDeclaration innerClass : getClassDeclarations().keySet()) {
                Applier.apply(finder, innerClass.getScope().getDeclarations().keySet().iterator());
                result = finder.getDecl();
                if (result != null) {
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Provide a list of types of the parameters of the given method declaration.
     * The types are simple type images. 
     * @param mnd the method declaration.
     * @return List of types
     */
    private List<TypedNameDeclaration> determineParameterTypes(MethodNameDeclaration mnd) {
        List<TypedNameDeclaration> parameterTypes = new ArrayList<TypedNameDeclaration>();
        List<ASTFormalParameter> parameters = mnd.getMethodNameDeclaratorNode().findDescendantsOfType(ASTFormalParameter.class);
        for (ASTFormalParameter p : parameters) {
            Class<?> resolvedType = this.getEnclosingScope(SourceFileScope.class).resolveType(p.getTypeNode().getTypeImage());
            parameterTypes.add(new SimpleTypedNameDeclaration(p.getTypeNode().getTypeImage(), resolvedType));
        }
        return parameterTypes;
    }

    /**
     * Provide a list of types of the arguments of the given method call.
     * The types are simple type images. If the argument type cannot be determined (e.g. because it is itself
     * the result of a method call), the parameter type is used - so it is assumed, it is of the correct type.
     * This might cause confusion when methods are overloaded.
     * @param occurrence the method call
     * @param parameterTypes the parameter types of the called method
     * @return the list of argument types
     */
    private List<TypedNameDeclaration> determineArgumentTypes(JavaNameOccurrence occurrence, List<TypedNameDeclaration> parameterTypes) {
        List<TypedNameDeclaration> argumentTypes = new ArrayList<TypedNameDeclaration>();
        ASTArgumentList arguments = null;
        Node nextSibling = null;
        if (occurrence.getLocation() instanceof ASTPrimarySuffix) {
            nextSibling = getNextSibling(occurrence.getLocation());
        } else {
            nextSibling = getNextSibling(occurrence.getLocation().jjtGetParent());
        }
        if (nextSibling != null) {
            arguments = nextSibling.getFirstDescendantOfType(ASTArgumentList.class);
        }

        if (arguments != null) {
            for (int i = 0; i < arguments.jjtGetNumChildren(); i++) {
                Node argument = arguments.jjtGetChild(i);
                Node child = null;
                if (argument.jjtGetNumChildren() > 0 && argument.jjtGetChild(0).jjtGetNumChildren() > 0
                        && argument.jjtGetChild(0).jjtGetChild(0).jjtGetNumChildren() > 0) {
                    child = argument.jjtGetChild(0).jjtGetChild(0).jjtGetChild(0);
                }
                TypedNameDeclaration type = null;
                if (child instanceof ASTName) {
                    ASTName name = (ASTName)child;
                    Scope s = name.getScope();
                    while (s != null) {
                        if (s.contains(new JavaNameOccurrence(name, name.getImage()))) {
                            break;
                        }
                        s = s.getParent();
                    }
                    if (s != null) {
                        Map<VariableNameDeclaration, List<NameOccurrence>> vars = s.getDeclarations(VariableNameDeclaration.class);
                        for (VariableNameDeclaration d : vars.keySet()) {
                            if (d.getImage().equals(name.getImage())) {
                                type = new SimpleTypedNameDeclaration(d.getTypeImage(),
                                        this.getEnclosingScope(SourceFileScope.class).resolveType(d.getTypeImage()));
                                break;
                            }
                        }
                    }
                } else if (child instanceof ASTLiteral) {
                    ASTLiteral literal = (ASTLiteral)child;
                    if (literal.isCharLiteral()) {
                        type = new SimpleTypedNameDeclaration("char", literal.getType());
                    } else if (literal.isStringLiteral()) {
                        type = new SimpleTypedNameDeclaration("String", literal.getType());
                    } else if (literal.isFloatLiteral()) {
                        type = new SimpleTypedNameDeclaration("float", literal.getType());
                    } else if (literal.isDoubleLiteral()) {
                        type = new SimpleTypedNameDeclaration("double", literal.getType());
                    } else if (literal.isIntLiteral()) {
                        type = new SimpleTypedNameDeclaration("int", literal.getType());
                    } else if (literal.isLongLiteral()) {
                        type = new SimpleTypedNameDeclaration("long", literal.getType());
                    }
                } else if (child instanceof ASTAllocationExpression && child.jjtGetChild(0) instanceof ASTClassOrInterfaceType) {
                    ASTClassOrInterfaceType classInterface = (ASTClassOrInterfaceType)child.jjtGetChild(0);
                    String typeImage = classInterface.getImage();
                    type = new SimpleTypedNameDeclaration(typeImage,
                            this.getEnclosingScope(SourceFileScope.class).resolveType(typeImage));
                }
                if (type == null && parameterTypes.size() > i) {
                    // replace the unknown type with the correct parameter type of the method.
                    // in case the argument is itself a method call, we can't determine the result type of the called
                    // method. Therefore the parameter type is used.
                    // This might cause confusion, if method overloading is used.
                    type = parameterTypes.get(i);
                }
                argumentTypes.add(type);
            }
        }
        return argumentTypes;
    }

    private Node getNextSibling(Node current) {
        Node nextSibling = null;
        for (int i = 0; i < current.jjtGetParent().jjtGetNumChildren() - 1; i++) {
            if (current.jjtGetParent().jjtGetChild(i) == current) {
                nextSibling = current.jjtGetParent().jjtGetChild(i + 1);
                break;
            }
        }
        return nextSibling;
    }

    public String toString() {
        StringBuilder res = new StringBuilder("ClassScope (").append(className).append("): ");
        Map<ClassNameDeclaration, List<NameOccurrence>> classDeclarations = getClassDeclarations();
        if (classDeclarations.isEmpty()) {
            res.append("Inner Classes ").append(glomNames(classDeclarations.keySet())).append("; ");
        }
        Map<MethodNameDeclaration, List<NameOccurrence>> methodDeclarations = getMethodDeclarations();
        if (!methodDeclarations.isEmpty()) {
            for (MethodNameDeclaration mnd: methodDeclarations.keySet()) {
                res.append(mnd.toString());
                int usages = methodDeclarations.get(mnd).size();
                res.append("(begins at line ").append(mnd.getNode().getBeginLine()).append(", ").append(usages).append(" usages)");
                res.append(", ");
            }
        }
        Map<VariableNameDeclaration, List<NameOccurrence>> variableDeclarations = getVariableDeclarations();
        if (!variableDeclarations.isEmpty()) {
            res.append("Variables ").append(glomNames(variableDeclarations.keySet()));
        }
        return res.toString();
    }

    private String clipClassName(String s) {
        return s.substring(s.indexOf('.') + 1);
    }
}
