/* This file is part of the OWL API.
 * The contents of this file are subject to the LGPL License, Version 3.0.
 * Copyright 2014, The University of Manchester
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 * Alternatively, the contents of this file may be used under the terms of the Apache License, Version 2.0 in which case, the provisions of the Apache License Version 2.0 are applicable instead of those above.
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions and limitations under the License. */
package org.semanticweb.owlapi.profiles;

import static org.semanticweb.owlapi.util.OWLAPIStreamUtils.asUnorderedSet;

import java.util.HashSet;
import java.util.Set;

import org.semanticweb.owlapi.model.*;
import org.semanticweb.owlapi.profiles.violations.UseOfIllegalAxiom;
import org.semanticweb.owlapi.profiles.violations.UseOfIllegalDataRange;
import org.semanticweb.owlapi.profiles.violations.UseOfNonEquivalentClassExpression;
import org.semanticweb.owlapi.profiles.violations.UseOfNonSubClassExpression;
import org.semanticweb.owlapi.profiles.violations.UseOfNonSuperClassExpression;
import org.semanticweb.owlapi.util.OWLOntologyWalker;
import org.semanticweb.owlapi.util.OWLOntologyWalkerVisitor;
import org.semanticweb.owlapi.vocab.OWL2Datatype;

/**
 * @author Matthew Horridge, The University of Manchester, Information
 *         Management Group
 */
public class OWL2RLProfile implements OWLProfile {

    protected static final Set<IRI> ALLOWED_DATATYPES = asUnorderedSet(OWL2Datatype.RL_DATATYPES.stream().map(i -> i
        .getIRI()));
    private final OWL2RLSubClassExpressionChecker subClassExpressionChecker = new OWL2RLSubClassExpressionChecker();
    private final OWL2RLSuperClassExpressionChecker superClassExpressionChecker = new OWL2RLSuperClassExpressionChecker();
    private final OWL2RLEquivalentClassExpressionChecker equivalentClassExpressionChecker = new OWL2RLEquivalentClassExpressionChecker();

    /**
     * Gets the name of the profile.
     * 
     * @return A string that represents the name of the profile
     */
    @Override
    public String getName() {
        return "OWL 2 RL";
    }

    @Override
    public IRI getIRI() {
        return Profiles.OWL2_RL.getIRI();
    }

    /**
     * Checks an ontology and its import closure to see if it is within this
     * profile.
     * 
     * @param ontology
     *        The ontology to be checked.
     * @return An {@code OWLProfileReport} that describes whether or not the
     *         ontology is within this profile.
     */
    @Override
    public OWLProfileReport checkOntology(OWLOntology ontology) {
        OWL2DLProfile profile = new OWL2DLProfile();
        OWLProfileReport report = profile.checkOntology(ontology);
        Set<OWLProfileViolation> violations = new HashSet<>();
        violations.addAll(report.getViolations());
        OWLOntologyProfileWalker walker = new OWLOntologyProfileWalker(ontology.importsClosure());
        OWL2RLObjectVisitor visitor = new OWL2RLObjectVisitor(walker);
        walker.walkStructure(visitor);
        violations.addAll(visitor.getProfileViolations());
        return new OWLProfileReport(this, violations);
    }

    private class OWL2RLObjectVisitor extends OWLOntologyWalkerVisitor {

        private final Set<OWLProfileViolation> violations = new HashSet<>();

        OWL2RLObjectVisitor(OWLOntologyWalker walker) {
            super(walker);
        }

        public Set<OWLProfileViolation> getProfileViolations() {
            return new HashSet<>(violations);
        }

        @Override
        public void visit(OWLClassAssertionAxiom axiom) {
            if (!isOWL2RLSuperClassExpression(axiom.getClassExpression())) {
                violations.add(new UseOfNonSuperClassExpression(getCurrentOntology(), axiom, axiom
                    .getClassExpression()));
            }
        }

        @Override
        public void visit(OWLDataPropertyDomainAxiom axiom) {
            if (!isOWL2RLSuperClassExpression(axiom.getDomain())) {
                violations.add(new UseOfNonSuperClassExpression(getCurrentOntology(), axiom, axiom.getDomain()));
            }
        }

        @Override
        public void visit(OWLDisjointClassesAxiom axiom) {
            axiom.classExpressions().filter(ce -> !isOWL2RLEquivalentClassExpression(ce)).forEach(ce -> violations.add(
                new UseOfNonSubClassExpression(getCurrentOntology(), axiom, ce)));
        }

        @Override
        public void visit(OWLDisjointUnionAxiom axiom) {
            violations.add(new UseOfIllegalAxiom(getCurrentOntology(), axiom));
        }

        @Override
        public void visit(OWLEquivalentClassesAxiom axiom) {
            axiom.classExpressions().filter(ce -> !isOWL2RLEquivalentClassExpression(ce)).forEach(ce -> violations.add(
                new UseOfNonEquivalentClassExpression(getCurrentOntology(), axiom, ce)));
        }

        @Override
        public void visit(OWLHasKeyAxiom axiom) {
            if (!isOWL2RLSubClassExpression(axiom.getClassExpression())) {
                violations.add(new UseOfNonSubClassExpression(getCurrentOntology(), axiom, axiom.getClassExpression()));
            }
        }

        @Override
        public void visit(OWLObjectPropertyDomainAxiom axiom) {
            if (!isOWL2RLSuperClassExpression(axiom.getDomain())) {
                violations.add(new UseOfNonSuperClassExpression(getCurrentOntology(), axiom, axiom.getDomain()));
            }
        }

        @Override
        public void visit(OWLObjectPropertyRangeAxiom axiom) {
            if (!isOWL2RLSuperClassExpression(axiom.getRange())) {
                violations.add(new UseOfNonSuperClassExpression(getCurrentOntology(), axiom, axiom.getRange()));
            }
        }

        @Override
        public void visit(OWLSubClassOfAxiom axiom) {
            if (!isOWL2RLSubClassExpression(axiom.getSubClass())) {
                violations.add(new UseOfNonSubClassExpression(getCurrentOntology(), axiom, axiom.getSubClass()));
            }
            if (!isOWL2RLSuperClassExpression(axiom.getSuperClass())) {
                violations.add(new UseOfNonSuperClassExpression(getCurrentOntology(), axiom, axiom.getSuperClass()));
            }
        }

        @Override
        public void visit(SWRLRule rule) {
            violations.add(new UseOfIllegalAxiom(getCurrentOntology(), rule));
        }

        @Override
        public void visit(OWLDataOneOf node) {
            violations.add(new UseOfIllegalDataRange(getCurrentOntology(), getCurrentAxiom(), node));
        }

        @Override
        public void visit(OWLDataComplementOf node) {
            violations.add(new UseOfIllegalDataRange(getCurrentOntology(), getCurrentAxiom(), node));
        }

        @Override
        public void visit(OWLDatatype node) {
            if (!ALLOWED_DATATYPES.contains(node.getIRI())) {
                violations.add(new UseOfIllegalDataRange(getCurrentOntology(), getCurrentAxiom(), node));
            }
        }

        @Override
        public void visit(OWLDatatypeRestriction node) {
            violations.add(new UseOfIllegalDataRange(getCurrentOntology(), getCurrentAxiom(), node));
        }

        @Override
        public void visit(OWLDataUnionOf node) {
            violations.add(new UseOfIllegalDataRange(getCurrentOntology(), getCurrentAxiom(), node));
        }

        @Override
        public void visit(OWLDatatypeDefinitionAxiom axiom) {
            violations.add(new UseOfIllegalAxiom(getCurrentOntology(), axiom));
        }
    }

    private class OWL2RLSubClassExpressionChecker implements OWLClassExpressionVisitorEx<Boolean> {

        OWL2RLSubClassExpressionChecker() {}

        @Override
        public Boolean doDefault(Object o) {
            return Boolean.FALSE;
        }

        @Override
        public Boolean visit(OWLClass ce) {
            return Boolean.valueOf(!ce.isOWLThing());
        }

        @Override
        public Boolean visit(OWLObjectIntersectionOf ce) {
            return Boolean.valueOf(!ce.operands().anyMatch(op -> !isOWL2RLSubClassExpression(op)));
        }

        @Override
        public Boolean visit(OWLObjectUnionOf ce) {
            return Boolean.valueOf(!ce.operands().anyMatch(op -> !isOWL2RLSubClassExpression(op)));
        }

        @Override
        public Boolean visit(OWLObjectSomeValuesFrom ce) {
            return Boolean.valueOf(ce.getFiller().isOWLThing() || isOWL2RLSubClassExpression(ce.getFiller()));
        }

        @Override
        public Boolean visit(OWLObjectHasValue ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLObjectOneOf ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLDataSomeValuesFrom ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLDataHasValue ce) {
            return Boolean.TRUE;
        }
    }

    protected boolean isOWL2RLSubClassExpression(OWLClassExpression ce) {
        return ce.accept(subClassExpressionChecker).booleanValue();
    }

    private class OWL2RLSuperClassExpressionChecker implements OWLClassExpressionVisitorEx<Boolean> {

        OWL2RLSuperClassExpressionChecker() {}

        @Override
        public Boolean doDefault(Object o) {
            return Boolean.FALSE;
        }

        @Override
        public Boolean visit(OWLClass ce) {
            return Boolean.valueOf(!ce.isOWLThing());
        }

        @Override
        public Boolean visit(OWLObjectIntersectionOf ce) {
            return Boolean.valueOf(!ce.operands().anyMatch(e -> !e.accept(this).booleanValue()));
        }

        // XXX difference in subclass and superclass - correct?
        @Override
        public Boolean visit(OWLObjectComplementOf ce) {
            return Boolean.valueOf(isOWL2RLSubClassExpression(ce.getOperand()));
        }

        @Override
        public Boolean visit(OWLObjectAllValuesFrom ce) {
            return ce.getFiller().accept(this);
        }

        @Override
        public Boolean visit(OWLObjectHasValue ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLObjectMaxCardinality ce) {
            return Boolean.valueOf((ce.getCardinality() == 0 || ce.getCardinality() == 1) && (ce.getFiller()
                .isOWLThing() || isOWL2RLSubClassExpression(ce.getFiller())));
        }

        @Override
        public Boolean visit(OWLDataAllValuesFrom ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLDataHasValue ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLDataMaxCardinality ce) {
            return Boolean.valueOf(ce.getCardinality() == 0 || ce.getCardinality() == 1);
        }
    }

    /**
     * @param ce
     *        class
     * @return true if OWL 2 RL superclass
     */
    public boolean isOWL2RLSuperClassExpression(OWLClassExpression ce) {
        return ce.accept(superClassExpressionChecker).booleanValue();
    }

    private static class OWL2RLEquivalentClassExpressionChecker implements OWLClassExpressionVisitorEx<Boolean> {

        OWL2RLEquivalentClassExpressionChecker() {}

        @Override
        public Boolean doDefault(Object o) {
            return Boolean.FALSE;
        }

        @Override
        public Boolean visit(OWLClass ce) {
            return Boolean.valueOf(!ce.isOWLThing());
        }

        @Override
        public Boolean visit(OWLObjectIntersectionOf ce) {
            return Boolean.valueOf(!ce.operands().anyMatch(e -> !e.accept(this).booleanValue()));
        }

        @Override
        public Boolean visit(OWLObjectHasValue ce) {
            return Boolean.TRUE;
        }

        @Override
        public Boolean visit(OWLDataHasValue ce) {
            return Boolean.TRUE;
        }
    }

    /**
     * @param ce
     *        class
     * @return true if equivalent classes expression
     */
    public boolean isOWL2RLEquivalentClassExpression(OWLClassExpression ce) {
        return ce.accept(equivalentClassExpressionChecker).booleanValue();
    }
}
