/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.time.Duration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.profile.config.navigate.RevocationLifetimeLookupFunction;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Default lookup function for fetching the chain revocation lifetime. This inherits the functionality of
 * {@link RevocationLifetimeLookupFunction} but also adds the configurable clock skew value and additional 5 minutes
 * to the returned value.
 */
public class DefaultChainRevocationLifetimeLookupStrategy extends RevocationLifetimeLookupFunction {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(DefaultChainRevocationLifetimeLookupStrategy.class);

    /** Positive clock skew adjustment to consider when calculating revocation lifetime. */
    @Nonnull private Duration clockSkew;

    /**
     * Constructor.
     */
    public DefaultChainRevocationLifetimeLookupStrategy() {
        clockSkew = Duration.ofMinutes(5);
    }

    /**
     * Set the clock skew.
     * 
     * @param skew clock skew to set
     */
    public void setClockSkew(@Nonnull final Duration skew) {
        clockSkew = Constraint.isNotNull(skew, "Clock skew cannot be null").abs();
    }

    /** {@inheritDoc} */
    @Override
    @Nullable
    public Duration apply(@Nullable final ProfileRequestContext input) {
        final Duration profileDuration = super.apply(input);
        if (profileDuration == null || profileDuration.isZero()) {
            log.debug("No chain expiration time could be resolved, returning null");
            return null;            
        }
        return profileDuration.plus(Duration.ofMinutes(5)).plus(clockSkew);
    }
    
}