/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseConsentContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.TokenRequestConsentedAttributesLookupFunction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Action that locates consent from authorization code / access token. For located consent
 * {@link OIDCAuthenticationResponseConsentContext} is created under {@link OIDCAuthenticationResponseContext} and the
 * consent placed there. Token and user info end points use the consent context for forming response.
 */
public class SetConsentFromTokenToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetConsentFromTokenToResponseContext.class);

    /** Strategy used to obtain the consented attributes. */
    @Nonnull private Function<ProfileRequestContext, List<Object>> consentedAttributesLookupStrategy;

    /**
     * Constructor.
     */
    public SetConsentFromTokenToResponseContext() {
        consentedAttributesLookupStrategy = new TokenRequestConsentedAttributesLookupFunction();
    }

    /**
     * Set the strategy used to locate the consented attributes.
     * 
     * @param strategy lookup strategy
     */
    public void setConsentedAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,List<Object>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        consentedAttributesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final List<Object> consentedAttributes = consentedAttributesLookupStrategy.apply(profileRequestContext);
        if (consentedAttributes != null) {
            final OIDCAuthenticationResponseConsentContext consentClaimsCtx =
                    getOidcResponseContext().getSubcontext(OIDCAuthenticationResponseConsentContext.class, true);
            consentClaimsCtx.getConsentedAttributes().addAll(consentedAttributes);
        }
    }

}