/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationMetadataPolicyContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCClientRegistrationTokenMetadataPolicyLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.DefaultMetadataPolicyMergingStrategy;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.oidc.metadata.policy.impl.DefaultMetadataPolicyValidator;
import net.shibboleth.oidc.profile.config.navigate.RegistrationMetadataPolicyLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Initializes the {@link OIDCClientRegistrationMetadataPolicyContext} and attaches it as a subcontext for the incoming
 * {@link MessageContext}. The context contains the merged metadata policy, containing the policy items defined in the
 * profile configuration and initial registration access token (if involved).
 */
public class InitializeRegistrationMetadataPolicyContext extends AbstractProfileAction {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(InitializeRegistrationMetadataPolicyContext.class);

    /** The strategy used to locate the request metadata policy configured for the profile. */
    @NonnullAfterInit private Function<ProfileRequestContext, Map<String, MetadataPolicy>>
        profileMetadataPolicyLookupStrategy;

    /** The strategy used to locate the request metadata policy encoded in the access token. */
    @NonnullAfterInit private Function<ProfileRequestContext, Map<String, MetadataPolicy>>
        tokenMetadataPolicyLookupStrategy;
    
    /** The strategy used to create or locate the metadata policy context. */
    @NonnullAfterInit private Function<ProfileRequestContext,OIDCClientRegistrationMetadataPolicyContext>
        registrationPolicyContextCreationStrategy;
    
    /** The strategy used for merging profile and token based metadata policies. */
    @NonnullAfterInit private BiFunction<Map<String, MetadataPolicy>, Map<String, MetadataPolicy>,
        Pair<Map<String, MetadataPolicy>, Boolean>> metadataPolicyMergingStrategy;
    
    /** The strategy used for validating token and merged metadata policies. */
    @NonnullAfterInit private Predicate<Map<String, MetadataPolicy>> metadataPolicyValidationStrategy;
    
    /** The metadata policy context to operate on. */
    private OIDCClientRegistrationMetadataPolicyContext metadataPolicyContext;

    /**
     * Constructor.
     */
    public InitializeRegistrationMetadataPolicyContext() {
        profileMetadataPolicyLookupStrategy = new RegistrationMetadataPolicyLookupFunction();
        tokenMetadataPolicyLookupStrategy = new DefaultOIDCClientRegistrationTokenMetadataPolicyLookupFunction();
        registrationPolicyContextCreationStrategy = 
                new ChildContextLookup<>(OIDCClientRegistrationMetadataPolicyContext.class, true).compose(
                        new InboundMessageContextLookup());
        metadataPolicyMergingStrategy = new DefaultMetadataPolicyMergingStrategy();
        metadataPolicyValidationStrategy = new DefaultMetadataPolicyValidator();
    }

    /**
     * Set the strategy used to locate the request metadata policy configured for the profile.
     * 
     * @param strategy What to set.
     */
    public void setProfileMetadataPolicyLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, Map<String, MetadataPolicy>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        profileMetadataPolicyLookupStrategy = Constraint.isNotNull(strategy,
                "Profile metadata policy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the request metadata policy encoded in the access token.
     * 
     * @param strategy What to set.
     */
    public void setTokenMetadataPolicyLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, Map<String, MetadataPolicy>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        tokenMetadataPolicyLookupStrategy = Constraint.isNotNull(strategy,
                "Token metadata policy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to create or return the {@link OIDCClientRegistrationMetadataPolicyContext}.
     * 
     * @param strategy What to set.
     */
    public void setRegistrationPolicyContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCClientRegistrationMetadataPolicyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        registrationPolicyContextCreationStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationMetadataPolicyContext creation strategy cannot be null");
    }
    
    /**
     * Set the strategy used for merging profile and token based metadata policies.
     * 
     * @param strategy What to set.
     */
    public void setMetadataPolicyMergingStrategy(@Nonnull final BiFunction<Map<String,MetadataPolicy>,
            Map<String,MetadataPolicy>, Pair<Map<String, MetadataPolicy>, Boolean>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        metadataPolicyMergingStrategy = Constraint.isNotNull(strategy,
                "Metadata policy merging strategy cannot be null");
    }
    
    /**
     * Set the strategy used for validating token and merged metadata policies.
     * 
     * @param strategy What to set.
     */
    public void setMetadataPolicyValidationStrategy(@Nonnull final Predicate<Map<String, MetadataPolicy>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        metadataPolicyValidationStrategy = Constraint.isNotNull(strategy,
                "Metadata policy validation strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        metadataPolicyContext = registrationPolicyContextCreationStrategy.apply(profileRequestContext);
        if (metadataPolicyContext == null) {
            log.error("{} Registration metadata policy context could not be created, invalid profile context",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final Map<String, MetadataPolicy> profileMetadataPolicy =
                profileMetadataPolicyLookupStrategy.apply(profileRequestContext);
        
        final Map<String, MetadataPolicy> tokenMetadataPolicy =
                tokenMetadataPolicyLookupStrategy.apply(profileRequestContext);
        
        if (!metadataPolicyValidationStrategy.test(tokenMetadataPolicy)) {
            log.warn("{} Metadata policy in token is invalid", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;            
        }
        
        final Pair<Map<String, MetadataPolicy>, Boolean> mergedResult =
                metadataPolicyMergingStrategy.apply(profileMetadataPolicy, tokenMetadataPolicy);
        
        if (!mergedResult.getSecond()) {
            log.warn("{} Metadata policies from profile and token could not be merged", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;
        }
        
        final Map<String, MetadataPolicy> mergedPolicy = mergedResult.getFirst();
        if (!metadataPolicyValidationStrategy.test(mergedPolicy)) {
            log.warn("{} Merged metadata policy is invalid", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;
        }
        
        metadataPolicyContext.setMetadataPolicy(mergedPolicy);
    }
    
}