/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.time.Instant;
import java.util.Date;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.client.ClientInformationResponse;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformationResponse;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationResponseContext;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * An action that uses the information from {@link OIDCClientRegistrationResponseContext} attached to the message
 * context for creating a new {@link ClientInformationResponse}. It will be set as the outbound message.
 */
public class BuildClientInformation extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(BuildClientInformation.class);
    
    /**
     * Strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given 
     * {@link MessageContext}.
     */
    @Nonnull private Function<MessageContext,OIDCClientRegistrationResponseContext> oidcResponseContextLookupStrategy;

    /** The {@link MessageContext} to operate on. */
    private MessageContext messageContext;

    /** The {@link OIDCClientRegistrationResponseContext} to operate on. */
    private OIDCClientRegistrationResponseContext oidcResponseContext;

    /** Constructor. */
    public BuildClientInformation() {
        oidcResponseContextLookupStrategy = new ChildContextLookup<>(OIDCClientRegistrationResponseContext.class);
    }
    
    /**
     * Set the strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given
     * {@link MessageContext}.
     * 
     * @param strategy strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a 
     *         given {@link MessageContext}
     */
    public void setOidcResponseContextLookupStrategy(
            @Nonnull final Function<MessageContext,OIDCClientRegistrationResponseContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        oidcResponseContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationResponseContext lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        messageContext = profileRequestContext.getOutboundMessageContext();
        if (messageContext == null) {
            log.error("{} No message context found", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        oidcResponseContext = oidcResponseContextLookupStrategy.apply(messageContext);
        if (oidcResponseContext == null) {
            log.error("{} No OIDC response context found", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }

        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final String id = oidcResponseContext.getClientId();
        if (StringSupport.trimOrNull(id) == null) {
            log.error("{} No client ID in the OIDC response context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return;
        }
        final ClientID clientId = new ClientID(id);
        
        final OIDCClientMetadata metadata = oidcResponseContext.getClientMetadata();
        if (metadata == null) {
            log.error("{} No client metadata in the OIDC response context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return;
        }

        final ClientAuthenticationMethod tokenAuthMethod = metadata.getTokenEndpointAuthMethod();

        final boolean secretNeeded = tokenAuthMethod == null ||
                tokenAuthMethod.equals(ClientAuthenticationMethod.CLIENT_SECRET_BASIC) ||
                tokenAuthMethod.equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT) ||
                tokenAuthMethod.equals(ClientAuthenticationMethod.CLIENT_SECRET_POST);

        final Secret clientSecret;
        if (secretNeeded) {
            if (StringSupport.trimOrNull(oidcResponseContext.getClientSecret()) == null) {
                log.error("{} No required client secret in the OIDC response context", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
                return;
            }
            final Instant secretExpiresAt = oidcResponseContext.getClientSecretExpiresAt();
            if (secretExpiresAt != null) {
                log.warn("{} client secret expiration time {} is ignored", getLogPrefix(), secretExpiresAt);
            }
            clientSecret = new Secret(oidcResponseContext.getClientSecret());
        } else {
            clientSecret = null;
        }
        
        final OIDCClientInformation clientInformation = new OIDCClientInformation(clientId, new Date(), 
                metadata, clientSecret);
        final OIDCClientInformationResponse response = new OIDCClientInformationResponse(clientInformation, true);
        messageContext.setMessage(response);
        log.info("{} Client information successfully added to the outbound context", getLogPrefix());
        
    }    
}