/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.decoding.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.decoder.MessageDecodingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.MoreObjects;
import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.client.ClientRegistrationRequest;
import com.nimbusds.oauth2.sdk.http.HTTPRequest;
import com.nimbusds.oauth2.sdk.http.ServletUtils;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientRegistrationRequest;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.plugin.oidc.op.oauth2.decoding.impl.BaseOAuth2RequestDecoder;

/**
 * Message decoder decoding OpenID Connect {@link ClientRegistrationRequest}s.
 */
public class OIDCClientRegistrationRequestDecoder extends BaseOAuth2RequestDecoder<OIDCClientRegistrationRequest> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(OIDCClientRegistrationRequestDecoder.class);

    /** {@inheritDoc} */
    @Override
    protected OIDCClientRegistrationRequest parseMessage() throws MessageDecodingException {
        try {
            final HTTPRequest httpRequest = ServletUtils.createHTTPRequest(getHttpServletRequest());
            getProtocolMessageLog().trace("Inbound request {}", RequestUtil.toString(httpRequest));
            final JSONObject requestJson = httpRequest.getQueryAsJSONObject();
            //TODO: Nimbus seems to be interpreting scope in different way as many RPs, currently the scope
            //is removed in this phase, better solution TODO.
            if (requestJson.containsKey("scope")) {
                log.debug("Removed 'scope'");
                requestJson.remove("scope");
                httpRequest.setQuery(requestJson.toJSONString());
            }
            
            log.trace("JSON object: {}", httpRequest.getQueryAsJSONObject().toJSONString());
            return OIDCClientRegistrationRequest.parse(httpRequest);
        } catch (final IOException e) {
            log.error("Could not create HTTP request from the request", e);
            throw new MessageDecodingException(e);
        } catch (final ParseException e) {
            log.error("Unable to decode oidc request: {}", e.getMessage());
            throw new MessageDecodingException(e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected String getMessageToLog(@Nullable final OIDCClientRegistrationRequest message) {
        return message == null ? null : MoreObjects.toStringHelper(this).omitNullValues()
                .add("accessToken", RequestUtil.getAccessTokenLog(message.getAccessToken()))
                .add("clientMetadata", message.getClientMetadata())
                .add("endpointURI", message.getEndpointURI())
                .add("oidcClientMetadata", message.getOIDCClientMetadata())
                .add("softwareStatement", message.getSoftwareStatement() == null ? null :
                    message.getSoftwareStatement().serialize())
                .toString();
    }

}