/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.attribute.filter.matcher.impl;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.navigate.RecursiveTypedParentContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.IdPRequestedAttribute;
import net.shibboleth.idp.attribute.filter.Matcher;
import net.shibboleth.idp.attribute.filter.context.AttributeFilterContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentSupport;

/** Class for matching attribute to requested claims. */
public class AttributeInOIDCRequestedClaimsMatcher extends AbstractIdentifiableInitializableComponent
        implements Matcher {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(AttributeInOIDCRequestedClaimsMatcher.class);

    /** Whether to return a match if the request contains no requested claims. */
    private boolean matchIfRequestedClaimsSilent;

    /** Whether to look for a match only in id token part. */
    private boolean matchOnlyIDToken;

    /** Whether to look for a match only in user info part. */
    private boolean matchOnlyUserInfo;

    /** Whether to drop non essential claims. */
    private boolean onlyIfEssential;

    /** The String used to prefix log message. */
    private String logPrefix;

    /**
     * Gets whether to drop non essential claims.
     * 
     * @return whether to drop non essential claims
     */
    public boolean getOnlyIfEssential() {
        return onlyIfEssential;
    }

    /**
     * Sets whether to drop non essential claims.
     * 
     * @param flag whether to drop non essential claims
     */
    public void setOnlyIfEssential(final boolean flag) {
        onlyIfEssential = flag;
    }

    /**
     * Gets whether to match only id token part of the requested claims.
     * 
     * @return whether to match only id token part of the requested claims
     */
    public boolean getMatchOnlyIDToken() {
        return matchOnlyIDToken;
    }

    /**
     * Sets whether to match only id token part of the requested claims.
     * 
     * @param flag whether to match only id token part of the requested claims
     */
    public void setMatchOnlyIDToken(final boolean flag) {
        this.matchOnlyIDToken = flag;
    }

    /**
     * Gets whether to match only user info part of the requested claims.
     * 
     * @return whether to match only user info part of the requested claims
     */
    public boolean getMatchOnlyUserInfo() {
        return matchOnlyUserInfo;
    }

    /**
     * Sets whether to match only user info part of the requested claims.
     * 
     * @param flag whether to match only user info part of the requested claims
     */
    public void setMatchOnlyUserInfo(final boolean flag) {
        this.matchOnlyUserInfo = flag;
    }

    /**
     * Gets whether to matched if the request contains no requested claims.
     * 
     * @return whether to match if the request contains no requested claims
     */
    public boolean getMatchIRequestedClaimsSilent() {
        return matchIfRequestedClaimsSilent;
    }

    /**
     * Sets whether to match if the request contains no requested claims.
     * 
     * @param flag whether to match if the request contains no requested claims
     */
    public void setMatchIfRequestedClaimsSilent(final boolean flag) {
        matchIfRequestedClaimsSilent = flag;
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    public Set<IdPAttributeValue> getMatchingValues(@Nonnull final IdPAttribute attribute,
            @Nonnull final AttributeFilterContext filtercontext) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);

        final ProfileRequestContext profileRequestContext = new RecursiveTypedParentContextLookup<>(
                ProfileRequestContext.class).apply(filtercontext);
        if (profileRequestContext == null || profileRequestContext.getOutboundMessageContext() == null) {
            log.trace("{} No outbound message context", getLogPrefix());
            return null;
        }
        final OIDCAuthenticationResponseContext respCtx = profileRequestContext.getOutboundMessageContext()
                .getSubcontext(OIDCAuthenticationResponseContext.class);
        if (respCtx == null) {
            // This is always a failure.
            log.debug("{} No oidc response ctx for this comparison", getLogPrefix());
            return null;
        }
        
        if (respCtx.getMappedIdTokenRequestedClaims() == null && respCtx.getMappedUserinfoRequestedClaims() == null) {
            log.debug("{} No requested claims", getLogPrefix());
            if (getMatchIRequestedClaimsSilent()) {
                log.debug("{} all values matched as in silent mode", getLogPrefix());
                return Set.copyOf(attribute.getValues());
            } else {
                log.debug("{} none of the values matched as not silent mode", getLogPrefix());
                return Collections.emptySet();
            }
        }
        
        // Are we able to release the values based on claim being requested for id token?
        if (respCtx.getMappedIdTokenRequestedClaims() != null && !getMatchOnlyUserInfo()) {
            if (respCtx.getMappedIdTokenRequestedClaims().get().containsKey(attribute.getId())) {
                if (verifyEssentiality(respCtx.getMappedIdTokenRequestedClaims().get().get(attribute.getId()))) {
                    log.debug("{} All values matched, as {} is a requested ID token claim", getLogPrefix(),
                            attribute.getId());
                    return Set.copyOf(attribute.getValues());
                }
            }
        }
        
        // Are we able to release the values based on claim being requested for user info response?
        if (respCtx.getMappedUserinfoRequestedClaims() != null && !getMatchOnlyIDToken()) {
            if (respCtx.getMappedUserinfoRequestedClaims().get().containsKey(attribute.getId())) {
                if (verifyEssentiality(respCtx.getMappedUserinfoRequestedClaims().get().get(attribute.getId()))) {
                    log.debug("{} All values matched, as {} is a requested Userinfo claim", getLogPrefix(),
                            attribute.getId());
                    return Set.copyOf(attribute.getValues());
                }
            }
        }

        log.debug("{} Attribute {} was not a requested claim, none of the values matched", getLogPrefix(),
                attribute.getId());
        return Collections.emptySet();
    }
// Checkstyle: CyclomaticComplexity ON

    /**
     * Checks whether any of the matching requested claims have the required flag set if necessary.
     * 
     * @param claims claims request claims
     * 
     * @return whether any of the matching requested claims have the required flag set if necessary
     */
    private boolean verifyEssentiality(final Collection<IdPAttribute> claims) {
        if (onlyIfEssential) {
            return claims
                    .stream()
                    .filter(IdPRequestedAttribute.class::isInstance)
                    .map(IdPRequestedAttribute.class::cast)
                    .anyMatch(a -> a.isRequired());
        }
        
        return true;
    }
    
    /**
     * Return a string which is to be prepended to all log messages.
     * 
     * @return "Attribute Filter '&lt;filterID&gt;' :"
     */
    @Nonnull
    protected String getLogPrefix() {
        String prefix = logPrefix;
        if (null == prefix) {
            final StringBuilder builder = new StringBuilder("Attribute Filter '").append(getId()).append("':");
            prefix = builder.toString();
            if (null == logPrefix) {
                logPrefix = prefix;
            }
        }
        return prefix;
    }

}
