/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import com.nimbusds.openid.connect.sdk.SubjectType;

import net.shibboleth.utilities.java.support.logic.Constraint;

/** Activation condition returning true if public subject is requested. */
public class PublicSubjectActivationCondition implements Predicate<ProfileRequestContext> {
    
    /** Strategy used to obtain subject type. */
    @Nonnull private Function<ProfileRequestContext,SubjectType> subjectTypeLookupStrategy;
    

    /**
     * Constructor.
     */
    public PublicSubjectActivationCondition() {
        subjectTypeLookupStrategy = new DefaultSubjectTypeStrategy();
    }

    /**
     * Get the strategy used to locate subject type.
     * 
     * @return lookup strategy
     */
    @Nonnull public Function<ProfileRequestContext,SubjectType> getSubjectTypeLookupStrategy() {
        return subjectTypeLookupStrategy;
    }
    
    /**
     * Set the strategy used to locate subject type.
     * 
     * @param strategy lookup strategy
     */
    public void setSubjectTypeLookupStrategy(@Nonnull final Function<ProfileRequestContext, SubjectType> strategy) {
        subjectTypeLookupStrategy =
                Constraint.isNotNull(strategy, "SubjectTypeLookupStrategy lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    public boolean test(@Nullable final ProfileRequestContext input) {
        return SubjectType.PUBLIC.equals(subjectTypeLookupStrategy.apply(input));
    }
    
}