/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.TokenRequestSubjectLookupFunction;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.SubjectType;

/**
 * Action that locates subject using strategy. Located subject is set to {@link OIDCAuthenticationResponseContext}.
 **/
public class SetSubjectToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetSubjectToResponseContext.class);

    /** Strategy used to obtain the subject. */
    @Nonnull private Function<ProfileRequestContext, String> subjectLookupStrategy;

    /** Strategy used to determine the subject type to try. */
    @Nonnull private Function<ProfileRequestContext, SubjectType> subjectTypeStrategy;

    /** Subject type. */
    @Nonnull private SubjectType subjectType;

    /**
     * Constructor.
     */
    public SetSubjectToResponseContext() {
        subjectLookupStrategy = new TokenRequestSubjectLookupFunction();
    }

    /**
     * Set the strategy function to use to obtain the subject type.
     * 
     * @param strategy subject type lookup strategy
     */
    public void setSubjectTypeLookupStrategy(@Nonnull final Function<ProfileRequestContext, SubjectType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        subjectTypeStrategy = Constraint.isNotNull(strategy, "Subject type lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate subject.
     * 
     * @param strategy lookup strategy
     */
    public void setSubjectLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        subjectLookupStrategy = Constraint.isNotNull(strategy, "Subject lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final String subject = subjectLookupStrategy.apply(profileRequestContext);
        if (subject == null) {
            log.error("{} Subject may not be null", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_SUBJECT);
            return;
        }
        getOidcResponseContext().setSubject(subject);
        if (subjectTypeStrategy != null) {
            getOidcResponseContext()
                    .setSubjectType(SubjectType.PUBLIC.equals(subjectTypeStrategy.apply(profileRequestContext))
                            ? "public" : "pairwise");
        }
    }

}