/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Collection;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.nimbusds.openid.connect.sdk.OIDCClaimsRequest;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSetRequest;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.attribute.AttributeDecodingException;
import net.shibboleth.idp.attribute.AttributesMapContainer;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoder;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscoderSupport;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestedClaimsLookupFunction;
import net.shibboleth.idp.profile.ActionSupport;
import net.shibboleth.utilities.java.support.annotation.constraint.Live;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.service.ReloadableService;
import net.shibboleth.utilities.java.support.service.ServiceableComponent;

/**
 * Action that sets requested claims to response context. For instance attribute filtering may use this information.
 * 
 * <p>This also executes a reverse mapping of the requested claims to map them back into RequestedIdPAttributes so
 * the filtering mechanism is freed from that role.</p>
 */
public class SetRequestedClaimsToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SetRequestedClaimsToResponseContext.class);
    
    /** Strategy used to obtain the requested claims of request. */
    @Nonnull private Function<ProfileRequestContext, OIDCClaimsRequest> requestedClaimsLookupStrategy;

    /** Transcoder registry service object. */
    @NonnullAfterInit private ReloadableService<AttributeTranscoderRegistry> transcoderRegistry;

    /**
     * Constructor.
     */
    public SetRequestedClaimsToResponseContext() {
        requestedClaimsLookupStrategy = new DefaultRequestedClaimsLookupFunction();
    }

    /**
     * Sets the registry of transcoding rules to apply to encode attributes.
     * 
     * @param registry registry service interface
     */
    public void setTranscoderRegistry(@Nonnull final ReloadableService<AttributeTranscoderRegistry> registry) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        transcoderRegistry = Constraint.isNotNull(registry, "AttributeTranscoderRegistry cannot be null");
    }

    /**
     * Set the strategy used to locate the requested claims of request.
     * 
     * @param strategy lookup strategy
     */
    public void setRequestedClaimsLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCClaimsRequest> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        requestedClaimsLookupStrategy =
                Constraint.isNotNull(strategy, "RequestedClaimsLookupStrategy lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (transcoderRegistry == null) {
            throw new ComponentInitializationException("AttributeTranscoderRegistry cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final OIDCClaimsRequest cr = requestedClaimsLookupStrategy.apply(profileRequestContext);
        getOidcResponseContext().setRequestedClaims(cr);
        if (cr == null) {
            getOidcResponseContext().setMappedIdTokenRequestedClaims(null);
            getOidcResponseContext().setMappedUserinfoRequestedClaims(null);
            return;
        }
        
        ServiceableComponent<AttributeTranscoderRegistry> component = null;
        try {
            component = transcoderRegistry.getServiceableComponent();
            if (component == null) {
                log.error("Attribute transoding service unavailable");
                ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_PROC_ERROR);
                return;
            }
            
            // This is awful but it's a stopgap. Wrap the ClaimsRequest entry in a JSONObject
            // wrapper to maintain compatibility with the transcoders. The TODO is to build
            // a new claims and requested claims abstraction ourselves and map the data in and
            // out.
            
            Multimap<String,IdPAttribute> results = HashMultimap.create();
            if (cr.getIDTokenClaimsRequest() != null) {
                for (final ClaimsSetRequest.Entry entry : cr.getIDTokenClaimsRequest().getEntries()) {
                                
                    final JSONObject wrapper = new JSONObject();
                    wrapper.put(entry.getClaimName(), entry);
                    
                    final Collection<TranscodingRule> transcodingRules =
                            component.getComponent().getTranscodingRules(wrapper);
                    decodeAttribute(profileRequestContext, transcodingRules, wrapper, results);
                }
            }
            getOidcResponseContext().setMappedIdTokenRequestedClaims(results.isEmpty() ? null :
                    new AttributesMapContainer(results));
            
            results = HashMultimap.create();
            if (cr.getUserInfoClaimsRequest() != null) {
                for (final ClaimsSetRequest.Entry entry : cr.getUserInfoClaimsRequest().getEntries()) {
                                
                    final JSONObject wrapper = new JSONObject();
                    wrapper.put(entry.getClaimName(), entry);
                
                    final Collection<TranscodingRule> transcodingRules =
                            component.getComponent().getTranscodingRules(wrapper);
                    decodeAttribute(profileRequestContext, transcodingRules, wrapper, results);
                }
            }
            getOidcResponseContext().setMappedUserinfoRequestedClaims(results.isEmpty() ? null :
                new AttributesMapContainer(results));
            
        } finally {
            if (component != null) {
                component.unpinComponent();
            }
        }
    }

    /**
     * Access the registry of transcoding rules to decode the input object.
     * 
     * @param profileRequestContext profile request context
     * @param rules transcoding rules
     * @param input input object
     * @param results collection to add results to
     */
    private void decodeAttribute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull @NonnullElements final Collection<TranscodingRule> rules, @Nonnull final JSONObject input,
            @Nonnull @NonnullElements @Live final Multimap<String,IdPAttribute> results) {
                
        for (final TranscodingRule rule : rules) {
            final AttributeTranscoder<JSONObject> transcoder = TranscoderSupport.getTranscoder(rule);
            try {
                final IdPAttribute decodedAttribute = transcoder.decode(profileRequestContext, input, rule);
                if (decodedAttribute != null) {
                    results.put(decodedAttribute.getId(), decodedAttribute);
                }
            } catch (final AttributeDecodingException e) {
                log.warn("{} Failed to decode requested claim '{}'", getLogPrefix(), input.keySet().iterator().next());
            }
        }
    }

}