/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseConsentContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.profile.config.logic.AttributeConsentFlowEnabledPredicate;
import net.shibboleth.oidc.profile.config.logic.EncodeConsentPredicate;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.context.AttributeContext;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that checks for adds the currently existing attributes from {@link AttributeContext} for token delivery. They
 * are assumed to be consented, if they exist in the context. The (consent) information is stored to
 * {@link OIDCAuthenticationResponseTokenClaimsContext} that is created under {@link OIDCAuthenticationResponseContext}.
 **/
public class SetConsentToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetConsentToResponseContext.class);

    /**
     * Strategy used to locate the {@link AttributeContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext,AttributeContext> attributeContextLookupStrategy;
    
    /**
     * Predicate used to check if consent is enabled.
     */
    @Nonnull private Predicate<ProfileRequestContext> consentEnabledPredicate;

    /**
     * Predicate used to check if consent should be embedded in tokens. 
     */
    @Nonnull private Predicate<ProfileRequestContext> encodeConsentPredicate;

    /** AttributeContext to use. */
    @Nullable private AttributeContext attributeCtx;

    /** Constructor. */
    SetConsentToResponseContext() {
        attributeContextLookupStrategy = new ChildContextLookup<>(AttributeContext.class).compose(
                new ChildContextLookup<>(RelyingPartyContext.class));
        consentEnabledPredicate = new AttributeConsentFlowEnabledPredicate();
        encodeConsentPredicate = new EncodeConsentPredicate();
    }

    /**
     * Set the strategy used to locate the {@link AttributeContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link AttributeContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setAttributeContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, AttributeContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        attributeContextLookupStrategy =
                Constraint.isNotNull(strategy, "AttributeContext lookup strategy cannot be null");
    }

    /**
     * Set the predicate used to check if consent is enable.
     * 
     * @param predicate predicate used to check if consent is enabled
     */
    public void setConsentEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        consentEnabledPredicate = Constraint.isNotNull(predicate, "Predicate cannot be null");
    }

    /**
     * Set the predicate used to check if consent should be encoded.
     * 
     * @param predicate predicate used to check if consent should be encoded
     */
    public void setEncodeConsentPredicate(@Nonnull final Predicate<ProfileRequestContext> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        encodeConsentPredicate = Constraint.isNotNull(predicate, "Predicate cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        if (!consentEnabledPredicate.test(profileRequestContext)) {
            log.debug("{} Attribute consent has not been enabled, nothing to do", getLogPrefix());
            return false;
        }
        
        attributeCtx = attributeContextLookupStrategy.apply(profileRequestContext);
        if (attributeCtx == null) {
            log.debug("{} No AttributeSubcontext available, nothing to do", getLogPrefix());
            return false;
        }

        if (!attributeCtx.isConsented() && !encodeConsentPredicate.test(profileRequestContext)) {
            log.debug("{} Consent is not being encoded into tokens, nothing to do", getLogPrefix());
            return false;
        }
                
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final OIDCAuthenticationResponseConsentContext oidcConsentCtx =
                getOidcResponseContext().getSubcontext(OIDCAuthenticationResponseConsentContext.class, true);

        final Map<String, IdPAttribute> consented = attributeCtx.getIdPAttributes();
        oidcConsentCtx.getConsentedAttributes().addAll(consented.keySet());
        log.debug("{} Set to response context consented attributes {} and consentable attributes {}", getLogPrefix(),
                oidcConsentCtx.getConsentedAttributes().toJSONString());

    }
}