/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.sdk.impl;


import java.text.ParseException;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.duosecurity.Client;
import com.duosecurity.exception.DuoException;
import com.duosecurity.model.HealthCheckResponse;
import com.duosecurity.model.Token;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.authn.duo.AbstractDuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.model.DuoHealthCheck;
import net.shibboleth.idp.plugin.authn.duo.model.DuoHealthCheckResponse;
import net.shibboleth.oidc.security.impl.JWSAssemblyUtils;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.codec.EncodingException;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * <p>An Object Adaptor class for bridging between the Duo SDK implementation 
 * and the internal {@link DuoOIDCClient} interface.</p>
 */
@ThreadSafe
@Immutable
public final class DuoSDKClientAdaptor extends AbstractDuoOIDCClient{
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DuoSDKClientAdaptor.class);
    
    /** The wrapped Duo native client.*/
    @Nonnull private final Client client;
    
    /** 
     * Function to map the native Duo {@link HealthCheckResponse} object to the 
     * interface {@link DuoHealthCheck} object.
     */
    @Nonnull private final Function<HealthCheckResponse,DuoHealthCheck> healthCheckResponseConverter;
    
    /** Function to map the native Duo {@link Token} object to the interface {@link JWT} object.*/
    @Nonnull private final BiFunction<Token, DuoOIDCIntegration, JWT> tokenResponseConverter;  
    
    /** Save off the integration to help generate the JWT.*/
    @Nonnull private final DuoOIDCIntegration duoIntegration;
    
    /**
     * 
     * Package-private constructor. Initialises the native Duo SDK client.
     * 
     * <p>Should only be instantiated by the {@link DuoSDKClientFactory}.</p>
     *
     * @param integration the Duo integration to initialize the client from. Never {@code null}.
     * @param caCerts the list of CA Certificates used to validate connections to Duo. Can be {@code null}.
     * 
     * @throws DuoClientException if there is an error instantiating the client
     */
     DuoSDKClientAdaptor(@Nonnull final DuoOIDCIntegration integration, 
            @Nullable final List<String> caCerts) throws DuoClientException {
        super();
        duoIntegration = Constraint.isNotNull(integration,"Duo SDK Client requires a non-null Duo Integration");
        healthCheckResponseConverter = new DefaultHealthCheckResponseConverter();
        tokenResponseConverter = new DefaultTokenResponseConverter();

        try {
            if (caCerts == null) {
                //will use the default certs in the Client if the caCerts are null
                client = new Client.Builder(integration.getClientId(), integration.getSecretKey(),
                        integration.getAPIHost(), integration.getRedirectURI()).setUseDuoCodeAttribute(false).build();
            } else {
                client = new Client.Builder(integration.getClientId(), integration.getSecretKey(),
                        integration.getAPIHost(), integration.getRedirectURI()).setCACerts(caCerts.toArray(new String[caCerts.size()]))
                        .setUseDuoCodeAttribute(false).build();
            }
        } catch (final DuoException e) {
            //wrap exception and throw
            throw new DuoClientException(e);
        }
 
    }


    /** {@inheritDoc} */
    @Override
    @Nonnull public DuoHealthCheck healthCheck() throws DuoClientException {        
        try {
            final HealthCheckResponse response = client.healthCheck();
            if (response == null) {
                throw new DuoClientException("Duo health check response was null");
            }
            return healthCheckResponseConverter.apply(response);
            
            
        } catch (final DuoException e) {
          //wrap duo specific exception.
           throw new DuoClientException(e);
        }
        
    }

    /** 
     * {@inheritDoc}
     *  
     * <p>The Duo WebSDK Client does not support either the {@code nonce} or {@code redirectURIOverride} 
     * parameters.</p>
     * 
     */
    @Override
    @Nonnull public String createAuthUrl(@Nonnull @NotEmpty final String username, 
            @Nonnull @NotEmpty final String state, @Nullable final String nonce,
            @Nullable final String redirectURIOverride) 
            throws DuoClientException {
        Constraint.isNotEmpty(username, "Username can not be null or empty");
        Constraint.isNotEmpty(state, "State can not be null or empty");
        //does not support the nonce or redirect_uri override
        try {
            return client.createAuthUrl(username, state);
        } catch (final DuoException e) {
            //wrap duo specific exception.
            throw new DuoClientException(e);
        }
    }

    /** 
     * {@inheritDoc}
     *  
     * <p>The Duo WebSDK Client does not support the {@code redirectURIOverride} parameter.</p>
     * 
     */
    @Override
    @Nonnull public JWT exchangeAuthorizationCodeFor2FAResult(@Nonnull final String code, 
            @Nonnull final String username, @Nullable final String redirectURIOverride) 
            throws DuoClientException {
        Constraint.isNotEmpty(code, "Auth_code can not be null");
        try {
            final Token token = client.exchangeAuthorizationCodeFor2FAResult(code,username);
            if (token == null) {
                throw new DuoClientException("Duo token was null");
            }
            final JWT tokenAsJWT = tokenResponseConverter.apply(token,duoIntegration);
            if (tokenAsJWT == null) {
                throw new DuoClientException("Duo token could not be converted to a JWT");
            }
            return tokenAsJWT;
         } catch (final DuoException e) {
            //wrap duo specific exception.
            throw new DuoClientException(e);
        }
    }


    /** Default health check response converter. */
    @ThreadSafe
    private class DefaultHealthCheckResponseConverter implements Function<HealthCheckResponse,DuoHealthCheck>{

        @Override
        public DuoHealthCheck apply(@Nonnull final HealthCheckResponse response) {
            
            return DuoHealthCheck.builder().withStatus(response.getStat()).withCode(response.getCode())
                    .withMessage(response.getMessage()).withMessageDetail(response.getMessage_detail())
                    .withResponse(new DuoHealthCheckResponse(response.getResponse().getTimestamp()))
                    .withTimestamp(response.getTimestamp()).build();
        }
        
    }

    /**
     * Default Duo token converter. Creates a JWT by converting the Duo token (back) to a JSON String which it uses to
     * create a **signed** JWT. As the signature is not returned from the Duo SDK and the flow requires a signed JWT, 
     * a new HMAC signature is computed using the integrations secret key.
     */
    @ThreadSafe
    private final class DefaultTokenResponseConverter implements BiFunction<Token, DuoOIDCIntegration, JWT>{
        
        /** Thread-safe JSON object mapper. */
        @Nonnull private ObjectMapper objectMapper;
        
        /** Constructor. */
        private DefaultTokenResponseConverter() {
            objectMapper = new ObjectMapper();
        }
    
        @Override
        @Nullable public JWT apply(@Nonnull final Token t, @Nonnull final DuoOIDCIntegration integ) {                 
            try {               
                final String duoTokenAsJson = objectMapper.writeValueAsString(t);
                final JWTClaimsSet claims = JWTClaimsSet.parse(duoTokenAsJson);
                //re-sign the JWT using an incompatible key for the given algorithm!
                //FIXME please change this Duo!
                return JWSAssemblyUtils.assembleMacJws(JWSAlgorithm.HS512,claims,
                        JWSAssemblyUtils.getSecretBytes(integ.getSecretKey()));

            } catch (final JsonProcessingException | ParseException | JOSEException | EncodingException e) {
                log.error("Could not convert Duo Token to a Nimbus JWT Token",e);
               return null;
            }      
        }        
    }


    @Override
    public boolean isSupportsNonce() {
        return false;
    }

}
