/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.AbstractDuoAuthenticationAction;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * Authentication action that validates the Duo response state parameter (which is <b>required</b> in the Duo flow) 
 * matches that in the 2FA request.
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link AuthnEventIds#NO_CREDENTIALS}
 * @pre <pre>
 *      ProfileRequestContext.getSubcontext(AuthenticationContext.class, false) != null
 *      </pre>
 * 
 * @pre <pre>
 *      AuthenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class, false) != null
 *      </pre>
 * 
 */
public class ValidateDuoResponseState extends AbstractDuoAuthenticationAction {    
    
    /** Class logger. */
    @Nonnull @NotEmpty private final Logger log = LoggerFactory.getLogger(ValidateDuoResponseState.class);
    
    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext, 
            @Nonnull final DuoOIDCAuthenticationContext duoContext) {
        
        log.trace("{} Duo 2FA request state '{}' was returned in the response as '{}'",
                getLogPrefix(),duoContext.getRequestState(),duoContext.getResponseState() );
        
        if (duoContext.getRequestState() == null || duoContext.getResponseState() == null) {
            log.error("{} The state parameter was not present in either the request or response, "
                    + "state is mandatory for Duo 2FA requests",getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            return;
        }
        if (!duoContext.getRequestState().equals(duoContext.getResponseState())) {
            log.error("{} Duo request state did not match response state, has it been tampered with!",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            //blank request and response for safety
            blankState(duoContext);
            return;
        } 
        // else state is fine. Blank request and response for safety.
        blankState(duoContext);
        log.debug("{} Duo 2FA request and response state match, continuing",getLogPrefix());       
        
    }
    
    /**
     * Set the request and response states to null so they can't be reused. There is no control
     * on how long they persist in-memory.
     * 
     * @param context the duo context.
     */
    private void blankState(@Nonnull final DuoOIDCAuthenticationContext context) {
        context.setRequestState(null);
        context.setResponseState(null);
    }

}
