/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.AbstractDuoAuthenticationAction;
import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCAuthAPI;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.model.DuoHealthCheck;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * An action that checks the health of the Duo 2FA endpoint for the established Duo integration. 
 * An {@link AuthnEventIds#AUTHN_EXCEPTION} event is emitted if the health endpoint is unreachable
 * or unhealthy (unhealthy can also mean the client details are incorrect e.g. the client is not 
 * registered with Duo).
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#IO_ERROR}
 * @pre <pre>
 *      ProfileRequestContext.getSubcontext(AuthenticationContext.class, false) != null
 *      </pre>
 * 
 * @pre <pre>
 *      AuthenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class, false) != null
 *      </pre>
 */
public class HealthCheckDuoOIDCAuthAPI extends AbstractDuoAuthenticationAction{      
    
    /** Class logger. */
    @Nonnull @NotEmpty private final Logger log = LoggerFactory.getLogger(HealthCheckDuoOIDCAuthAPI.class);
    
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext duoContext) {        
        
        try {
            final DuoOIDCClient client = duoContext.getClient();
            if (client == null) {
                throw new DuoClientException("Duo client is null, has the context been created correctly?");
            }
            //Native duo WebSDK v4 will throw an exception if anything other than OK is returned. 
            final DuoHealthCheck healthCheckResponse = client.healthCheck();            
            log.trace("{} Duo health check response '{}'",getLogPrefix(),healthCheckResponse);           
            
            //These checks are redundant if using the Duo WebSDK v4 client as it throws an exception if not 'OK'.
            //They are still included  to be compatible with other implementations which return the full response. 
            if (DuoOIDCAuthAPI.DUO_RESPONSE_STATUS_OK.equalsIgnoreCase(healthCheckResponse.getStatus())) {
                log.trace("{} Duo 2FA endpoints and client are healthy!",getLogPrefix());
                return;
                
            } else if (DuoOIDCAuthAPI.DUO_RESPONSE_STATUS_FAIL.equalsIgnoreCase(healthCheckResponse.getStatus())) {
                //2FA is unavailable for the given client integration. 
                log.error("{} Duo 2FA health check failed, current status '{}',"
                        + " message '{}', message detail '{}'",getLogPrefix(),
                        healthCheckResponse.getStatus(),healthCheckResponse.getMessage(),
                        healthCheckResponse.getMessageDetail());
                throw new DuoClientException("Duo 2FA health check responded with a failure status of: "+
                        healthCheckResponse.getMessage());
            
            } else {
                log.error("{} Duo health check response contained an unknown status of '{}', "
                        + " message '{}', message detail '{}'",getLogPrefix(),
                        healthCheckResponse.getStatus(),
                        healthCheckResponse.getMessage(),healthCheckResponse.getMessageDetail());
                throw new DuoClientException("Duo 2FA health check returned an unknown status response: "+
                        healthCheckResponse.getMessage());
            }           
   
        } catch (final DuoClientException e) {
           log.error("{} Duo API health check failed",getLogPrefix(),e);
           ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.AUTHN_EXCEPTION);
           return;
        }
        
    }


}
