/*===========================================================================
  Copyright (C) 2016-2017 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
===========================================================================*/

package net.sf.okapi.filters.openxml;

import static net.sf.okapi.filters.openxml.ContentTypes.Types.Common.CORE_PROPERTIES_TYPE;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;

import net.sf.okapi.filters.openxml.ContentTypes.Types.Drawing;
import net.sf.okapi.filters.openxml.ContentTypes.Types.Powerpoint;
import net.sf.okapi.filters.openxml.Relationships.Rel;

class PowerpointDocument extends DocumentType {
	private static final String SLIDE_LAYOUT = "/slideLayout";
	private static final String COMMENTS = "/comments";
	private static final String NOTES_SLIDE = "/notesSlide";
	private static final String NOTES_MASTER = "/notesMaster";
	private static final String CHART = "/chart";
	private static final String DIAGRAM_DATA = "/diagramData";

	private static final Pattern RELS_NAME_PATTERN = Pattern.compile(".+slide\\d+\\.xml\\.rels");
	private Matcher relsNameMatcher = RELS_NAME_PATTERN.matcher("").reset();

	private StyleDefinitions presentationNotesStyleDefinitions;

	private List<String> slideNames = new ArrayList<>();
	private List<String> slideLayoutNames = new ArrayList<>();

	/**
	 * Uses the slide name as key and the comment name as value.
	 */
	private Map<String, String> slidesByComment = new HashMap<>();

	/**
	 * Uses the slide name as key and the note name as value.
	 */
	private Map<String, String> slidesByNote = new HashMap<>();

	/**
	 * Uses the slide name as key and the chart name as value.
	 */
	private Map<String, String> slidesByChart = new HashMap<>();

	/**
	 * Uses the slide name as key and the diagram name as value.
	 */
	private Map<String, String> slidesByDiagramData = new HashMap<>();

	PowerpointDocument(OpenXMLZipFile zipFile,
			ConditionalParameters params) {
		super(zipFile, params);
	}

	@Override
	boolean isClarifiablePart(String contentType) {
		return Powerpoint.MAIN_DOCUMENT_TYPE.equals(contentType);
	}

	@Override
	boolean isStyledTextPart(String entryName, String type) {
		if (type.equals(Powerpoint.SLIDE_TYPE)) return true;
		if (type.equals(Drawing.DIAGRAM_TYPE)) return true;
		if (isMasterPart(entryName, type)) return true;
		if (getParams().getTranslatePowerpointNotes() && type.equals(Powerpoint.NOTES_TYPE)) return true;
		if (type.equals(Drawing.CHART_TYPE)) return true;
		return false;
	}

	boolean isMasterPart(String entryName, String type) {
		if (getParams().getTranslatePowerpointMasters()) {
			if (type.equals(Powerpoint.MASTERS_TYPE)) return true;
			// Layouts are translatable if we are translating masters and this particular layout is
			// in use by a slide
			if (type.equals(Powerpoint.LAYOUT_TYPE)
					&& slideLayoutNames.contains(entryName)) return true;
		}

		return false;
	}

	@Override
	void initialize() throws IOException, XMLStreamException {
		presentationNotesStyleDefinitions = parsePresentationNotesStyleDefinitions();
		slideNames = findSlides();
		slideLayoutNames = findSlideLayouts(slideNames);
		slidesByComment = findComments(slideNames);
		slidesByNote = findNotes(slideNames);
		slidesByChart = findCharts(slideNames);
		slidesByDiagramData = findDiagramDatas(slideNames);
	}

	private StyleDefinitions parsePresentationNotesStyleDefinitions() throws IOException, XMLStreamException {
		String relationshipTarget = getRelationshipTarget(
			getZipFile().documentRelationshipsNamespace().uri().concat(NOTES_MASTER)
		);

		if (null == relationshipTarget) {
			return new EmptyStyleDefinitions();
		}

		Reader reader = getZipFile().getPartReader(relationshipTarget);

		return new PresentationNotesStylesParser(
				getZipFile().getEventFactory(),
				getZipFile().getInputFactory(),
				reader,
				getParams()
		).parse();
	}

	@Override
	OpenXMLPartHandler getHandlerForFile(ZipEntry entry, String contentType) throws IOException, XMLStreamException {
		relsNameMatcher.reset(entry.getName());
		if (isRelationshipsPart(contentType) && relsNameMatcher.matches() && getParams().getExtractExternalHyperlinks()) {
			return new RelationshipsPartHandler(getParams(), getZipFile(), entry);
		}

		// Check to see if this is non-translatable
		if (!isTranslatableType(entry.getName(), contentType)) {
			if (isClarifiablePart(contentType)) {
				return new ClarifiablePartHandler(getZipFile(), entry);
			}

			return new NonTranslatablePartHandler(getZipFile(), entry);
		}

		final StyleOptimisation styleOptimisation = new StyleOptimisation.Default(
			new StyleOptimisation.Bypass(),
			getParams(),
			getZipFile().getEventFactory(),
			Namespaces.DrawingML.getQName("pPr", Namespace.PREFIX_A),
			Namespaces.DrawingML.getQName("defRPr", Namespace.PREFIX_A)
		);
		if (isMasterPart(entry.getName(), contentType)) {
			return new MasterPartHandler(getParams(), getZipFile(), entry, new EmptyStyleDefinitions(), styleOptimisation);

		} else if (isStyledTextPart(entry.getName(), contentType)) {
			StyleDefinitions styleDefinitions;

			switch (contentType) {
				case Powerpoint.NOTES_TYPE:
					styleDefinitions = presentationNotesStyleDefinitions;
					break;
				default:
					styleDefinitions = new EmptyStyleDefinitions();
			}

			return new StyledTextPartHandler(
				getParams(),
				getZipFile(),
				entry,
				styleDefinitions,
				styleOptimisation
			);
		}

		OpenXMLContentFilter openXMLContentFilter = new OpenXMLContentFilter(getParams(), entry.getName());
		ParseType parseType = getParseType(contentType);
		getParams().nFileType = parseType;

		openXMLContentFilter.setUpConfig(parseType);

		// Other configuration
		return new StandardPartHandler(openXMLContentFilter, getParams(), getZipFile(), entry);
	}

    /**
     * @param entryName ZIP entry name
     * @param contentType the entry's content type
     * @return {@code true} if the entry is to be excluded due to
     * {@link ConditionalParameters#getPowerpointIncludedSlideNumbersOnly()} and
     * {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
     */
    private boolean isExcluded(String entryName, String contentType) {
        return isExcludedSlide(entryName, contentType)
                || isExcludedNote(entryName, contentType)
                || isExcludedComment(entryName, contentType)
                || isExcludedChart(entryName, contentType)
                || isExcludedDiagramData(entryName, contentType);
    }

    private ParseType getParseType(String contentType) {
        ParseType parseType;
		if (contentType.equals(CORE_PROPERTIES_TYPE)) {
			parseType = ParseType.MSWORDDOCPROPERTIES;
		}
		else if (contentType.equals(Powerpoint.COMMENTS_TYPE)) {
			parseType = ParseType.MSPOWERPOINTCOMMENTS;
		}
		else {
			throw new IllegalStateException("Unexpected content type " + contentType);
		}

		return parseType;
	}

	private boolean isTranslatableType(String entryName, String type) throws IOException, XMLStreamException {
		if (!entryName.endsWith(".xml")) return false;
        if (isExcluded(entryName, type)) return false;
		if (isHidden(entryName, type)) return false;
		if (isStyledTextPart(entryName, type)) return true;
		if (getParams().getTranslateDocProperties() && type.equals(CORE_PROPERTIES_TYPE)) return true;
		if (getParams().getTranslateComments() && type.equals(Powerpoint.COMMENTS_TYPE)) return true;
		if (type.equals(Drawing.DIAGRAM_TYPE)) return true;
		if (getParams().getTranslatePowerpointNotes() && type.equals(Powerpoint.NOTES_TYPE)) return true;
		return false;
	}

	private boolean isHidden(String entryName, String type) throws IOException, XMLStreamException {
		if (!getParams().getTranslatePowerpointHidden() && Powerpoint.SLIDE_TYPE.equals(type)) {
			return isHiddenSlide(entryName);
		}
		return false;
	}

	private boolean isHiddenSlide(String entryName) throws IOException, XMLStreamException {
		XMLEventReader eventReader = null;
		try {
			eventReader = getZipFile().getInputFactory().createXMLEventReader(getZipFile().getPartReader(entryName));
			return PresentationSlide.fromXMLEventReader(eventReader).isHidden();
		} finally {
			if (null != eventReader) {
				eventReader.close();
			}
		}
	}

	/**
	 * Do additional reordering of the entries for PPTX files to make
	 * sure that slides are parsed in the correct order.  This is done
	 * by scraping information from one of the rels files and the
	 * presentation itself in order to determine the proper order, rather
	 * than relying on the order in which things appeared in the zip.
	 * @return the sorted enum of ZipEntry
	 * @throws IOException if any error is encountered while reading the stream
	 * @throws XMLStreamException if any error is encountered while parsing the XML
	 */
	@Override
	Enumeration<? extends ZipEntry> getZipFileEntries() throws IOException, XMLStreamException {
		Enumeration<? extends ZipEntry> entries = getZipFile().entries();
		List<? extends ZipEntry> entryList = Collections.list(entries);
		entryList.sort(new ZipEntryComparator(reorderedPartNames()));
		return Collections.enumeration(entryList);
	}

	private List<String> reorderedPartNames() throws IOException, XMLStreamException {
		final List<String> names = new LinkedList<>();
		for (final String slideName : this.slideNames) {
			names.add(slideName);
			if (getParams().getReorderPowerpointNotesAndComments()) {
				final String namespaceUri = getZipFile().documentRelationshipsNamespace().uri();
				names.addAll(slideRelationshipTargetsForType(slideName, namespaceUri.concat(NOTES_SLIDE)));
				names.addAll(slideRelationshipTargetsForType(slideName, namespaceUri.concat(COMMENTS)));
			}
		}
		return names;
	}

	private List<String> slideRelationshipTargetsForType(final String slideName, final String typeUri) throws IOException, XMLStreamException {
		List<Rel> rels = getZipFile().getRelationshipsForTarget(slideName).getRelByType(typeUri);
		return rels == null ? Collections.emptyList() :
			rels.stream()
				.map(r -> r.target)
				.collect(Collectors.toList());
	}

	List<String> findSlides() throws IOException, XMLStreamException {
		OpenXMLZipFile zipFile = getZipFile();
		// XXX Not strictly correct, I should really look for the main document and then go from there
		Relationships rels = zipFile.getRelationships("ppt/_rels/presentation.xml.rels");
		Presentation pres = new Presentation(zipFile.getInputFactory(), rels);
		pres.parseFromXML(zipFile.getPartReader("ppt/presentation.xml"));
		return pres.getSlidePartNames();
	}

	/**
	 * Examine relationship information to find all layouts that are used in
	 * a slide in this document.  Return a list of their entry names, in order.
	 * @return list of entry names.
	 * @throws XMLStreamException See {@link OpenXMLZipFile#getRelationshipsForTarget(String)}
	 * @throws IOException See {@link OpenXMLZipFile#getRelationshipsForTarget(String)}
	 */
	List<String> findSlideLayouts(List<String> slideNames) throws IOException, XMLStreamException {
		List<String> layouts = new ArrayList<>();
		OpenXMLZipFile zipFile = getZipFile();
		final String typeUri = getZipFile().documentRelationshipsNamespace().uri().concat(SLIDE_LAYOUT);
		for (String slideName : slideNames) {
			List<Relationships.Rel> rels =
					zipFile.getRelationshipsForTarget(slideName).getRelByType(typeUri);
			if (!rels.isEmpty()) {
				layouts.add(rels.get(0).target);
			}
		}
		return layouts;
	}

	/**
	 * @param entryName the entry name
	 * @param contentType the entry's content type
	 * @return {@code true} if the given entry represents a slide that was not included using
	 * option {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
	 */
	private boolean isExcludedSlide(String entryName, String contentType) {
		if (!Powerpoint.SLIDE_TYPE.equals(contentType)) {
			return false;
		}

		if (!getParams().getPowerpointIncludedSlideNumbersOnly()) {
			return false;
		}

		int slideIndex = slideNames.indexOf(entryName);
		if (slideIndex == -1) {
			return false;
		}

		int slideNumber = slideIndex + 1; // human readable / 1-based slide numbers
		return !getParams().tsPowerpointIncludedSlideNumbers.contains(slideNumber);
	}

	/**
	 * @param entryName the entry name
	 * @param contentType the entry's content type
	 * @return {@code true} if the given entry represents a note that is used on a slide that was
	 * not included using option {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
	 */
	private boolean isExcludedNote(String entryName, String contentType) {
		if (!Powerpoint.NOTES_TYPE.equals(contentType)
				|| !slidesByNote.containsKey(entryName)) {
			return false;
		}

		String slideName = slidesByNote.get(entryName);
		return isExcludedSlide(slideName, Powerpoint.SLIDE_TYPE);
	}

	/**
	 * @param entryName the entry name
	 * @param contentType the entry's content type
	 * @return {@code true} if the given entry represents a comment that is used on a slide that was
	 * not included using option {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
	 */
	private boolean isExcludedComment(String entryName, String contentType) {
		if (!Powerpoint.COMMENTS_TYPE.equals(contentType)
				|| !slidesByComment.containsKey(entryName)) {
			return false;
		}

		String slideName = slidesByComment.get(entryName);
		return isExcludedSlide(slideName, Powerpoint.SLIDE_TYPE);
	}

	/**
	 * @param entryName the entry name
	 * @param contentType the entry's content type
	 * @return {@code true} if the given entry represents a chart that is used on a slide that was
	 * not included using option {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
	 */
	private boolean isExcludedChart(String entryName, String contentType) {
		if (!Drawing.CHART_TYPE.equals(contentType)
				|| !slidesByChart.containsKey(entryName)) {
			return false;
		}

		String slideName = slidesByChart.get(entryName);
		return isExcludedSlide(slideName, Powerpoint.SLIDE_TYPE);
	}

	/**
	 * "Diagram data" is used by SmartArt, for example.
	 *
	 * @param entryName the entry name
	 * @param contentType the entry's content type
	 * @return {@code true} if the given entry represents a diagram that is used on a slide that was
	 * not included using option {@link ConditionalParameters#tsPowerpointIncludedSlideNumbers}
	 */
	private boolean isExcludedDiagramData(String entryName, String contentType) {
		if (!Drawing.DIAGRAM_TYPE.equals(contentType)
				|| !slidesByDiagramData.containsKey(entryName)) {
			return false;
		}

		String slideName = slidesByDiagramData.get(entryName);
		return isExcludedSlide(slideName, Powerpoint.SLIDE_TYPE);
	}

	/**
	 * Initializes the relationships of type {@link #NOTES_SLIDE}.
	 */
	private Map<String, String> findNotes(List<String> slideNames)
			throws IOException, XMLStreamException {
		final String namespaceUri = getZipFile().documentRelationshipsNamespace().uri();
		return initializeRelsByEntry(slideNames, namespaceUri.concat(NOTES_SLIDE));
	}

	/**
	 * Initializes the relationships of type {@link #COMMENTS}.
	 */
	private Map<String, String> findComments(List<String> slideNames)
			throws IOException, XMLStreamException {
		final String namespaceUri = getZipFile().documentRelationshipsNamespace().uri();
		return initializeRelsByEntry(slideNames, namespaceUri.concat(COMMENTS));
	}

	/**
	 * Initializes the relationships of type {@link #CHART}.
	 */
	private Map<String, String> findCharts(List<String> slideNames)
			throws IOException, XMLStreamException {
		final String namespaceUri = getZipFile().documentRelationshipsNamespace().uri();
		return initializeRelsByEntry(slideNames, namespaceUri.concat(CHART));
	}

	/**
	 * Initializes the relationships of type {@link #DIAGRAM_DATA}.
	 */
	private Map<String, String> findDiagramDatas(List<String> slideNames)
			throws IOException, XMLStreamException {
		final String namespaceUri = getZipFile().documentRelationshipsNamespace().uri();
		return initializeRelsByEntry(slideNames, namespaceUri.concat(DIAGRAM_DATA));
	}
}
