package net.andreinc.mockneat.abstraction;

import java.util.function.Supplier;
import java.util.stream.DoubleStream;

import static java.util.stream.DoubleStream.generate;
import static java.util.stream.IntStream.range;
import static net.andreinc.mockneat.utils.ValidationUtils.SIZE_BIGGER_THAN_ZERO;
import static net.andreinc.mockneat.utils.ValidationUtils.isTrue;

public interface MockUnitFloat extends MockUnit<Float> {

    /**
     * <p>Transforms an existing {@code MockUnitFloat} into a {@code MockUnit<DoubleStream>}.</p>
     *
     * <p>The DoubleStream will be "infinite" and will contain values generated by the internal {@code MockUnitFloat} supplier.</p>
     *
     * <p>Note: The reason we are returning a {@code DoubleStream} is because no {@code FloatStream} exists.</p>
     *
     * @return A new {@code MockUnit<DoubleStream>}
     */
    default MockUnit<DoubleStream> doubleStream() {
        Supplier<DoubleStream> supp = () -> generate(supplier()::get);
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitFloat} into a {@code MockUnit<float[]>}.</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitFloat} supplier.</p>
     *
     * @param size The size of the Array
     * @return A new {@code MockUnit<float[]>}
     */
    default MockUnit<float[]> arrayPrimitive(int size) {
        isTrue(size>=0, SIZE_BIGGER_THAN_ZERO);
        Supplier<float[]> supp = () -> {
            final float[] result = new float[size];
            range(0, size).forEach(i -> result[i] = val());
            return result;
        };
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitFloat} into a {@code MockUnit<Float[]>}.</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitFloat} supplier.</p>
     *
     * @param size The size of the Array
     * @return A new {@code MockUnit<Float[]>}
     */
    default MockUnit<Float[]> array(int size) {
        isTrue(size>=0, SIZE_BIGGER_THAN_ZERO);
        Supplier<Float[]> supp = () -> {
            final Float[] result = new Float[size];
            range(0, size).forEach(i -> result[i] = val());
            return result;
        };
        return () -> supp;
    }
}
