package net.andreinc.mockneat.abstraction;

import net.andreinc.mockneat.utils.ValidationUtils;

import java.util.function.Supplier;
import java.util.stream.LongStream;

import static java.util.stream.IntStream.range;
import static net.andreinc.mockneat.utils.ValidationUtils.isTrue;

public interface MockUnitLong extends MockUnit<Long> {

    /**
     * <p>Transforms an existing {@code MockUnitLong} into a {@code MockUnit<LongStream>}.</p>
     *
     * <p>The {@code LongStream} will be "infinite" and will contain values generated by the internal {@code MockUnitLong} supplier.</p>
     *
     * @return A new {@code MockUnit<LongStream>}
     */
    default MockUnit<LongStream> longStream() {
        Supplier<LongStream> supp = () -> LongStream.generate(supplier()::get);
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitLong} into a {@code MockUnit<long[]>}.</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitLong} supplier.</p>
     *
     * @param size The size of the Array
     * @return A new {@code MockUnit<long[]>}
     */
    default MockUnit<long[]> arrayPrimitive(int size) {
        isTrue(size>=0, ValidationUtils.SIZE_BIGGER_THAN_ZERO);
        Supplier<long[]> supp = () -> {
            long[] array = new long[size];
            range(0, size).forEach(i -> array[i] = val());
            return array;
        };
        return () -> supp;
    }

    /**
     * <p>Transforms an existing {@code MockUnitLong} into a {@code MockUnit<Long[]>}.</p>
     *
     * <p>The values in the array will be generated using the {@code MockUnitLong} supplier.</p>
     *
     * @param size The size of the Array
     * @return A new {@code MockUnit<Long[]>}
     */
    default MockUnit<Long[]> array(int size) {
        isTrue(size>=0, ValidationUtils.SIZE_BIGGER_THAN_ZERO);
        Supplier<Long[]> supp = () -> {
            final Long[] result = new Long[size];
            range(0, size).forEach(i -> result[i] = val());
            return result;
        };
        return () -> supp;
    }
}
