package me.piebridge.brevent.protocol;

import android.content.pm.PackageManager;
import android.content.pm.PermissionInfo;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresPermission;
import android.support.annotation.WorkerThread;

import java.io.IOException;
import java.util.List;

/**
 * Brevent disable module
 * Created by thom on 2018/3/6.
 */
@WorkerThread
public class BreventDisabled extends BreventModule {

    public static final String PERMISSION = "me.piebridge.brevent.permission.BREVENT_DISABLED";

    /**
     * call <pre>Brevent.with(context, new BreventDisabled());</pre> first
     *
     * @return BreventDisabled
     */
    public static BreventDisabled getInstance() {
        return Brevent.getModule(BreventDisabled.class);
    }

    /**
     * check whether has a supported brevent
     *
     * @return true or false
     */
    public boolean hasBrevent() {
        try {
            PermissionInfo permission = getContext().getPackageManager()
                    .getPermissionInfo(PERMISSION, 0);
            return (permission.protectionLevel & PermissionInfo.PROTECTION_DANGEROUS) != 0;
        } catch (PackageManager.NameNotFoundException ignore) { // NOSONAR
            return false;
        }
    }

    /**
     * Check whether brevent disabled isAvailable
     *
     * @return true or false
     * @throws IOException       network exception
     * @throws SecurityException security exception
     */
    @RequiresPermission(PERMISSION)
    public boolean isAvailable() throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledStatus(true));
        return response instanceof BreventDisabledStatus
                && ((BreventDisabledStatus) response).enabled;
    }

    /**
     * get apps in disabled-user state
     *
     * @param uid UserHandler.getIdentifier(), can use hashCode or toString directly
     * @return apps in disabled-user state
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#getApplicationEnabledSetting(String)
     */
    @Nullable
    @RequiresPermission(PERMISSION)
    public List<String> getDisabledPackages(int uid) throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledPackagesRequest(uid));
        if (response instanceof BreventDisabledPackagesResponse) {
            return ((BreventDisabledPackagesResponse) response).packageNames;
        } else {
            return null;
        }
    }

    /**
     * check whether apps in disabled-user state<br>
     *
     * @param packageName package name
     * @param uid         UserHandler.getIdentifier(), can use hashCode or toString directly
     * @return true  or false
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#getApplicationEnabledSetting(String)
     */
    @RequiresPermission(PERMISSION)
    public boolean isDisabled(String packageName, int uid) throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledGetState(packageName, uid, false));
        return response instanceof BreventDisabledGetState &&
                ((BreventDisabledGetState) response).disabled;
    }

    /**
     * update app's state, can only between enabled and disabled-user
     *
     * @param packageName package name
     * @param uid         UserHandler.getIdentifier(), can use hashCode or toString directly
     * @param enable      ture for enabled, false for disabled-user
     * @return true for success, false for fail
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#setApplicationEnabledSetting(String, int, int)
     */
    @RequiresPermission(PERMISSION)
    public boolean setPackageEnabled(String packageName, int uid, boolean enable)
            throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledSetState(packageName, uid, enable));
        return response instanceof BreventDisabledSetState &&
                ((BreventDisabledSetState) response).enable;
    }

}

