package me.piebridge.brevent.protocol;

import android.support.annotation.Nullable;
import android.support.annotation.RequiresPermission;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.Socket;
import java.util.List;
import java.util.Objects;

/**
 * Brevent disabled module<br>
 * 1. keep token as safe as possible<br>
 * 2. catch IOException and SecurityException<br>
 * 3. target sdk 23+, and declare BreventDisabledModule.PERMISSION<br>
 * Created by thom on 2018/2/25.
 */
@WorkerThread
public class BreventDisabledModule {

    public static final String PERMISSION = "me.piebridge.brevent.permission.BREVENT_DISABLED";

    private static final int TIMEOUT = 5000;

    private static String token;

    /**
     * get token
     *
     * @return token token for request
     */
    public static String getToken() {
        return token;
    }

    /**
     * set token<br>
     * You can declare packageName + ".receiver.BreventServerReceiver" to receiver token too.<br>
     * action: "me.piebridge.brevent.intent.action.BREVENT"<br>
     * extra: Intent.EXTRA_REMOTE_INTENT_TOKEN<br>
     *
     * @param token token for request, if it's null, package must be a foreground app
     */
    public static void setToken(@Nullable String token) {
        BreventDisabledModule.token = token;
    }

    /**
     * Check whether brevent disabled isAvailable
     *
     * @return true or false
     * @throws IOException       network exception
     * @throws SecurityException security exception
     */
    @RequiresPermission(PERMISSION)
    public static boolean isAvailable() throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledStatus(true));
        return response instanceof BreventDisabledStatus
                && ((BreventDisabledStatus) response).enabled;
    }

    /**
     * get apps in disabled-user state
     *
     * @param uid UserHandler.getIdentifier(), can use hashCode or toString directly
     * @return apps in disabled-user state
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#getApplicationEnabledSetting(String)
     */
    @Nullable
    @RequiresPermission(PERMISSION)
    public static List<String> getDisabledPackages(int uid) throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledPackagesRequest(uid));
        if (response instanceof BreventDisabledPackagesResponse) {
            return ((BreventDisabledPackagesResponse) response).packageNames;
        } else {
            return null;
        }
    }

    /**
     * check whether apps in disabled-user state<br>
     *
     * @param packageName package name
     * @param uid         UserHandler.getIdentifier(), can use hashCode or toString directly
     * @return true  or false
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#getApplicationEnabledSetting(String)
     */
    @RequiresPermission(PERMISSION)
    public static boolean isDisabled(String packageName, int uid) throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledGetState(packageName, uid, false));
        return response instanceof BreventDisabledGetState &&
                ((BreventDisabledGetState) response).disabled;
    }

    /**
     * update app's state, can only between enabled and disabled-user
     *
     * @param packageName package name
     * @param uid         UserHandler.getIdentifier(), can use hashCode or toString directly
     * @param enable      ture for enabled, false for disabled-user
     * @return true for success, false for fail
     * @throws IOException       network exception
     * @throws SecurityException security exception
     * @see android.content.pm.PackageManager#setApplicationEnabledSetting(String, int, int)
     */
    @RequiresPermission(PERMISSION)
    public static boolean setPackageEnabled(String packageName, int uid, boolean enable)
            throws IOException {
        BaseBreventProtocol response = request(new BreventDisabledSetState(packageName, uid, enable));
        return response instanceof BreventDisabledSetState &&
                ((BreventDisabledSetState) response).enable;
    }

    private static BaseBreventProtocol request(BaseBreventProtocol request) throws IOException {
        request.token = token;
        try (
                Socket socket = new Socket(BaseBreventProtocol.HOST, BaseBreventProtocol.PORT);
                DataOutputStream os = new DataOutputStream(socket.getOutputStream());
                DataInputStream is = new DataInputStream(socket.getInputStream())
        ) {
            socket.setSoTimeout(TIMEOUT);
            BaseBreventProtocol.writeTo(request, os);
            os.flush();
            BaseBreventProtocol response = BaseBreventProtocol.readFromBase(is);
            if (response == BaseBreventOK.INSTANCE) {
                throw new SecurityException("no permission");
            }
            if (response != null
                    && !TextUtils.isEmpty(response.token)
                    && !Objects.equals(token, response.token)) {
                token = response.token;
            }
            return response;
        }
    }

}
