/**
 * Copyright 2011-2016 Terracotta, Inc.
 * Copyright 2011-2016 Oracle America Incorporated
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package javax.cache.integration;

import javax.cache.configuration.CompleteConfiguration;
import java.util.Map;

/**
 * Used when a cache is read-through or when loading data into a cache via the
 * {@link javax.cache.Cache#loadAll(java.util.Set, boolean,
 * CompletionListener)} method.
 *
 * @param <K> the type of keys handled by this loader
 * @param <V> the type of values generated by this loader
 * @author Greg Luck
 * @author Yannis Cosmadopoulos
 * @see CompleteConfiguration#isReadThrough()
 * @see CacheWriter
 * @since 1.0
 */
public interface CacheLoader<K, V> {

  /**
   * Loads an object. Application developers should implement this
   * method to customize the loading of a value for a cache entry. This method
   * is called by a cache when a requested entry is not in the cache. If
   * the object can't be loaded <code>null</code> should be returned.
   *
   * @param key the key identifying the object being loaded
   * @return The value for the entry that is to be stored in the cache or
   *         <code>null</code> if the object can't be loaded
   * @throws CacheLoaderException if there is problem executing the loader.
   */
  V load(K key) throws CacheLoaderException;



  /**
   * Loads multiple objects. Application developers should implement this
   * method to customize the loading of cache entries. This method is called
   * when the requested object is not in the cache. If an object can't be loaded,
   * it is not returned in the resulting map.
   *
   * @param keys keys identifying the values to be loaded
   * @return A map of key, values to be stored in the cache.
   * @throws CacheLoaderException if there is problem executing the loader.
   */
  Map<K, V> loadAll(Iterable<? extends K> keys) throws CacheLoaderException;
}
