package io.ticofab.androidgpxparser.parser;

import android.os.AsyncTask;
import android.util.Xml;

import com.google.android.gms.maps.model.LatLng;

import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import io.ticofab.androidgpxparser.parser.domain.Gpx;
import io.ticofab.androidgpxparser.parser.domain.Track;
import io.ticofab.androidgpxparser.parser.domain.TrackPoint;
import io.ticofab.androidgpxparser.parser.domain.TrackSegment;

public class GPXParser {

    static final String TAG_GPX = "gpx";
    static final String TAG_TRACK = "trk";
    static final String TAG_SEGMENT = "trkseg";
    static final String TAG_POINT = "trkpt";
    static final String TAG_LAT = "lat";
    static final String TAG_LON = "lon";
    static final String TAG_ELEVATION = "ele";
    static final String TAG_TIME = "time";

    static final String ns = null;

    public void parse(String gpxUrl, FetchAndParseGpxTask.GpxFetchedAndParsed listener) {
        new FetchAndParseGpxTask(gpxUrl, listener).execute();
    }

    public Gpx parse(InputStream in) throws XmlPullParserException, IOException {
        try {
            XmlPullParser parser = Xml.newPullParser();
            parser.setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, true);
            parser.setInput(in, null);
            parser.nextTag();
            return readGpx(parser);
        } finally {
            in.close();
        }
    }

    Gpx readGpx(XmlPullParser parser) throws XmlPullParserException, IOException {
        List<Track> tracks = new ArrayList<>();

        parser.require(XmlPullParser.START_TAG, ns, TAG_GPX);
        while (parser.next() != XmlPullParser.END_TAG) {
            if (parser.getEventType() != XmlPullParser.START_TAG) {
                continue;
            }
            String name = parser.getName();
            // Starts by looking for the entry tag
            switch (name) {
                case TAG_TRACK:
                    tracks.add(readTrack(parser));
                    break;
                default:
                    skip(parser);
                    break;
            }
        }
        parser.require(XmlPullParser.END_TAG, ns, TAG_GPX);
        return new Gpx.Builder()
                .setTracks(tracks)
                .build();
    }

    // Parses the contents of an entry. If it encounters a title, summary, or link tag, hands them off
    // to their respective "read" methods for processing. Otherwise, skips the tag.
    Track readTrack(XmlPullParser parser) throws XmlPullParserException, IOException {
        List<TrackSegment> segments = new ArrayList<>();
        parser.require(XmlPullParser.START_TAG, ns, TAG_TRACK);
        while (parser.next() != XmlPullParser.END_TAG) {
            if (parser.getEventType() != XmlPullParser.START_TAG) {
                continue;
            }
            String name = parser.getName();
            switch (name) {
                case TAG_SEGMENT:
                    segments.add(readSegment(parser));
                    break;
                default:
                    skip(parser);
                    break;
            }
        }
        parser.require(XmlPullParser.END_TAG, ns, TAG_TRACK);
        return new Track.Builder()
                .setTrackSegments(segments)
                .build();
    }

    // Processes summary tags in the feed.
    TrackSegment readSegment(XmlPullParser parser) throws IOException, XmlPullParserException {
        List<TrackPoint> points = new ArrayList<>();
        parser.require(XmlPullParser.START_TAG, ns, TAG_SEGMENT);
        while (parser.next() != XmlPullParser.END_TAG) {
            if (parser.getEventType() != XmlPullParser.START_TAG) {
                continue;
            }
            String name = parser.getName();
            switch (name) {
                case TAG_POINT:
                    points.add(readPoint(parser));
                    break;
                default:
                    skip(parser);
                    break;
            }
        }
        parser.require(XmlPullParser.END_TAG, ns, TAG_SEGMENT);
        return new TrackSegment.Builder()
                .setTrackPoints(points)
                .build();
    }

    // Processes summary tags in the feed.
    TrackPoint readPoint(XmlPullParser parser) throws IOException, XmlPullParserException {
        parser.require(XmlPullParser.START_TAG, ns, TAG_POINT);
        Double lat = Double.valueOf(parser.getAttributeValue(null, TAG_LAT));
        Double lng = Double.valueOf(parser.getAttributeValue(null, TAG_LON));
        Double ele = null;
        DateTime time = null;
        while (parser.next() != XmlPullParser.END_TAG) {
            if (parser.getEventType() != XmlPullParser.START_TAG) {
                continue;
            }
            String name = parser.getName();
            switch (name) {
                case TAG_ELEVATION:
                    ele = readElevation(parser);
                    break;
                case TAG_TIME:
                    time = readTime(parser);
                    break;
                default:
                    skip(parser);
                    break;
            }
        }
        parser.require(XmlPullParser.END_TAG, ns, TAG_POINT);
        return new TrackPoint.Builder()
                .setElevation(ele)
                .setLatLng(new LatLng(lat, lng))
                .setTime(time)
                .build();
    }

    Double readElevation(XmlPullParser parser) throws IOException, XmlPullParserException {
        parser.require(XmlPullParser.START_TAG, ns, TAG_ELEVATION);
        Double ele = Double.valueOf(readText(parser));
        parser.require(XmlPullParser.END_TAG, ns, TAG_ELEVATION);
        return ele;
    }

    DateTime readTime(XmlPullParser parser) throws IOException, XmlPullParserException {
        parser.require(XmlPullParser.START_TAG, ns, TAG_TIME);
        DateTime time = ISODateTimeFormat.dateTimeParser().parseDateTime(readText(parser));
        parser.require(XmlPullParser.END_TAG, ns, TAG_TIME);
        return time;
    }

    String readText(XmlPullParser parser) throws IOException, XmlPullParserException {
        String result = "";
        if (parser.next() == XmlPullParser.TEXT) {
            result = parser.getText();
            parser.nextTag();
        }
        return result;
    }

    void skip(XmlPullParser parser) throws XmlPullParserException, IOException {
        if (parser.getEventType() != XmlPullParser.START_TAG) {
            throw new IllegalStateException();
        }
        int depth = 1;
        while (depth != 0) {
            switch (parser.next()) {
                case XmlPullParser.END_TAG:
                    depth--;
                    break;
                case XmlPullParser.START_TAG:
                    depth++;
                    break;
            }
        }
    }

    public static class FetchAndParseGpxTask extends AsyncTask<Void, Void, Gpx> {

        public interface GpxFetchedAndParsed {
            void onGpxFetchedAndParsed(Gpx gpx);
        }

        final String mGpxUrl;
        final GpxFetchedAndParsed mListener;
        final GPXParser mParser = new GPXParser();

        public FetchAndParseGpxTask(String gpxUrl, GpxFetchedAndParsed listener) {
            mGpxUrl = gpxUrl;
            mListener = listener;
        }

        @Override
        protected Gpx doInBackground(Void... unused) {
            Gpx parsedGpx = null;
            try {
                URL url = new URL(mGpxUrl);
                HttpURLConnection client = (HttpURLConnection) url.openConnection();
                InputStream in = new BufferedInputStream(client.getInputStream());
                parsedGpx = mParser.parse(in);
            } catch (IOException | XmlPullParserException e) {
                e.printStackTrace();
            }
            return parsedGpx;
        }

        @Override
        protected void onPostExecute(Gpx gpx) {
            mListener.onGpxFetchedAndParsed(gpx);
        }
    }
}
