/**
 * Copyright (c) 2016-present, RxJava Contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See
 * the License for the specific language governing permissions and limitations under the License.
 */
package io.reactivex.rxjava3.core;

import java.util.*;
import java.util.concurrent.*;
import java.util.stream.*;

import org.reactivestreams.*;

import io.reactivex.rxjava3.annotations.*;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.exceptions.*;
import io.reactivex.rxjava3.flowables.*;
import io.reactivex.rxjava3.functions.*;
import io.reactivex.rxjava3.internal.functions.*;
import io.reactivex.rxjava3.internal.fuseable.ScalarSupplier;
import io.reactivex.rxjava3.internal.jdk8.*;
import io.reactivex.rxjava3.internal.operators.flowable.*;
import io.reactivex.rxjava3.internal.operators.maybe.MaybeToFlowable;
import io.reactivex.rxjava3.internal.operators.mixed.*;
import io.reactivex.rxjava3.internal.operators.observable.ObservableFromPublisher;
import io.reactivex.rxjava3.internal.operators.single.SingleToFlowable;
import io.reactivex.rxjava3.internal.schedulers.ImmediateThinScheduler;
import io.reactivex.rxjava3.internal.subscribers.*;
import io.reactivex.rxjava3.internal.util.*;
import io.reactivex.rxjava3.parallel.ParallelFlowable;
import io.reactivex.rxjava3.plugins.RxJavaPlugins;
import io.reactivex.rxjava3.schedulers.*;
import io.reactivex.rxjava3.subscribers.*;

/**
 * The {@code Flowable} class that implements the <a href="https://github.com/reactive-streams/reactive-streams-jvm">Reactive Streams</a> {@link Publisher}
 * Pattern and offers factory methods, intermediate operators and the ability to consume reactive dataflows.
 * <p>
 * <em>Reactive Streams</em> operates with {@code Publisher}s which {@code Flowable} extends. Many operators
 * therefore accept general {@code Publisher}s directly and allow direct interoperation with other
 * <em>Reactive Streams</em> implementations.
 * <p>
 * The {@code Flowable} hosts the default buffer size of 128 elements for operators, accessible via {@link #bufferSize()},
 * that can be overridden globally via the system parameter {@code rx3.buffer-size}. Most operators, however, have
 * overloads that allow setting their internal buffer size explicitly.
 * <p>
 * The documentation for this class makes use of marble diagrams. The following legend explains these diagrams:
 * <p>
 * <img width="640" height="317" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/legend.v3.png" alt="">
 * <p>
 * The {@code Flowable} follows the protocol
 * <pre><code>
 *      onSubscribe onNext* (onError | onComplete)?
 * </code></pre>
 * where the stream can be disposed through the {@link Subscription} instance provided to consumers through
 * {@link Subscriber#onSubscribe(Subscription)}.
 * Unlike the {@code Observable.subscribe()} of version 1.x, {@link #subscribe(Subscriber)} does not allow external cancellation
 * of a subscription and the {@link Subscriber} instance is expected to expose such capability if needed.
 * <p>
 * {@code Flowable}s support backpressure and require {@code Subscriber}s to signal demand via {@link Subscription#request(long)}.
 * <p>
 * Example:
 * <pre><code>
 * Disposable d = Flowable.just("Hello world!")
 *     .delay(1, TimeUnit.SECONDS)
 *     .subscribeWith(new DisposableSubscriber&lt;String&gt;() {
 *         &#64;Override public void onStart() {
 *             System.out.println("Start!");
 *             request(1);
 *         }
 *         &#64;Override public void onNext(String t) {
 *             System.out.println(t);
 *             request(1);
 *         }
 *         &#64;Override public void onError(Throwable t) {
 *             t.printStackTrace();
 *         }
 *         &#64;Override public void onComplete() {
 *             System.out.println("Done!");
 *         }
 *     });
 *
 * Thread.sleep(500);
 * // the sequence can now be cancelled via dispose()
 * d.dispose();
 * </code></pre>
 * <p>
 * The <em>Reactive Streams</em> specification is relatively strict when defining interactions between {@code Publisher}s and {@code Subscriber}s, so much so
 * that there is a significant performance penalty due certain timing requirements and the need to prepare for invalid
 * request amounts via {@link Subscription#request(long)}.
 * Therefore, RxJava has introduced the {@link FlowableSubscriber} interface that indicates the consumer can be driven with relaxed rules.
 * All RxJava operators are implemented with these relaxed rules in mind.
 * If the subscribing {@code Subscriber} does not implement this interface, for example, due to it being from another <em>Reactive Streams</em> compliant
 * library, the {@code Flowable} will automatically apply a compliance wrapper around it.
 * <p>
 * {@code Flowable} is an abstract class, but it is not advised to implement sources and custom operators by extending the class directly due
 * to the large amounts of <a href="https://github.com/reactive-streams/reactive-streams-jvm#specification">Reactive Streams</a>
 * rules to be followed to the letter. See <a href="https://github.com/ReactiveX/RxJava/wiki/Writing-operators-for-2.0">the wiki</a> for
 * some guidance if such custom implementations are necessary.
 * <p>
 * The recommended way of creating custom {@code Flowable}s is by using the {@link #create(FlowableOnSubscribe, BackpressureStrategy)} factory method:
 * <pre><code>
 * Flowable&lt;String&gt; source = Flowable.create(new FlowableOnSubscribe&lt;String&gt;() {
 *     &#64;Override
 *     public void subscribe(FlowableEmitter&lt;String&gt; emitter) throws Exception {
 *
 *         // signal an item
 *         emitter.onNext("Hello");
 *
 *         // could be some blocking operation
 *         Thread.sleep(1000);
 *
 *         // the consumer might have cancelled the flow
 *         if (emitter.isCancelled()) {
 *             return;
 *         }
 *
 *         emitter.onNext("World");
 *
 *         Thread.sleep(1000);
 *
 *         // the end-of-sequence has to be signaled, otherwise the
 *         // consumers may never finish
 *         emitter.onComplete();
 *     }
 * }, BackpressureStrategy.BUFFER);
 *
 * System.out.println("Subscribe!");
 *
 * source.subscribe(System.out::println);
 *
 * System.out.println("Done!");
 * </code></pre>
 * <p>
 * RxJava reactive sources, such as {@code Flowable}, are generally synchronous and sequential in nature. In the ReactiveX design, the location (thread)
 * where operators run is <i>orthogonal</i> to when the operators can work with data. This means that asynchrony and parallelism
 * has to be explicitly expressed via operators such as {@link #subscribeOn(Scheduler)}, {@link #observeOn(Scheduler)} and {@link #parallel()}. In general,
 * operators featuring a {@link Scheduler} parameter are introducing this type of asynchrony into the flow.
 * <p>
 * For more information see the <a href="http://reactivex.io/documentation/Publisher.html">ReactiveX documentation</a>.
 *
 * @param <T>
 *            the type of the items emitted by the {@code Flowable}
 * @see Observable
 * @see ParallelFlowable
 * @see io.reactivex.rxjava3.subscribers.DisposableSubscriber
 */
public abstract class Flowable<@NonNull T> implements Publisher<T> {
    /** The default buffer size. */
    static final int BUFFER_SIZE;
    static {
        BUFFER_SIZE = Math.max(1, Integer.getInteger("rx3.buffer-size", 128));
    }

    /**
     * Mirrors the one {@link Publisher} in an {@link Iterable} of several {@code Publisher}s that first either emits an item or sends
     * a termination notification.
     * <p>
     * <img width="640" height="417" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.amb.png" alt="">
     * <p>
     * When one of the {@code Publisher}s signal an item or terminates first, all subscriptions to the other
     * {@code Publisher}s are canceled.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator itself doesn't interfere with backpressure which is determined by the winning
     *  {@code Publisher}'s backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code amb} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>
     *     If any of the losing {@code Publisher}s signals an error, the error is routed to the global
     *     error handler via {@link RxJavaPlugins#onError(Throwable)}.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element type
     * @param sources
     *            an {@code Iterable} of {@code Publisher}s sources competing to react first. A subscription to each {@code Publisher} will
     *            occur in the same order as in this {@code Iterable}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/amb.html">ReactiveX operators documentation: Amb</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> amb(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        Objects.requireNonNull(sources, "sources is null");
        return RxJavaPlugins.onAssembly(new FlowableAmb<>(null, sources));
    }

    /**
     * Mirrors the one {@link Publisher} in an array of several {@code Publisher}s that first either emits an item or sends
     * a termination notification.
     * <p>
     * <img width="640" height="417" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.ambArray.png" alt="">
     * <p>
     * When one of the {@code Publisher}s signal an item or terminates first, all subscriptions to the other
     * {@code Publisher}s are canceled.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator itself doesn't interfere with backpressure which is determined by the winning
     *  {@code Publisher}'s backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code ambArray} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>
     *     If any of the losing {@code Publisher}s signals an error, the error is routed to the global
     *     error handler via {@link RxJavaPlugins#onError(Throwable)}.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element type
     * @param sources
     *            an array of {@code Publisher} sources competing to react first. A subscription to each {@code Publisher} will
     *            occur in the same order as in this array.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/amb.html">ReactiveX operators documentation: Amb</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    public static <T> Flowable<T> ambArray(@NonNull Publisher<@NonNull ? extends T>... sources) {
        Objects.requireNonNull(sources, "sources is null");
        int len = sources.length;
        if (len == 0) {
            return empty();
        } else
        if (len == 1) {
            return fromPublisher(sources[0]);
        }
        return RxJavaPlugins.onAssembly(new FlowableAmb<>(sources, null));
    }

    /**
     * Returns the default internal buffer size used by most async operators.
     * <p>The value can be overridden via system parameter {@code rx3.buffer-size}
     * <em>before</em> the {@code Flowable} class is loaded.
     * @return the default internal buffer size.
     */
    @CheckReturnValue
    public static int bufferSize() {
        return BUFFER_SIZE;
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided array of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public static <T, @NonNull R> Flowable<R> combineLatestArray(@NonNull Publisher<@NonNull ? extends T>[] sources, @NonNull Function<? super Object[], ? extends R> combiner) {
        return combineLatestArray(sources, combiner, bufferSize());
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided array of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @param bufferSize
     *            the internal buffer size and prefetch amount applied to every source {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    public static <T, R> Flowable<R> combineLatestArray(@NonNull Publisher<@NonNull ? extends T>[] sources, @NonNull Function<? super Object[], ? extends R> combiner, int bufferSize) {
        Objects.requireNonNull(sources, "sources is null");
        if (sources.length == 0) {
            return empty();
        }
        Objects.requireNonNull(combiner, "combiner is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableCombineLatest<>(sources, combiner, bufferSize, false));
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided iterable of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public static <T, @NonNull R> Flowable<R> combineLatest(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources,
            @NonNull Function<? super Object[], ? extends R> combiner) {
        return combineLatest(sources, combiner, bufferSize());
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided iterable of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting any items and
     * without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @param bufferSize
     *            the internal buffer size and prefetch amount applied to every source {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    public static <T, R> Flowable<R> combineLatest(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources,
            @NonNull Function<? super Object[], ? extends R> combiner, int bufferSize) {
        Objects.requireNonNull(sources, "sources is null");
        Objects.requireNonNull(combiner, "combiner is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableCombineLatest<>(sources, combiner, bufferSize, false));
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided array of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestArrayDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public static <T, @NonNull R> Flowable<R> combineLatestArrayDelayError(@NonNull Publisher<@NonNull ? extends T>[] sources,
            @NonNull Function<? super Object[], ? extends R> combiner) {
        return combineLatestArrayDelayError(sources, combiner, bufferSize());
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function and delays any error from the sources until
     * all source {@code Publisher}s terminate.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided array of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestArrayDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @param bufferSize
     *            the internal buffer size and prefetch amount applied to every source {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    public static <T, @NonNull R> Flowable<R> combineLatestArrayDelayError(@NonNull Publisher<@NonNull ? extends T>[] sources,
            @NonNull Function<? super Object[], ? extends R> combiner, int bufferSize) {
        Objects.requireNonNull(sources, "sources is null");
        Objects.requireNonNull(combiner, "combiner is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        if (sources.length == 0) {
            return empty();
        }
        return RxJavaPlugins.onAssembly(new FlowableCombineLatest<>(sources, combiner, bufferSize, true));
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function and delays any error from the sources until
     * all source {@code Publisher}s terminate.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided iterable of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public static <T, @NonNull R> Flowable<R> combineLatestDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources,
            @NonNull Function<? super Object[], ? extends R> combiner) {
        return combineLatestDelayError(sources, combiner, bufferSize());
    }

    /**
     * Combines a collection of source {@link Publisher}s by emitting an item that aggregates the latest values of each of
     * the source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function and delays any error from the sources until
     * all source {@code Publisher}s terminate.
     * <p>
     * Note on method signature: since Java doesn't allow creating a generic array with {@code new T[]}, the
     * implementation of this operator has to create an {@code Object[]} instead. Unfortunately, a
     * {@code Function<Integer[], R>} passed to the method would trigger a {@link ClassCastException}.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * If the provided iterable of source {@code Publisher}s is empty, the resulting sequence completes immediately without emitting
     * any items and without any calls to the combiner function.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatestDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the common base type of source values
     * @param <R>
     *            the result type
     * @param sources
     *            the collection of source {@code Publisher}s
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @param bufferSize
     *            the internal buffer size and prefetch amount applied to every source {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public static <T, R> Flowable<R> combineLatestDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources,
            @NonNull Function<? super Object[], ? extends R> combiner, int bufferSize) {
        Objects.requireNonNull(sources, "sources is null");
        Objects.requireNonNull(combiner, "combiner is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableCombineLatest<>(sources, combiner, bufferSize, true));
    }

    /**
     * Combines two source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from either of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T1, T2, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull BiFunction<? super T1, ? super T2, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines three source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Function3<? super T1, ? super T2, ? super T3, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines four source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Function4<? super T1, ? super T2, ? super T3, ? super T4, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines five source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <T5> the element type of the fifth source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param source5
     *            the fifth source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Publisher<@NonNull ? extends T5> source5,
            @NonNull Function5<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4, source5 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines six source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <T5> the element type of the fifth source
     * @param <T6> the element type of the sixth source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param source5
     *            the fifth source {@code Publisher}
     * @param source6
     *            the sixth source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Function6<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4, source5, source6 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines seven source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <T5> the element type of the fifth source
     * @param <T6> the element type of the sixth source
     * @param <T7> the element type of the seventh source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param source5
     *            the fifth source {@code Publisher}
     * @param source6
     *            the sixth source {@code Publisher}
     * @param source7
     *            the seventh source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7,
            @NonNull Function7<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4, source5, source6, source7 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines eight source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <T5> the element type of the fifth source
     * @param <T6> the element type of the sixth source
     * @param <T7> the element type of the seventh source
     * @param <T8> the element type of the eighth source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param source5
     *            the fifth source {@code Publisher}
     * @param source6
     *            the sixth source {@code Publisher}
     * @param source7
     *            the seventh source {@code Publisher}
     * @param source8
     *            the eighth source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7}, {@code source8} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, T8, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7, @NonNull Publisher<@NonNull ? extends T8> source8,
            @NonNull Function8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(source8, "source8 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4, source5, source6, source7, source8 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Combines nine source {@link Publisher}s by emitting an item that aggregates the latest values of each of the
     * source {@code Publisher}s each time an item is received from any of the source {@code Publisher}s, where this
     * aggregation is defined by a specified function.
     * <p>
     * If any of the sources never produces an item but only terminates (normally or with an error), the
     * resulting sequence terminates immediately (normally or with all the errors accumulated until that point).
     * If that input source is also synchronous, other sources after it will not be subscribed to.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/combineLatest.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Publisher} honors backpressure from downstream. The source {@code Publisher}s
     *   are requested in a bounded manner, however, their backpressure is not enforced (the operator won't signal
     *   {@link MissingBackpressureException}) and may lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code combineLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the element type of the first source
     * @param <T2> the element type of the second source
     * @param <T3> the element type of the third source
     * @param <T4> the element type of the fourth source
     * @param <T5> the element type of the fifth source
     * @param <T6> the element type of the sixth source
     * @param <T7> the element type of the seventh source
     * @param <T8> the element type of the eighth source
     * @param <T9> the element type of the ninth source
     * @param <R> the combined output type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            the second source {@code Publisher}
     * @param source3
     *            the third source {@code Publisher}
     * @param source4
     *            the fourth source {@code Publisher}
     * @param source5
     *            the fifth source {@code Publisher}
     * @param source6
     *            the sixth source {@code Publisher}
     * @param source7
     *            the seventh source {@code Publisher}
     * @param source8
     *            the eighth source {@code Publisher}
     * @param source9
     *            the ninth source {@code Publisher}
     * @param combiner
     *            the aggregation function used to combine the items emitted by the source {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7}, {@code source8}, {@code source9}
     *                              or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/combinelatest.html">ReactiveX operators documentation: CombineLatest</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, T8, T9, R> Flowable<R> combineLatest(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull Publisher<@NonNull ? extends T3> source3, @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7, @NonNull Publisher<@NonNull ? extends T8> source8,
            @NonNull Publisher<@NonNull ? extends T9> source9,
            @NonNull Function9<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? super T9, ? extends R> combiner) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(source8, "source8 is null");
        Objects.requireNonNull(source9, "source9 is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return combineLatestArray(new Publisher[] { source1, source2, source3, source4, source5, source6, source7, source8, source9 }, Functions.toFunction(combiner), bufferSize());
    }

    /**
     * Concatenates elements of each {@link Publisher} provided via an {@link Iterable} sequence into a single sequence
     * of elements without interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the common value type of the sources
     * @param sources the {@code Iterable} sequence of {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> concat(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        Objects.requireNonNull(sources, "sources is null");
        // unlike general sources, fromIterable can only throw on a boundary because it is consumed only there
        return fromIterable(sources).concatMapDelayError((Function)Functions.identity(), false, 2);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by each of the {@link Publisher}s emitted by the source
     * {@code Publisher}, one after the other, without interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}
     *  sources are expected to honor backpressure as well. If the outer violates this, a
     *  {@link MissingBackpressureException} is signaled. If any of the inner {@code Publisher}s violates
     *  this, it <em>may</em> throw an {@link IllegalStateException} when an inner {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concat(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concat(sources, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by each of the {@link Publisher}s emitted by the source
     * {@code Publisher}, one after the other, without interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}
     *  sources are expected to honor backpressure as well. If the outer violates this, a
     *  {@link MissingBackpressureException} is signaled. If any of the inner {@code Publisher}s violates
     *  this, it <em>may</em> throw an {@link IllegalStateException} when an inner {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @param prefetch
     *            the number of {@code Publisher}s to prefetch from the sources sequence.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concat(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int prefetch) {
        return fromPublisher(sources).concatMap((Function)Functions.identity(), prefetch);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by two {@link Publisher}s, one after the other, without
     * interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that source {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be concatenated
     * @param source2
     *            a {@code Publisher} to be concatenated
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1} or {@code source2} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> concat(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        return concatArray(source1, source2);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by three {@link Publisher}s, one after the other, without
     * interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that source {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be concatenated
     * @param source2
     *            a {@code Publisher} to be concatenated
     * @param source3
     *            a {@code Publisher} to be concatenated
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code source3} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> concat(
            @NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull Publisher<@NonNull ? extends T> source3) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        return concatArray(source1, source2, source3);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by four {@link Publisher}s, one after the other, without
     * interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that source {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be concatenated
     * @param source2
     *            a {@code Publisher} to be concatenated
     * @param source3
     *            a {@code Publisher} to be concatenated
     * @param source4
     *            a {@code Publisher} to be concatenated
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3} or {@code source4} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> concat(
            @NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull Publisher<@NonNull ? extends T> source3, @NonNull Publisher<@NonNull ? extends T> source4) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        return concatArray(source1, source2, source3, source4);
    }

    /**
     * Concatenates a variable number of {@link Publisher} sources.
     * <p>
     * Note: named this way because of overload conflict with {@code concat(Publisher<Publisher<T>>}).
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that source {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param sources the array of source {@code Publisher}s
     * @param <T> the common base value type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> concatArray(@NonNull Publisher<@NonNull ? extends T>... sources) {
        Objects.requireNonNull(sources, "sources is null");
        if (sources.length == 0) {
            return empty();
        } else
        if (sources.length == 1) {
            return fromPublisher(sources[0]);
        }
        return RxJavaPlugins.onAssembly(new FlowableConcatArray<>(sources, false));
    }

    /**
     * Concatenates a variable number of {@link Publisher} sources and delays errors from any of them
     * till all terminate.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when that source {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatArrayDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param sources the array of source {@code Publisher}s
     * @param <T> the common base value type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> concatArrayDelayError(@NonNull Publisher<@NonNull ? extends T>... sources) {
        Objects.requireNonNull(sources, "sources is null");
        if (sources.length == 0) {
            return empty();
        } else
        if (sources.length == 1) {
            return fromPublisher(sources[0]);
        }
        return RxJavaPlugins.onAssembly(new FlowableConcatArray<>(sources, true));
    }

    /**
     * Concatenates an array of {@link Publisher}s eagerly into a single stream of values.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatArrayEager.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, the operator will signal a
     *  {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources an array of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> concatArrayEager(@NonNull Publisher<@NonNull ? extends T>... sources) {
        return concatArrayEager(bufferSize(), bufferSize(), sources);
    }

    /**
     * Concatenates an array of {@link Publisher}s eagerly into a single stream of values.
     * <p>
     * <img width="640" height="406" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatArrayEager.nn.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, the operator will signal a
     *  {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources an array of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrent subscriptions at a time, {@link Integer#MAX_VALUE}
     *                       is interpreted as an indication to subscribe to all sources at once
     * @param prefetch the number of elements to prefetch from each {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings({ "rawtypes", "unchecked" })
    @SafeVarargs
    public static <T> Flowable<T> concatArrayEager(int maxConcurrency, int prefetch, @NonNull Publisher<@NonNull ? extends T>... sources) {
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEager(new FlowableFromArray(sources), Functions.identity(), maxConcurrency, prefetch, ErrorMode.IMMEDIATE));
    }

    /**
     * Concatenates an array of {@link Publisher}s eagerly into a single stream of values
     * and delaying any errors until all sources terminate.
     * <p>
     * <img width="640" height="358" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatArrayEagerDelayError.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s
     * and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, the operator will signal a
     *  {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources an array of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 2.2.1 - experimental
     */
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> concatArrayEagerDelayError(@NonNull Publisher<@NonNull ? extends T>... sources) {
        return concatArrayEagerDelayError(bufferSize(), bufferSize(), sources);
    }

    /**
     * Concatenates an array of {@link Publisher}s eagerly into a single stream of values
     * and delaying any errors until all sources terminate.
     * <p>
     * <img width="640" height="359" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatArrayEagerDelayError.nn.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s
     * and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the source {@code Publisher}s violate this, the operator will signal a
     *  {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources an array of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrent subscriptions at a time, {@link Integer#MAX_VALUE}
     *                       is interpreted as indication to subscribe to all sources at once
     * @param prefetch the number of elements to prefetch from each {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.2.1 - experimental
     */
    @SuppressWarnings({ "rawtypes", "unchecked" })
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> concatArrayEagerDelayError(int maxConcurrency, int prefetch, @NonNull Publisher<@NonNull ? extends T>... sources) {
        return fromArray(sources).concatMapEagerDelayError((Function)Functions.identity(), true, maxConcurrency, prefetch);
    }

    /**
     * Concatenates the {@link Iterable} sequence of {@link Publisher}s into a single sequence by subscribing to each {@code Publisher},
     * one after the other, one at a time and delays any errors till the all inner {@code Publisher}s terminate.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}
     *  sources are expected to honor backpressure as well. If the outer violates this, a
     *  {@link MissingBackpressureException} is signaled. If any of the inner {@code Publisher}s violates
     *  this, it <em>may</em> throw an {@link IllegalStateException} when an inner {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources the {@code Iterable} sequence of {@code Publisher}s
     * @return the new {@code Flowable} with the concatenating behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> concatDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        Objects.requireNonNull(sources, "sources is null");
        return fromIterable(sources).concatMapDelayError((Function)Functions.identity());
    }

    /**
     * Concatenates the {@link Publisher} sequence of {@code Publisher}s into a single sequence by subscribing to each inner {@code Publisher},
     * one after the other, one at a time and delays any errors till the all inner and the outer {@code Publisher}s terminate.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>{@code concatDelayError} fully supports backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources the {@code Publisher} sequence of {@code Publisher}s
     * @return the new {@code Flowable} with the concatenating behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concatDelayError(sources, bufferSize(), true);
    }

    /**
     * Concatenates the {@link Publisher} sequence of {@code Publisher}s into a single sequence by subscribing to each inner {@code Publisher},
     * one after the other, one at a time and delays any errors till the all inner and the outer {@code Publisher}s terminate.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>{@code concatDelayError} fully supports backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources the {@code Publisher} sequence of {@code Publisher}s
     * @param prefetch the number of elements to prefetch from the outer {@code Publisher}
     * @param tillTheEnd if {@code true}, exceptions from the outer and all inner {@code Publisher}s are delayed to the end
     *                   if {@code false}, exception from the outer {@code Publisher} is delayed till the current inner {@code Publisher} terminates
     * @return the new {@code Flowable} with the concatenating behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is {@code null}
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int prefetch, boolean tillTheEnd) {
        return fromPublisher(sources).concatMapDelayError((Function)Functions.identity(), tillTheEnd, prefetch);
    }

    /**
     * Concatenates a sequence of {@link Publisher}s eagerly into a single stream of values.
     * <p>
     * <img width="640" height="422" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEager.i.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and the inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatEager(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concatEager(sources, bufferSize(), bufferSize());
    }

    /**
     * Concatenates a sequence of {@link Publisher}s eagerly into a single stream of values and
     * runs a limited number of inner sequences at once.
     * <p>
     * <img width="640" height="375" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEager.in.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrently running inner {@code Publisher}s; {@link Integer#MAX_VALUE}
     *                       is interpreted as all inner {@code Publisher}s can be active at the same time
     * @param prefetch the number of elements to prefetch from each inner {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings({ "rawtypes", "unchecked" })
    public static <T> Flowable<T> concatEager(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int prefetch) {
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEager(new FlowableFromIterable(sources), Functions.identity(), maxConcurrency, prefetch, ErrorMode.BOUNDARY));
    }

    /**
     * Concatenates a {@link Publisher} sequence of {@code Publisher}s eagerly into a single stream of values.
     * <p>
     * <img width="640" height="490" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEager.p.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * emitted source {@code Publisher}s as they are observed. The operator buffers the values emitted by these
     * {@code Publisher}s and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatEager(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concatEager(sources, bufferSize(), bufferSize());
    }

    /**
     * Concatenates a {@link Publisher} sequence of {@code Publisher}s eagerly into a single stream of values and
     * runs a limited number of inner sequences at once.
     * <p>
     * <img width="640" height="421" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEager.pn.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * emitted source {@code Publisher}s as they are observed. The operator buffers the values emitted by these
     * {@code Publisher}s and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrently running inner {@code Publisher}s; {@link Integer#MAX_VALUE}
     *                       is interpreted as all inner {@code Publisher}s can be active at the same time
     * @param prefetch the number of elements to prefetch from each inner {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings({ "rawtypes", "unchecked" })
    public static <T> Flowable<T> concatEager(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int prefetch) {
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEagerPublisher(sources, Functions.identity(), maxConcurrency, prefetch, ErrorMode.IMMEDIATE));
    }

    /**
     * Concatenates a sequence of {@link Publisher}s eagerly into a single stream of values,
     * delaying errors until all the inner sequences terminate.
     * <p>
     * <img width="640" height="428" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEagerDelayError.i.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and the inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatEagerDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concatEagerDelayError(sources, bufferSize(), bufferSize());
    }

    /**
     * Concatenates a sequence of {@link Publisher}s eagerly into a single stream of values,
     * delaying errors until all the inner sequences terminate and runs a limited number
     * of inner sequences at once.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEagerDelayError.in.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * source {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them
     * in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrently running inner {@code Publisher}s; {@link Integer#MAX_VALUE}
     *                       is interpreted as all inner {@code Publisher}s can be active at the same time
     * @param prefetch the number of elements to prefetch from each inner {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings({ "rawtypes", "unchecked" })
    public static <T> Flowable<T> concatEagerDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int prefetch) {
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEager(new FlowableFromIterable(sources), Functions.identity(), maxConcurrency, prefetch, ErrorMode.END));
    }

    /**
     * Concatenates a {@link Publisher} sequence of {@code Publisher}s eagerly into a single stream of values,
     * delaying errors until all the inner and the outer sequences terminate.
     * <p>
     * <img width="640" height="496" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEagerDelayError.p.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * emitted source {@code Publisher}s as they are observed. The operator buffers the values emitted by these
     * {@code Publisher}s and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> concatEagerDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return concatEagerDelayError(sources, bufferSize(), bufferSize());
    }

    /**
     * Concatenates a {@link Publisher} sequence of {@code Publisher}s eagerly into a single stream of values,
     * delaying errors until all the inner and outer sequences terminate and runs a limited number of inner
     * sequences at once.
     * <p>
     * <img width="640" height="421" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.concatEagerDelayError.pn.png" alt="">
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * emitted source {@code Publisher}s as they are observed. The operator buffers the values emitted by these
     * {@code Publisher}s and then drains them in order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream and both the outer and inner {@code Publisher}s are
     *  expected to support backpressure. Violating this assumption, the operator will
     *  signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type
     * @param sources a sequence of {@code Publisher}s that need to be eagerly concatenated
     * @param maxConcurrency the maximum number of concurrently running inner {@code Publisher}s; {@link Integer#MAX_VALUE}
     *                       is interpreted as all inner {@code Publisher}s can be active at the same time
     * @param prefetch the number of elements to prefetch from each inner {@code Publisher} source
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings({ "rawtypes", "unchecked" })
    public static <T> Flowable<T> concatEagerDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int prefetch) {
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEagerPublisher(sources, Functions.identity(), maxConcurrency, prefetch, ErrorMode.END));
    }

    /**
     * Provides an API (via a cold {@code Flowable}) that bridges the reactive world with the callback-style,
     * generally non-backpressured world.
     * <p>
     * Example:
     * <pre><code>
     * Flowable.&lt;Event&gt;create(emitter -&gt; {
     *     Callback listener = new Callback() {
     *         &#64;Override
     *         public void onEvent(Event e) {
     *             emitter.onNext(e);
     *             if (e.isLast()) {
     *                 emitter.onComplete();
     *             }
     *         }
     *
     *         &#64;Override
     *         public void onFailure(Exception e) {
     *             emitter.onError(e);
     *         }
     *     };
     *
     *     AutoCloseable c = api.someMethod(listener);
     *
     *     emitter.setCancellable(c::close);
     *
     * }, BackpressureStrategy.BUFFER);
     * </code></pre>
     * <p>
     * Whenever a {@link Subscriber} subscribes to the returned {@code Flowable}, the provided
     * {@link FlowableOnSubscribe} callback is invoked with a fresh instance of a {@link FlowableEmitter}
     * that will interact only with that specific {@code Subscriber}. If this {@code Subscriber}
     * cancels the flow (making {@link FlowableEmitter#isCancelled} return {@code true}),
     * other observers subscribed to the same returned {@code Flowable} are not affected.
     * <p>
     * You should call the {@link FlowableEmitter#onNext(Object)}, {@link FlowableEmitter#onError(Throwable)}
     * and {@link FlowableEmitter#onComplete()} methods in a serialized fashion. The
     * rest of its methods are thread-safe.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The backpressure behavior is determined by the {@code mode} parameter.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code create} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the element type
     * @param source the emitter that is called when a {@code Subscriber} subscribes to the returned {@code Flowable}
     * @param mode the backpressure mode to apply if the downstream {@code Subscriber} doesn't request (fast) enough
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source} or {@code mode} is {@code null}
     * @see FlowableOnSubscribe
     * @see BackpressureStrategy
     * @see Cancellable
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> create(@NonNull FlowableOnSubscribe<T> source, @NonNull BackpressureStrategy mode) {
        Objects.requireNonNull(source, "source is null");
        Objects.requireNonNull(mode, "mode is null");
        return RxJavaPlugins.onAssembly(new FlowableCreate<>(source, mode));
    }

    /**
     * Returns a {@code Flowable} that calls a {@link Publisher} factory to create a {@code Publisher} for each new {@link Subscriber}
     * that subscribes. That is, for each subscriber, the actual {@code Publisher} that subscriber observes is
     * determined by the factory function.
     * <p>
     * <img width="640" height="340" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/defer.v3.png" alt="">
     * <p>
     * The defer {@code Subscriber} allows you to defer or delay emitting items from a {@code Publisher} until such time as a
     * {@code Subscriber} subscribes to the {@code Publisher}. This allows a {@code Subscriber} to easily obtain updates or a
     * refreshed version of the sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator itself doesn't interfere with backpressure which is determined by the {@code Publisher}
     *  returned by the {@code supplier}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code defer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param supplier
     *            the {@code Publisher} factory function to invoke for each {@code Subscriber} that subscribes to the
     *            resulting {@code Flowable}
     * @param <T>
     *            the type of the items emitted by the {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code supplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/defer.html">ReactiveX operators documentation: Defer</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> defer(@NonNull Supplier<? extends Publisher<@NonNull ? extends T>> supplier) {
        Objects.requireNonNull(supplier, "supplier is null");
        return RxJavaPlugins.onAssembly(new FlowableDefer<>(supplier));
    }

    /**
     * Returns a {@code Flowable} that emits no items to the {@link Subscriber} and immediately invokes its
     * {@link Subscriber#onComplete onComplete} method.
     * <p>
     * <img width="640" height="190" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/empty.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code empty} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the type of the items (ostensibly) emitted by the {@link Publisher}
     * @return the shared {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/empty-never-throw.html">ReactiveX operators documentation: Empty</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings("unchecked")
    @NonNull
    public static <T> Flowable<T> empty() {
        return RxJavaPlugins.onAssembly((Flowable<T>) FlowableEmpty.INSTANCE);
    }

    /**
     * Returns a {@code Flowable} that invokes a {@link Subscriber}'s {@link Subscriber#onError onError} method when the
     * {@code Subscriber} subscribes to it.
     * <p>
     * <img width="640" height="190" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/error.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code error} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param supplier
     *            a {@link Supplier} factory to return a {@link Throwable} for each individual {@code Subscriber}
     * @param <T>
     *            the type of the items (ostensibly) emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code supplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/empty-never-throw.html">ReactiveX operators documentation: Throw</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> error(@NonNull Supplier<? extends Throwable> supplier) {
        Objects.requireNonNull(supplier, "supplier is null");
        return RxJavaPlugins.onAssembly(new FlowableError<>(supplier));
    }

    /**
     * Returns a {@code Flowable} that invokes a {@link Subscriber}'s {@link Subscriber#onError onError} method when the
     * {@code Subscriber} subscribes to it.
     * <p>
     * <img width="640" height="190" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/error.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code error} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param throwable
     *            the particular {@link Throwable} to pass to {@link Subscriber#onError onError}
     * @param <T>
     *            the type of the items (ostensibly) emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code throwable} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/empty-never-throw.html">ReactiveX operators documentation: Throw</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> error(@NonNull Throwable throwable) {
        Objects.requireNonNull(throwable, "throwable is null");
        return error(Functions.justSupplier(throwable));
    }

    /**
     * Returns a {@code Flowable} instance that runs the given {@link Action} for each {@link Subscriber} and
     * emits either its exception or simply completes.
     * <p>
     * <img width="640" height="285" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromAction.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromAction} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd> If the {@code Action} throws an exception, the respective {@link Throwable} is
     *  delivered to the downstream via {@link Subscriber#onError(Throwable)},
     *  except when the downstream has canceled the resulting {@code Flowable} source.
     *  In this latter case, the {@code Throwable} is delivered to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} as an {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}.
     *  </dd>
     * </dl>
     * @param <T> the target type
     * @param action the {@code Action} to run for each {@code Subscriber}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code action} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    public static <T> Flowable<T> fromAction(@NonNull Action action) {
        Objects.requireNonNull(action, "action is null");
        return RxJavaPlugins.onAssembly(new FlowableFromAction<>(action));
    }

    /**
     * Converts an array into a {@link Publisher} that emits the items in the array.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/from.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and iterates the given {@code array}
     *  on demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param items
     *            the array of elements
     * @param <T>
     *            the type of items in the array and the type of items to be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code items} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/from.html">ReactiveX operators documentation: From</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    public static <@NonNull T> Flowable<T> fromArray(@NonNull T... items) {
        Objects.requireNonNull(items, "items is null");
        if (items.length == 0) {
            return empty();
        }
        if (items.length == 1) {
            return just(items[0]);
        }
        return RxJavaPlugins.onAssembly(new FlowableFromArray<>(items));
    }

    /**
     * Returns a {@code Flowable} that, when a {@link Subscriber} subscribes to it, invokes a function you specify and then
     * emits the value returned from that function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/fromCallable.v3.png" alt="">
     * <p>
     * This allows you to defer the execution of the function you specify until a {@code Subscriber} subscribes to the
     * {@link Publisher}. That is to say, it makes the function "lazy."
     * <dl>
     *   <dt><b>Backpressure:</b></dt>
     *   <dd>The operator honors backpressure from downstream.</dd>
     *   <dt><b>Scheduler:</b></dt>
     *   <dd>{@code fromCallable} does not operate by default on a particular {@link Scheduler}.</dd>
     *   <dt><b>Error handling:</b></dt>
     *   <dd> If the {@link Callable} throws an exception, the respective {@link Throwable} is
     *   delivered to the downstream via {@link Subscriber#onError(Throwable)},
     *   except when the downstream has canceled this {@code Flowable} source.
     *   In this latter case, the {@code Throwable} is delivered to the global error handler via
     *   {@link RxJavaPlugins#onError(Throwable)} as an {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}.
     *   </dd>
     * </dl>
     *
     * @param callable
     *         a function, the execution of which should be deferred; {@code fromCallable} will invoke this
     *         function only when a {@code Subscriber} subscribes to the {@code Publisher} that {@code fromCallable} returns
     * @param <T>
     *         the type of the item emitted by the {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code callable} is {@code null}
     * @see #defer(Supplier)
     * @see #fromSupplier(Supplier)
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> fromCallable(@NonNull Callable<? extends T> callable) {
        Objects.requireNonNull(callable, "callable is null");
        return RxJavaPlugins.onAssembly(new FlowableFromCallable<>(callable));
    }

    /**
     * Wraps a {@link CompletableSource} into a {@code Flowable}.
     * <p>
     * <img width="640" height="278" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromCompletable.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the target type
     * @param completableSource the {@code CompletableSource} to convert from
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code completableSource} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    public static <T> Flowable<T> fromCompletable(@NonNull CompletableSource completableSource) {
        Objects.requireNonNull(completableSource, "completableSource is null");
        return RxJavaPlugins.onAssembly(new FlowableFromCompletable<>(completableSource));
    }

    /**
     * Converts a {@link Future} into a {@link Publisher}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/from.Future.v3.png" alt="">
     * <p>
     * The operator calls {@link Future#get()}, which is a blocking method, on the subscription thread.
     * It is recommended applying {@link #subscribeOn(Scheduler)} to move this blocking wait to a
     * background thread, and if the {@link Scheduler} supports it, interrupt the wait when the flow
     * is disposed.
     * <p>
     * Also note that this operator will consume a {@link CompletionStage}-based {@code Future} subclass (such as
     * {@link CompletableFuture}) in a blocking manner as well. Use the {@link #fromCompletionStage(CompletionStage)}
     * operator to convert and consume such sources in a non-blocking fashion instead.
     * <p>
     * Unlike 1.x, canceling the {@code Flowable} won't cancel the future. If necessary, one can use composition to achieve the
     * cancellation effect: {@code futurePublisher.doOnCancel(() -> future.cancel(true));}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromFuture} does not operate by default on a particular {@code Scheduler}.</dd>
     * </dl>
     *
     * @param future
     *            the source {@code Future}
     * @param <T>
     *            the type of object that the {@code Future} returns, and also the type of item to be emitted by
     *            the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code future} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/from.html">ReactiveX operators documentation: From</a>
     * @see #fromCompletionStage(CompletionStage)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> fromFuture(@NonNull Future<? extends T> future) {
        Objects.requireNonNull(future, "future is null");
        return RxJavaPlugins.onAssembly(new FlowableFromFuture<>(future, 0L, null));
    }

    /**
     * Converts a {@link Future} into a {@link Publisher}, with a timeout on the {@code Future}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/from.Future.v3.png" alt="">
     * <p>
     * The operator calls {@link Future#get(long, TimeUnit)}, which is a blocking method, on the subscription thread.
     * It is recommended applying {@link #subscribeOn(Scheduler)} to move this blocking wait to a
     * background thread, and if the {@link Scheduler} supports it, interrupt the wait when the flow
     * is disposed.
     * <p>
     * Unlike 1.x, canceling the {@code Flowable} won't cancel the future. If necessary, one can use composition to achieve the
     * cancellation effect: {@code futurePublisher.doOnCancel(() -> future.cancel(true));}.
     * <p>
     * Also note that this operator will consume a {@link CompletionStage}-based {@code Future} subclass (such as
     * {@link CompletableFuture}) in a blocking manner as well. Use the {@link #fromCompletionStage(CompletionStage)}
     * operator to convert and consume such sources in a non-blocking fashion instead.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromFuture} does not operate by default on a particular {@code Scheduler}.</dd>
     * </dl>
     *
     * @param future
     *            the source {@code Future}
     * @param timeout
     *            the maximum time to wait before calling {@code get}
     * @param unit
     *            the {@link TimeUnit} of the {@code timeout} argument
     * @param <T>
     *            the type of object that the {@code Future} returns, and also the type of item to be emitted by
     *            the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code future} or {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/from.html">ReactiveX operators documentation: From</a>
     * @see #fromCompletionStage(CompletionStage)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> fromFuture(@NonNull Future<? extends T> future, long timeout, @NonNull TimeUnit unit) {
        Objects.requireNonNull(future, "future is null");
        Objects.requireNonNull(unit, "unit is null");
        return RxJavaPlugins.onAssembly(new FlowableFromFuture<>(future, timeout, unit));
    }

    /**
     * Converts an {@link Iterable} sequence into a {@link Publisher} that emits the items in the sequence.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/from.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and iterates the given {@code iterable}
     *  on demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param source
     *            the source {@code Iterable} sequence
     * @param <T>
     *            the type of items in the {@code Iterable} sequence and the type of items to be emitted by the
     *            resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/from.html">ReactiveX operators documentation: From</a>
     * @see #fromStream(Stream)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> fromIterable(@NonNull Iterable<@NonNull ? extends T> source) {
        Objects.requireNonNull(source, "source is null");
        return RxJavaPlugins.onAssembly(new FlowableFromIterable<>(source));
    }

    /**
     * Returns a {@code Flowable} instance that when subscribed to, subscribes to the {@link MaybeSource} instance and
     * emits {@code onSuccess} as a single item or forwards any {@code onComplete} or
     * {@code onError} signal.
     * <p>
     * <img width="640" height="226" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromMaybe.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type of the {@code MaybeSource} element
     * @param maybe the {@code MaybeSource} instance to subscribe to, not {@code null}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code maybe} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public static <T> Flowable<T> fromMaybe(@NonNull MaybeSource<T> maybe) {
        Objects.requireNonNull(maybe, "maybe is null");
        return RxJavaPlugins.onAssembly(new MaybeToFlowable<>(maybe));
    }

    /**
     * Converts the given {@link ObservableSource} into a {@code Flowable} by applying the specified backpressure strategy.
     * <p>
     * Marble diagrams for the various backpressure strategies are as follows:
     * <ul>
     * <li>{@link BackpressureStrategy#BUFFER}
     * <p>
     * <img width="640" height="264" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromObservable.buffer.png" alt="">
     * </li>
     * <li>{@link BackpressureStrategy#DROP}
     * <p>
     * <img width="640" height="374" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromObservable.drop.png" alt="">
     * </li>
     * <li>{@link BackpressureStrategy#LATEST}
     * <p>
     * <img width="640" height="284" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromObservable.latest.png" alt="">
     * </li>
     * <li>{@link BackpressureStrategy#ERROR}
     * <p>
     * <img width="640" height="365" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromObservable.error.png" alt="">
     * </li>
     * <li>{@link BackpressureStrategy#MISSING}
     * <p>
     * <img width="640" height="397" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromObservable.missing.png" alt="">
     * </li>
     * </ul>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator applies the chosen backpressure strategy of {@link BackpressureStrategy} enum.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromObservable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the element type of the source and resulting sequence
     * @param source the {@code ObservableSource} to convert
     * @param strategy the backpressure strategy to apply
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source} or {@code strategy} is {@code null}
     */
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> fromObservable(@NonNull ObservableSource<T> source, @NonNull BackpressureStrategy strategy) {
        Objects.requireNonNull(source, "source is null");
        Objects.requireNonNull(strategy, "strategy is null");
        Flowable<T> f = new FlowableFromObservable<>(source);
        switch (strategy) {
            case DROP:
                return f.onBackpressureDrop();
            case LATEST:
                return f.onBackpressureLatest();
            case MISSING:
                return f;
            case ERROR:
                return RxJavaPlugins.onAssembly(new FlowableOnBackpressureError<>(f));
            default:
                return f.onBackpressureBuffer();
        }
    }

    /**
     * Converts an arbitrary <em>Reactive Streams</em> {@link Publisher} into a {@code Flowable} if not already a
     * {@code Flowable}.
     * <p>
     * The {@code Publisher} must follow the
     * <a href="https://github.com/reactive-streams/reactive-streams-jvm#reactive-streams">Reactive-Streams specification</a>.
     * Violating the specification may result in undefined behavior.
     * <p>
     * If possible, use {@link #create(FlowableOnSubscribe, BackpressureStrategy)} to create a
     * source-like {@code Flowable} instead.
     * <p>
     * Note that even though {@code Publisher} appears to be a functional interface, it
     * is not recommended to implement it through a lambda as the specification requires
     * state management that is not achievable with a stateless lambda.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator is a pass-through for backpressure and its behavior is determined by the
     *  backpressure behavior of the wrapped publisher.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromPublisher} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type of the flow
     * @param publisher the {@code Publisher} to convert
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code publisher} is {@code null}
     * @see #create(FlowableOnSubscribe, BackpressureStrategy)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings("unchecked")
    public static <T> Flowable<T> fromPublisher(@NonNull Publisher<@NonNull ? extends T> publisher) {
        if (publisher instanceof Flowable) {
            return RxJavaPlugins.onAssembly((Flowable<T>)publisher);
        }
        Objects.requireNonNull(publisher, "publisher is null");

        return RxJavaPlugins.onAssembly(new FlowableFromPublisher<>(publisher));
    }

    /**
     * Returns a {@code Flowable} instance that runs the given {@link Runnable} for each {@link Subscriber} and
     * emits either its unchecked exception or simply completes.
     * <p>
     * <img width="640" height="286" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromRunnable.png" alt="">
     * <p>
     * If the code to be wrapped needs to throw a checked or more broader {@link Throwable} exception, that
     * exception has to be converted to an unchecked exception by the wrapped code itself. Alternatively,
     * use the {@link #fromAction(Action)} method which allows the wrapped code to throw any {@code Throwable}
     * exception and will signal it to observers as-is.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromRunnable} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd> If the {@code Runnable} throws an exception, the respective {@code Throwable} is
     *  delivered to the downstream via {@link Subscriber#onError(Throwable)},
     *  except when the downstream has canceled the resulting {@code Flowable} source.
     *  In this latter case, the {@code Throwable} is delivered to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} as an {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}.
     *  </dd>
     * </dl>
     * @param <T> the target type
     * @param run the {@code Runnable} to run for each {@code Subscriber}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code run} is {@code null}
     * @since 3.0.0
     * @see #fromAction(Action)
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    public static <T> Flowable<T> fromRunnable(@NonNull Runnable run) {
        Objects.requireNonNull(run, "run is null");
        return RxJavaPlugins.onAssembly(new FlowableFromRunnable<>(run));
    }

    /**
     * Returns a {@code Flowable} instance that when subscribed to, subscribes to the {@link SingleSource} instance and
     * emits {@code onSuccess} as a single item or forwards the {@code onError} signal.
     * <p>
     * <img width="640" height="341" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromSingle.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code fromSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type of the {@code SingleSource} element
     * @param source the {@code SingleSource} instance to subscribe to, not {@code null}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public static <T> Flowable<T> fromSingle(@NonNull SingleSource<T> source) {
        Objects.requireNonNull(source, "source is null");
        return RxJavaPlugins.onAssembly(new SingleToFlowable<>(source));
    }

    /**
     * Returns a {@code Flowable} that, when a {@link Subscriber} subscribes to it, invokes a supplier function you specify and then
     * emits the value returned from that function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.fromSupplier.v3.png" alt="">
     * <p>
     * This allows you to defer the execution of the function you specify until a {@code Subscriber} subscribes to the
     * {@link Publisher}. That is to say, it makes the function "lazy."
     * <dl>
     *   <dt><b>Backpressure:</b></dt>
     *   <dd>The operator honors backpressure from downstream.</dd>
     *   <dt><b>Scheduler:</b></dt>
     *   <dd>{@code fromSupplier} does not operate by default on a particular {@link Scheduler}.</dd>
     *   <dt><b>Error handling:</b></dt>
     *   <dd> If the {@link Supplier} throws an exception, the respective {@link Throwable} is
     *   delivered to the downstream via {@link Subscriber#onError(Throwable)},
     *   except when the downstream has canceled this {@code Flowable} source.
     *   In this latter case, the {@code Throwable} is delivered to the global error handler via
     *   {@link RxJavaPlugins#onError(Throwable)} as an {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}.
     *   </dd>
     * </dl>
     *
     * @param supplier
     *         a function, the execution of which should be deferred; {@code fromSupplier} will invoke this
     *         function only when a {@code Subscriber} subscribes to the {@code Publisher} that {@code fromSupplier} returns
     * @param <T>
     *         the type of the item emitted by the {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code supplier} is {@code null}
     * @see #defer(Supplier)
     * @see #fromCallable(Callable)
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> fromSupplier(@NonNull Supplier<? extends T> supplier) {
        Objects.requireNonNull(supplier, "supplier is null");
        return RxJavaPlugins.onAssembly(new FlowableFromSupplier<>(supplier));
    }

    /**
     * Returns a cold, synchronous, stateless and backpressure-aware generator of values.
     * <p>
     * Note that the {@link Emitter#onNext}, {@link Emitter#onError} and
     * {@link Emitter#onComplete} methods provided to the function via the {@link Emitter} instance should be called synchronously,
     * never concurrently and only while the function body is executing. Calling them from multiple threads
     * or outside the function call is not supported and leads to an undefined behavior.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code generate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the generated value type
     * @param generator the {@link Consumer} called whenever a particular downstream {@link Subscriber} has
     * requested a value. The callback then should call {@code onNext}, {@code onError} or
     * {@code onComplete} to signal a value or a terminal event. Signaling multiple {@code onNext}
     * in a call will make the operator signal {@link IllegalStateException}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code generator} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> generate(@NonNull Consumer<@NonNull Emitter<T>> generator) {
        Objects.requireNonNull(generator, "generator is null");
        return generate(Functions.nullSupplier(),
                FlowableInternalHelper.simpleGenerator(generator),
                Functions.emptyConsumer());
    }

    /**
     * Returns a cold, synchronous, stateful and backpressure-aware generator of values.
     * <p>
     * Note that the {@link Emitter#onNext}, {@link Emitter#onError} and
     * {@link Emitter#onComplete} methods provided to the function via the {@link Emitter} instance should be called synchronously,
     * never concurrently and only while the function body is executing. Calling them from multiple threads
     * or outside the function call is not supported and leads to an undefined behavior.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code generate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <S> the type of the per-{@link Subscriber} state
     * @param <T> the generated value type
     * @param initialState the {@link Supplier} to generate the initial state for each {@code Subscriber}
     * @param generator the {@link Consumer} called with the current state whenever a particular downstream {@code Subscriber} has
     * requested a value. The callback then should call {@code onNext}, {@code onError} or
     * {@code onComplete} to signal a value or a terminal event. Signaling multiple {@code onNext}
     * in a call will make the operator signal {@link IllegalStateException}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code initialState} or {@code generator} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, S> Flowable<T> generate(@NonNull Supplier<S> initialState, @NonNull BiConsumer<S, Emitter<T>> generator) {
        Objects.requireNonNull(generator, "generator is null");
        return generate(initialState, FlowableInternalHelper.simpleBiGenerator(generator),
                Functions.emptyConsumer());
    }

    /**
     * Returns a cold, synchronous, stateful and backpressure-aware generator of values.
     * <p>
     * Note that the {@link Emitter#onNext}, {@link Emitter#onError} and
     * {@link Emitter#onComplete} methods provided to the function via the {@link Emitter} instance should be called synchronously,
     * never concurrently and only while the function body is executing. Calling them from multiple threads
     * or outside the function call is not supported and leads to an undefined behavior.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code generate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <S> the type of the per-{@link Subscriber} state
     * @param <T> the generated value type
     * @param initialState the {@link Supplier} to generate the initial state for each {@code Subscriber}
     * @param generator the {@link Consumer} called with the current state whenever a particular downstream {@code Subscriber} has
     * requested a value. The callback then should call {@code onNext}, {@code onError} or
     * {@code onComplete} to signal a value or a terminal event. Signaling multiple {@code onNext}
     * in a call will make the operator signal {@link IllegalStateException}.
     * @param disposeState the {@code Consumer} that is called with the current state when the generator
     * terminates the sequence or it gets canceled
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code initialState}, {@code generator} or {@code disposeState} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, S> Flowable<T> generate(@NonNull Supplier<S> initialState, @NonNull BiConsumer<S, Emitter<T>> generator,
            @NonNull Consumer<? super S> disposeState) {
        Objects.requireNonNull(generator, "generator is null");
        return generate(initialState, FlowableInternalHelper.simpleBiGenerator(generator), disposeState);
    }

    /**
     * Returns a cold, synchronous, stateful and backpressure-aware generator of values.
     * <p>
     * Note that the {@link Emitter#onNext}, {@link Emitter#onError} and
     * {@link Emitter#onComplete} methods provided to the function via the {@link Emitter} instance should be called synchronously,
     * never concurrently and only while the function body is executing. Calling them from multiple threads
     * or outside the function call is not supported and leads to an undefined behavior.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code generate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <S> the type of the per-{@link Subscriber} state
     * @param <T> the generated value type
     * @param initialState the {@link Supplier} to generate the initial state for each {@code Subscriber}
     * @param generator the {@link Function} called with the current state whenever a particular downstream {@code Subscriber} has
     * requested a value. The callback then should call {@code onNext}, {@code onError} or
     * {@code onComplete} to signal a value or a terminal event and should return a (new) state for
     * the next invocation. Signaling multiple {@code onNext}
     * in a call will make the operator signal {@link IllegalStateException}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code initialState} or {@code generator} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T, S> Flowable<T> generate(@NonNull Supplier<S> initialState, @NonNull BiFunction<S, @NonNull Emitter<T>, S> generator) {
        return generate(initialState, generator, Functions.emptyConsumer());
    }

    /**
     * Returns a cold, synchronous, stateful and backpressure-aware generator of values.
     * <p>
     * Note that the {@link Emitter#onNext}, {@link Emitter#onError} and
     * {@link Emitter#onComplete} methods provided to the function via the {@link Emitter} instance should be called synchronously,
     * never concurrently and only while the function body is executing. Calling them from multiple threads
     * or outside the function call is not supported and leads to an undefined behavior.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code generate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <S> the type of the per-{@link Subscriber} state
     * @param <T> the generated value type
     * @param initialState the {@link Supplier} to generate the initial state for each {@code Subscriber}
     * @param generator the {@link Function} called with the current state whenever a particular downstream {@code Subscriber} has
     * requested a value. The callback then should call {@code onNext}, {@code onError} or
     * {@code onComplete} to signal a value or a terminal event and should return a (new) state for
     * the next invocation. Signaling multiple {@code onNext}
     * in a call will make the operator signal {@link IllegalStateException}.
     * @param disposeState the {@link Consumer} that is called with the current state when the generator
     * terminates the sequence or it gets canceled
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code initialState}, {@code generator} or {@code disposeState} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, S> Flowable<T> generate(@NonNull Supplier<S> initialState, @NonNull BiFunction<S, @NonNull Emitter<T>, S> generator, @NonNull Consumer<? super S> disposeState) {
        Objects.requireNonNull(initialState, "initialState is null");
        Objects.requireNonNull(generator, "generator is null");
        Objects.requireNonNull(disposeState, "disposeState is null");
        return RxJavaPlugins.onAssembly(new FlowableGenerate<>(initialState, generator, disposeState));
    }

    /**
     * Returns a {@code Flowable} that emits a {@code 0L} after the {@code initialDelay} and ever-increasing numbers
     * after each {@code period} of time thereafter.
     * <p>
     * <img width="640" height="200" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timer.p.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator generates values based on time and ignores downstream backpressure which
     *  may lead to {@link MissingBackpressureException} at some point in the chain.
     *  Downstream consumers should consider applying one of the {@code onBackpressureXXX} operators as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code interval} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param initialDelay
     *            the initial delay time to wait before emitting the first value of 0L
     * @param period
     *            the period of time between emissions of the subsequent numbers
     * @param unit
     *            the time unit for both {@code initialDelay} and {@code period}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/interval.html">ReactiveX operators documentation: Interval</a>
     * @since 1.0.12
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public static Flowable<Long> interval(long initialDelay, long period, @NonNull TimeUnit unit) {
        return interval(initialDelay, period, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits a {@code 0L} after the {@code initialDelay} and ever-increasing numbers
     * after each {@code period} of time thereafter, on a specified {@link Scheduler}.
     * <p>
     * <img width="640" height="200" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timer.ps.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator generates values based on time and ignores downstream backpressure which
     *  may lead to {@link MissingBackpressureException} at some point in the chain.
     *  Downstream consumers should consider applying one of the {@code onBackpressureXXX} operators as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param initialDelay
     *            the initial delay time to wait before emitting the first value of 0L
     * @param period
     *            the period of time between emissions of the subsequent numbers
     * @param unit
     *            the time unit for both {@code initialDelay} and {@code period}
     * @param scheduler
     *            the {@code Scheduler} on which the waiting happens and items are emitted
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/interval.html">ReactiveX operators documentation: Interval</a>
     * @since 1.0.12
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public static Flowable<Long> interval(long initialDelay, long period, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableInterval(Math.max(0L, initialDelay), Math.max(0L, period), unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that emits a sequential number every specified interval of time.
     * <p>
     * <img width="640" height="195" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/interval.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator signals a {@link MissingBackpressureException} if the downstream
     *  is not ready to receive the next value.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code interval} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param period
     *            the period size in time units (see below)
     * @param unit
     *            time units to use for the interval size
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/interval.html">ReactiveX operators documentation: Interval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public static Flowable<Long> interval(long period, @NonNull TimeUnit unit) {
        return interval(period, period, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits a sequential number every specified interval of time, on a
     * specified {@link Scheduler}.
     * <p>
     * <img width="640" height="200" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/interval.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator generates values based on time and ignores downstream backpressure which
     *  may lead to {@link MissingBackpressureException} at some point in the chain.
     *  Downstream consumers should consider applying one of the {@code onBackpressureXXX} operators as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param period
     *            the period size in time units (see below)
     * @param unit
     *            time units to use for the interval size
     * @param scheduler
     *            the {@code Scheduler} to use for scheduling the items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/interval.html">ReactiveX operators documentation: Interval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public static Flowable<Long> interval(long period, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return interval(period, period, unit, scheduler);
    }

    /**
     * Signals a range of long values, the first after some initial delay and the rest periodically after.
     * <p>
     * The sequence completes immediately after the last value {@code (start + count - 1)} has been reached.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator signals a {@link MissingBackpressureException} if the downstream can't keep up.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code intervalRange} by default operates on the {@link Schedulers#computation() computation} {@link Scheduler}.</dd>
     * </dl>
     * @param start that start value of the range
     * @param count the number of values to emit in total, if zero, the operator emits an {@code onComplete} after the initial delay.
     * @param initialDelay the initial delay before signaling the first value (the start)
     * @param period the period between subsequent values
     * @param unit the unit of measure of the {@code initialDelay} and {@code period} amounts
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero, or if {@code start} + {@code count} &minus; 1 exceeds
     *             {@link Long#MAX_VALUE}
     * @see #range(int, int)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    public static Flowable<Long> intervalRange(long start, long count, long initialDelay, long period, @NonNull TimeUnit unit) {
        return intervalRange(start, count, initialDelay, period, unit, Schedulers.computation());
    }

    /**
     * Signals a range of long values, the first after some initial delay and the rest periodically after.
     * <p>
     * The sequence completes immediately after the last value (start + count - 1) has been reached.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator signals a {@link MissingBackpressureException} if the downstream can't keep up.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>you provide the {@link Scheduler}.</dd>
     * </dl>
     * @param start that start value of the range
     * @param count the number of values to emit in total, if zero, the operator emits an {@code onComplete} after the initial delay.
     * @param initialDelay the initial delay before signaling the first value (the start)
     * @param period the period between subsequent values
     * @param unit the unit of measure of the {@code initialDelay} and {@code period} amounts
     * @param scheduler the target {@code Scheduler} where the values and terminal signals will be emitted
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero, or if {@code start} + {@code count} &minus; 1 exceeds
     *             {@link Long#MAX_VALUE}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public static Flowable<Long> intervalRange(long start, long count, long initialDelay, long period, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        if (count < 0L) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        }
        if (count == 0L) {
            return Flowable.<Long>empty().delay(initialDelay, unit, scheduler);
        }

        long end = start + (count - 1);
        if (start > 0 && end < 0) {
            throw new IllegalArgumentException("Overflow! start + count is bigger than Long.MAX_VALUE");
        }
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");

        return RxJavaPlugins.onAssembly(new FlowableIntervalRange(start, end, Math.max(0L, initialDelay), Math.max(0L, period), unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that signals the given (constant reference) item and then completes.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.v3.png" alt="">
     * <p>
     * Note that the item is taken and re-emitted as is and not computed by any means by {@code just}. Use {@link #fromCallable(Callable)}
     * to generate a single item on demand (when {@link Subscriber}s subscribe to it).
     * <p>
     * See the multi-parameter overloads of {@code just} to emit more than one (constant reference) items one after the other.
     * Use {@link #fromArray(Object...)} to emit an arbitrary number of items that are known upfront.
     * <p>
     * To emit the items of an {@link Iterable} sequence (such as a {@link java.util.List}), use {@link #fromIterable(Iterable)}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item
     *            the item to emit
     * @param <T>
     *            the type of that item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     * @see #just(Object, Object)
     * @see #fromCallable(Callable)
     * @see #fromArray(Object...)
     * @see #fromIterable(Iterable)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item) {
        Objects.requireNonNull(item, "item is null");
        return RxJavaPlugins.onAssembly(new FlowableJust<>(item));
    }

    /**
     * Converts two items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1} or {@code item2} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");

        return fromArray(item1, item2);
    }

    /**
     * Converts three items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2} or {@code item3} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");

        return fromArray(item1, item2, item3);
    }

    /**
     * Converts four items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              or {@code item4} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");

        return fromArray(item1, item2, item3, item4);
    }

    /**
     * Converts five items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4} or {@code item5} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");

        return fromArray(item1, item2, item3, item4, item5);
    }

    /**
     * Converts six items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param item6
     *            sixth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4}, {@code item5} or {@code item6} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5, T item6) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");
        Objects.requireNonNull(item6, "item6 is null");

        return fromArray(item1, item2, item3, item4, item5, item6);
    }

    /**
     * Converts seven items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param item6
     *            sixth item
     * @param item7
     *            seventh item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4}, {@code item5}, {@code item6}
     *                              or {@code item7} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5, T item6, T item7) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");
        Objects.requireNonNull(item6, "item6 is null");
        Objects.requireNonNull(item7, "item7 is null");

        return fromArray(item1, item2, item3, item4, item5, item6, item7);
    }

    /**
     * Converts eight items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param item6
     *            sixth item
     * @param item7
     *            seventh item
     * @param item8
     *            eighth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4}, {@code item5}, {@code item6},
     *                              {@code item7} or {@code item8} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5, T item6, T item7, T item8) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");
        Objects.requireNonNull(item6, "item6 is null");
        Objects.requireNonNull(item7, "item7 is null");
        Objects.requireNonNull(item8, "item8 is null");

        return fromArray(item1, item2, item3, item4, item5, item6, item7, item8);
    }

    /**
     * Converts nine items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param item6
     *            sixth item
     * @param item7
     *            seventh item
     * @param item8
     *            eighth item
     * @param item9
     *            ninth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4}, {@code item5}, {@code item6},
     *                              {@code item7}, {@code item8} or {@code item9} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5, T item6, T item7, T item8, T item9) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");
        Objects.requireNonNull(item6, "item6 is null");
        Objects.requireNonNull(item7, "item7 is null");
        Objects.requireNonNull(item8, "item8 is null");
        Objects.requireNonNull(item9, "item9 is null");

        return fromArray(item1, item2, item3, item4, item5, item6, item7, item8, item9);
    }

    /**
     * Converts ten items into a {@link Publisher} that emits those items.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/just.m.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals each value on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code just} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item1
     *            first item
     * @param item2
     *            second item
     * @param item3
     *            third item
     * @param item4
     *            fourth item
     * @param item5
     *            fifth item
     * @param item6
     *            sixth item
     * @param item7
     *            seventh item
     * @param item8
     *            eighth item
     * @param item9
     *            ninth item
     * @param item10
     *            tenth item
     * @param <T>
     *            the type of these items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item1}, {@code item2}, {@code item3},
     *                              {@code item4}, {@code item5}, {@code item6},
     *                              {@code item7}, {@code item8}, {@code item9},
     *                              or {@code item10} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/just.html">ReactiveX operators documentation: Just</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <@NonNull T> Flowable<T> just(T item1, T item2, T item3, T item4, T item5, T item6, T item7, T item8, T item9, T item10) {
        Objects.requireNonNull(item1, "item1 is null");
        Objects.requireNonNull(item2, "item2 is null");
        Objects.requireNonNull(item3, "item3 is null");
        Objects.requireNonNull(item4, "item4 is null");
        Objects.requireNonNull(item5, "item5 is null");
        Objects.requireNonNull(item6, "item6 is null");
        Objects.requireNonNull(item7, "item7 is null");
        Objects.requireNonNull(item8, "item8 is null");
        Objects.requireNonNull(item9, "item9 is null");
        Objects.requireNonNull(item10, "item10 is null");

        return fromArray(item1, item2, item3, item4, item5, item6, item7, item8, item9, item10);
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, without any transformation, while limiting the
     * number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Iterable, int, int)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param bufferSize
     *            the number of items to prefetch from each inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Iterable, int, int)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> merge(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int bufferSize) {
        return fromIterable(sources).flatMap((Function)Functions.identity(), false, maxConcurrency, bufferSize);
    }

    /**
     * Flattens an array of {@link Publisher}s into one {@code Publisher}, without any transformation, while limiting the
     * number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeArray} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeArrayDelayError(int, int, Publisher[])} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the array of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param bufferSize
     *            the number of items to prefetch from each inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeArrayDelayError(int, int, Publisher...)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> mergeArray(int maxConcurrency, int bufferSize, @NonNull Publisher<@NonNull ? extends T>... sources) {
        return fromArray(sources).flatMap((Function)Functions.identity(), false, maxConcurrency, bufferSize);
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Iterable)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Iterable)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> merge(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return fromIterable(sources).flatMap((Function)Functions.identity());
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, without any transformation, while limiting the
     * number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Iterable, int)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code maxConcurrency} is less than or equal to 0
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Iterable, int)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> merge(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency) {
        return fromIterable(sources).flatMap((Function)Functions.identity(), maxConcurrency);
    }

    /**
     * Flattens a {@link Publisher} that emits {@code Publisher}s into a single {@code Publisher} that emits the items emitted by
     * thos {@code Publisher}s , without any transformation.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.oo.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed
     *  in unbounded mode (i.e., no backpressure is applied to it). The inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Publisher)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Publisher)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> merge(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return merge(sources, bufferSize());
    }

    /**
     * Flattens a {@link Publisher} that emits {@code Publisher}s into a single {@code Publisher} that emits the items emitted by
     * those {@code Publisher}s, without any transformation, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.oo.v3.png" alt="">
     * <p>
     * You can combine the items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Publisher, int)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code maxConcurrency} is less than or equal to 0
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Publisher, int)
     * @since 1.1.0
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> merge(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency) {
        return fromPublisher(sources).flatMap((Function)Functions.identity(), maxConcurrency);
    }

    /**
     * Flattens an array of {@link Publisher}s into one {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.io.v3.png" alt="">
     * <p>
     * You can combine items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeArray} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeArrayDelayError(Publisher...)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the array of {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeArrayDelayError(Publisher...)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> mergeArray(@NonNull Publisher<@NonNull ? extends T>... sources) {
        return fromArray(sources).flatMap((Function)Functions.identity(), sources.length);
    }

    /**
     * Flattens two {@link Publisher}s into a single {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Publisher, Publisher)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1} or {@code source2} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Publisher, Publisher)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> merge(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        return fromArray(source1, source2).flatMap((Function)Functions.identity(), false, 2);
    }

    /**
     * Flattens three {@link Publisher}s into a single {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Publisher, Publisher, Publisher)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @param source3
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code source3} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Publisher, Publisher, Publisher)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> merge(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2, @NonNull Publisher<@NonNull ? extends T> source3) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        return fromArray(source1, source2, source3).flatMap((Function)Functions.identity(), false, 3);
    }

    /**
     * Flattens four {@link Publisher}s into a single {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code merge} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code merge} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If any of the source {@code Publisher}s signal a {@link Throwable} via {@code onError}, the resulting
     *  {@code Flowable} terminates with that {@code Throwable} and all other source {@code Publisher}s are canceled.
     *  If more than one {@code Publisher} signals an error, the resulting {@code Flowable} may terminate with the
     *  first one's error or, depending on the concurrency of the sources, may terminate with a
     *  {@link CompositeException} containing two or more of the various error signals.
     *  {@code Throwable}s that didn't make into the composite will be sent (individually) to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors. Similarly, {@code Throwable}s
     *  signaled by source(s) after the returned {@code Flowable} has been canceled or terminated with a
     *  (composite) error will be sent to the same global error handler.
     *  Use {@link #mergeDelayError(Publisher, Publisher, Publisher, Publisher)} to merge sources and terminate only when all source {@code Publisher}s
     *  have completed or failed with an error.
     *  </dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @param source3
     *            a {@code Publisher} to be merged
     * @param source4
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3} or {@code source4} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @see #mergeDelayError(Publisher, Publisher, Publisher, Publisher)
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> merge(
            @NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull Publisher<@NonNull ? extends T> source3, @NonNull Publisher<@NonNull ? extends T> source4) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        return fromArray(source1, source2, source3, source4).flatMap((Function)Functions.identity(), false, 4);
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. All inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> mergeDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return fromIterable(sources).flatMap((Function)Functions.identity(), true);
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them, while limiting the number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. All inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param bufferSize
     *            the number of items to prefetch from each inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> mergeDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency, int bufferSize) {
        return fromIterable(sources).flatMap((Function)Functions.identity(), true, maxConcurrency, bufferSize);
    }

    /**
     * Flattens an array of {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them, while limiting the number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. All source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeArrayDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the array of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param bufferSize
     *            the number of items to prefetch from each inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> mergeArrayDelayError(int maxConcurrency, int bufferSize, @NonNull Publisher<@NonNull ? extends T>... sources) {
        return fromArray(sources).flatMap((Function)Functions.identity(), true, maxConcurrency, bufferSize);
    }

    /**
     * Flattens an {@link Iterable} of {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them, while limiting the number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. All inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the {@code Iterable} of {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> mergeDelayError(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency) {
        return fromIterable(sources).flatMap((Function)Functions.identity(), true, maxConcurrency);
    }

    /**
     * Flattens a {@link Publisher} that emits {@code Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to
     * receive all successfully emitted items from all of the source {@code Publisher}s without being interrupted by
     * an error notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed
     *  in unbounded mode (i.e., no backpressure is applied to it). The inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> mergeDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return mergeDelayError(sources, bufferSize());
    }

    /**
     * Flattens a {@link Publisher} that emits {@code Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to
     * receive all successfully emitted items from all of the source {@code Publisher}s without being interrupted by
     * an error notification from one of them, while limiting the
     * number of concurrent subscriptions to these {@code Publisher}s.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            a {@code Publisher} that emits {@code Publisher}s
     * @param maxConcurrency
     *            the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     * @since 2.0
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> mergeDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int maxConcurrency) {
        return fromPublisher(sources).flatMap((Function)Functions.identity(), true, maxConcurrency);
    }

    /**
     * Flattens an array of {@link Publisher}s into one {@code Flowable}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher)} except that if any of the merged {@code Publisher}s notify of an
     * error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from propagating that
     * error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both the outer and inner {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeArrayDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param sources
     *            the array of {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public static <T> Flowable<T> mergeArrayDelayError(@NonNull Publisher<@NonNull ? extends T>... sources) {
        return fromArray(sources).flatMap((Function)Functions.identity(), true, sources.length);
    }

    /**
     * Flattens two {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from each of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher, Publisher)} except that if any of the merged {@code Publisher}s
     * notify of an error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain from
     * propagating that error notification until all of the merged {@code Publisher}s have finished emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if both merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1} or {@code source2} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> mergeDelayError(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        return fromArray(source1, source2).flatMap((Function)Functions.identity(), true, 2);
    }

    /**
     * Flattens three {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from all of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher, Publisher, Publisher)} except that if any of the merged
     * {@code Publisher}s notify of an error via {@link Subscriber#onError onError}, {@code mergeDelayError} will refrain
     * from propagating that error notification until all of the merged {@code Publisher}s have finished emitting
     * items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @param source3
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code source3} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> mergeDelayError(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2, @NonNull Publisher<@NonNull ? extends T> source3) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        return fromArray(source1, source2, source3).flatMap((Function)Functions.identity(), true, 3);
    }

    /**
     * Flattens four {@link Publisher}s into one {@code Publisher}, in a way that allows a {@link Subscriber} to receive all
     * successfully emitted items from all of the source {@code Publisher}s without being interrupted by an error
     * notification from one of them.
     * <p>
     * This behaves like {@link #merge(Publisher, Publisher, Publisher, Publisher)} except that if any of
     * the merged {@code Publisher}s notify of an error via {@link Subscriber#onError onError}, {@code mergeDelayError}
     * will refrain from propagating that error notification until all of the merged {@code Publisher}s have finished
     * emitting items.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeDelayError.v3.png" alt="">
     * <p>
     * Even if multiple merged {@code Publisher}s send {@code onError} notifications, {@code mergeDelayError} will only
     * invoke the {@code onError} method of its {@code Subscriber}s once.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element base type
     * @param source1
     *            a {@code Publisher} to be merged
     * @param source2
     *            a {@code Publisher} to be merged
     * @param source3
     *            a {@code Publisher} to be merged
     * @param source4
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3} or {@code source4} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> mergeDelayError(
            @NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull Publisher<@NonNull ? extends T> source3, @NonNull Publisher<@NonNull ? extends T> source4) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        return fromArray(source1, source2, source3, source4).flatMap((Function)Functions.identity(), true, 4);
    }

    /**
     * Returns a {@code Flowable} that never sends any items or notifications to a {@link Subscriber}.
     * <p>
     * <img width="640" height="185" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/never.v3.png" alt="">
     * <p>
     * This {@link Publisher} is useful primarily for testing purposes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This source doesn't produce any elements and effectively ignores downstream backpressure.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code never} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T>
     *            the type of items (not) emitted by the {@code Publisher}
     * @return the shared {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/empty-never-throw.html">ReactiveX operators documentation: Never</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SuppressWarnings("unchecked")
    @NonNull
    public static <T> Flowable<T> never() {
        return RxJavaPlugins.onAssembly((Flowable<T>) FlowableNever.INSTANCE);
    }

    /**
     * Returns a {@code Flowable} that emits a sequence of {@link Integer}s within a specified range.
     * <p>
     * <img width="640" height="195" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/range.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals values on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code range} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param start
     *            the value of the first {@code Integer} in the sequence
     * @param count
     *            the number of sequential {@code Integer}s to generate
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero, or if {@code start} + {@code count} &minus; 1 exceeds
     *             {@link Integer#MAX_VALUE}
     * @see <a href="http://reactivex.io/documentation/operators/range.html">ReactiveX operators documentation: Range</a>
     * @see #rangeLong(long, long)
     * @see #intervalRange(long, long, long, long, TimeUnit)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static Flowable<Integer> range(int start, int count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        } else
        if (count == 0) {
            return empty();
        } else
        if (count == 1) {
            return just(start);
        } else
        if ((long)start + (count - 1) > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Integer overflow");
        }
        return RxJavaPlugins.onAssembly(new FlowableRange(start, count));
    }

    /**
     * Returns a {@code Flowable} that emits a sequence of {@link Long}s within a specified range.
     * <p>
     * <img width="640" height="195" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/range.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and signals values on-demand (i.e., when requested).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code rangeLong} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param start
     *            the value of the first {@code Long} in the sequence
     * @param count
     *            the number of sequential {@code Long}s to generate
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero, or if {@code start} + {@code count} &minus; 1 exceeds
     *             {@link Long#MAX_VALUE}
     * @see <a href="http://reactivex.io/documentation/operators/range.html">ReactiveX operators documentation: Range</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static Flowable<Long> rangeLong(long start, long count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        }

        if (count == 0) {
            return empty();
        }

        if (count == 1) {
            return just(start);
        }

        long end = start + (count - 1);
        if (start > 0 && end < 0) {
            throw new IllegalArgumentException("Overflow! start + count is bigger than Long.MAX_VALUE");
        }

        return RxJavaPlugins.onAssembly(new FlowableRangeLong(start, count));
    }

    /**
     * Returns a {@link Single} that emits a {@link Boolean} value that indicates whether two {@link Publisher} sequences are the
     * same by comparing the items emitted by each {@code Publisher} pairwise.
     * <p>
     * <img width="640" height="385" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sequenceEqual.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator honors downstream backpressure and expects both of its sources
     *  to honor backpressure as well. If violated, the operator will emit a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sequenceEqual} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param source1
     *            the first {@code Publisher} to compare
     * @param source2
     *            the second {@code Publisher} to compare
     * @param <T>
     *            the type of items emitted by each {@code Publisher}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code source1} or {@code source2} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sequenceequal.html">ReactiveX operators documentation: SequenceEqual</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Single<Boolean> sequenceEqual(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2) {
        return sequenceEqual(source1, source2, ObjectHelper.equalsPredicate(), bufferSize());
    }

    /**
     * Returns a {@link Single} that emits a {@link Boolean} value that indicates whether two {@link Publisher} sequences are the
     * same by comparing the items emitted by each {@code Publisher} pairwise based on the results of a specified
     * equality function.
     * <p>
     * <img width="640" height="385" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sequenceEqual.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator signals a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sequenceEqual} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param source1
     *            the first {@code Publisher} to compare
     * @param source2
     *            the second {@code Publisher} to compare
     * @param isEqual
     *            a function used to compare items emitted by each {@code Publisher}
     * @param <T>
     *            the type of items emitted by each {@code Publisher}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code isEqual} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sequenceequal.html">ReactiveX operators documentation: SequenceEqual</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Single<Boolean> sequenceEqual(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull BiPredicate<? super T, ? super T> isEqual) {
        return sequenceEqual(source1, source2, isEqual, bufferSize());
    }

    /**
     * Returns a {@link Single} that emits a {@link Boolean} value that indicates whether two {@link Publisher} sequences are the
     * same by comparing the items emitted by each {@code Publisher} pairwise based on the results of a specified
     * equality function.
     * <p>
     * <img width="640" height="385" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sequenceEqual.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The source {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator signals a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sequenceEqual} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param source1
     *            the first {@code Publisher} to compare
     * @param source2
     *            the second {@code Publisher} to compare
     * @param isEqual
     *            a function used to compare items emitted by each {@code Publisher}
     * @param bufferSize
     *            the number of items to prefetch from the first and second source {@code Publisher}
     * @param <T>
     *            the type of items emitted by each {@code Publisher}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code isEqual} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/sequenceequal.html">ReactiveX operators documentation: SequenceEqual</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Single<Boolean> sequenceEqual(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2,
            @NonNull BiPredicate<? super T, ? super T> isEqual, int bufferSize) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(isEqual, "isEqual is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableSequenceEqualSingle<>(source1, source2, isEqual, bufferSize));
    }

    /**
     * Returns a {@link Single} that emits a {@link Boolean} value that indicates whether two {@link Publisher} sequences are the
     * same by comparing the items emitted by each {@code Publisher} pairwise.
     * <p>
     * <img width="640" height="385" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sequenceEqual.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator honors downstream backpressure and expects both of its sources
     *  to honor backpressure as well. If violated, the operator will emit a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sequenceEqual} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param source1
     *            the first {@code Publisher} to compare
     * @param source2
     *            the second {@code Publisher} to compare
     * @param bufferSize
     *            the number of items to prefetch from the first and second source {@code Publisher}
     * @param <T>
     *            the type of items emitted by each {@code Publisher}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code source1} or {@code source2} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/sequenceequal.html">ReactiveX operators documentation: SequenceEqual</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Single<Boolean> sequenceEqual(@NonNull Publisher<@NonNull ? extends T> source1, @NonNull Publisher<@NonNull ? extends T> source2, int bufferSize) {
        return sequenceEqual(source1, source2, ObjectHelper.equalsPredicate(), bufferSize);
    }

    /**
     * Converts a {@link Publisher} that emits {@code Publisher}s into a {@code Publisher} that emits the items emitted by the
     * most recently emitted of those {@code Publisher}s.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchDo.v3.png" alt="">
     * <p>
     * {@code switchOnNext} subscribes to a {@code Publisher} that emits {@code Publisher}s. Each time it observes one of
     * these emitted {@code Publisher}s, the {@code Publisher} returned by {@code switchOnNext} begins emitting the items
     * emitted by that {@code Publisher}. When a new {@code Publisher} is emitted, {@code switchOnNext} stops emitting items
     * from the earlier-emitted {@code Publisher} and begins emitting items from the new one.
     * <p>
     * The resulting {@code Flowable} completes if both the outer {@code Publisher} and the last inner {@code Publisher}, if any, complete.
     * If the outer {@code Publisher} signals an {@code onError}, the inner {@code Publisher} is canceled and the error delivered in-sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchOnNext} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the item type
     * @param sources
     *            the source {@code Publisher} that emits {@code Publisher}s
     * @param bufferSize
     *            the number of items to prefetch from the inner {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/switch.html">ReactiveX operators documentation: Switch</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> switchOnNext(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int bufferSize) {
        return fromPublisher(sources).switchMap((Function)Functions.identity(), bufferSize);
    }

    /**
     * Converts a {@link Publisher} that emits {@code Publisher}s into a {@code Publisher} that emits the items emitted by the
     * most recently emitted of those {@code Publisher}s.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchDo.v3.png" alt="">
     * <p>
     * {@code switchOnNext} subscribes to a {@code Publisher} that emits {@code Publisher}s. Each time it observes one of
     * these emitted {@code Publisher}s, the {@code Publisher} returned by {@code switchOnNext} begins emitting the items
     * emitted by that {@code Publisher}. When a new {@code Publisher} is emitted, {@code switchOnNext} stops emitting items
     * from the earlier-emitted {@code Publisher} and begins emitting items from the new one.
     * <p>
     * The resulting {@code Flowable} completes if both the outer {@code Publisher} and the last inner {@code Publisher}, if any, complete.
     * If the outer {@code Publisher} signals an {@code onError}, the inner {@code Publisher} is canceled and the error delivered in-sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchOnNext} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the item type
     * @param sources
     *            the source {@code Publisher} that emits {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/switch.html">ReactiveX operators documentation: Switch</a>
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> switchOnNext(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return fromPublisher(sources).switchMap((Function)Functions.identity());
    }

    /**
     * Converts a {@link Publisher} that emits {@code Publisher}s into a {@code Publisher} that emits the items emitted by the
     * most recently emitted of those {@code Publisher}s and delays any exception until all {@code Publisher}s terminate.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchDo.v3.png" alt="">
     * <p>
     * {@code switchOnNext} subscribes to a {@code Publisher} that emits {@code Publisher}s. Each time it observes one of
     * these emitted {@code Publisher}s, the {@code Publisher} returned by {@code switchOnNext} begins emitting the items
     * emitted by that {@code Publisher}. When a new {@code Publisher} is emitted, {@code switchOnNext} stops emitting items
     * from the earlier-emitted {@code Publisher} and begins emitting items from the new one.
     * <p>
     * The resulting {@code Flowable} completes if both the main {@code Publisher} and the last inner {@code Publisher}, if any, complete.
     * If the main {@code Publisher} signals an {@code onError}, the termination of the last inner {@code Publisher} will emit that error as is
     * or wrapped into a {@link CompositeException} along with the other possible errors the former inner {@code Publisher}s signaled.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchOnNextDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the item type
     * @param sources
     *            the source {@code Publisher} that emits {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/switch.html">ReactiveX operators documentation: Switch</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> switchOnNextDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources) {
        return switchOnNextDelayError(sources, bufferSize());
    }

    /**
     * Converts a {@link Publisher} that emits {@code Publisher}s into a {@code Publisher} that emits the items emitted by the
     * most recently emitted of those {@code Publisher}s and delays any exception until all {@code Publisher}s terminate.
     * <p>
     * <img width="640" height="370" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchDo.v3.png" alt="">
     * <p>
     * {@code switchOnNext} subscribes to a {@code Publisher} that emits {@code Publisher}s. Each time it observes one of
     * these emitted {@code Publisher}s, the {@code Publisher} returned by {@code switchOnNext} begins emitting the items
     * emitted by that {@code Publisher}. When a new {@code Publisher} is emitted, {@code switchOnNext} stops emitting items
     * from the earlier-emitted {@code Publisher} and begins emitting items from the new one.
     * <p>
     * The resulting {@code Flowable} completes if both the main {@code Publisher} and the last inner {@code Publisher}, if any, complete.
     * If the main {@code Publisher} signals an {@code onError}, the termination of the last inner {@code Publisher} will emit that error as is
     * or wrapped into a {@link CompositeException} along with the other possible errors the former inner {@code Publisher}s signaled.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchOnNextDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the item type
     * @param sources
     *            the source {@code Publisher} that emits {@code Publisher}s
     * @param prefetch
     *            the number of items to prefetch from the inner {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/switch.html">ReactiveX operators documentation: Switch</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T> Flowable<T> switchOnNextDelayError(@NonNull Publisher<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, int prefetch) {
        return fromPublisher(sources).switchMapDelayError(Functions.<Publisher<@NonNull ? extends T>>identity(), prefetch);
    }

    /**
     * Returns a {@code Flowable} that emits {@code 0L} after a specified delay, and then completes.
     * <p>
     * <img width="640" height="200" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. If the downstream needs a slower rate
     *      it should slow the timer or use something like {@link #onBackpressureDrop}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code timer} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param delay
     *            the initial delay before emitting a single {@code 0L}
     * @param unit
     *            time units to use for {@code delay}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timer.html">ReactiveX operators documentation: Timer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public static Flowable<Long> timer(long delay, @NonNull TimeUnit unit) {
        return timer(delay, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits {@code 0L} after a specified delay, on a specified {@link Scheduler}, and then
     * completes.
     * <p>
     * <img width="640" height="200" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timer.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. If the downstream needs a slower rate
     *      it should slow the timer or use something like {@link #onBackpressureDrop}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param delay
     *            the initial delay before emitting a single 0L
     * @param unit
     *            time units to use for {@code delay}
     * @param scheduler
     *            the {@code Scheduler} to use for scheduling the item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timer.html">ReactiveX operators documentation: Timer</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public static Flowable<Long> timer(long delay, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");

        return RxJavaPlugins.onAssembly(new FlowableTimer(Math.max(0L, delay), unit, scheduler));
    }

    /**
     * Create a {@code Flowable} by wrapping a {@link Publisher} <em>which has to be implemented according
     * to the <b>Reactive Streams</b> specification by handling backpressure and
     * cancellation correctly; no safeguards are provided by the {@code Flowable} itself</em>.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator is a pass-through for backpressure and the behavior is determined by the
     *  provided {@code Publisher} implementation.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code unsafeCreate} by default doesn't operate on any particular {@link Scheduler}.</dd>
     * </dl>
     * @param <T> the value type emitted
     * @param onSubscribe the {@code Publisher} instance to wrap
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onSubscribe} is {@code null}
     * @throws IllegalArgumentException if {@code onSubscribe} is a subclass of {@code Flowable}; such
     * instances don't need conversion and is possibly a port remnant from 1.x or one should use {@link #hide()}
     * instead.
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.NONE)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T> Flowable<T> unsafeCreate(@NonNull Publisher<T> onSubscribe) {
        Objects.requireNonNull(onSubscribe, "onSubscribe is null");
        if (onSubscribe instanceof Flowable) {
            throw new IllegalArgumentException("unsafeCreate(Flowable) should be upgraded");
        }
        return RxJavaPlugins.onAssembly(new FlowableFromPublisher<>(onSubscribe));
    }

    /**
     * Constructs a {@code Flowable} that creates a dependent resource object, a {@link Publisher} with
     * that resource and calls the provided {@code resourceDisposer} function if this inner source terminates or the
     * downstream cancels the flow.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/using.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator is a pass-through for backpressure and otherwise depends on the
     *  backpressure support of the {@code Publisher} returned by the {@code resourceFactory}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code using} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the element type of the generated {@code Publisher}
     * @param <D> the type of the resource associated with the output sequence
     * @param resourceSupplier
     *            the factory function to create a resource object that depends on the {@code Publisher}
     * @param sourceSupplier
     *            the factory function to create a {@code Publisher}
     * @param resourceCleanup
     *            the function that will dispose of the resource
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code resourceSupplier}, {@code sourceSupplier} or {@code resourceCleanup} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/using.html">ReactiveX operators documentation: Using</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public static <T, D> Flowable<T> using(
            @NonNull Supplier<? extends D> resourceSupplier,
            @NonNull Function<? super D, ? extends Publisher<@NonNull ? extends T>> sourceSupplier,
            @NonNull Consumer<? super D> resourceCleanup) {
        return using(resourceSupplier, sourceSupplier, resourceCleanup, true);
    }

    /**
     * Constructs a {@code Flowable} that creates a dependent resource object, a {@link Publisher} with
     * that resource and calls the provided {@code resourceDisposer} function if this inner source terminates or the
     * downstream disposes the flow; doing it before these end-states have been reached if {@code eager == true}, after otherwise.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/using.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator is a pass-through for backpressure and otherwise depends on the
     *  backpressure support of the {@code Publisher} returned by the {@code resourceFactory}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code using} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the element type of the generated {@code Publisher}
     * @param <D> the type of the resource associated with the output sequence
     * @param resourceSupplier
     *            the factory function to create a resource object that depends on the {@code Publisher}
     * @param sourceSupplier
     *            the factory function to create a {@code Publisher}
     * @param resourceCleanup
     *            the function that will dispose of the resource
     * @param eager
     *            If {@code true}, the resource disposal will happen either on a {@code cancel()} call before the upstream is disposed
     *            or just before the emission of a terminal event ({@code onComplete} or {@code onError}).
     *            If {@code false} the resource disposal will happen either on a {@code cancel()} call after the upstream is disposed
     *            or just after the emission of a terminal event ({@code onComplete} or {@code onError}).
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code resourceSupplier}, {@code sourceSupplier} or {@code resourceCleanup} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/using.html">ReactiveX operators documentation: Using</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, D> Flowable<T> using(
            @NonNull Supplier<? extends D> resourceSupplier,
            @NonNull Function<? super D, ? extends Publisher<@NonNull ? extends T>> sourceSupplier,
            @NonNull Consumer<? super D> resourceCleanup,
            boolean eager) {
        Objects.requireNonNull(resourceSupplier, "resourceSupplier is null");
        Objects.requireNonNull(sourceSupplier, "sourceSupplier is null");
        Objects.requireNonNull(resourceCleanup, "resourceCleanup is null");
        return RxJavaPlugins.onAssembly(new FlowableUsing<T, D>(resourceSupplier, sourceSupplier, resourceCleanup, eager));
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * items emitted, in sequence, by an {@link Iterable} of other {@link Publisher}s.
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each of the source {@code Publisher}s;
     * the second item emitted by the new {@code Publisher} will be the result of the function applied to the second
     * item emitted by each of those {@code Publisher}s; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@code onNext} as many times as
     * the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(Arrays.asList(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2)), (a) -&gt; a)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common value type
     * @param <R> the zipped result type
     * @param sources
     *            an {@code Iterable} of source {@code Publisher}s
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, R> Flowable<R> zip(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources, @NonNull Function<? super Object[], ? extends R> zipper) {
        Objects.requireNonNull(zipper, "zipper is null");
        Objects.requireNonNull(sources, "sources is null");
        return RxJavaPlugins.onAssembly(new FlowableZip<>(null, sources, zipper, bufferSize(), false));
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * items emitted, in sequence, by an {@link Iterable} of other {@link Publisher}s.
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each of the source {@code Publisher}s;
     * the second item emitted by the new {@code Publisher} will be the result of the function applied to the second
     * item emitted by each of those {@code Publisher}s; and so forth.
     * <p>
     * The resulting {@code Floawble} returned from {@code zip} will invoke {@code onNext} as many times as
     * the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(Arrays.asList(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2)), (a) -&gt; a)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     *
     * @param sources
     *            an {@code Iterable} of source {@code Publisher}s
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @param delayError
     *            delay errors signaled by any of the source {@code Publisher} until all {@code Publisher}s terminate
     * @param bufferSize
     *            the number of elements to prefetch from each source {@code Publisher}
     * @param <T> the common source value type
     * @param <R> the zipped result type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code zipper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T, R> Flowable<R> zip(@NonNull Iterable<@NonNull ? extends Publisher<@NonNull ? extends T>> sources,
            @NonNull Function<? super Object[], ? extends R> zipper, boolean delayError,
            int bufferSize) {
        Objects.requireNonNull(zipper, "zipper is null");
        Objects.requireNonNull(sources, "sources is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableZip<>(null, sources, zipper, bufferSize, delayError));
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * two items emitted, in sequence, by two other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1} and the first item
     * emitted by {@code o2}; the second item emitted by the new {@code Publisher} will be the result of the function
     * applied to the second item emitted by {@code o1} and the second item emitted by {@code o2}; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), (a, b) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results
     *            in an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull BiFunction<? super T1, ? super T2, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * two items emitted, in sequence, by two other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1} and the first item
     * emitted by {@code o2}; the second item emitted by the new {@code Publisher} will be the result of the function
     * applied to the second item emitted by {@code o1} and the second item emitted by {@code o2}; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), (a, b) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results
     *            in an item that will be emitted by the resulting {@code Flowable}
     * @param delayError delay errors from any of the source {@code Publisher}s till the other terminates
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull BiFunction<? super T1, ? super T2, ? extends R> zipper, boolean delayError) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), delayError, bufferSize(), source1, source2);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * two items emitted, in sequence, by two other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1} and the first item
     * emitted by {@code o2}; the second item emitted by the new {@code Publisher} will be the result of the function
     * applied to the second item emitted by {@code o1} and the second item emitted by {@code o2}; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), (a, b) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results
     *            in an item that will be emitted by the resulting {@code Flowable}
     * @param delayError delay errors from any of the source {@code Publisher}s till the other terminates
     * @param bufferSize the number of elements to prefetch from each source {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2} or {@code zipper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2,
            @NonNull BiFunction<? super T1, ? super T2, ? extends R> zipper, boolean delayError, int bufferSize) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), delayError, bufferSize, source1, source2);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * three items emitted, in sequence, by three other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1}, the first item
     * emitted by {@code o2}, and the first item emitted by {@code o3}; the second item emitted by the new
     * {@code Publisher} will be the result of the function applied to the second item emitted by {@code o1}, the
     * second item emitted by {@code o2}, and the second item emitted by {@code o3}; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Function3<? super T1, ? super T2, ? super T3, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * four items emitted, in sequence, by four other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1}, the first item
     * emitted by {@code o2}, the first item emitted by {@code o3}, and the first item emitted by {@code 04};
     * the second item emitted by the new {@code Publisher} will be the result of the function applied to the second
     * item emitted by each of those {@code Publisher}s; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4,
            @NonNull Function4<? super T1, ? super T2, ? super T3, ? super T4, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * five items emitted, in sequence, by five other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by {@code o1}, the first item
     * emitted by {@code o2}, the first item emitted by {@code o3}, the first item emitted by {@code o4}, and
     * the first item emitted by {@code o5}; the second item emitted by the new {@code Publisher} will be the result of
     * the function applied to the second item emitted by each of those {@code Publisher}s; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d, e) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <T5> the value type of the fifth source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param source5
     *            a fifth source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4, @NonNull Publisher<@NonNull ? extends T5> source5,
            @NonNull Function5<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4, source5);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * six items emitted, in sequence, by six other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each source {@code Publisher}, the
     * second item emitted by the new {@code Publisher} will be the result of the function applied to the second item
     * emitted by each of those {@code Publisher}s, and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d, e, f) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <T5> the value type of the fifth source
     * @param <T6> the value type of the sixth source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param source5
     *            a fifth source {@code Publisher}
     * @param source6
     *            a sixth source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6}
     *                              or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4, @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Function6<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4, source5, source6);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * seven items emitted, in sequence, by seven other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each source {@code Publisher}, the
     * second item emitted by the new {@code Publisher} will be the result of the function applied to the second item
     * emitted by each of those {@code Publisher}s, and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d, e, f, g) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <T5> the value type of the fifth source
     * @param <T6> the value type of the sixth source
     * @param <T7> the value type of the seventh source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param source5
     *            a fifth source {@code Publisher}
     * @param source6
     *            a sixth source {@code Publisher}
     * @param source7
     *            a seventh source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4, @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7,
            @NonNull Function7<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4, source5, source6, source7);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * eight items emitted, in sequence, by eight other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each source {@code Publisher}, the
     * second item emitted by the new {@code Publisher} will be the result of the function applied to the second item
     * emitted by each of those {@code Publisher}s, and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d, e, f, g, h) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <T5> the value type of the fifth source
     * @param <T6> the value type of the sixth source
     * @param <T7> the value type of the seventh source
     * @param <T8> the value type of the eighth source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param source5
     *            a fifth source {@code Publisher}
     * @param source6
     *            a sixth source {@code Publisher}
     * @param source7
     *            a seventh source {@code Publisher}
     * @param source8
     *            an eighth source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7}, {@code source8} or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, T8, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4, @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7, @NonNull Publisher<@NonNull ? extends T8> source8,
            @NonNull Function8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? extends R> zipper) {
        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(source8, "source8 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4, source5, source6, source7, source8);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * nine items emitted, in sequence, by nine other {@link Publisher}s.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each source {@code Publisher}, the
     * second item emitted by the new {@code Publisher} will be the result of the function applied to the second item
     * emitted by each of those {@code Publisher}s, and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@link Subscriber#onNext onNext}
     * as many times as the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest
     * items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2), ..., (a, b, c, d, e, f, g, h, i) -&gt; a + b)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T1> the value type of the first source
     * @param <T2> the value type of the second source
     * @param <T3> the value type of the third source
     * @param <T4> the value type of the fourth source
     * @param <T5> the value type of the fifth source
     * @param <T6> the value type of the sixth source
     * @param <T7> the value type of the seventh source
     * @param <T8> the value type of the eighth source
     * @param <T9> the value type of the ninth source
     * @param <R> the zipped result type
     * @param source1
     *            the first source {@code Publisher}
     * @param source2
     *            a second source {@code Publisher}
     * @param source3
     *            a third source {@code Publisher}
     * @param source4
     *            a fourth source {@code Publisher}
     * @param source5
     *            a fifth source {@code Publisher}
     * @param source6
     *            a sixth source {@code Publisher}
     * @param source7
     *            a seventh source {@code Publisher}
     * @param source8
     *            an eighth source {@code Publisher}
     * @param source9
     *            a ninth source {@code Publisher}
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code source1}, {@code source2}, {@code source3},
     *                              {@code source4}, {@code source5}, {@code source6},
     *                              {@code source7}, {@code source8}, {@code source9}
     *                              or {@code zipper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public static <T1, T2, T3, T4, T5, T6, T7, T8, T9, R> Flowable<R> zip(
            @NonNull Publisher<@NonNull ? extends T1> source1, @NonNull Publisher<@NonNull ? extends T2> source2, @NonNull Publisher<@NonNull ? extends T3> source3,
            @NonNull Publisher<@NonNull ? extends T4> source4, @NonNull Publisher<@NonNull ? extends T5> source5, @NonNull Publisher<@NonNull ? extends T6> source6,
            @NonNull Publisher<@NonNull ? extends T7> source7, @NonNull Publisher<@NonNull ? extends T8> source8, @NonNull Publisher<@NonNull ? extends T9> source9,
            @NonNull Function9<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? super T9, ? extends R> zipper) {

        Objects.requireNonNull(source1, "source1 is null");
        Objects.requireNonNull(source2, "source2 is null");
        Objects.requireNonNull(source3, "source3 is null");
        Objects.requireNonNull(source4, "source4 is null");
        Objects.requireNonNull(source5, "source5 is null");
        Objects.requireNonNull(source6, "source6 is null");
        Objects.requireNonNull(source7, "source7 is null");
        Objects.requireNonNull(source8, "source8 is null");
        Objects.requireNonNull(source9, "source9 is null");
        Objects.requireNonNull(zipper, "zipper is null");
        return zipArray(Functions.toFunction(zipper), false, bufferSize(), source1, source2, source3, source4, source5, source6, source7, source8, source9);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified combiner function applied to combinations of
     * items emitted, in sequence, by an array of other {@link Publisher}s.
     * <p>
     * {@code zip} applies this function in strict sequence, so the first item emitted by the new {@code Publisher}
     * will be the result of the function applied to the first item emitted by each of the source {@code Publisher}s;
     * the second item emitted by the new {@code Publisher} will be the result of the function applied to the second
     * item emitted by each of those {@code Publisher}s; and so forth.
     * <p>
     * The resulting {@code Flowable} returned from {@code zip} will invoke {@code onNext} as many times as
     * the number of {@code onNext} invocations of the source {@code Publisher} that emits the fewest items.
     * <p>
     * The operator subscribes to its sources in the order they are specified and completes eagerly if
     * one of the sources is shorter than the rest while canceling the other sources. Therefore, it
     * is possible those other sources will never be able to run to completion (and thus not calling
     * {@code doOnComplete()}). This can also happen if the sources are exactly the same length; if
     * source A completes and B has been consumed and is about to complete, the operator detects A won't
     * be sending further values and it will cancel B immediately. For example:
     * <pre><code>zip(new Publisher[]{range(1, 5).doOnComplete(action1), range(6, 5).doOnComplete(action2)}, (a) -&gt;
     * a)</code></pre>
     * {@code action1} will be called but {@code action2} won't.
     * <br>To work around this termination property,
     * use {@link #doOnCancel(Action)} as well or use {@code using()} to do cleanup in case of completion
     * or cancellation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/zip.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from the sources and honors backpressure from the downstream.
     *  (I.e., zipping with {@link #interval(long, TimeUnit)} may result in {@link MissingBackpressureException}, use
     *  one of the {@code onBackpressureX} to handle similar, backpressure-ignoring sources.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code zipArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <T> the common element type
     * @param <R> the result type
     * @param sources
     *            an array of source {@code Publisher}s
     * @param zipper
     *            a function that, when applied to an item emitted by each of the source {@code Publisher}s, results in
     *            an item that will be emitted by the resulting {@code Flowable}
     * @param delayError
     *            delay errors signaled by any of the source {@code Publisher} until all {@code Publisher}s terminate
     * @param bufferSize
     *            the number of elements to prefetch from each source {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sources} or {@code zipper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/zip.html">ReactiveX operators documentation: Zip</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    public static <T, R> Flowable<R> zipArray(@NonNull Function<? super Object[], ? extends R> zipper,
            boolean delayError, int bufferSize, @NonNull Publisher<@NonNull ? extends T>... sources) {
        Objects.requireNonNull(sources, "sources is null");
        if (sources.length == 0) {
            return empty();
        }
        Objects.requireNonNull(zipper, "zipper is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableZip<>(sources, null, zipper, bufferSize, delayError));
    }

    // ***************************************************************************************************
    // Instance operators
    // ***************************************************************************************************

    /**
     * Returns a {@link Single} that emits a {@link Boolean} that indicates whether all of the items emitted by the current
     * {@code Flowable} satisfy a condition.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/all.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code all} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            a function that evaluates an item and returns a {@code Boolean}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/all.html">ReactiveX operators documentation: All</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<Boolean> all(@NonNull Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");
        return RxJavaPlugins.onAssembly(new FlowableAllSingle<>(this, predicate));
    }

    /**
     * Mirrors the {@link Publisher} (current or provided) that first either emits an item or sends a termination
     * notification.
     * <p>
     * <img width="640" height="376" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.ambWith.png" alt="">
     * <p>
     * When the current {@code Flowable} signals an item or terminates first, the subscription to the other
     * {@code Publisher} is canceled. If the other {@code Publisher} signals an item or terminates first,
     * the subscription to the current {@code Flowable} is canceled.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator itself doesn't interfere with backpressure which is determined by the winning
     *  {@code Publisher}'s backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code ambWith} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>
     *     If the losing {@code Publisher} signals an error, the error is routed to the global
     *     error handler via {@link RxJavaPlugins#onError(Throwable)}.
     *  </dd>
     * </dl>
     *
     * @param other
     *            a {@code Publisher} competing to react first. A subscription to this provided {@code Publisher} will occur after subscribing
     *            to the current {@code Flowable}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/amb.html">ReactiveX operators documentation: Amb</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> ambWith(@NonNull Publisher<@NonNull ? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return ambArray(this, other);
    }

    /**
     * Returns a {@link Single} that emits {@code true} if any item emitted by the current {@code Flowable} satisfies a
     * specified condition, otherwise {@code false}. <em>Note:</em> this always emits {@code false} if the
     * current {@code Flowable} is empty.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/exists.v3.png" alt="">
     * <p>
     * In Rx.Net this is the {@code any} operator but we renamed it in RxJava to better match Java naming
     * idioms.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code any} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            the condition to test items emitted by the current {@code Flowable}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/contains.html">ReactiveX operators documentation: Contains</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<Boolean> any(@NonNull Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");
        return RxJavaPlugins.onAssembly(new FlowableAnySingle<>(this, predicate));
    }

    /**
     * Returns the first item emitted by this {@code Flowable}, or throws
     * {@link NoSuchElementException} if it emits no items.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingFirst} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @throws NoSuchElementException
     *             if this {@code Flowable} emits no items
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingFirst() {
        BlockingFirstSubscriber<T> s = new BlockingFirstSubscriber<>();
        subscribe(s);
        T v = s.blockingGet();
        if (v != null) {
            return v;
        }
        throw new NoSuchElementException();
    }

    /**
     * Returns the first item emitted by this {@code Flowable}, or a default value if it emits no
     * items.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingFirst} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @param defaultItem
     *            a default value to return if this {@code Flowable} emits no items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingFirst(@NonNull T defaultItem) {
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        BlockingFirstSubscriber<T> s = new BlockingFirstSubscriber<>();
        subscribe(s);
        T v = s.blockingGet();
        return v != null ? v : defaultItem;
    }

    /**
     * Consumes the current {@code Flowable} in a blocking fashion and invokes the given
     * {@link Consumer} with each upstream item on the <em>current thread</em> until the
     * upstream terminates.
     * <p>
     * <img width="640" height="330" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.forEach.v3.png" alt="">
     * <p>
     * <em>Note:</em> the method will only return if the upstream terminates or the current
     * thread is interrupted.
     * <p>
     * This method executes the {@code Consumer} on the current thread while
     * {@link #subscribe(Consumer)} executes the consumer on the original caller thread of the
     * sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator requests {@link Flowable#bufferSize()} upfront, then 75% of this
     *  amount when 75% is received.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingForEach} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @param onNext
     *            the {@code Consumer} to invoke for each item emitted by the {@code Flowable}
     * @throws NullPointerException if {@code onNext} is {@code null}
     * @throws RuntimeException
     *             if an error occurs; {@code Error}s and {@code RuntimeException}s are rethrown
     *             as they are, checked {@code Exception}s are wrapped into {@code RuntimeException}s
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX documentation: Subscribe</a>
     * @see #subscribe(Consumer)
     * @see #blockingForEach(Consumer, int)
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingForEach(@NonNull Consumer<? super T> onNext) {
        blockingForEach(onNext, bufferSize());
    }

    /**
     * Consumes the current {@code Flowable} in a blocking fashion and invokes the given
     * {@link Consumer} with each upstream item on the <em>current thread</em> until the
     * upstream terminates.
     * <p>
     * <img width="640" height="330" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.forEach.v3.png" alt="">
     * <p>
     * <em>Note:</em> the method will only return if the upstream terminates or the current
     * thread is interrupted.
     * <p>
     * This method executes the {@code Consumer} on the current thread while
     * {@link #subscribe(Consumer)} executes the consumer on the original caller thread of the
     * sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator requests the given {@code prefetch} amount upfront, then 75% of this
     *  amount when 75% is received.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingForEach} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @param onNext
     *            the {@code Consumer} to invoke for each item emitted by the {@code Flowable}
     * @param bufferSize
     *            the number of items to prefetch upfront, then 75% of it after 75% received
     * @throws NullPointerException if {@code onNext} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @throws RuntimeException
     *             if an error occurs; {@code Error}s and {@code RuntimeException}s are rethrown
     *             as they are, checked {@code Exception}s are wrapped into {@code RuntimeException}s
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX documentation: Subscribe</a>
     * @see #subscribe(Consumer)
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingForEach(@NonNull Consumer<? super T> onNext, int bufferSize) {
        Objects.requireNonNull(onNext, "onNext is null");
        Iterator<T> it = blockingIterable(bufferSize).iterator();
        while (it.hasNext()) {
            try {
                onNext.accept(it.next());
            } catch (Throwable e) {
                Exceptions.throwIfFatal(e);
                ((Disposable)it).dispose();
                throw ExceptionHelper.wrapOrThrow(e);
            }
        }
    }

    /**
     * Converts this {@code Flowable} into an {@link Iterable}.
     * <p>
     * <img width="640" height="315" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.toIterable.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to honor backpressure otherwise the returned
     *  {@code Iterable}'s iterator will throw a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Iterable} instance
     * @see <a href="http://reactivex.io/documentation/operators/to.html">ReactiveX documentation: To</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Iterable<T> blockingIterable() {
        return blockingIterable(bufferSize());
    }

    /**
     * Converts this {@code Flowable} into an {@link Iterable}.
     * <p>
     * <img width="640" height="315" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.toIterable.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to honor backpressure otherwise the returned
     *  {@code Iterable}'s iterator will throw a {@link MissingBackpressureException}.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param bufferSize the number of items to prefetch from the current {@code Flowable}
     * @return the new {@code Iterable} instance
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/to.html">ReactiveX documentation: To</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Iterable<T> blockingIterable(int bufferSize) {
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return new BlockingFlowableIterable<>(this, bufferSize);
    }

    /**
     * Returns the last item emitted by this {@code Flowable}, or throws
     * {@link NoSuchElementException} if this {@code Flowable} emits no items.
     * <p>
     * <img width="640" height="315" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.last.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingLast} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @throws NoSuchElementException
     *             if this {@code Flowable} emits no items
     * @see <a href="http://reactivex.io/documentation/operators/last.html">ReactiveX documentation: Last</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingLast() {
        BlockingLastSubscriber<T> s = new BlockingLastSubscriber<>();
        subscribe(s);
        T v = s.blockingGet();
        if (v != null) {
            return v;
        }
        throw new NoSuchElementException();
    }

    /**
     * Returns the last item emitted by this {@code Flowable}, or a default value if it emits no
     * items.
     * <p>
     * <img width="640" height="310" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.lastOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingLast} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @param defaultItem
     *            a default value to return if this {@code Flowable} emits no items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/last.html">ReactiveX documentation: Last</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingLast(@NonNull T defaultItem) {
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        BlockingLastSubscriber<T> s = new BlockingLastSubscriber<>();
        subscribe(s);
        T v = s.blockingGet();
        return v != null ? v : defaultItem;
    }

    /**
     * Returns an {@link Iterable} that returns the latest item emitted by this {@code Flowable},
     * waiting if necessary for one to become available.
     * <p>
     * If this {@code Flowable} produces items faster than {@code Iterator.next} takes them,
     * {@code onNext} events might be skipped, but {@code onError} or {@code onComplete} events are not.
     * <p>
     * Note also that an {@code onNext} directly followed by {@code onComplete} might hide the {@code onNext}
     * event.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Iterable} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Iterable<T> blockingLatest() {
        return new BlockingFlowableLatest<>(this);
    }

    /**
     * Returns an {@link Iterable} that always returns the item most recently emitted by this
     * {@code Flowable}.
     * <p>
     * <img width="640" height="490" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.mostRecent.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingMostRecent} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param initialItem
     *            the initial item that the {@code Iterable} sequence will yield if this
     *            {@code Flowable} has not yet emitted an item
     * @return the new {@code Iterable} instance
     * @throws NullPointerException if {@code initialItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Iterable<T> blockingMostRecent(@NonNull T initialItem) {
        Objects.requireNonNull(initialItem, "initialItem is null");
        return new BlockingFlowableMostRecent<>(this, initialItem);
    }

    /**
     * Returns an {@link Iterable} that blocks until this {@code Flowable} emits another item, then
     * returns that item.
     * <p>
     * <img width="640" height="490" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.next.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingNext} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Iterable} instance
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Iterable<T> blockingNext() {
        return new BlockingFlowableNext<>(this);
    }

    /**
     * If this {@code Flowable} completes after emitting a single item, return that item, otherwise
     * throw a {@link NoSuchElementException}.
     * <p>
     * <img width="640" height="315" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.single.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingSingle() {
        return singleOrError().blockingGet();
    }

    /**
     * If this {@code Flowable} completes after emitting a single item, return that item; if it emits
     * more than one item, throw an {@link IllegalArgumentException}; if it emits no items, return a default
     * value.
     * <p>
     * <img width="640" height="315" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/B.singleOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the source signals an error, the operator wraps a checked {@link Exception}
     *  into {@link RuntimeException} and throws that. Otherwise, {@code RuntimeException}s and
     *  {@link Error}s are rethrown as they are.</dd>
     * </dl>
     *
     * @param defaultItem
     *            a default value to return if this {@code Flowable} emits no items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final T blockingSingle(@NonNull T defaultItem) {
        return single(defaultItem).blockingGet();
    }

    /**
     * Returns a {@link Future} representing the only value emitted by this {@code Flowable}.
     * <p>
     * <img width="640" height="311" src="https://github.com/ReactiveX/RxJava/wiki/images/rx-operators/Flowable.toFuture.png" alt="">
     * <p>
     * If the {@code Flowable} emits more than one item, {@link java.util.concurrent.Future} will receive an
     * {@link java.lang.IndexOutOfBoundsException}. If the {@code Flowable} is empty, {@link java.util.concurrent.Future}
     * will receive a {@link java.util.NoSuchElementException}. The {@code Flowable} source has to terminate in order
     * for the returned {@code Future} to terminate as well.
     * <p>
     * If the {@code Flowable} may emit more than one item, use {@code Flowable.toList().toFuture()}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code toFuture} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Future} instance
     * @see <a href="http://reactivex.io/documentation/operators/to.html">ReactiveX documentation: To</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Future<T> toFuture() {
        return subscribeWith(new FutureSubscriber<>());
    }

    /**
     * Runs the current {@code Flowable} to a terminal event, ignoring any values and rethrowing any exception.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @since 2.0
     * @see #blockingSubscribe(Consumer)
     * @see #blockingSubscribe(Consumer, Consumer)
     * @see #blockingSubscribe(Consumer, Consumer, Action)
     */
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe() {
        FlowableBlockingSubscribe.subscribe(this);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * If the {@code Flowable} emits an error, it is wrapped into an
     * {@link io.reactivex.rxjava3.exceptions.OnErrorNotImplementedException OnErrorNotImplementedException}
     * and routed to the {@link RxJavaPlugins#onError(Throwable)} handler.
     * Using the overloads {@link #blockingSubscribe(Consumer, Consumer)}
     * or {@link #blockingSubscribe(Consumer, Consumer, Action)} instead is recommended.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param onNext the callback action for each source value
     * @throws NullPointerException if {@code onNext} is {@code null}
     * @since 2.0
     * @see #blockingSubscribe(Consumer, Consumer)
     * @see #blockingSubscribe(Consumer, Consumer, Action)
     */
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext) {
        FlowableBlockingSubscribe.subscribe(this, onNext, Functions.ON_ERROR_MISSING, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * If the {@code Flowable} emits an error, it is wrapped into an
     * {@link io.reactivex.rxjava3.exceptions.OnErrorNotImplementedException OnErrorNotImplementedException}
     * and routed to the {@link RxJavaPlugins#onError(Throwable)} handler.
     * Using the overloads {@link #blockingSubscribe(Consumer, Consumer)}
     * or {@link #blockingSubscribe(Consumer, Consumer, Action)} instead is recommended.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an bounded manner (up to bufferSize
     *  outstanding request amount for items).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.15 - experimental
     * @param onNext the callback action for each source value
     * @param bufferSize the size of the buffer
     * @throws NullPointerException if {@code onNext} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see #blockingSubscribe(Consumer, Consumer)
     * @see #blockingSubscribe(Consumer, Consumer, Action)
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext, int bufferSize) {
        FlowableBlockingSubscribe.subscribe(this, onNext, Functions.ON_ERROR_MISSING, Functions.EMPTY_ACTION, bufferSize);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param onNext the callback action for each source value
     * @param onError the callback action for an error event
     * @throws NullPointerException if {@code onNext} or {@code onError} is {@code null}
     * @since 2.0
     * @see #blockingSubscribe(Consumer, Consumer, Action)
     */
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError) {
        FlowableBlockingSubscribe.subscribe(this, onNext, onError, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an bounded manner (up to bufferSize
     *  outstanding request amount for items).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.15 - experimental
     * @param onNext the callback action for each source value
     * @param onError the callback action for an error event
     * @param bufferSize the size of the buffer
     * @throws NullPointerException if {@code onNext} or {@code onError} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @since 2.2
     * @see #blockingSubscribe(Consumer, Consumer, Action)
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError,
        int bufferSize) {
        FlowableBlockingSubscribe.subscribe(this, onNext, onError, Functions.EMPTY_ACTION, bufferSize);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner
     *  (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param onNext the callback action for each source value
     * @param onError the callback action for an error event
     * @param onComplete the callback action for the completion event.
     * @throws NullPointerException if {@code onNext}, {@code onError} or {@code onComplete} is {@code null}
     * @since 2.0
     */
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError, @NonNull Action onComplete) {
        FlowableBlockingSubscribe.subscribe(this, onNext, onError, onComplete);
    }

    /**
     * Subscribes to the source and calls the given callbacks <strong>on the current thread</strong>.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally or with an error. Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an bounded manner (up to bufferSize
     *  outstanding request amount for items).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.15 - experimental
     * @param onNext the callback action for each source value
     * @param onError the callback action for an error event
     * @param onComplete the callback action for the completion event.
     * @param bufferSize the size of the buffer
     * @throws NullPointerException if {@code onNext}, {@code onError} or {@code onComplete} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError, @NonNull Action onComplete,
            int bufferSize) {
        FlowableBlockingSubscribe.subscribe(this, onNext, onError, onComplete, bufferSize);
    }

    /**
     * Subscribes to the source and calls the {@link Subscriber} methods <strong>on the current thread</strong>.
     * <p>
     * Note that calling this method will block the caller thread until the upstream terminates
     * normally, with an error or the {@code Subscriber} cancels the {@link Subscription} it receives via
     * {@link Subscriber#onSubscribe(Subscription)}.
     * Therefore, calling this method from special threads such as the
     * Android Main Thread or the Swing Event Dispatch Thread is not recommended.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The supplied {@code Subscriber} determines how backpressure is applied.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code blockingSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * The cancellation and backpressure is composed through.
     * @param subscriber the subscriber to forward events and calls to in the current thread
     * @throws NullPointerException if {@code subscriber} is {@code null}
     * @since 2.0
     */
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void blockingSubscribe(@NonNull Subscriber<@NonNull ? super T> subscriber) {
        Objects.requireNonNull(subscriber, "subscriber is null");
        FlowableBlockingSubscribe.subscribe(this, subscriber);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each containing {@code count} items. When the current
     * {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the notification from the
     * current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event is passed on
     * immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer3.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and expects the current {@code Flowable} to honor it as
     *  well, although not enforced; violation <em>may</em> lead to {@link MissingBackpressureException} somewhere
     *  downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items in each buffer before it should be emitted
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code count} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<List<T>> buffer(int count) {
        return buffer(count, count);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits buffers every {@code skip} items, each containing {@code count} items. When the current
     * {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the notification from the
     * current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event is passed on
     * immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer4.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and expects the current {@code Flowable} to honor it as
     *  well, although not enforced; violation <em>may</em> lead to {@link MissingBackpressureException} somewhere
     *  downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the maximum size of each buffer before it should be emitted
     * @param skip
     *            how many items emitted by the current {@code Flowable} should be skipped before starting a new
     *            buffer. Note that when {@code skip} and {@code count} are equal, this is the same operation as
     *            {@link #buffer(int)}.
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code count} or {@code skip} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<List<T>> buffer(int count, int skip) {
        return buffer(count, skip, ArrayListSupplier.asSupplier());
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits buffers every {@code skip} items, each containing {@code count} items. When the current
     * {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the notification from the
     * current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event is passed on
     * immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer4.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and expects the current {@code Flowable} to honor it as
     *  well, although not enforced; violation <em>may</em> lead to {@link MissingBackpressureException} somewhere
     *  downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param count
     *            the maximum size of each buffer before it should be emitted
     * @param skip
     *            how many items emitted by the current {@code Flowable} should be skipped before starting a new
     *            buffer. Note that when {@code skip} and {@code count} are equal, this is the same operation as
     *            {@link #buffer(int)}.
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code bufferSupplier} is {@code null}
     * @throws IllegalArgumentException if {@code count} or {@code skip} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U extends Collection<? super T>> Flowable<U> buffer(int count, int skip, @NonNull Supplier<U> bufferSupplier) {
        ObjectHelper.verifyPositive(count, "count");
        ObjectHelper.verifyPositive(skip, "skip");
        Objects.requireNonNull(bufferSupplier, "bufferSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableBuffer<>(this, count, skip, bufferSupplier));
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each containing {@code count} items. When the current
     * {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the notification from the
     * current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event is passed on
     * immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer3.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and expects the current {@code Flowable} to honor it as
     *  well, although not enforced; violation <em>may</em> lead to {@link MissingBackpressureException} somewhere
     *  downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param count
     *            the maximum number of items in each buffer before it should be emitted
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code bufferSupplier} is {@code null}
     * @throws IllegalArgumentException if {@code count} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U extends Collection<? super T>> Flowable<U> buffer(int count, @NonNull Supplier<U> bufferSupplier) {
        return buffer(count, count, bufferSupplier);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} starts a new buffer periodically, as determined by the {@code timeskip} argument. It emits
     * each buffer after a fixed timespan, specified by the {@code timespan} argument. When the current
     * {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the notification from the
     * current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event is passed on
     * immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer7.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted
     * @param timeskip
     *            the period of time after which a new buffer will be created
     * @param unit
     *            the unit of time that applies to the {@code timespan} and {@code timeskip} arguments
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, long timeskip, @NonNull TimeUnit unit) {
        return buffer(timespan, timeskip, unit, Schedulers.computation(), ArrayListSupplier.asSupplier());
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} starts a new buffer periodically, as determined by the {@code timeskip} argument, and on the
     * specified {@code scheduler}. It emits each buffer after a fixed timespan, specified by the
     * {@code timespan} argument. When the current {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer
     * and propagates the notification from the current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError}
     * notification the event is passed on immediately without first emitting the buffer it is in the process of
     * assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer7.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted
     * @param timeskip
     *            the period of time after which a new buffer will be created
     * @param unit
     *            the unit of time that applies to the {@code timespan} and {@code timeskip} arguments
     * @param scheduler
     *            the {@code Scheduler} to use when determining the end and start of a buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, long timeskip, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return buffer(timespan, timeskip, unit, scheduler, ArrayListSupplier.asSupplier());
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} starts a new buffer periodically, as determined by the {@code timeskip} argument, and on the
     * specified {@code scheduler}. It emits each buffer after a fixed timespan, specified by the
     * {@code timespan} argument. When the current {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer
     * and propagates the notification from the current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError}
     * notification the event is passed on immediately without first emitting the buffer it is in the process of
     * assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer7.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param timespan
     *            the period of time each buffer collects items before it is emitted
     * @param timeskip
     *            the period of time after which a new buffer will be created
     * @param unit
     *            the unit of time that applies to the {@code timespan} and {@code timeskip} arguments
     * @param scheduler
     *            the {@code Scheduler} to use when determining the end and start of a buffer
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit}, {@code scheduler} or {@code bufferSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <U extends Collection<? super T>> Flowable<U> buffer(long timespan, long timeskip, @NonNull TimeUnit unit,
            @NonNull Scheduler scheduler, @NonNull Supplier<U> bufferSupplier) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        Objects.requireNonNull(bufferSupplier, "bufferSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableBufferTimed<>(this, timespan, timeskip, unit, scheduler, bufferSupplier, Integer.MAX_VALUE, false));
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each of a fixed duration specified by the
     * {@code timespan} argument. When the current {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer
     * and propagates the notification from the current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError}
     * notification the event is passed on immediately without first emitting the buffer it is in the process of
     * assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer5.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted and replaced with a new
     *            buffer
     * @param unit
     *            the unit of time that applies to the {@code timespan} argument
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, @NonNull TimeUnit unit) {
        return buffer(timespan, unit, Schedulers.computation(), Integer.MAX_VALUE);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each of a fixed duration specified by the
     * {@code timespan} argument or a maximum size specified by the {@code count} argument (whichever is reached
     * first). When the current {@code Flowable} completes, the resulting {@code Flowable} emits the current buffer and propagates the
     * notification from the current {@code Flowable}. Note that if the current {@code Flowable} issues an {@code onError} notification the event
     * is passed on immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer6.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted and replaced with a new
     *            buffer
     * @param unit
     *            the unit of time which applies to the {@code timespan} argument
     * @param count
     *            the maximum size of each buffer before it is emitted
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @throws IllegalArgumentException if {@code count} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, @NonNull TimeUnit unit, int count) {
        return buffer(timespan, unit, Schedulers.computation(), count);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each of a fixed duration specified by the
     * {@code timespan} argument as measured on the specified {@code scheduler}, or a maximum size specified by
     * the {@code count} argument (whichever is reached first). When the current {@code Flowable} completes, the resulting
     * {@code Flowable} emits the current buffer and propagates the notification from the current {@code Flowable}. Note that if the
     * current {@code Flowable} issues an {@code onError} notification the event is passed on immediately without first emitting the
     * buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer6.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted and replaced with a new
     *            buffer
     * @param unit
     *            the unit of time which applies to the {@code timespan} argument
     * @param scheduler
     *            the {@code Scheduler} to use when determining the end and start of a buffer
     * @param count
     *            the maximum size of each buffer before it is emitted
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code count} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, int count) {
        return buffer(timespan, unit, scheduler, count, ArrayListSupplier.asSupplier(), false);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each of a fixed duration specified by the
     * {@code timespan} argument as measured on the specified {@code scheduler}, or a maximum size specified by
     * the {@code count} argument (whichever is reached first). When the current {@code Flowable} completes, the resulting
     * {@code Flowable} emits the current buffer and propagates the notification from the current {@code Flowable}. Note that if the
     * current {@code Flowable} issues an {@code onError} notification the event is passed on immediately without first emitting the
     * buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer6.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param timespan
     *            the period of time each buffer collects items before it is emitted and replaced with a new
     *            buffer
     * @param unit
     *            the unit of time which applies to the {@code timespan} argument
     * @param scheduler
     *            the {@code Scheduler} to use when determining the end and start of a buffer
     * @param count
     *            the maximum size of each buffer before it is emitted
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @param restartTimerOnMaxSize if {@code true}, the time window is restarted when the max capacity of the current buffer
     *            is reached
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit}, {@code scheduler} or {@code bufferSupplier} is {@code null}
     * @throws IllegalArgumentException if {@code count} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final <U extends Collection<? super T>> Flowable<U> buffer(
            long timespan, @NonNull TimeUnit unit,
            @NonNull Scheduler scheduler, int count,
            @NonNull Supplier<U> bufferSupplier,
            boolean restartTimerOnMaxSize) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        Objects.requireNonNull(bufferSupplier, "bufferSupplier is null");
        ObjectHelper.verifyPositive(count, "count");
        return RxJavaPlugins.onAssembly(new FlowableBufferTimed<>(this, timespan, timespan, unit, scheduler, bufferSupplier, count, restartTimerOnMaxSize));
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits connected, non-overlapping buffers, each of a fixed duration specified by the
     * {@code timespan} argument and on the specified {@code scheduler}. When the current {@code Flowable} completes, the
     * resulting {@code Flowable} emits the current buffer and propagates the notification from the current {@code Flowable}. Note that
     * if the current {@code Flowable} issues an {@code onError} notification the event is passed on immediately without first emitting
     * the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer5.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time. It requests {@link Long#MAX_VALUE}
     *      upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timespan
     *            the period of time each buffer collects items before it is emitted and replaced with a new
     *            buffer
     * @param unit
     *            the unit of time which applies to the {@code timespan} argument
     * @param scheduler
     *            the {@code Scheduler} to use when determining the end and start of a buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<List<T>> buffer(long timespan, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return buffer(timespan, unit, scheduler, Integer.MAX_VALUE, ArrayListSupplier.asSupplier(), false);
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits buffers that it creates when the specified {@code openingIndicator} {@link Publisher} emits an
     * item, and closes when the {@code Publisher} returned from {@code closingIndicator} emits an item. If any of the current
     * {@code PFlowable}, {@code openingIndicator} or {@code closingIndicator} issues an {@code onError} notification the event is passed
     * on immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="470" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer2.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it is instead controlled by the given {@code Publisher}s and
     *      buffers data. It requests {@link Long#MAX_VALUE} upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <TOpening> the element type of the buffer-opening {@code Publisher}
     * @param <TClosing> the element type of the individual buffer-closing {@code Publisher}s
     * @param openingIndicator
     *            the {@code Publisher} that, when it emits an item, causes a new buffer to be created
     * @param closingIndicator
     *            the {@link Function} that is used to produce a {@code Publisher} for every buffer created. When this
     *            {@code Publisher} emits an item, the associated buffer is emitted.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code openingIndicator} or {@code closingIndicator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <TOpening, TClosing> Flowable<List<T>> buffer(
            @NonNull Publisher<@NonNull ? extends TOpening> openingIndicator,
            @NonNull Function<? super TOpening, ? extends Publisher<@NonNull ? extends TClosing>> closingIndicator) {
        return buffer(openingIndicator, closingIndicator, ArrayListSupplier.asSupplier());
    }

    /**
     * Returns a {@code Flowable} that emits buffers of items it collects from the current {@code Flowable}. The resulting
     * {@code Flowable} emits buffers that it creates when the specified {@code openingIndicator} {@link Publisher} emits an
     * item, and closes when the {@code Publisher} returned from {@code closingIndicator} emits an item. If any of the current
     * {@code Flowable}, {@code openingIndicator} or {@code closingIndicator} issues an {@code onError} notification the event is passed
     * on immediately without first emitting the buffer it is in the process of assembling.
     * <p>
     * <img width="640" height="470" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer2.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it is instead controlled by the given {@code Publisher}s and
     *      buffers data. It requests {@link Long#MAX_VALUE} upstream and does not obey downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param <TOpening> the element type of the buffer-opening {@code Publisher}
     * @param <TClosing> the element type of the individual buffer-closing {@code Publisher}s
     * @param openingIndicator
     *            the {@code Publisher} that, when it emits an item, causes a new buffer to be created
     * @param closingIndicator
     *            the {@link Function} that is used to produce a {@code Publisher} for every buffer created. When this
     *            {@code Publisher} emits an item, the associated buffer is emitted.
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code openingIndicator}, {@code closingIndicator} or {@code bufferSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <TOpening, TClosing, U extends Collection<? super T>> Flowable<U> buffer(
            @NonNull Publisher<@NonNull ? extends TOpening> openingIndicator,
            @NonNull Function<? super TOpening, ? extends Publisher<@NonNull ? extends TClosing>> closingIndicator,
            @NonNull Supplier<U> bufferSupplier) {
        Objects.requireNonNull(openingIndicator, "openingIndicator is null");
        Objects.requireNonNull(closingIndicator, "closingIndicator is null");
        Objects.requireNonNull(bufferSupplier, "bufferSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableBufferBoundary<T, U, TOpening, TClosing>(this, openingIndicator, closingIndicator, bufferSupplier));
    }

    /**
     * Returns a {@code Flowable} that emits non-overlapping buffered items from the current {@code Flowable} each time the
     * specified boundary {@link Publisher} emits an item.
     * <p>
     * <img width="640" height="395" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer8.v3.png" alt="">
     * <p>
     * Completion of either the source or the boundary {@code Publisher} causes the returned {@code Publisher} to emit the
     * latest buffer and complete. If either the current {@code Flowable} or the boundary {@code Publisher} issues an {@code onError} notification
     * the event is passed on immediately without first emitting the buffer it is in the process of assembling.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it is instead controlled by the {@code Publisher}
     *      {@code boundary} and buffers data. It requests {@link Long#MAX_VALUE} upstream and does not obey
     *      downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <B>
     *            the boundary value type (ignored)
     * @param boundaryIndicator
     *            the boundary {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code boundaryIndicator} is {@code null}
     * @see #buffer(Publisher, int)
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <B> Flowable<List<T>> buffer(@NonNull Publisher<B> boundaryIndicator) {
        return buffer(boundaryIndicator, ArrayListSupplier.asSupplier());
    }

    /**
     * Returns a {@code Flowable} that emits non-overlapping buffered items from the current {@code Flowable} each time the
     * specified boundary {@link Publisher} emits an item.
     * <p>
     * <img width="640" height="395" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer8.v3.png" alt="">
     * <p>
     * Completion of either the source or the boundary {@code Publisher} causes the returned {@code Publisher} to emit the
     * latest buffer and complete. If either the current {@code Flowable} or the boundary {@code Publisher} issues an {@code onError} notification
     * the event is passed on immediately without first emitting the buffer it is in the process of assembling.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it is instead controlled by the {@code Publisher}
     *      {@code boundary} and buffers data. It requests {@link Long#MAX_VALUE} upstream and does not obey
     *      downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <B>
     *            the boundary value type (ignored)
     * @param boundaryIndicator
     *            the boundary {@code Publisher}
     * @param initialCapacity
     *            the initial capacity of each buffer chunk
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code boundaryIndicator} is {@code null}
     * @throws IllegalArgumentException if {@code initialCapacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     * @see #buffer(Publisher)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <B> Flowable<List<T>> buffer(@NonNull Publisher<B> boundaryIndicator, int initialCapacity) {
        ObjectHelper.verifyPositive(initialCapacity, "initialCapacity");
        return buffer(boundaryIndicator, Functions.createArrayList(initialCapacity));
    }

    /**
     * Returns a {@code Flowable} that emits non-overlapping buffered items from the current {@code Flowable} each time the
     * specified boundary {@link Publisher} emits an item.
     * <p>
     * <img width="640" height="395" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/buffer8.v3.png" alt="">
     * <p>
     * Completion of either the source or the boundary {@code Publisher} causes the returned {@code Publisher} to emit the
     * latest buffer and complete. If either the current {@code Flowable} or the boundary {@code Publisher} issues an {@code onError} notification
     * the event is passed on immediately without first emitting the buffer it is in the process of assembling.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it is instead controlled by the {@code Publisher}
     *      {@code boundary} and buffers data. It requests {@link Long#MAX_VALUE} upstream and does not obey
     *      downstream requests.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code buffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the collection subclass type to buffer into
     * @param <B>
     *            the boundary value type (ignored)
     * @param boundaryIndicator
     *            the boundary {@code Publisher}
     * @param bufferSupplier
     *            a factory function that returns an instance of the collection subclass to be used and returned
     *            as the buffer
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code boundaryIndicator} or {@code bufferSupplier} is {@code null}
     * @see #buffer(Publisher, int)
     * @see <a href="http://reactivex.io/documentation/operators/buffer.html">ReactiveX operators documentation: Buffer</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <B, U extends Collection<? super T>> Flowable<U> buffer(@NonNull Publisher<B> boundaryIndicator, @NonNull Supplier<U> bufferSupplier) {
        Objects.requireNonNull(boundaryIndicator, "boundaryIndicator is null");
        Objects.requireNonNull(bufferSupplier, "bufferSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableBufferExactBoundary<>(this, boundaryIndicator, bufferSupplier));
    }

    /**
     * Returns a {@code Flowable} that subscribes to this {@link Publisher} lazily, caches all of its events
     * and replays them, in the same order as received, to all the downstream subscribers.
     * <p>
     * <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/cache.v3.png" alt="">
     * <p>
     * This is useful when you want a {@code Publisher} to cache responses and you can't control the
     * subscribe/cancel behavior of all the {@link Subscriber}s.
     * <p>
     * The operator subscribes only when the first downstream subscriber subscribes and maintains
     * a single subscription towards this {@code Publisher}. In contrast, the operator family of {@link #replay()}
     * that return a {@link ConnectableFlowable} require an explicit call to {@link ConnectableFlowable#connect()}.
     * <p>
     * <em>Note:</em> You sacrifice the ability to cancel the origin when you use the {@code cache}
     * operator so be careful not to use this operator on {@code Publisher}s that emit an infinite or very large number
     * of items that will use up memory.
     * A possible workaround is to apply {@link #takeUntil(Publisher)} with a predicate or
     * another source before (and perhaps after) the application of {@code cache()}.
     * <pre><code>
     * AtomicBoolean shouldStop = new AtomicBoolean();
     *
     * source.takeUntil(v -&gt; shouldStop.get())
     *       .cache()
     *       .takeUntil(v -&gt; shouldStop.get())
     *       .subscribe(...);
     * </code></pre>
     * Since the operator doesn't allow clearing the cached values either, the possible workaround is
     * to forget all references to it via {@link #onTerminateDetach()} applied along with the previous
     * workaround:
     * <pre><code>
     * AtomicBoolean shouldStop = new AtomicBoolean();
     *
     * source.takeUntil(v -&gt; shouldStop.get())
     *       .onTerminateDetach()
     *       .cache()
     *       .takeUntil(v -&gt; shouldStop.get())
     *       .onTerminateDetach()
     *       .subscribe(...);
     * </code></pre>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes this {@code Publisher} in an unbounded fashion but respects the backpressure
     *  of each downstream {@code Subscriber} individually.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code cache} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #takeUntil(Predicate)
     * @see #takeUntil(Publisher)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> cache() {
        return cacheWithInitialCapacity(16);
    }

    /**
     * Returns a {@code Flowable} that subscribes to this {@link Publisher} lazily, caches all of its events
     * and replays them, in the same order as received, to all the downstream subscribers.
     * <p>
     * <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/cache.v3.png" alt="">
     * <p>
     * This is useful when you want a {@code Publisher} to cache responses and you can't control the
     * subscribe/cancel behavior of all the {@link Subscriber}s.
     * <p>
     * The operator subscribes only when the first downstream subscriber subscribes and maintains
     * a single subscription towards this {@code Publisher}. In contrast, the operator family of {@link #replay()}
     * that return a {@link ConnectableFlowable} require an explicit call to {@link ConnectableFlowable#connect()}.
     * <p>
     * <em>Note:</em> You sacrifice the ability to cancel the origin when you use the {@code cache}
     * operator so be careful not to use this operator on {@code Publisher}s that emit an infinite or very large number
     * of items that will use up memory.
     * A possible workaround is to apply {@link #takeUntil(Publisher)} with a predicate or
     * another source before (and perhaps after) the application of {@code cacheWithInitialCapacity()}.
     * <pre><code>
     * AtomicBoolean shouldStop = new AtomicBoolean();
     *
     * source.takeUntil(v -&gt; shouldStop.get())
     *       .cache()
     *       .takeUntil(v -&gt; shouldStop.get())
     *       .subscribe(...);
     * </code></pre>
     * Since the operator doesn't allow clearing the cached values either, the possible workaround is
     * to forget all references to it via {@link #onTerminateDetach()} applied along with the previous
     * workaround:
     * <pre><code>
     * AtomicBoolean shouldStop = new AtomicBoolean();
     *
     * source.takeUntil(v -&gt; shouldStop.get())
     *       .onTerminateDetach()
     *       .cache()
     *       .takeUntil(v -&gt; shouldStop.get())
     *       .onTerminateDetach()
     *       .subscribe(...);
     * </code></pre>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes this {@code Publisher} in an unbounded fashion but respects the backpressure
     *  of each downstream {@code Subscriber} individually.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code cacheWithInitialCapacity} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>
     * <em>Note:</em> The capacity hint is not an upper bound on cache size. For that, consider
     * {@link #replay(int)} in combination with {@link ConnectableFlowable#autoConnect()} or similar.
     *
     * @param initialCapacity hint for number of items to cache (for optimizing underlying data structure)
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code initialCapacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #takeUntil(Predicate)
     * @see #takeUntil(Publisher)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> cacheWithInitialCapacity(int initialCapacity) {
        ObjectHelper.verifyPositive(initialCapacity, "initialCapacity");
        return RxJavaPlugins.onAssembly(new FlowableCache<>(this, initialCapacity));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable}, converted to the specified
     * type.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/cast.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code cast} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the output value type cast to
     * @param clazz
     *            the target class type that {@code cast} will cast the items emitted by the current {@code Flowable}
     *            into before emitting them from the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code clazz} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/map.html">ReactiveX operators documentation: Map</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<U> cast(@NonNull Class<U> clazz) {
        Objects.requireNonNull(clazz, "clazz is null");
        return map(Functions.castFunction(clazz));
    }

    /**
     * Collects items emitted by the finite source {@link Publisher} into a single mutable data structure and returns
     * a {@link Single} that emits this structure.
     * <p>
     * <img width="640" height="330" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/collect.v3.png" alt="">
     * <p>
     * This is a simplified version of {@code reduce} that does not need to return the state on each pass.
     * <p>
     * Note that this operator requires the upstream to signal {@code onComplete} for the accumulator object to
     * be emitted. Sources that are infinite and never complete will never emit anything through this
     * operator and an infinite source may lead to a fatal {@link OutOfMemoryError}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure because by intent it will receive all values and reduce
     *      them to a single {@code onNext}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code collect} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the accumulator and output type
     * @param initialItemSupplier
     *           the mutable data structure that will collect the items
     * @param collector
     *           a function that accepts the {@code state} and an emitted item, and modifies {@code state}
     *           accordingly
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code initialItemSupplier} or {@code collector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/reduce.html">ReactiveX operators documentation: Reduce</a>
     * @see #collect(Collector)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Single<U> collect(@NonNull Supplier<? extends U> initialItemSupplier, @NonNull BiConsumer<? super U, ? super T> collector) {
        Objects.requireNonNull(initialItemSupplier, "initialItemSupplier is null");
        Objects.requireNonNull(collector, "collector is null");
        return RxJavaPlugins.onAssembly(new FlowableCollectSingle<>(this, initialItemSupplier, collector));
    }

    /**
     * Collects items emitted by the finite source {@link Publisher} into a single mutable data structure and returns
     * a {@link Single} that emits this structure.
     * <p>
     * <img width="640" height="330" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/collect.v3.png" alt="">
     * <p>
     * This is a simplified version of {@code reduce} that does not need to return the state on each pass.
     * <p>
     * Note that this operator requires the upstream to signal {@code onComplete} for the accumulator object to
     * be emitted. Sources that are infinite and never complete will never emit anything through this
     * operator and an infinite source may lead to a fatal {@link OutOfMemoryError}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure because by intent it will receive all values and reduce
     *      them to a single {@code onNext}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code collectInto} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the accumulator and output type
     * @param initialItem
     *           the mutable data structure that will collect the items
     * @param collector
     *           a function that accepts the {@code state} and an emitted item, and modifies {@code state}
     *           accordingly
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code initialItem} or {@code collector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/reduce.html">ReactiveX operators documentation: Reduce</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull U> Single<U> collectInto(U initialItem, @NonNull BiConsumer<? super U, ? super T> collector) {
        Objects.requireNonNull(initialItem, "initialItem is null");
        return collect(Functions.justSupplier(initialItem), collector);
    }

    /**
     * Transform the current {@code Flowable} by applying a particular {@link FlowableTransformer} function to it.
     * <p>
     * This method operates on the {@code Flowable} itself whereas {@link #lift} operates on the {@code Flowable}'s
     * {@link Subscriber}s.
     * <p>
     * If the operator you are creating is designed to act on the individual items emitted by a current
     * {@code Flowable}, use {@link #lift}. If your operator is designed to transform the current {@code Flowable} as a whole
     * (for instance, by applying a particular set of existing RxJava operators to it) use {@code compose}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator itself doesn't interfere with the backpressure behavior which only depends
     *  on what kind of {@link Publisher} the {@code FlowableTransformer} returns.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code compose} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the output {@code Publisher}
     * @param composer implements the function that transforms the current {@code Flowable}
     * @return the new composed {@code Flowable} instance
     * @throws NullPointerException if {@code composer} is {@code null}
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Implementing-Your-Own-Operators">RxJava wiki: Implementing Your Own Operators</a>
     */
    @SuppressWarnings("unchecked")
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> compose(@NonNull FlowableTransformer<? super T, ? extends R> composer) {
        return fromPublisher(((FlowableTransformer<T, R>) Objects.requireNonNull(composer, "composer is null")).apply(this));
    }

    /**
     * Returns a new {@code Flowable} that emits items resulting from applying a function that you supply to each item
     * emitted by the current {@code Flowable}, where that function returns a {@link Publisher}, and then emitting the items
     * that result from concatenating those returned {@code Publisher}s.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <p>
     * Note that there is no guarantee where the given {@code mapper} function will be executed; it could be on the subscribing thread,
     * on the upstream thread signaling the new item to be mapped or on the thread where the inner source terminates. To ensure
     * the {@code mapper} function is confined to a known thread, use the {@link #concatMap(Function, int, Scheduler)} overload.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the type of the inner {@code Publisher} sources and thus the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return concatMap(mapper, 2);
    }

    /**
     * Returns a new {@code Flowable} that emits items resulting from applying a function that you supply to each item
     * emitted by the current {@code Flowable}, where that function returns a {@link Publisher}, and then emitting the items
     * that result from concatenating those returned {@code Publisher}s.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <p>
     * Note that there is no guarantee where the given {@code mapper} function will be executed; it could be on the subscribing thread,
     * on the upstream thread signaling the new item to be mapped or on the thread where the inner source terminates. To ensure
     * the {@code mapper} function is confined to a known thread, use the {@link #concatMap(Function, int, Scheduler)} overload.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the type of the inner {@code Publisher} sources and thus the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @see #concatMap(Function, int, Scheduler)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        if (this instanceof ScalarSupplier) {
            @SuppressWarnings("unchecked")
            T v = ((ScalarSupplier<T>)this).get();
            if (v == null) {
                return empty();
            }
            return FlowableScalarXMap.scalarXMap(v, mapper);
        }
        return RxJavaPlugins.onAssembly(new FlowableConcatMap<>(this, mapper, prefetch, ErrorMode.IMMEDIATE));
    }

    /**
     * Returns a new {@code Flowable} that emits items resulting from applying a function (on a designated scheduler)
     * that you supply to each item emitted by the current {@code Flowable}, where that function returns a {@link Publisher}, and then emitting the items
     * that result from concatenating those returned {@code Publisher}s.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <p>
     * The difference between {@link #concatMap(Function, int)} and this operator is that this operator guarantees the {@code mapper}
     * function is executed on the specified scheduler.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMap} executes the given {@code mapper} function on the provided {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the type of the inner {@code Publisher} sources and thus the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @param scheduler
     *            the scheduler where the {@code mapper} function will be executed
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 3.0.0
     * @see #concatMap(Function, int)
     * @see #concatMapDelayError(Function, boolean, int, Scheduler)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> concatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int prefetch, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapScheduler<>(this, mapper, prefetch, ErrorMode.IMMEDIATE, scheduler));
    }

    /**
     * Maps the upstream items into {@link CompletableSource}s and subscribes to them one after the
     * other completes.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with the upstream item and should return
     *               a {@code CompletableSource} to become the next source to
     *               be subscribed to
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapCompletableDelayError(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public final Completable concatMapCompletable(@NonNull Function<? super T, ? extends CompletableSource> mapper) {
        return concatMapCompletable(mapper, 2);
    }

    /**
     * Maps the upstream items into {@link CompletableSource}s and subscribes to them one after the
     * other completes.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with the upstream item and should return
     *               a {@code CompletableSource} to become the next source to
     *               be subscribed to
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code CompletableSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code CompletableSource}s.
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapCompletableDelayError(Function, boolean, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public final Completable concatMapCompletable(@NonNull Function<? super T, ? extends CompletableSource> mapper, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapCompletable<>(this, mapper, ErrorMode.IMMEDIATE, prefetch));
    }

    /**
     * Maps the upstream items into {@link CompletableSource}s and subscribes to them one after the
     * other terminates, delaying all errors till both this {@code Flowable} and all
     * inner {@code CompletableSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapCompletableDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with the upstream item and should return
     *               a {@code CompletableSource} to become the next source to
     *               be subscribed to
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapCompletable(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public final Completable concatMapCompletableDelayError(@NonNull Function<? super T, ? extends CompletableSource> mapper) {
        return concatMapCompletableDelayError(mapper, true, 2);
    }

    /**
     * Maps the upstream items into {@link CompletableSource}s and subscribes to them one after the
     * other terminates, optionally delaying all errors till both this {@code Flowable} and all
     * inner {@code CompletableSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapCompletableDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with the upstream item and should return
     *               a {@code CompletableSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code CompletableSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code CompletableSource} terminates and only then is
     *                   it emitted to the downstream.
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapCompletable(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    @NonNull
    public final Completable concatMapCompletableDelayError(@NonNull Function<? super T, ? extends CompletableSource> mapper, boolean tillTheEnd) {
        return concatMapCompletableDelayError(mapper, tillTheEnd, 2);
    }

    /**
     * Maps the upstream items into {@link CompletableSource}s and subscribes to them one after the
     * other terminates, optionally delaying all errors till both this {@code Flowable} and all
     * inner {@code CompletableSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapCompletableDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with the upstream item and should return
     *               a {@code CompletableSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code CompletableSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code CompletableSource} terminates and only then is
     *                   it emitted to the downstream.
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code CompletableSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code CompletableSource}s.
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapCompletable(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public final Completable concatMapCompletableDelayError(@NonNull Function<? super T, ? extends CompletableSource> mapper, boolean tillTheEnd, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapCompletable<>(this, mapper, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY, prefetch));
    }

    /**
     * Maps each of the items into a {@link Publisher}, subscribes to them one after the other,
     * one at a time and emits their values in order
     * while delaying any error from either this or any of the inner {@code Publisher}s
     * till all of them terminate.
     * <p>
     * Note that there is no guarantee where the given {@code mapper} function will be executed; it could be on the subscribing thread,
     * on the upstream thread signaling the new item to be mapped or on the thread where the inner source terminates. To ensure
     * the {@code mapper} function is confined to a known thread, use the {@link #concatMapDelayError(Function, boolean, int, Scheduler)} overload.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the result value type
     * @param mapper the function that maps the items of this {@code Publisher} into the inner {@code Publisher}s.
     * @return the new {@code Flowable} instance with the concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapDelayError(Function, boolean, int, Scheduler)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return concatMapDelayError(mapper, true, 2);
    }

    /**
     * Maps each of the items into a {@link Publisher}, subscribes to them one after the other,
     * one at a time and emits their values in order
     * while delaying any error from either this or any of the inner {@code Publisher}s
     * till all of them terminate.
     * <p>
     * Note that there is no guarantee where the given {@code mapper} function will be executed; it could be on the subscribing thread,
     * on the upstream thread signaling the new item to be mapped or on the thread where the inner source terminates. To ensure
     * the {@code mapper} function is confined to a known thread, use the {@link #concatMapDelayError(Function, boolean, int, Scheduler)} overload.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the result value type
     * @param mapper the function that maps the items of this {@code Publisher} into the inner {@code Publisher}s.
     * @param tillTheEnd
     *            if {@code true}, all errors from the outer and inner {@code Publisher} sources are delayed until the end,
     *            if {@code false}, an error from the main source is signaled when the current inner {@code Publisher} source terminates
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance with the concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapDelayError(Function, boolean, int, Scheduler)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            boolean tillTheEnd, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        if (this instanceof ScalarSupplier) {
            @SuppressWarnings("unchecked")
            T v = ((ScalarSupplier<T>)this).get();
            if (v == null) {
                return empty();
            }
            return FlowableScalarXMap.scalarXMap(v, mapper);
        }
        return RxJavaPlugins.onAssembly(new FlowableConcatMap<>(this, mapper, prefetch, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY));
    }

    /**
     * Maps each of the upstream items into a {@link Publisher}, subscribes to them one after the other,
     * one at a time and emits their values in order
     * while executing the mapper function on the designated scheduler, delaying any error from either this or any of the
     * inner {@code Publisher}s till all of them terminate.
     * <p>
     * The difference between {@link #concatMapDelayError(Function, boolean, int)} and this operator is that this operator guarantees the {@code mapper}
     * function is executed on the specified scheduler.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the inner {@code Publisher}s are
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}. If any of the inner {@code Publisher}s doesn't honor
     *  backpressure, that <em>may</em> throw an {@link IllegalStateException} when that
     *  {@code Publisher} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapDelayError} executes the given {@code mapper} function on the provided {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the result value type
     * @param mapper the function that maps the items of this {@code Publisher} into the inner {@code Publisher}s.
     * @param tillTheEnd
     *            if {@code true}, all errors from the outer and inner {@code Publisher} sources are delayed until the end,
     *            if {@code false}, an error from the main source is signaled when the current inner {@code Publisher} source terminates
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @param scheduler
     *            the scheduler where the {@code mapper} function will be executed
     * @return the new {@code Flowable} instance with the concatenation behavior
     * @throws NullPointerException if {@code mapper} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapDelayError(Function, boolean, int)
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> concatMapDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            boolean tillTheEnd, int prefetch, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapScheduler<>(this, mapper, prefetch, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY, scheduler));
    }

    /**
     * Maps a sequence of values into {@link Publisher}s and concatenates these {@code Publisher}s eagerly into a single
     * {@code Publisher}.
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * inner {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them in
     * order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream, however, due to the eagerness requirement, sources
     *      are subscribed to in unbounded mode and their values are queued up in an unbounded buffer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the value type
     * @param mapper the function that maps a sequence of values into a sequence of {@code Publisher}s that will be
     *               eagerly concatenated
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapEager(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return concatMapEager(mapper, bufferSize(), bufferSize());
    }

    /**
     * Maps a sequence of values into {@link Publisher}s and concatenates these {@code Publisher}s eagerly into a single
     * {@code Publisher}.
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * inner {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them in
     * order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream, however, due to the eagerness requirement, sources
     *      are subscribed to in unbounded mode and their values are queued up in an unbounded buffer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the value type
     * @param mapper the function that maps a sequence of values into a sequence of {@code Publisher}s that will be
     *               eagerly concatenated
     * @param maxConcurrency the maximum number of concurrent subscribed {@code Publisher}s
     * @param prefetch hints about the number of expected values from each inner {@code Publisher}, must be positive
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapEager(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            int maxConcurrency, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEager<>(this, mapper, maxConcurrency, prefetch, ErrorMode.IMMEDIATE));
    }

    /**
     * Maps a sequence of values into {@link Publisher}s and concatenates these {@code Publisher}s eagerly into a single
     * {@code Publisher}.
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * inner {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them in
     * order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream, however, due to the eagerness requirement, sources
     *      are subscribed to in unbounded mode and their values are queued up in an unbounded buffer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the value type
     * @param mapper the function that maps a sequence of values into a sequence of {@code Publisher}s that will be
     *               eagerly concatenated
     * @param tillTheEnd
     *            if {@code true}, all errors from the outer and inner {@code Publisher} sources are delayed until the end,
     *            if {@code false}, an error from the main source is signaled when the current inner {@code Publisher} source terminates
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapEagerDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            boolean tillTheEnd) {
        return concatMapEagerDelayError(mapper, tillTheEnd, bufferSize(), bufferSize());
    }

    /**
     * Maps a sequence of values into {@link Publisher}s and concatenates these {@code Publisher}s eagerly into a single
     * {@code Flowable} sequence.
     * <p>
     * Eager concatenation means that once a subscriber subscribes, this operator subscribes to all of the
     * inner {@code Publisher}s. The operator buffers the values emitted by these {@code Publisher}s and then drains them in
     * order, each one after the previous one completes.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>Backpressure is honored towards the downstream, however, due to the eagerness requirement, sources
     *      are subscribed to in unbounded mode and their values are queued up in an unbounded buffer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the value type
     * @param mapper the function that maps a sequence of values into a sequence of {@code Publisher}s that will be
     *               eagerly concatenated
     * @param tillTheEnd
     *               if {@code true}, exceptions from the current {@code Flowable} and all the inner {@code Publisher}s are delayed until
     *               all of them terminate, if {@code false}, exception from the current {@code Flowable} is delayed until the
     *               currently running {@code Publisher} terminates
     * @param maxConcurrency the maximum number of concurrent subscribed {@code Publisher}s
     * @param prefetch
     *               the number of elements to prefetch from each source {@code Publisher}
     * @return the new {@code Flowable} instance with the specified concatenation behavior
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code prefetch} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapEagerDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            boolean tillTheEnd, int maxConcurrency, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapEager<>(this, mapper, maxConcurrency, prefetch, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY));
    }

    /**
     * Returns a {@code Flowable} that concatenate each item emitted by the current {@code Flowable} with the values in an
     * {@link Iterable} corresponding to that item that is generated by a selector.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}s is
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of item emitted by the resulting {@code Flowable}
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for when given an item emitted by the
     *            current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U> Flowable<U> concatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper) {
        return concatMapIterable(mapper, 2);
    }

    /**
     * Returns a {@code Flowable} that concatenate each item emitted by the current {@code Flowable} with the values in an
     * {@link Iterable} corresponding to that item that is generated by a selector.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable} is
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of item emitted by the resulting {@code Flowable}
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for when given an item emitted by the
     *            current {@code Flowable}
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<U> concatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableFlattenIterable<>(this, mapper, prefetch));
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and subscribes to them one after the
     * other succeeds or completes, emits their success value if available or terminates immediately if
     * either this {@code Flowable} or the current inner {@code MaybeSource} fail.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code MaybeSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code MaybeSource} to become the next source to
     *               be subscribed to
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapMaybeDelayError(Function)
     * @see #concatMapMaybe(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapMaybe(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper) {
        return concatMapMaybe(mapper, 2);
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and subscribes to them one after the
     * other succeeds or completes, emits their success value if available or terminates immediately if
     * either this {@code Flowable} or the current inner {@code MaybeSource} fail.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code MaybeSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code MaybeSource} to become the next source to
     *               be subscribed to
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code MaybeSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code MaybeSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapMaybe(Function)
     * @see #concatMapMaybeDelayError(Function, boolean, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapMaybe(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapMaybe<>(this, mapper, ErrorMode.IMMEDIATE, prefetch));
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and subscribes to them one after the
     * other terminates, emits their success value if available and delaying all errors
     * till both this {@code Flowable} and all inner {@code MaybeSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapMaybeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code MaybeSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code MaybeSource} to become the next source to
     *               be subscribed to
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapMaybe(Function)
     * @see #concatMapMaybeDelayError(Function, boolean)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapMaybeDelayError(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper) {
        return concatMapMaybeDelayError(mapper, true, 2);
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and subscribes to them one after the
     * other terminates, emits their success value if available and optionally delaying all errors
     * till both this {@code Flowable} and all inner {@code MaybeSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapMaybeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code MaybeSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code MaybeSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code MaybeSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code MaybeSource} terminates and only then is
     *                   it emitted to the downstream.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapMaybe(Function, int)
     * @see #concatMapMaybeDelayError(Function, boolean, int)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapMaybeDelayError(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper, boolean tillTheEnd) {
        return concatMapMaybeDelayError(mapper, tillTheEnd, 2);
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and subscribes to them one after the
     * other terminates, emits their success value if available and optionally delaying all errors
     * till both this {@code Flowable} and all inner {@code MaybeSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapMaybeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code MaybeSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code MaybeSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code MaybeSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code MaybeSource} terminates and only then is
     *                   it emitted to the downstream.
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code MaybeSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code MaybeSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapMaybe(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapMaybeDelayError(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper, boolean tillTheEnd, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapMaybe<>(this, mapper, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY, prefetch));
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and subscribes to them one after the
     * other succeeds, emits their success values or terminates immediately if
     * either this {@code Flowable} or the current inner {@code SingleSource} fail.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code SingleSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code SingleSource} to become the next source to
     *               be subscribed to
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapSingleDelayError(Function)
     * @see #concatMapSingle(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapSingle(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper) {
        return concatMapSingle(mapper, 2);
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and subscribes to them one after the
     * other succeeds, emits their success values or terminates immediately if
     * either this {@code Flowable} or the current inner {@code SingleSource} fail.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code SingleSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code SingleSource} to become the next source to
     *               be subscribed to
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code SingleSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code SingleSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapSingle(Function)
     * @see #concatMapSingleDelayError(Function, boolean, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapSingle(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapSingle<>(this, mapper, ErrorMode.IMMEDIATE, prefetch));
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and subscribes to them one after the
     * other succeeds or fails, emits their success values and delays all errors
     * till both this {@code Flowable} and all inner {@code SingleSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapSingleDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code SingleSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code SingleSource} to become the next source to
     *               be subscribed to
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapSingle(Function)
     * @see #concatMapSingleDelayError(Function, boolean)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapSingleDelayError(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper) {
        return concatMapSingleDelayError(mapper, true, 2);
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and subscribes to them one after the
     * other succeeds or fails, emits their success values and optionally delays all errors
     * till both this {@code Flowable} and all inner {@code SingleSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapSingleDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code SingleSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code SingleSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code SingleSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code SingleSource} terminates and only then is
     *                   it emitted to the downstream.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #concatMapSingle(Function, int)
     * @see #concatMapSingleDelayError(Function, boolean, int)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> concatMapSingleDelayError(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper, boolean tillTheEnd) {
        return concatMapSingleDelayError(mapper, tillTheEnd, 2);
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and subscribes to them one after the
     * other succeeds or fails, emits their success values and optionally delays errors
     * till both this {@code Flowable} and all inner {@code SingleSource}s terminate.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the upstream to support backpressure and honors
     *  the backpressure from downstream. If this {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatMapSingleDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the result type of the inner {@code SingleSource}s
     * @param mapper the function called with the upstream item and should return
     *               a {@code SingleSource} to become the next source to
     *               be subscribed to
     * @param tillTheEnd If {@code true}, errors from this {@code Flowable} or any of the
     *                   inner {@code SingleSource}s are delayed until all
     *                   of them terminate. If {@code false}, an error from this
     *                   {@code Flowable} is delayed until the current inner
     *                   {@code SingleSource} terminates and only then is
     *                   it emitted to the downstream.
     * @param prefetch The number of upstream items to prefetch so that fresh items are
     *                 ready to be mapped when a previous {@code SingleSource} terminates.
     *                 The operator replenishes after half of the prefetch amount has been consumed
     *                 and turned into {@code SingleSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see #concatMapSingle(Function, int)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> concatMapSingleDelayError(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper, boolean tillTheEnd, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowableConcatMapSingle<>(this, mapper, tillTheEnd ? ErrorMode.END : ErrorMode.BOUNDARY, prefetch));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted from the current {@code Flowable}, then the next, one after
     * the other, without interleaving them.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the {@code other} {@link Publisher}s
     *  are expected to honor backpressure as well. If any of then violates this rule, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param other
     *            a {@code Publisher} to be concatenated after the current
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/concat.html">ReactiveX operators documentation: Concat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> concatWith(@NonNull Publisher<@NonNull ? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return concat(this, other);
    }

    /**
     * Returns a {@code Flowable} that emits the items from this {@code Flowable} followed by the success item or error event
     * of the other {@link SingleSource}.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator supports backpressure and makes sure the success item of the other {@code SingleSource}
     *  is only emitted when there is a demand for it.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code SingleSource} whose signal should be emitted after this {@code Flowable} completes normally.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> concatWith(@NonNull SingleSource<? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableConcatWithSingle<>(this, other));
    }

    /**
     * Returns a {@code Flowable} that emits the items from this {@code Flowable} followed by the success item or terminal events
     * of the other {@link MaybeSource}.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator supports backpressure and makes sure the success item of the other {@code MaybeSource}
     *  is only emitted when there is a demand for it.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code MaybeSource} whose signal should be emitted after this {@code Flowable} completes normally.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> concatWith(@NonNull MaybeSource<? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableConcatWithMaybe<>(this, other));
    }

    /**
     * Returns a {@code Flowable} that emits items from this {@code Flowable} and when it completes normally, the
     * other {@link CompletableSource} is subscribed to and the returned {@code Flowable} emits its terminal events.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/concat.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator does not interfere with backpressure between the current {@code Flowable} and the
     *  downstream consumer (i.e., acts as pass-through). When the operator switches to the
     *  {@link Completable}, backpressure is no longer present because {@code Completable} doesn't
     *  have items to apply backpressure to.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code concatWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code CompletableSource} to subscribe to once the current {@code Flowable} completes normally
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> concatWith(@NonNull CompletableSource other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableConcatWithCompletable<>(this, other));
    }

    /**
     * Returns a {@link Single} that emits a {@link Boolean} that indicates whether the current {@code Flowable} emitted a
     * specified item.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/contains.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code contains} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item
     *            the item to search for in the emissions from the current {@code Flowable}
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code item} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/contains.html">ReactiveX operators documentation: Contains</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<Boolean> contains(@NonNull Object item) {
        Objects.requireNonNull(item, "item is null");
        return any(Functions.equalsWith(item));
    }

    /**
     * Returns a {@link Single} that counts the total number of items emitted by the current {@code Flowable} and emits
     * this count as a 64-bit {@link Long}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/longCount.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code count} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Single} instance
     * @see <a href="http://reactivex.io/documentation/operators/count.html">ReactiveX operators documentation: Count</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<Long> count() {
        return RxJavaPlugins.onAssembly(new FlowableCountSingle<>(this));
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, except that it drops items emitted by the
     * current {@code Flowable} that are followed by another item within a computed debounce duration.
     * <p>
     * <img width="640" height="425" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/debounce.f.v3.png" alt="">
     * <p>
     * The delivery of the item happens on the thread of the first {@code onNext} or {@code onComplete}
     * signal of the generated {@link Publisher} sequence,
     * which if takes too long, a newer item may arrive from the upstream, causing the
     * generated sequence to get cancelled, which may also interrupt any downstream blocking operation
     * (yielding an {@code InterruptedException}). It is recommended processing items
     * that may take long time to be moved to another thread via {@link #observeOn} applied after
     * {@code debounce} itself.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses the {@code debounceSelector} to mark
     *      boundaries.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code debounce} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the debounce value type (ignored)
     * @param debounceIndicator
     *            function to retrieve a sequence that indicates the throttle duration for each item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code debounceIndicator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/debounce.html">ReactiveX operators documentation: Debounce</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> debounce(@NonNull Function<? super T, ? extends Publisher<U>> debounceIndicator) {
        Objects.requireNonNull(debounceIndicator, "debounceIndicator is null");
        return RxJavaPlugins.onAssembly(new FlowableDebounce<>(this, debounceIndicator));
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, except that it drops items emitted by the
     * current {@code Flowable} that are followed by newer items before a timeout value expires. The timer resets on
     * each emission.
     * <p>
     * <em>Note:</em> If items keep being emitted by the current {@code Flowable} faster than the timeout then no items
     * will be emitted by the resulting {@code Flowable}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/debounce.v3.png" alt="">
     * <p>
     * Delivery of the item after the grace period happens on the {@code computation} {@link Scheduler}'s
     * {@code Worker} which if takes too long, a newer item may arrive from the upstream, causing the
     * {@code Worker}'s task to get disposed, which may also interrupt any downstream blocking operation
     * (yielding an {@code InterruptedException}). It is recommended processing items
     * that may take long time to be moved to another thread via {@link #observeOn} applied after
     * {@code debounce} itself.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code debounce} operates by default on the {@code computation} {@code Scheduler}.</dd>
     * </dl>
     *
     * @param timeout
     *            the length of the window of time that must pass after the emission of an item from the current
     *            {@code Flowable} in which it emits no items in order for the item to be emitted by the
     *            resulting {@code Flowable}
     * @param unit
     *            the unit of time for the specified {@code timeout}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/debounce.html">ReactiveX operators documentation: Debounce</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleWithTimeout(long, TimeUnit)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> debounce(long timeout, @NonNull TimeUnit unit) {
        return debounce(timeout, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, except that it drops items emitted by the
     * current {@code Flowable} that are followed by newer items before a timeout value expires on a specified
     * {@link Scheduler}. The timer resets on each emission.
     * <p>
     * <em>Note:</em> If items keep being emitted by the current {@code Flowable} faster than the timeout then no items
     * will be emitted by the resulting {@code Flowable}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/debounce.s.v3.png" alt="">
     * <p>
     * Delivery of the item after the grace period happens on the given {@code Scheduler}'s
     * {@code Worker} which if takes too long, a newer item may arrive from the upstream, causing the
     * {@code Worker}'s task to get disposed, which may also interrupt any downstream blocking operation
     * (yielding an {@code InterruptedException}). It is recommended processing items
     * that may take long time to be moved to another thread via {@link #observeOn} applied after
     * {@code debounce} itself.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timeout
     *            the time each item has to be "the most recent" of those emitted by the current {@code Flowable} to
     *            ensure that it's not dropped
     * @param unit
     *            the unit of time for the specified {@code timeout}
     * @param scheduler
     *            the {@code Scheduler} to use internally to manage the timers that handle the timeout for each
     *            item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/debounce.html">ReactiveX operators documentation: Debounce</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleWithTimeout(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> debounce(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableDebounceTimed<>(this, timeout, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} or a specified default item
     * if the current {@code Flowable} is empty.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/defaultIfEmpty.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>If the current {@code Flowable} is empty, this operator is guaranteed to honor backpressure from downstream.
     *  If the current {@code Flowable} is non-empty, it is expected to honor backpressure as well; if the rule is violated,
     *  a {@link MissingBackpressureException} <em>may</em> get signaled somewhere downstream.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code defaultIfEmpty} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param defaultItem
     *            the item to emit if the current {@code Flowable} emits no items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/defaultifempty.html">ReactiveX operators documentation: DefaultIfEmpty</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> defaultIfEmpty(@NonNull T defaultItem) {
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        return switchIfEmpty(just(defaultItem));
    }

    /**
     * Returns a {@code Flowable} that delays the emissions of the current {@code Flowable} via another {@link Publisher} on a
     * per-item basis.
     * <p>
     * <img width="640" height="450" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.o.v3.png" alt="">
     * <p>
     * <em>Note:</em> the resulting {@code Flowable} will immediately propagate any {@code onError} notification
     * from the current {@code Flowable}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.
     *  All of the other {@code Publisher}s supplied by the function are consumed
     *  in an unbounded manner (i.e., no backpressure applied to them).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code delay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the item delay value type (ignored)
     * @param itemDelayIndicator
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}, which is
     *            then used to delay the emission of that item by the resulting {@code Flowable} until the {@code Publisher}
     *            returned from {@code itemDelay} emits an item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code itemDelayIndicator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> delay(@NonNull Function<? super T, ? extends Publisher<U>> itemDelayIndicator) {
        Objects.requireNonNull(itemDelayIndicator, "itemDelayIndicator is null");
        return flatMap(FlowableInternalHelper.itemDelay(itemDelayIndicator));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} shifted forward in time by a
     * specified delay. The {@code onError} notification from the current {@code Flowable} is not delayed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code delay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the delay to shift the source by
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> delay(long time, @NonNull TimeUnit unit) {
        return delay(time, unit, Schedulers.computation(), false);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} shifted forward in time by a
     * specified delay. If {@code delayError} is {@code true}, error notifications will also be delayed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code delay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the delay to shift the source by
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @param delayError
     *            if {@code true}, the upstream exception is signaled with the given delay, after all preceding normal elements,
     *            if {@code false}, the upstream exception is signaled immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> delay(long time, @NonNull TimeUnit unit, boolean delayError) {
        return delay(time, unit, Schedulers.computation(), delayError);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} shifted forward in time by a
     * specified delay. The {@code onError} notification from the current {@code Flowable} is not delayed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the delay to shift the source by
     * @param unit
     *            the time unit of {@code delay}
     * @param scheduler
     *            the {@code Scheduler} to use for delaying
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> delay(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return delay(time, unit, scheduler, false);
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} shifted forward in time by a
     * specified delay. If {@code delayError} is {@code true}, error notifications will also be delayed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the delay to shift the source by
     * @param unit
     *            the time unit of {@code delay}
     * @param scheduler
     *            the {@code Scheduler} to use for delaying
     * @param delayError
     *            if {@code true}, the upstream exception is signaled with the given delay, after all preceding normal elements,
     *            if {@code false}, the upstream exception is signaled immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> delay(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");

        return RxJavaPlugins.onAssembly(new FlowableDelay<>(this, Math.max(0L, time), unit, scheduler, delayError));
    }

    /**
     * Returns a {@code Flowable} that delays the subscription to and emissions from the current {@code Flowable} via another
     * {@link Publisher} on a per-item basis.
     * <p>
     * <img width="640" height="450" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delay.oo.v3.png" alt="">
     * <p>
     * <em>Note:</em> the resulting {@code Flowable} will immediately propagate any {@code onError} notification
     * from the current {@code Flowable}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.
     *  All of the other {@code Publisher}s supplied by the functions are consumed
     *  in an unbounded manner (i.e., no backpressure applied to them).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code delay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the subscription delay value type (ignored)
     * @param <V>
     *            the item delay value type (ignored)
     * @param subscriptionIndicator
     *            a function that returns a {@code Publisher} that triggers the subscription to the current {@code Flowable}
     *            once it emits any item
     * @param itemDelayIndicator
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}, which is
     *            then used to delay the emission of that item by the resulting {@code Flowable} until the {@code Publisher}
     *            returned from {@code itemDelay} emits an item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code subscriptionIndicator} and {@code itemDelayIndicator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U, V> Flowable<T> delay(@NonNull Publisher<U> subscriptionIndicator,
            @NonNull Function<? super T, ? extends Publisher<V>> itemDelayIndicator) {
        return delaySubscription(subscriptionIndicator).delay(itemDelayIndicator);
    }

    /**
     * Returns a {@code Flowable} that delays the subscription to this {@link Publisher}
     * until the other {@code Publisher} emits an element or completes normally.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator forwards the backpressure requests to this {@code Publisher} once
     *  the subscription happens and requests {@link Long#MAX_VALUE} from the other {@code Publisher}</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This method does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the value type of the other {@code Publisher}, irrelevant
     * @param subscriptionIndicator the other {@code Publisher} that should trigger the subscription
     *        to this {@code Publisher}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code subscriptionIndicator} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> delaySubscription(@NonNull Publisher<U> subscriptionIndicator) {
        Objects.requireNonNull(subscriptionIndicator, "subscriptionIndicator is null");
        return RxJavaPlugins.onAssembly(new FlowableDelaySubscriptionOther<>(this, subscriptionIndicator));
    }

    /**
     * Returns a {@code Flowable} that delays the subscription to the current {@code Flowable} by a given amount of time.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delaySubscription.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code delaySubscription} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the time to delay the subscription
     * @param unit
     *            the time unit of {@code delay}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> delaySubscription(long time, @NonNull TimeUnit unit) {
        return delaySubscription(time, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that delays the subscription to the current {@code Flowable} by a given amount of time,
     * both waiting and subscribing on a given {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/delaySubscription.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with the backpressure behavior which is determined by the current {@code Flowable}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the time to delay the subscription
     * @param unit
     *            the time unit of {@code delay}
     * @param scheduler
     *            the {@code Scheduler} on which the waiting and subscription will happen
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/delay.html">ReactiveX operators documentation: Delay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> delaySubscription(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return delaySubscription(timer(time, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that reverses the effect of {@link #materialize materialize} by transforming the
     * {@link Notification} objects extracted from the source items via a selector function
     * into their respective {@link Subscriber} signal types.
     * <p>
     * <img width="640" height="335" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/dematerialize.v3.png" alt="">
     * <p>
     * The intended use of the {@code selector} function is to perform a
     * type-safe identity mapping (see example) on a source that is already of type
     * {@code Notification<T>}. The Java language doesn't allow
     * limiting instance methods to a certain generic argument shape, therefore,
     * a function is used to ensure the conversion remains type safe.
     * <p>
     * When the upstream signals an {@link Notification#createOnError(Throwable) onError} or
     * {@link Notification#createOnComplete() onComplete} item, the
     * returned {@code Flowable} cancels of the flow and terminates with that type of terminal event:
     * <pre><code>
     * Flowable.just(createOnNext(1), createOnComplete(), createOnNext(2))
     * .doOnCancel(() -&gt; System.out.println("Canceled!"));
     * .dematerialize(notification -&gt; notification)
     * .test()
     * .assertResult(1);
     * </code></pre>
     * If the upstream signals {@code onError} or {@code onComplete} directly, the flow is terminated
     * with the same event.
     * <pre><code>
     * Flowable.just(createOnNext(1), createOnNext(2))
     * .dematerialize(notification -&gt; notification)
     * .test()
     * .assertResult(1, 2);
     * </code></pre>
     * If this behavior is not desired, the completion can be suppressed by applying {@link #concatWith(Publisher)}
     * with a {@link #never()} source.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code dematerialize} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.2.4 - experimental
     *
     * @param <R> the output value type
     * @param selector function that returns the upstream item and should return a {@code Notification} to signal
     * the corresponding {@code Subscriber} event to the downstream.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/materialize-dematerialize.html">ReactiveX operators documentation: Dematerialize</a>
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    public final <R> Flowable<R> dematerialize(@NonNull Function<@NonNull ? super T, @NonNull Notification<R>> selector) {
        Objects.requireNonNull(selector, "selector is null");
        return RxJavaPlugins.onAssembly(new FlowableDematerialize<>(this, selector));
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct
     * based on {@link Object#equals(Object)} comparison.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinct.v3.png" alt="">
     * <p>
     * It is recommended the elements' class {@code T} in the flow overrides the default {@code Object.equals()} and {@link Object#hashCode()} to provide
     * a meaningful comparison between items as the default Java implementation only considers reference equivalence.
     * <p>
     * By default, {@code distinct()} uses an internal {@link java.util.HashSet} per {@link Subscriber} to remember
     * previously seen items and uses {@link java.util.Set#add(Object)} returning {@code false} as the
     * indicator for duplicates.
     * <p>
     * Note that this internal {@link HashSet} may grow unbounded as items won't be removed from it by
     * the operator. Therefore, using very long or infinite upstream (with very distinct elements) may lead
     * to {@link OutOfMemoryError}.
     * <p>
     * Customizing the retention policy can happen only by providing a custom {@link java.util.Collection} implementation
     * to the {@link #distinct(Function, Supplier)} overload.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinct} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     * @see #distinct(Function)
     * @see #distinct(Function, Supplier)
     */
    @SuppressWarnings({ "rawtypes", "unchecked" })
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> distinct() {
        return distinct((Function)Functions.identity(), Functions.<T>createHashSet());
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct according
     * to a key selector function and based on {@link Object#equals(Object)} comparison of the objects
     * returned by the key selector function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinct.key.v3.png" alt="">
     * <p>
     * It is recommended the keys' class {@code K} overrides the default {@code Object.equals()} and {@link Object#hashCode()} to provide
     * a meaningful comparison between the key objects as the default Java implementation only considers reference equivalence.
     * <p>
     * By default, {@code distinct()} uses an internal {@link java.util.HashSet} per {@link Subscriber} to remember
     * previously seen keys and uses {@link java.util.Set#add(Object)} returning {@code false} as the
     * indicator for duplicates.
     * <p>
     * Note that this internal {@link HashSet} may grow unbounded as keys won't be removed from it by
     * the operator. Therefore, using very long or infinite upstream (with very distinct keys) may lead
     * to {@link OutOfMemoryError}.
     * <p>
     * Customizing the retention policy can happen only by providing a custom {@link java.util.Collection} implementation
     * to the {@link #distinct(Function, Supplier)} overload.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinct} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <K> the key type
     * @param keySelector
     *            a function that projects an emitted item to a key value that is used to decide whether an item
     *            is distinct from another one or not
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     * @see #distinct(Function, Supplier)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K> Flowable<T> distinct(@NonNull Function<? super T, K> keySelector) {
        return distinct(keySelector, Functions.createHashSet());
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct according
     * to a key selector function and based on {@link Object#equals(Object)} comparison of the objects
     * returned by the key selector function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinct.key.v3.png" alt="">
     * <p>
     * It is recommended the keys' class {@code K} overrides the default {@code Object.equals()} and {@link Object#hashCode()} to provide
     * a meaningful comparison between the key objects as the default Java implementation only considers reference equivalence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinct} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <K> the key type
     * @param keySelector
     *            a function that projects an emitted item to a key value that is used to decide whether an item
     *            is distinct from another one or not
     * @param collectionSupplier
     *            function called for each individual {@link Subscriber} to return a {@link Collection} subtype for holding the extracted
     *            keys and whose add() method's return indicates uniqueness.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} or {@code collectionSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K> Flowable<T> distinct(@NonNull Function<? super T, K> keySelector,
            @NonNull Supplier<? extends Collection<? super K>> collectionSupplier) {
        Objects.requireNonNull(keySelector, "keySelector is null");
        Objects.requireNonNull(collectionSupplier, "collectionSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableDistinct<>(this, keySelector, collectionSupplier));
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct from their
     * immediate predecessors based on {@link Object#equals(Object)} comparison.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinctUntilChanged.v3.png" alt="">
     * <p>
     * It is recommended the elements' class {@code T} in the flow overrides the default {@code Object.equals()} to provide
     * a meaningful comparison between items as the default Java implementation only considers reference equivalence.
     * Alternatively, use the {@link #distinctUntilChanged(BiPredicate)} overload and provide a comparison function
     * in case the class {@code T} can't be overridden with custom {@code equals()} or the comparison itself
     * should happen on different terms or properties of the class {@code T}.
     * <p>
     * Note that the operator always retains the latest item from upstream regardless of the comparison result
     * and uses it in the next comparison with the next upstream item.
     * <p>
     * Note that if element type {@code T} in the flow is mutable, the comparison of the previous and current
     * item may yield unexpected results if the items are mutated externally. Common cases are mutable
     * {@link CharSequence}s or {@link List}s where the objects will actually have the same
     * references when they are modified and {@code distinctUntilChanged} will evaluate subsequent items as same.
     * To avoid such situation, it is recommended that mutable data is converted to an immutable one,
     * for example using {@code map(CharSequence::toString)} or {@code map(list -> Collections.unmodifiableList(new ArrayList<>(list)))}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinctUntilChanged} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     * @see #distinctUntilChanged(BiPredicate)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> distinctUntilChanged() {
        return distinctUntilChanged(Functions.identity());
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct from their
     * immediate predecessors, according to a key selector function and based on {@link Object#equals(Object)} comparison
     * of those objects returned by the key selector function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinctUntilChanged.key.v3.png" alt="">
     * <p>
     * It is recommended the keys' class {@code K} overrides the default {@code Object.equals()} to provide
     * a meaningful comparison between the key objects as the default Java implementation only considers reference equivalence.
     * Alternatively, use the {@link #distinctUntilChanged(BiPredicate)} overload and provide a comparison function
     * in case the class {@code K} can't be overridden with custom {@code equals()} or the comparison itself
     * should happen on different terms or properties of the item class {@code T} (for which the keys can be
     * derived via a similar selector).
     * <p>
     * Note that the operator always retains the latest key from upstream regardless of the comparison result
     * and uses it in the next comparison with the next key derived from the next upstream item.
     * <p>
     * Note that if element type {@code T} in the flow is mutable, the comparison of the previous and current
     * item may yield unexpected results if the items are mutated externally. Common cases are mutable
     * {@link CharSequence}s or {@link List}s where the objects will actually have the same
     * references when they are modified and {@code distinctUntilChanged} will evaluate subsequent items as same.
     * To avoid such situation, it is recommended that mutable data is converted to an immutable one,
     * for example using {@code map(CharSequence::toString)} or {@code map(list -> Collections.unmodifiableList(new ArrayList<>(list)))}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinctUntilChanged} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <K> the key type
     * @param keySelector
     *            a function that projects an emitted item to a key value that is used to decide whether an item
     *            is distinct from another one or not
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K> Flowable<T> distinctUntilChanged(@NonNull Function<? super T, K> keySelector) {
        Objects.requireNonNull(keySelector, "keySelector is null");
        return RxJavaPlugins.onAssembly(new FlowableDistinctUntilChanged<>(this, keySelector, ObjectHelper.equalsPredicate()));
    }

    /**
     * Returns a {@code Flowable} that emits all items emitted by the current {@code Flowable} that are distinct from their
     * immediate predecessors when compared with each other via the provided comparator function.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/distinctUntilChanged.v3.png" alt="">
     * <p>
     * Note that the operator always retains the latest item from upstream regardless of the comparison result
     * and uses it in the next comparison with the next upstream item.
     * <p>
     * Note that if element type {@code T} in the flow is mutable, the comparison of the previous and current
     * item may yield unexpected results if the items are mutated externally. Common cases are mutable
     * {@link CharSequence}s or {@link List}s where the objects will actually have the same
     * references when they are modified and {@code distinctUntilChanged} will evaluate subsequent items as same.
     * To avoid such situation, it is recommended that mutable data is converted to an immutable one,
     * for example using {@code map(CharSequence::toString)} or {@code map(list -> Collections.unmodifiableList(new ArrayList<>(list)))}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code distinctUntilChanged} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param comparer the function that receives the previous item and the current item and is
     *                   expected to return {@code true} if the two are equal, thus skipping the current value.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code comparer} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/distinct.html">ReactiveX operators documentation: Distinct</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> distinctUntilChanged(@NonNull BiPredicate<? super T, ? super T> comparer) {
        Objects.requireNonNull(comparer, "comparer is null");
        return RxJavaPlugins.onAssembly(new FlowableDistinctUntilChanged<>(this, Functions.identity(), comparer));
    }

    /**
     * Calls the specified action after this {@code Flowable} signals {@code onError} or {@code onComplete} or gets canceled by
     * the downstream.
     * <p>In case of a race between a terminal event and a cancellation, the provided {@code onFinally} action
     * is executed once per subscription.
     * <p>Note that the {@code onFinally} action is shared between subscriptions and as such
     * should be thread-safe.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doFinally} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Operator-fusion:</b></dt>
     *  <dd>This operator supports normal and conditional {@link Subscriber}s as well as boundary-limited
     *  synchronous or asynchronous queue-fusion.</dd>
     * </dl>
     * <p>History: 2.0.1 - experimental
     * @param onFinally the action called when this {@code Flowable} terminates or gets canceled
     * @throws NullPointerException if {@code onFinally} is {@code null}
     * @return the new {@code Flowable} instance
     * @since 2.1
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doFinally(@NonNull Action onFinally) {
        Objects.requireNonNull(onFinally, "onFinally is null");
        return RxJavaPlugins.onAssembly(new FlowableDoFinally<>(this, onFinally));
    }

    /**
     * Calls the specified consumer with the current item after this item has been emitted to the downstream.
     * <p>Note that the {@code onAfterNext} action is shared between subscriptions and as such
     * should be thread-safe.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doAfterNext} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Operator-fusion:</b></dt>
     *  <dd>This operator supports normal and conditional {@link Subscriber}s as well as boundary-limited
     *  synchronous or asynchronous queue-fusion.</dd>
     * </dl>
     * <p>History: 2.0.1 - experimental
     * @param onAfterNext the {@link Consumer} that will be called after emitting an item from upstream to the downstream
     * @throws NullPointerException if {@code onAfterNext} is {@code null}
     * @return the new {@code Flowable} instance
     * @since 2.1
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doAfterNext(@NonNull Consumer<? super T> onAfterNext) {
        Objects.requireNonNull(onAfterNext, "onAfterNext is null");
        return RxJavaPlugins.onAssembly(new FlowableDoAfterNext<>(this, onAfterNext));
    }

    /**
     * Registers an {@link Action} to be called when this {@link Publisher} invokes either
     * {@link Subscriber#onComplete onComplete} or {@link Subscriber#onError onError}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/finallyDo.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doAfterTerminate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onAfterTerminate
     *            an {@code Action} to be invoked when the current {@code Flowable} finishes
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onAfterTerminate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     * @see #doOnTerminate(Action)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doAfterTerminate(@NonNull Action onAfterTerminate) {
        return doOnEach(Functions.emptyConsumer(), Functions.emptyConsumer(),
                Functions.EMPTY_ACTION, onAfterTerminate);
    }

    /**
     * Calls the cancel {@link Action} if the downstream cancels the sequence.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnUnsubscribe.v3.png" alt="">
     * <p>
     * The action is shared between subscriptions and thus may be called concurrently from multiple
     * threads; the action must be thread-safe.
     * <p>
     * If the action throws a runtime exception, that exception is rethrown by the {@code onCancel()} call,
     * sometimes as a {@link CompositeException} if there were multiple exceptions along the way.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>{@code doOnCancel} does not interact with backpressure requests or value delivery; backpressure
     *  behavior is preserved between its upstream and its downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnCancel} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onCancel
     *            the action that gets called when the current {@code Flowable}'s {@link Subscription} is canceled
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onCancel} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnCancel(@NonNull Action onCancel) {
        return doOnLifecycle(Functions.emptyConsumer(), Functions.EMPTY_LONG_CONSUMER, onCancel);
    }

    /**
     * Invokes an {@link Action} just before the current {@code Flowable} calls {@code onComplete}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnComplete.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnComplete} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onComplete
     *            the action to invoke when the current {@code Flowable} calls {@code onComplete}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onComplete} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnComplete(@NonNull Action onComplete) {
        return doOnEach(Functions.emptyConsumer(), Functions.emptyConsumer(),
                onComplete, Functions.EMPTY_ACTION);
    }

    /**
     * Calls the appropriate onXXX consumer (shared between all subscribers) whenever a signal with the same type
     * passes through, before forwarding them to downstream.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnEach.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnEach} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onNext}, {@code onError}, {@code onComplete} or {@code onAfterTerminate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    private Flowable<T> doOnEach(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError,
            Action onComplete, Action onAfterTerminate) {
        Objects.requireNonNull(onNext, "onNext is null");
        Objects.requireNonNull(onError, "onError is null");
        Objects.requireNonNull(onComplete, "onComplete is null");
        Objects.requireNonNull(onAfterTerminate, "onAfterTerminate is null");
        return RxJavaPlugins.onAssembly(new FlowableDoOnEach<>(this, onNext, onError, onComplete, onAfterTerminate));
    }

    /**
     * Invokes a {@link Consumer} with a {@link Notification} instances matching the signals emitted by the current {@code Flowable}
     * before they are forwarded to the downstream.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnEach.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnEach} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNotification
     *            the action to invoke for each item emitted by the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onNotification} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> doOnEach(@NonNull Consumer<@NonNull ? super Notification<T>> onNotification) {
        Objects.requireNonNull(onNotification, "onNotification is null");
        return doOnEach(
                Functions.notificationOnNext(onNotification),
                Functions.notificationOnError(onNotification),
                Functions.notificationOnComplete(onNotification),
                Functions.EMPTY_ACTION
            );
    }

    /**
     * Calls the appropriate methods of the given {@link Subscriber} when the current {@code Flowable} signals events before forwarding it
     * to the downstream.
     * <p>
     * In case the {@code onError} of the supplied {@code Subscriber} throws, the downstream will receive a composite
     * exception containing the original exception and the exception thrown by {@code onError}. If either the
     * {@code onNext} or the {@code onComplete} method of the supplied {@code Subscriber} throws, the downstream will be
     * terminated and will receive this thrown exception.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnEach.o.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnEach} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param subscriber
     *            the {@code Subscriber} to be notified about {@code onNext}, {@code onError} and {@code onComplete} events on its
     *            respective methods before the actual downstream {@code Subscriber} gets notified.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code subscriber} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> doOnEach(@NonNull Subscriber<@NonNull ? super T> subscriber) {
        Objects.requireNonNull(subscriber, "subscriber is null");
        return doOnEach(
                FlowableInternalHelper.subscriberOnNext(subscriber),
                FlowableInternalHelper.subscriberOnError(subscriber),
                FlowableInternalHelper.subscriberOnComplete(subscriber),
                Functions.EMPTY_ACTION);
    }

    /**
     * Calls the given {@link Consumer} with the error {@link Throwable} if the current {@code Flowable} failed before forwarding it to
     * the downstream.
     * <p>
     * In case the {@code onError} action throws, the downstream will receive a composite exception containing
     * the original exception and the exception thrown by {@code onError}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnError.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onError
     *            the action to invoke if the current {@code Flowable} calls {@code onError}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onError} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnError(@NonNull Consumer<? super Throwable> onError) {
        return doOnEach(Functions.emptyConsumer(), onError,
                Functions.EMPTY_ACTION, Functions.EMPTY_ACTION);
    }

    /**
     * Calls the appropriate {@code onXXX} method (shared between all {@link Subscriber}s) for the lifecycle events of
     * the sequence (subscription, cancellation, requesting).
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnNext.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnLifecycle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onSubscribe
     *              a {@link Consumer} called with the {@link Subscription} sent via {@link Subscriber#onSubscribe(Subscription)}
     * @param onRequest
     *              a {@link LongConsumer} called with the request amount sent via {@link Subscription#request(long)}
     * @param onCancel
     *              called when the downstream cancels the {@code Subscription} via {@link Subscription#cancel()}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onSubscribe}, {@code onRequest} or {@code onCancel} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> doOnLifecycle(@NonNull Consumer<? super Subscription> onSubscribe,
            @NonNull LongConsumer onRequest, @NonNull Action onCancel) {
        Objects.requireNonNull(onSubscribe, "onSubscribe is null");
        Objects.requireNonNull(onRequest, "onRequest is null");
        Objects.requireNonNull(onCancel, "onCancel is null");
        return RxJavaPlugins.onAssembly(new FlowableDoOnLifecycle<>(this, onSubscribe, onRequest, onCancel));
    }

    /**
     * Calls the given {@link Consumer} with the value emitted by the current {@code Flowable} before forwarding it to the downstream.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnNext.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnNext} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *            the action to invoke when the current {@code Flowable} calls {@code onNext}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onNext} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     * @see #doAfterNext(Consumer)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnNext(@NonNull Consumer<? super T> onNext) {
        return doOnEach(onNext, Functions.emptyConsumer(),
                Functions.EMPTY_ACTION, Functions.EMPTY_ACTION);
    }

    /**
     * Calls the given {@link LongConsumer} with the request amount from the downstream before forwarding it
     * to the current {@code Flowable}.
     * <p>
     * <b>Note:</b> This operator is for tracing the internal behavior of back-pressure request
     * patterns and generally intended for debugging use.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     * <dt><b>Scheduler:</b></dt>
     * <dd>{@code doOnRequest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onRequest
     *            the action that gets called when a {@link Subscriber} requests items from the current
     *            {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onRequest} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators
     *      documentation: Do</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnRequest(@NonNull LongConsumer onRequest) {
        return doOnLifecycle(Functions.emptyConsumer(), onRequest, Functions.EMPTY_ACTION);
    }

    /**
     * Calls the given {@link Consumer} with the {@link Subscription} provided by the current {@code Flowable} upon
     * subscription from the downstream before forwarding it to the subscriber's
     * {@link Subscriber#onSubscribe(Subscription) onSubscribe} method.
     * <p>
     * <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnSubscribe.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onSubscribe
     *            the {@code Consumer} that gets called when a {@link Subscriber} subscribes to the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onSubscribe} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnSubscribe(@NonNull Consumer<? super Subscription> onSubscribe) {
        return doOnLifecycle(onSubscribe, Functions.EMPTY_LONG_CONSUMER, Functions.EMPTY_ACTION);
    }

    /**
     * Calls the given {@link Action} when the current {@code Flowable} completes normally or with an error before those signals
     * are forwarded to the downstream.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/doOnTerminate.v3.png" alt="">
     * <p>
     * This differs from {@code doAfterTerminate} in that this happens <em>before</em> the {@code onComplete} or
     * {@code onError} notification.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code doOnTerminate} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onTerminate
     *            the action to invoke when the current {@code Flowable} calls {@code onComplete} or {@code onError}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onTerminate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/do.html">ReactiveX operators documentation: Do</a>
     * @see #doAfterTerminate(Action)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> doOnTerminate(@NonNull Action onTerminate) {
        return doOnEach(Functions.emptyConsumer(), Functions.actionConsumer(onTerminate),
                onTerminate, Functions.EMPTY_ACTION);
    }

    /**
     * Returns a {@link Maybe} that emits the single item at a specified index in a sequence of emissions from
     * this {@code Flowable} or completes if this {@code Flowable} sequence has fewer elements than index.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/elementAt.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code elementAt} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param index
     *            the zero-based index of the item to retrieve
     * @return the new {@code Maybe} instance
     * @throws IndexOutOfBoundsException if {@code index} is negative
     * @see <a href="http://reactivex.io/documentation/operators/elementat.html">ReactiveX operators documentation: ElementAt</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Maybe<T> elementAt(long index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("index >= 0 required but it was " + index);
        }
        return RxJavaPlugins.onAssembly(new FlowableElementAtMaybe<>(this, index));
    }

    /**
     * Returns a {@link Single} that emits the item found at a specified index in a sequence of emissions from
     * this {@code Flowable}, or a default item if that index is out of range.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/elementAtOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code elementAt} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param index
     *            the zero-based index of the item to retrieve
     * @param defaultItem
     *            the default item
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @throws IndexOutOfBoundsException
     *             if {@code index} is negative
     * @see <a href="http://reactivex.io/documentation/operators/elementat.html">ReactiveX operators documentation: ElementAt</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<T> elementAt(long index, @NonNull T defaultItem) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("index >= 0 required but it was " + index);
        }
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        return RxJavaPlugins.onAssembly(new FlowableElementAtSingle<>(this, index, defaultItem));
    }

    /**
     * Returns a {@link Single} that emits the item found at a specified index in a sequence of emissions from
     * this {@code Flowable} or signals a {@link NoSuchElementException} if this {@code Flowable} has fewer elements than index.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/elementAtOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code elementAtOrError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param index
     *            the zero-based index of the item to retrieve
     * @return the new {@code Single} instance
     * @throws IndexOutOfBoundsException
     *             if {@code index} is less than 0
     * @see <a href="http://reactivex.io/documentation/operators/elementat.html">ReactiveX operators documentation: ElementAt</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<T> elementAtOrError(long index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("index >= 0 required but it was " + index);
        }
        return RxJavaPlugins.onAssembly(new FlowableElementAtSingle<>(this, index, null));
    }

    /**
     * Filters items emitted by the current {@code Flowable} by only emitting those that satisfy a specified predicate.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/filter.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code filter} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            a function that evaluates each item emitted by the current {@code Flowable}, returning {@code true}
     *            if it passes the filter
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/filter.html">ReactiveX operators documentation: Filter</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> filter(@NonNull Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");
        return RxJavaPlugins.onAssembly(new FlowableFilter<>(this, predicate));
    }

    /**
     * Returns a {@link Maybe} that emits only the very first item emitted by this {@code Flowable} or
     * completes if this {@code Flowable} is empty.
     * <p>
     * <img width="640" height="286" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/firstElement.m.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code firstElement} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Maybe} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Maybe<T> firstElement() {
        return elementAt(0);
    }

    /**
     * Returns a {@link Single} that emits only the very first item emitted by this {@code Flowable}, or a default
     * item if this {@code Flowable} completes without emitting anything.
     * <p>
     * <img width="640" height="285" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/first.s.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code first} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param defaultItem
     *            the default item to emit if the current {@code Flowable} doesn't emit anything
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<T> first(@NonNull T defaultItem) {
        return elementAt(0, defaultItem);
    }

    /**
     * Returns a {@link Single} that emits only the very first item emitted by this {@code Flowable} or
     * signals a {@link NoSuchElementException} if this {@code Flowable} is empty.
     * <p>
     * <img width="640" height="237" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/firstOrError.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code firstOrError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Single} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL) // take may trigger UNBOUNDED_IN
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<T> firstOrError() {
        return elementAtOrError(0);
    }

    /**
     * Returns a {@code Flowable} that emits items based on applying a function that you supply to each item emitted
     * by the current {@code Flowable}, where that function returns a {@link Publisher}, and then merging those resulting
     * {@code Publisher}s and emitting the results of this merger.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@link #bufferSize()} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the inner {@code Publisher}s and the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return flatMap(mapper, false, bufferSize(), bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits items based on applying a function that you supply to each item emitted
     * by the current {@code Flowable}, where that function returns a {@link Publisher}, and then merging those resulting
     * {@code Publisher}s and emitting the results of this merger.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@link #bufferSize()} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the inner {@code Publisher}s and the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, boolean delayErrors) {
        return flatMap(mapper, delayErrors, bufferSize(), bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits items based on applying a function that you supply to each item emitted
     * by the current {@code Flowable}, where that function returns a {@link Publisher}, and then merging those resulting
     * {@code Publisher}s and emitting the results of this merger, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMap.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the inner {@code Publisher}s and the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int maxConcurrency) {
        return flatMap(mapper, false, maxConcurrency, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits items based on applying a function that you supply to each item emitted
     * by the current {@code Flowable}, where that function returns a {@link Publisher}, and then merging those resulting
     * {@code Publisher}s and emitting the results of this merger, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMap.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the inner {@code Publisher}s and the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, boolean delayErrors, int maxConcurrency) {
        return flatMap(mapper, delayErrors, maxConcurrency, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits items based on applying a function that you supply to each item emitted
     * by the current {@code Flowable}, where that function returns a {@link Publisher}, and then merging those resulting
     * {@code Publisher}s and emitting the results of this merger, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMap.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the value type of the inner {@code Publisher}s and the output type
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @param bufferSize
     *            the number of elements to prefetch from each inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper,
            boolean delayErrors, int maxConcurrency, int bufferSize) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        if (this instanceof ScalarSupplier) {
            @SuppressWarnings("unchecked")
            T v = ((ScalarSupplier<T>)this).get();
            if (v == null) {
                return empty();
            }
            return FlowableScalarXMap.scalarXMap(v, mapper);
        }
        return RxJavaPlugins.onAssembly(new FlowableFlatMap<>(this, mapper, delayErrors, maxConcurrency, bufferSize));
    }

    /**
     * Returns a {@code Flowable} that applies a function to each item emitted or notification raised by the current
     * {@code Flowable} and then flattens the {@link Publisher}s returned from these functions and emits the resulting items.
     * <p>
     * <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.nce.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@link #bufferSize()} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the result type
     * @param onNextMapper
     *            a function that returns a {@code Publisher} to merge for each item emitted by the current {@code Flowable}
     * @param onErrorMapper
     *            a function that returns a {@code Publisher} to merge for an {@code onError} notification from the current
     *            {@code Flowable}
     * @param onCompleteSupplier
     *            a function that returns a {@code Publisher} to merge for an {@code onComplete} notification from the current
     *            {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onNextMapper}, {@code onErrorMapper} or {@code onCompleteSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> flatMap(
            @NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> onNextMapper,
            @NonNull Function<? super Throwable, ? extends Publisher<@NonNull ? extends R>> onErrorMapper,
            @NonNull Supplier<? extends Publisher<@NonNull ? extends R>> onCompleteSupplier) {
        Objects.requireNonNull(onNextMapper, "onNextMapper is null");
        Objects.requireNonNull(onErrorMapper, "onErrorMapper is null");
        Objects.requireNonNull(onCompleteSupplier, "onCompleteSupplier is null");
        return merge(new FlowableMapNotification<>(this, onNextMapper, onErrorMapper, onCompleteSupplier));
    }

    /**
     * Returns a {@code Flowable} that applies a function to each item emitted or notification raised by the current
     * {@code Flowable} and then flattens the {@link Publisher}s returned from these functions and emits the resulting items,
     * while limiting the maximum number of concurrent subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.nce.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the result type
     * @param onNextMapper
     *            a function that returns a {@code Publisher} to merge for each item emitted by the current {@code Flowable}
     * @param onErrorMapper
     *            a function that returns a {@code Publisher} to merge for an {@code onError} notification from the current
     *            {@code Flowable}
     * @param onCompleteSupplier
     *            a function that returns a {@code Publisher} to merge for an {@code onComplete} notification from the current
     *            {@code Flowable}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onNextMapper}, {@code onErrorMapper} or {@code onCompleteSupplier} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> flatMap(
            @NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> onNextMapper,
            @NonNull Function<Throwable, ? extends Publisher<@NonNull ? extends R>> onErrorMapper,
            @NonNull Supplier<? extends Publisher<@NonNull ? extends R>> onCompleteSupplier,
            int maxConcurrency) {
        Objects.requireNonNull(onNextMapper, "onNextMapper is null");
        Objects.requireNonNull(onErrorMapper, "onErrorMapper is null");
        Objects.requireNonNull(onCompleteSupplier, "onCompleteSupplier is null");
        return merge(new FlowableMapNotification<>(
                this, onNextMapper, onErrorMapper, onCompleteSupplier), maxConcurrency);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified function to the pair of values emitted by the
     * current {@code Flowable} and a specified collection {@link Publisher}.
     * <p>
     * <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.r.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of items emitted by the inner {@code Publisher}s
     * @param <R>
     *            the type of items emitted by the combiner function
     * @param mapper
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}
     * @param combiner
     *            a function that combines one item emitted by each of the source and collection {@code Publisher}s and
     *            returns an item to be emitted by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U, R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends R> combiner) {
        return flatMap(mapper, combiner, false, bufferSize(), bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified function to the pair of values emitted by the
     * current {@code Flowable} and a specified inner {@link Publisher}.
     * <p>
     * <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.r.v3.png" alt="">
     * <dl>
     * <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@link #bufferSize()} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of items emitted by the inner {@code Publisher}s
     * @param <R>
     *            the type of items emitted by the combiner functions
     * @param mapper
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}
     * @param combiner
     *            a function that combines one item emitted by each of the source and collection {@code Publisher}s and
     *            returns an item to be emitted by the resulting {@code Flowable}
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U, R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends R> combiner, boolean delayErrors) {
        return flatMap(mapper, combiner, delayErrors, bufferSize(), bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified function to the pair of values emitted by the
     * current {@code Flowable} and a specified collection {@link Publisher}, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.r.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of items emitted by the inner {@code Publisher}s
     * @param <R>
     *            the type of items emitted by the combiner function
     * @param mapper
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}
     * @param combiner
     *            a function that combines one item emitted by each of the source and collection {@code Publisher}s and
     *            returns an item to be emitted by the resulting {@code Flowable}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U, R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends R> combiner, boolean delayErrors, int maxConcurrency) {
        return flatMap(mapper, combiner, delayErrors, maxConcurrency, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified function to the pair of values emitted by the
     * current {@code Flowable} and a specified collection {@link Publisher}, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.r.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@code maxConcurrency} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of items emitted by the inner {@code Publisher}s
     * @param <R>
     *            the type of items emitted by the combiner function
     * @param mapper
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}
     * @param combiner
     *            a function that combines one item emitted by each of the source and collection {@code Publisher}s and
     *            returns an item to be emitted by the resulting {@code Flowable}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @param delayErrors
     *            if {@code true}, exceptions from the current {@code Flowable} and all inner {@code Publisher}s are delayed until all of them terminate
     *            if {@code false}, the first one signaling an exception will terminate the whole sequence immediately
     * @param bufferSize
     *            the number of elements to prefetch from the inner {@code Publisher}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U, R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends R> combiner, boolean delayErrors, int maxConcurrency, int bufferSize) {
        Objects.requireNonNull(mapper, "mapper is null");
        Objects.requireNonNull(combiner, "combiner is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return flatMap(FlowableInternalHelper.flatMapWithCombiner(mapper, combiner), delayErrors, maxConcurrency, bufferSize);
    }

    /**
     * Returns a {@code Flowable} that emits the results of a specified function to the pair of values emitted by the
     * current {@code Flowable} and a specified collection {@link Publisher}, while limiting the maximum number of concurrent
     * subscriptions to these {@code Publisher}s.
     * <!-- <p> -->
     * <!-- <img width="640" height="390" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/mergeMap.r.v3.png" alt=""> -->
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The upstream {@code Flowable} is consumed
     *  in a bounded manner (up to {@link #bufferSize()} outstanding request amount for items).
     *  The inner {@code Publisher}s are expected to honor backpressure; if violated,
     *  the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of items emitted by the inner {@code Publisher}s
     * @param <R>
     *            the type of items emitted by the combiner function
     * @param mapper
     *            a function that returns a {@code Publisher} for each item emitted by the current {@code Flowable}
     * @param combiner
     *            a function that combines one item emitted by each of the source and collection {@code Publisher}s and
     *            returns an item to be emitted by the resulting {@code Flowable}
     * @param maxConcurrency
     *         the maximum number of {@code Publisher}s that may be subscribed to concurrently
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U, R> Flowable<R> flatMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends R> combiner, int maxConcurrency) {
        return flatMap(mapper, combiner, false, maxConcurrency, bufferSize());
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link CompletableSource}s, subscribes to them and
     * waits until the upstream and all {@code CompletableSource}s complete.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the upstream in an unbounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param mapper the function that received each source value and transforms them into {@code CompletableSource}s.
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Completable flatMapCompletable(@NonNull Function<? super T, ? extends CompletableSource> mapper) {
        return flatMapCompletable(mapper, false, Integer.MAX_VALUE);
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link CompletableSource}s, subscribes to them and
     * waits until the upstream and all {@code CompletableSource}s complete, optionally delaying all errors.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>If {@code maxConcurrency == }{@link Integer#MAX_VALUE} the operator consumes the upstream in an unbounded manner.
     *  Otherwise, the operator expects the upstream to honor backpressure. If the upstream doesn't support backpressure
     *  the operator behaves as if {@code maxConcurrency == }{@link Integer#MAX_VALUE} was used.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param mapper the function that received each source value and transforms them into {@code CompletableSource}s.
     * @param delayErrors if {@code true}, errors from the upstream and inner {@code CompletableSource}s are delayed until each of them
     * terminates.
     * @param maxConcurrency the maximum number of active subscriptions to the {@code CompletableSource}s.
     * @return the new {@link Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Completable flatMapCompletable(@NonNull Function<? super T, ? extends CompletableSource> mapper, boolean delayErrors, int maxConcurrency) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        return RxJavaPlugins.onAssembly(new FlowableFlatMapCompletableCompletable<>(this, mapper, delayErrors, maxConcurrency));
    }

    /**
     * Merges {@link Iterable}s generated by a mapper {@link Function} for each individual item emitted by
     * the current {@code Flowable} into a single {@code Flowable} sequence.
     * <p>
     * <img width="640" height="342" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMapIterable.f.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}s is
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the output type and the element type of the {@code Iterable}s
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for when given an item emitted by the
     *            current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <U> Flowable<U> flatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper) {
        return flatMapIterable(mapper, bufferSize());
    }

    /**
     * Merges {@link Iterable}s generated by a mapper {@link Function} for each individual item emitted by
     * the current {@code Flowable} into a single {@code Flowable} sequence.
     * <p>
     * <img width="640" height="342" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMapIterable.f.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}s is
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the type of item emitted by the resulting {@code Iterable}
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for when given an item emitted by the
     *            current {@code Flowable}
     * @param bufferSize
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<U> flatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper, int bufferSize) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableFlattenIterable<>(this, mapper, bufferSize));
    }

    /**
     * Merges {@link Iterable}s generated by a mapper {@link Function} for each individual item emitted by
     * the current {@code Flowable} into a single {@code Flowable} sequence where the resulting items will
     * be the combination of the original item and each inner item of the respective {@code Iterable} as returned
     * by the {@code resultSelector} {@link BiFunction}.
     * <p>
     * <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMapIterable.f.r.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and the current {@code Flowable}s is
     *  consumed in a bounded manner (requesting {@link #bufferSize()} items upfront, then 75% of it after 75% received).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the element type of the {@code Iterable}s
     * @param <V>
     *            the output type as determined by the {@code resultSelector} function
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for each item emitted by the current
     *            {@code Flowable}
     * @param combiner
     *            a function that returns an item based on the item emitted by the current {@code Flowable} and the
     *            {@code Iterable} returned for that item by the {@code collectionSelector}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U, V> Flowable<V> flatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends V> combiner) {
        Objects.requireNonNull(mapper, "mapper is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return flatMap(FlowableInternalHelper.flatMapIntoIterable(mapper), combiner, false, bufferSize(), bufferSize());
    }

    /**
     * Merges {@link Iterable}s generated by a mapper {@link Function} for each individual item emitted by
     * the current {@code Flowable} into a single {@code Flowable} sequence where the resulting items will
     * be the combination of the original item and each inner item of the respective {@code Iterable} as returned
     * by the {@code resultSelector} {@link BiFunction}.
     * <p>
     * <img width="640" height="410" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flatMapIterable.f.r.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}s is
     *  expected to honor backpressure as well. If the current {@code Flowable} violates the rule, the operator will
     *  signal a {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U>
     *            the element type of the inner {@code Iterable} sequences
     * @param <V>
     *            the type of item emitted by the resulting {@code Flowable}
     * @param mapper
     *            a function that returns an {@code Iterable} sequence of values for when given an item emitted by the
     *            current {@code Flowable}
     * @param combiner
     *            a function that returns an item based on the item emitted by the current {@code Flowable} and the
     *            {@code Iterable} returned for that item by the {@code collectionSelector}
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} or {@code combiner} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U, V> Flowable<V> flatMapIterable(@NonNull Function<? super T, ? extends Iterable<@NonNull ? extends U>> mapper,
            @NonNull BiFunction<? super T, ? super U, ? extends V> combiner, int prefetch) {
        Objects.requireNonNull(mapper, "mapper is null");
        Objects.requireNonNull(combiner, "combiner is null");
        return flatMap(FlowableInternalHelper.flatMapIntoIterable(mapper), combiner, false, bufferSize(), prefetch);
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link MaybeSource}s, subscribes to all of them
     * and merges their {@code onSuccess} values, in no particular order, into a single {@code Flowable} sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the upstream in an unbounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the result value type
     * @param mapper the function that received each source value and transforms them into {@code MaybeSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMapMaybe(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper) {
        return flatMapMaybe(mapper, false, Integer.MAX_VALUE);
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link MaybeSource}s, subscribes to at most
     * {@code maxConcurrency} {@code MaybeSource}s at a time and merges their {@code onSuccess} values,
     * in no particular order, into a single {@code Flowable} sequence, optionally delaying all errors.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>If {@code maxConcurrency == }{@link Integer#MAX_VALUE} the operator consumes the upstream in an unbounded manner.
     *  Otherwise, the operator expects the upstream to honor backpressure. If the upstream doesn't support backpressure
     *  the operator behaves as if {@code maxConcurrency == }{@link Integer#MAX_VALUE} was used.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the result value type
     * @param mapper the function that received each source value and transforms them into {@code MaybeSource}s.
     * @param delayErrors if {@code true}, errors from the upstream and inner {@code MaybeSource}s are delayed until each of them
     * terminates.
     * @param maxConcurrency the maximum number of active subscriptions to the {@code MaybeSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> flatMapMaybe(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper, boolean delayErrors, int maxConcurrency) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        return RxJavaPlugins.onAssembly(new FlowableFlatMapMaybe<>(this, mapper, delayErrors, maxConcurrency));
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link SingleSource}s, subscribes to all of them
     * and merges their {@code onSuccess} values, in no particular order, into a single {@code Flowable} sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the upstream in an unbounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the result value type
     * @param mapper the function that received each source value and transforms them into {@code SingleSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> flatMapSingle(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper) {
        return flatMapSingle(mapper, false, Integer.MAX_VALUE);
    }

    /**
     * Maps each element of the upstream {@code Flowable} into {@link SingleSource}s, subscribes to at most
     * {@code maxConcurrency} {@code SingleSource}s at a time and merges their {@code onSuccess} values,
     * in no particular order, into a single {@code Flowable} sequence, optionally delaying all errors.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>If {@code maxConcurrency == }{@link Integer#MAX_VALUE} the operator consumes the upstream in an unbounded manner.
     *  Otherwise, the operator expects the upstream to honor backpressure. If the upstream doesn't support backpressure
     *  the operator behaves as if {@code maxConcurrency == }{@link Integer#MAX_VALUE} was used.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code flatMapSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the result value type
     * @param mapper the function that received each source value and transforms them into {@code SingleSource}s.
     * @param delayErrors if {@code true}, errors from the upstream and inner {@code SingleSources} are delayed until each of them
     * terminates.
     * @param maxConcurrency the maximum number of active subscriptions to the {@code SingleSource}s.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code maxConcurrency} is non-positive
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> flatMapSingle(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper, boolean delayErrors, int maxConcurrency) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(maxConcurrency, "maxConcurrency");
        return RxJavaPlugins.onAssembly(new FlowableFlatMapSingle<>(this, mapper, delayErrors, maxConcurrency));
    }

    /**
     * Subscribes to the current {@code Flowable} and receives notifications for each element.
     * <p>
     * Alias to {@link #subscribe(Consumer)}
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code forEach} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *            {@link Consumer} to execute for each item.
     * @return
     *            a {@link Disposable} that allows canceling an asynchronous sequence
     * @throws NullPointerException
     *             if {@code onNext} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.NONE)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable forEach(@NonNull Consumer<? super T> onNext) {
        return subscribe(onNext);
    }

    /**
     * Subscribes to the current {@code Flowable} and receives notifications for each element until the
     * {@code onNext} Predicate returns {@code false}.
     * <p>
     * If the {@code Flowable} emits an error, it is wrapped into an
     * {@link io.reactivex.rxjava3.exceptions.OnErrorNotImplementedException OnErrorNotImplementedException}
     * and routed to the {@link RxJavaPlugins#onError(Throwable)} handler.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code forEachWhile} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *            {@link Predicate} to execute for each item.
     * @return
     *            a {@link Disposable} that allows canceling an asynchronous sequence
     * @throws NullPointerException
     *             if {@code onNext} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.NONE)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable forEachWhile(@NonNull Predicate<? super T> onNext) {
        return forEachWhile(onNext, Functions.ON_ERROR_MISSING, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the current {@code Flowable} and receives notifications for each element and error events until the
     * {@code onNext} Predicate returns {@code false}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code forEachWhile} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *            {@link Predicate} to execute for each item.
     * @param onError
     *            {@link Consumer} to execute when an error is emitted.
     * @return
     *            a {@link Disposable} that allows canceling an asynchronous sequence
     * @throws NullPointerException
     *             if {@code onNext} or {@code onError} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.NONE)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable forEachWhile(@NonNull Predicate<? super T> onNext, @NonNull Consumer<? super Throwable> onError) {
        return forEachWhile(onNext, onError, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the current {@code Flowable} and receives notifications for each element and the terminal events until the
     * {@code onNext} Predicate returns {@code false}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code forEachWhile} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *            {@link Predicate} to execute for each item.
     * @param onError
     *            {@link Consumer} to execute when an error is emitted.
     * @param onComplete
     *            {@link Action} to execute when completion is signaled.
     * @return
     *            a {@link Disposable} that allows canceling an asynchronous sequence
     * @throws NullPointerException
     *             if {@code onNext}, {@code onError} or {@code onComplete} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.NONE)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Disposable forEachWhile(@NonNull Predicate<? super T> onNext, @NonNull Consumer<? super Throwable> onError,
            @NonNull Action onComplete) {
        Objects.requireNonNull(onNext, "onNext is null");
        Objects.requireNonNull(onError, "onError is null");
        Objects.requireNonNull(onComplete, "onComplete is null");

        ForEachWhileSubscriber<T> s = new ForEachWhileSubscriber<>(onNext, onError, onComplete);
        subscribe(s);
        return s;
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission.
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     *
     * @param keySelector
     *            a function that extracts the key for each item
     * @param <K>
     *            the key type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     * @see #groupBy(Function, boolean)
     * @see #groupBy(Function, Function)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K> Flowable<GroupedFlowable<K, T>> groupBy(@NonNull Function<? super T, ? extends K> keySelector) {
        return groupBy(keySelector, Functions.identity(), false, bufferSize());
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission.
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     *
     * @param keySelector
     *            a function that extracts the key for each item
     * @param <K>
     *            the key type
     * @param delayError
     *            if {@code true}, the exception from the current {@code Flowable} is delayed in each group until that specific group emitted
     *            the normal values; if {@code false}, the exception bypasses values in the groups and is reported immediately.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K> Flowable<GroupedFlowable<K, T>> groupBy(@NonNull Function<? super T, ? extends K> keySelector, boolean delayError) {
        return groupBy(keySelector, Functions.identity(), delayError, bufferSize());
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission.
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     *
     * @param keySelector
     *            a function that extracts the key for each item
     * @param valueSelector
     *            a function that extracts the return element for each item
     * @param <K>
     *            the key type
     * @param <V>
     *            the element type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} or {@code valueSelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     * @see #groupBy(Function, Function, boolean)
     * @see #groupBy(Function, Function, boolean, int)
     * @see #groupBy(Function, Function, boolean, int, Function)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K, V> Flowable<GroupedFlowable<K, V>> groupBy(@NonNull Function<? super T, ? extends K> keySelector,
            @NonNull Function<? super T, ? extends V> valueSelector) {
        return groupBy(keySelector, valueSelector, false, bufferSize());
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission.
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     *
     * @param keySelector
     *            a function that extracts the key for each item
     * @param valueSelector
     *            a function that extracts the return element for each item
     * @param <K>
     *            the key type
     * @param <V>
     *            the element type
     * @param delayError
     *            if {@code true}, the exception from the current {@code Flowable} is delayed in each group until that specific group emitted
     *            the normal values; if {@code false}, the exception bypasses values in the groups and is reported immediately.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} or {@code valueSelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     * @see #groupBy(Function, Function, boolean, int)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <K, V> Flowable<GroupedFlowable<K, V>> groupBy(@NonNull Function<? super T, ? extends K> keySelector,
            @NonNull Function<? super T, ? extends V> valueSelector, boolean delayError) {
        return groupBy(keySelector, valueSelector, delayError, bufferSize());
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission.
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     *
     * @param keySelector
     *            a function that extracts the key for each item
     * @param valueSelector
     *            a function that extracts the return element for each item
     * @param delayError
     *            if {@code true}, the exception from the current {@code Flowable} is delayed in each group until that specific group emitted
     *            the normal values; if {@code false}, the exception bypasses values in the groups and is reported immediately.
     * @param bufferSize
     *            the hint for how many {@code GroupedFlowable}s and element in each {@code GroupedFlowable} should be buffered
     * @param <K>
     *            the key type
     * @param <V>
     *            the element type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector} or {@code valueSelector} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <K, V> Flowable<GroupedFlowable<K, V>> groupBy(@NonNull Function<? super T, ? extends K> keySelector,
            @NonNull Function<? super T, ? extends V> valueSelector,
            boolean delayError, int bufferSize) {
        Objects.requireNonNull(keySelector, "keySelector is null");
        Objects.requireNonNull(valueSelector, "valueSelector is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");

        return RxJavaPlugins.onAssembly(new FlowableGroupBy<>(this, keySelector, valueSelector, bufferSize, delayError, null));
    }

    /**
     * Groups the items emitted by the current {@code Flowable} according to a specified criterion, and emits these
     * grouped items as {@link GroupedFlowable}s. The emitted {@code GroupedFlowable} allows only a single
     * {@link Subscriber} during its lifetime and if this {@code Subscriber} cancels before the
     * source terminates, the next emission by the source having the same key will trigger a new
     * {@code GroupedFlowable} emission. The {@code evictingMapFactory} is used to create a map that will
     * be used to hold the {@code GroupedFlowable}s by key. The evicting map created by this factory must
     * notify the provided {@code Consumer<Object>} with the entry value (not the key!) when an entry in this
     * map has been evicted. The next source emission will bring about the completion of the evicted
     * {@code GroupedFlowable}s and the arrival of an item with the same key as a completed {@code GroupedFlowable}
     * will prompt the creation and emission of a new {@code GroupedFlowable} with that key.
     *
     * <p>A use case for specifying an {@code evictingMapFactory} is where the source is infinite and fast and
     * over time the number of keys grows enough to be a concern in terms of the memory footprint of the
     * internal hash map containing the {@code GroupedFlowable}s.
     *
     * <p>The map created by an {@code evictingMapFactory} must be thread-safe.
     *
     * <p>An example of an {@code evictingMapFactory} using <a href="https://google.github.io/guava/releases/24.0-jre/api/docs/com/google/common/cache/CacheBuilder.html">CacheBuilder</a> from the Guava library is below:
     *
     * <pre><code>
     * Function&lt;Consumer&lt;Object&gt;, Map&lt;Integer, Object&gt;&gt; evictingMapFactory =
     *   notify -&gt;
     *       CacheBuilder
     *         .newBuilder()
     *         .maximumSize(3)
     *         .removalListener(entry -&gt; {
     *              try {
     *                  // emit the value not the key!
     *                  notify.accept(entry.getValue());
     *              } catch (Exception e) {
     *                  throw new RuntimeException(e);
     *              }
     *            })
     *         .&lt;Integer, Object&gt; build()
     *         .asMap();
     *
     * // Emit 1000 items but ensure that the
     * // internal map never has more than 3 items in it
     *   Flowable
     *   .range(1, 1000)
     *   // note that number of keys is 10
     *   .groupBy(x -&gt; x % 10, x -&gt; x, true, 16, evictingMapFactory)
     *   .flatMap(g -&gt; g)
     *   .forEach(System.out::println);
     * </code></pre>
     *
     * <p>
     * <img width="640" height="360" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupBy.v3.png" alt="">
     * <p>
     * <em>Note:</em> A {@code GroupedFlowable} will cache the items it is to emit until such time as it
     * is subscribed to. For this reason, in order to avoid memory leaks, you should not simply ignore those
     * {@code GroupedFlowable}s that do not concern you. Instead, you can signal to them that they may
     * discard their buffers by applying an operator like {@link #ignoreElements} to them.
     * <p>
     * Note that the {@code GroupedFlowable}s should be subscribed to as soon as possible, otherwise,
     * the unconsumed groups may starve other groups due to the internal backpressure
     * coordination of the {@code groupBy} operator. Such hangs can be usually avoided by using
     * {@link #flatMap(Function, int)} or {@link #concatMapEager(Function, int, int)} and overriding the default maximum concurrency
     * value to be greater or equal to the expected number of groups, possibly using
     * {@link Integer#MAX_VALUE} if the number of expected groups is unknown.
     * <p>
     * Note also that ignoring groups or subscribing later (i.e., on another thread) will result in
     * so-called group abandonment where a group will only contain one element and the group will be
     * re-created over and over as new upstream items trigger a new group. The behavior is
     * a trade-off between no-dataloss, upstream cancellation and excessive group creation.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The consumer of the returned {@code Flowable} has to be ready to receive new {@code GroupedFlowable}s or else
     *  this operator will signal {@link MissingBackpressureException}. To avoid this exception, make
     *  sure a combining operator (such as {@code flatMap}) has adequate amount of buffering/prefetch configured.
     *  The inner {@code GroupedFlowable}s honor backpressure but due to the single-source multiple consumer
     *  nature of this operator, each group must be consumed so the whole operator can make progress and not hang.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupBy} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If the upstream signals or the callback(s) throw an exception, the returned {@code Flowable} and
     *  all active inner {@code GroupedFlowable}s will signal the same exception.</dd>
     * </dl>
     * <p>History: 2.1.10 - beta
     * @param keySelector
     *            a function that extracts the key for each item
     * @param valueSelector
     *            a function that extracts the return element for each item
     * @param delayError
     *            if {@code true}, the exception from the current {@code Flowable} is delayed in each group until that specific group emitted
     *            the normal values; if {@code false}, the exception bypasses values in the groups and is reported immediately.
     * @param bufferSize
     *            the hint for how many {@code GroupedFlowable}s and element in each {@code GroupedFlowable} should be buffered
     * @param evictingMapFactory
     *            The factory used to create a map that will be used by the implementation to hold the
     *            {@code GroupedFlowable}s. The evicting map created by this factory must
     *            notify the provided {@code Consumer<Object>} with the entry value (not the key!) when
     *            an entry in this map has been evicted. The next source emission will bring about the
     *            completion of the evicted {@code GroupedFlowable}s. See example above.
     * @param <K>
     *            the key type
     * @param <V>
     *            the element type
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code keySelector}, {@code valueSelector} or {@code evictingMapFactory} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/groupby.html">ReactiveX operators documentation: GroupBy</a>
     *
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <K, V> Flowable<GroupedFlowable<K, V>> groupBy(@NonNull Function<? super T, ? extends K> keySelector,
            @NonNull Function<? super T, ? extends V> valueSelector,
            boolean delayError, int bufferSize,
            @NonNull Function<? super Consumer<Object>, ? extends Map<K, Object>> evictingMapFactory) {
        Objects.requireNonNull(keySelector, "keySelector is null");
        Objects.requireNonNull(valueSelector, "valueSelector is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        Objects.requireNonNull(evictingMapFactory, "evictingMapFactory is null");

        return RxJavaPlugins.onAssembly(new FlowableGroupBy<>(this, keySelector, valueSelector, bufferSize, delayError, evictingMapFactory));
    }

    /**
     * Returns a {@code Flowable} that correlates two {@link Publisher}s when they overlap in time and groups the results.
     * <p>
     * There are no guarantees in what order the items get combined when multiple
     * items from one or both source {@code Publisher}s overlap.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/groupJoin.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure and consumes all participating {@code Publisher}s in
     *  an unbounded mode (i.e., not applying any backpressure to them).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code groupJoin} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <TRight> the value type of the right {@code Publisher} source
     * @param <TLeftEnd> the element type of the left duration {@code Publisher}s
     * @param <TRightEnd> the element type of the right duration {@code Publisher}s
     * @param <R> the result type
     * @param other
     *            the other {@code Publisher} to correlate items from the current {@code Flowable} with
     * @param leftEnd
     *            a function that returns a {@code Publisher} whose emissions indicate the duration of the values of
     *            the current {@code Flowable}
     * @param rightEnd
     *            a function that returns a {@code Publisher} whose emissions indicate the duration of the values of
     *            the {@code right} {@code Publisher}
     * @param resultSelector
     *            a function that takes an item emitted by each {@code Publisher} and returns the value to be emitted
     *            by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other}, {@code leftEnd}, {@code rightEnd} or {@code resultSelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/join.html">ReactiveX operators documentation: Join</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <TRight, TLeftEnd, TRightEnd, R> Flowable<R> groupJoin(
            @NonNull Publisher<@NonNull ? extends TRight> other,
            @NonNull Function<? super T, ? extends Publisher<@NonNull TLeftEnd>> leftEnd,
            @NonNull Function<? super TRight, ? extends Publisher<@NonNull TRightEnd>> rightEnd,
            @NonNull BiFunction<? super T, ? super Flowable<TRight>, ? extends R> resultSelector) {
        Objects.requireNonNull(other, "other is null");
        Objects.requireNonNull(leftEnd, "leftEnd is null");
        Objects.requireNonNull(rightEnd, "rightEnd is null");
        Objects.requireNonNull(resultSelector, "resultSelector is null");
        return RxJavaPlugins.onAssembly(new FlowableGroupJoin<>(
                this, other, leftEnd, rightEnd, resultSelector));
    }

    /**
     * Hides the identity of this {@code Flowable} and its {@link Subscription}.
     * <p>Allows hiding extra features such as {@link Processor}'s
     * {@link Subscriber} methods or preventing certain identity-based
     * optimizations (fusion).
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator is a pass-through for backpressure, the behavior is determined by the upstream's
     *  backpressure behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code hide} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @return the new {@code Flowable} instance
     *
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> hide() {
        return RxJavaPlugins.onAssembly(new FlowableHide<>(this));
    }

    /**
     * Ignores all items emitted by the current {@code Flowable} and only calls {@code onComplete} or {@code onError}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/ignoreElements.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator ignores backpressure as it doesn't emit any elements and consumes the current {@code Flowable}
     *  in an unbounded manner (i.e., no backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code ignoreElements} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@link Completable} instance
     * @see <a href="http://reactivex.io/documentation/operators/ignoreelements.html">ReactiveX operators documentation: IgnoreElements</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Completable ignoreElements() {
        return RxJavaPlugins.onAssembly(new FlowableIgnoreElementsCompletable<>(this));
    }

    /**
     * Returns a {@link Single} that emits {@code true} if the current {@code Flowable} is empty, otherwise {@code false}.
     * <p>
     * In Rx.Net this is negated as the {@code any} {@link Subscriber} but we renamed this in RxJava to better match Java
     * naming idioms.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/isEmpty.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code isEmpty} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Single} instance
     * @see <a href="http://reactivex.io/documentation/operators/contains.html">ReactiveX operators documentation: Contains</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<Boolean> isEmpty() {
        return all(Functions.alwaysFalse());
    }

    /**
     * Correlates the items emitted by two {@link Publisher}s based on overlapping durations.
     * <p>
     * There are no guarantees in what order the items get combined when multiple
     * items from one or both source {@code Publisher}s overlap.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/join_.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure and consumes all participating {@code Publisher}s in
     *  an unbounded mode (i.e., not applying any backpressure to them).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code join} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <TRight> the value type of the right {@code Publisher} source
     * @param <TLeftEnd> the element type of the left duration {@code Publisher}s
     * @param <TRightEnd> the element type of the right duration {@code Publisher}s
     * @param <R> the result type
     * @param other
     *            the second {@code Publisher} to join items from
     * @param leftEnd
     *            a function to select a duration for each item emitted by the current {@code Flowable}, used to
     *            determine overlap
     * @param rightEnd
     *            a function to select a duration for each item emitted by the {@code right} {@code Publisher}, used to
     *            determine overlap
     * @param resultSelector
     *            a function that computes an item to be emitted by the resulting {@code Flowable} for any two
     *            overlapping items emitted by the two {@code Publisher}s
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other}, {@code leftEnd}, {@code rightEnd} or {@code resultSelector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/join.html">ReactiveX operators documentation: Join</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <TRight, TLeftEnd, TRightEnd, R> Flowable<R> join(
            @NonNull Publisher<@NonNull ? extends TRight> other,
            @NonNull Function<? super T, ? extends Publisher<@NonNull TLeftEnd>> leftEnd,
            @NonNull Function<? super TRight, ? extends Publisher<@NonNull TRightEnd>> rightEnd,
            @NonNull BiFunction<? super T, ? super TRight, ? extends R> resultSelector) {
        Objects.requireNonNull(other, "other is null");
        Objects.requireNonNull(leftEnd, "leftEnd is null");
        Objects.requireNonNull(rightEnd, "rightEnd is null");
        Objects.requireNonNull(resultSelector, "resultSelector is null");
        return RxJavaPlugins.onAssembly(new FlowableJoin<T, TRight, TLeftEnd, TRightEnd, R>(
                this, other, leftEnd, rightEnd, resultSelector));
    }

    /**
     * Returns a {@link Maybe} that emits the last item emitted by this {@code Flowable} or completes if
     * this {@code Flowable} is empty.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/last.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code lastElement} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Maybe} instance
     * @see <a href="http://reactivex.io/documentation/operators/last.html">ReactiveX operators documentation: Last</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Maybe<T> lastElement() {
        return RxJavaPlugins.onAssembly(new FlowableLastMaybe<>(this));
    }

    /**
     * Returns a {@link Single} that emits only the last item emitted by this {@code Flowable}, or a default item
     * if this {@code Flowable} completes without emitting any items.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/lastOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code last} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param defaultItem
     *            the default item to emit if the current {@code Flowable} is empty
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/last.html">ReactiveX operators documentation: Last</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<T> last(@NonNull T defaultItem) {
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        return RxJavaPlugins.onAssembly(new FlowableLastSingle<>(this, defaultItem));
    }

    /**
     * Returns a {@link Single} that emits only the last item emitted by this {@code Flowable} or signals
     * a {@link NoSuchElementException} if this {@code Flowable} is empty.
     * <p>
     * <img width="640" height="236" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/lastOrError.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code lastOrError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Single} instance
     * @see <a href="http://reactivex.io/documentation/operators/last.html">ReactiveX operators documentation: Last</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<T> lastOrError() {
        return RxJavaPlugins.onAssembly(new FlowableLastSingle<>(this, null));
    }

    /**
     * <strong>This method requires advanced knowledge about building operators, please consider
     * other standard composition methods first;</strong>
     * Returns a {@code Flowable} which, when subscribed to, invokes the {@link FlowableOperator#apply(Subscriber) apply(Subscriber)} method
     * of the provided {@link FlowableOperator} for each individual downstream {@link Subscriber} and allows the
     * insertion of a custom operator by accessing the downstream's {@code Subscriber} during this subscription phase
     * and providing a new {@code Subscriber}, containing the custom operator's intended business logic, that will be
     * used in the subscription process going further upstream.
     * <p>
     * Generally, such a new {@code Subscriber} will wrap the downstream's {@code Subscriber} and forwards the
     * {@code onNext}, {@code onError} and {@code onComplete} events from the upstream directly or according to the
     * emission pattern the custom operator's business logic requires. In addition, such operator can intercept the
     * flow control calls of {@code cancel} and {@code request} that would have traveled upstream and perform
     * additional actions depending on the same business logic requirements.
     * <p>
     * Example:
     * <pre><code>
     * // Step 1: Create the consumer type that will be returned by the FlowableOperator.apply():
     *
     * public final class CustomSubscriber&lt;T&gt; implements FlowableSubscriber&lt;T&gt;, Subscription {
     *
     *     // The downstream's Subscriber that will receive the onXXX events
     *     final Subscriber&lt;? super String&gt; downstream;
     *
     *     // The connection to the upstream source that will call this class' onXXX methods
     *     Subscription upstream;
     *
     *     // The constructor takes the downstream subscriber and usually any other parameters
     *     public CustomSubscriber(Subscriber&lt;? super String&gt; downstream) {
     *         this.downstream = downstream;
     *     }
     *
     *     // In the subscription phase, the upstream sends a Subscription to this class
     *     // and subsequently this class has to send a Subscription to the downstream.
     *     // Note that relaying the upstream's Subscription instance directly is not allowed in RxJava
     *     &#64;Override
     *     public void onSubscribe(Subscription s) {
     *         if (upstream != null) {
     *             s.cancel();
     *         } else {
     *             upstream = s;
     *             downstream.onSubscribe(this);
     *         }
     *     }
     *
     *     // The upstream calls this with the next item and the implementation's
     *     // responsibility is to emit an item to the downstream based on the intended
     *     // business logic, or if it can't do so for the particular item,
     *     // request more from the upstream
     *     &#64;Override
     *     public void onNext(T item) {
     *         String str = item.toString();
     *         if (str.length() &lt; 2) {
     *             downstream.onNext(str);
     *         } else {
     *             upstream.request(1);
     *         }
     *     }
     *
     *     // Some operators may handle the upstream's error while others
     *     // could just forward it to the downstream.
     *     &#64;Override
     *     public void onError(Throwable throwable) {
     *         downstream.onError(throwable);
     *     }
     *
     *     // When the upstream completes, usually the downstream should complete as well.
     *     &#64;Override
     *     public void onComplete() {
     *         downstream.onComplete();
     *     }
     *
     *     // Some operators have to intercept the downstream's request calls to trigger
     *     // the emission of queued items while others can simply forward the request
     *     // amount as is.
     *     &#64;Override
     *     public void request(long n) {
     *         upstream.request(n);
     *     }
     *
     *     // Some operators may use their own resources which should be cleaned up if
     *     // the downstream cancels the flow before it completed. Operators without
     *     // resources can simply forward the cancellation to the upstream.
     *     // In some cases, a canceled flag may be set by this method so that other parts
     *     // of this class may detect the cancellation and stop sending events
     *     // to the downstream.
     *     &#64;Override
     *     public void cancel() {
     *         upstream.cancel();
     *     }
     * }
     *
     * // Step 2: Create a class that implements the FlowableOperator interface and
     * //         returns the custom consumer type from above in its apply() method.
     * //         Such class may define additional parameters to be submitted to
     * //         the custom consumer type.
     *
     * final class CustomOperator&lt;T&gt; implements FlowableOperator&lt;String&gt; {
     *     &#64;Override
     *     public Subscriber&lt;? super String&gt; apply(Subscriber&lt;? super T&gt; upstream) {
     *         return new CustomSubscriber&lt;T&gt;(upstream);
     *     }
     * }
     *
     * // Step 3: Apply the custom operator via lift() in a flow by creating an instance of it
     * //         or reusing an existing one.
     *
     * Flowable.range(5, 10)
     * .lift(new CustomOperator&lt;Integer&gt;())
     * .test()
     * .assertResult("5", "6", "7", "8", "9");
     * </code></pre>
     * <p>
     * Creating custom operators can be complicated and it is recommended one consults the
     * <a href="https://github.com/ReactiveX/RxJava/wiki/Writing-operators-for-2.0">RxJava wiki: Writing operators</a> page about
     * the tools, requirements, rules, considerations and pitfalls of implementing them.
     * <p>
     * Note that implementing custom operators via this {@code lift()} method adds slightly more overhead by requiring
     * an additional allocation and indirection per assembled flows. Instead, extending the abstract {@code Flowable}
     * class and creating a {@link FlowableTransformer} with it is recommended.
     * <p>
     * Note also that it is not possible to stop the subscription phase in {@code lift()} as the {@code apply()} method
     * requires a non-{@code null} {@code Subscriber} instance to be returned, which is then unconditionally subscribed to
     * the upstream {@code Flowable}. For example, if the operator decided there is no reason to subscribe to the
     * upstream source because of some optimization possibility or a failure to prepare the operator, it still has to
     * return a {@code Subscriber} that should immediately cancel the upstream's {@link Subscription} in its
     * {@code onSubscribe} method. Again, using a {@code FlowableTransformer} and extending the {@code Flowable} is
     * a better option as {@link #subscribeActual} can decide to not subscribe to its upstream after all.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The {@code Subscriber} instance returned by the {@code FlowableOperator} is responsible to be
     *  backpressure-aware or document the fact that the consumer of the returned {@link Publisher} has to apply one of
     *  the {@code onBackpressureXXX} operators.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code lift} does not operate by default on a particular {@link Scheduler}, however, the
     *  {@code FlowableOperator} may use a {@code Scheduler} to support its own asynchronous behavior.</dd>
     * </dl>
     *
     * @param <R> the output value type
     * @param lifter the {@code FlowableOperator} that receives the downstream's {@code Subscriber} and should return
     *               a {@code Subscriber} with custom behavior to be used as the consumer for the current
     *               {@code Flowable}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code lifter} is {@code null}
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Writing-operators-for-2.0">RxJava wiki: Writing operators</a>
     * @see #compose(FlowableTransformer)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> lift(@NonNull FlowableOperator<? extends R, ? super T> lifter) {
        Objects.requireNonNull(lifter, "lifter is null");
        return RxJavaPlugins.onAssembly(new FlowableLift<>(this, lifter));
    }

    /**
     * Returns a {@code Flowable} that applies a specified function to each item emitted by the current {@code Flowable} and
     * emits the results of these function applications.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/map.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code map} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the output type
     * @param mapper
     *            a function to apply to each item emitted by the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/map.html">ReactiveX operators documentation: Map</a>
     * @see #mapOptional(Function)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull R> Flowable<R> map(@NonNull Function<? super T, ? extends R> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableMap<>(this, mapper));
    }

    /**
     * Returns a {@code Flowable} that represents all of the emissions <em>and</em> notifications from the current
     * {@code Flowable} into emissions marked with their original types within {@link Notification} objects.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/materialize.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and expects it from the current {@code Flowable}.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code materialize} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/materialize-dematerialize.html">ReactiveX operators documentation: Materialize</a>
     * @see #dematerialize(Function)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<Notification<T>> materialize() {
        return RxJavaPlugins.onAssembly(new FlowableMaterialize<>(this));
    }

    /**
     * Flattens this and another {@link Publisher} into a single {@code Publisher}, without any transformation.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * You can combine items emitted by multiple {@code Publisher}s so that they appear as a single {@code Publisher}, by
     * using the {@code mergeWith} method.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. This and the other {@code Publisher}s are expected to honor
     *  backpressure; if violated, the operator <em>may</em> signal {@link MissingBackpressureException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param other
     *            a {@code Publisher} to be merged
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/merge.html">ReactiveX operators documentation: Merge</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> mergeWith(@NonNull Publisher<@NonNull ? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return merge(this, other);
    }

    /**
     * Merges the sequence of items of this {@code Flowable} with the success value of the other {@link SingleSource}.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * The success value of the other {@code SingleSource} can get interleaved at any point of this
     * {@code Flowable} sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and ensures the success item from the
     *  {@code SingleSource} is emitted only when there is a downstream demand.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code SingleSource} whose success value to merge with
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> mergeWith(@NonNull SingleSource<? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableMergeWithSingle<>(this, other));
    }

    /**
     * Merges the sequence of items of this {@code Flowable} with the success value of the other {@link MaybeSource}
     * or waits for both to complete normally if the {@code MaybeSource} is empty.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <p>
     * The success value of the other {@code MaybeSource} can get interleaved at any point of this
     * {@code Flowable} sequence.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and ensures the success item from the
     *  {@code MaybeSource} is emitted only when there is a downstream demand.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code MaybeSource} which provides a success value to merge with or completes
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> mergeWith(@NonNull MaybeSource<? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableMergeWithMaybe<>(this, other));
    }

    /**
     * Relays the items of this {@code Flowable} and completes only when the other {@link CompletableSource} completes
     * as well.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/merge.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code mergeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.10 - experimental
     * @param other the {@code CompletableSource} to await for completion
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> mergeWith(@NonNull CompletableSource other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableMergeWithCompletable<>(this, other));
    }

    /**
     * Signals the items and terminal signals of the current {@code Flowable} on the specified {@link Scheduler},
     * asynchronously with a bounded buffer of {@link #bufferSize()} slots.
     *
     * <p>Note that {@code onError} notifications will cut ahead of {@code onNext} notifications on the emission thread if {@code Scheduler} is truly
     * asynchronous. If strict event ordering is required, consider using the {@link #observeOn(Scheduler, boolean)} overload.
     * <p>
     * <img width="640" height="308" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/observeOn.v3.png" alt="">
     * <p>
     * This operator keeps emitting as many signals as it can on the given {@code Scheduler}'s Worker thread,
     * which may result in a longer than expected occupation of this thread. In other terms,
     * it does not allow per-signal fairness in case the worker runs on a shared underlying thread.
     * If such fairness and signal/work interleaving is preferred, use the delay operator with zero time instead.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator honors backpressure from downstream and expects it from the current {@code Flowable}. Violating this
     *  expectation will lead to {@link MissingBackpressureException}. This is the most common operator where the exception
     *  pops up; look for sources up the chain that don't support backpressure,
     *  such as {@link #interval(long, TimeUnit)}, {@link #timer(long, TimeUnit)},
     *  {@link io.reactivex.rxjava3.processors.PublishProcessor PublishProcessor} or
     *  {@link io.reactivex.rxjava3.processors.BehaviorProcessor BehaviorProcessor} and apply any
     *  of the {@code onBackpressureXXX} operators <strong>before</strong> applying {@code observeOn} itself.
     *  Note also that request amounts are not preserved between the immediate downstream and the
     *  immediate upstream. The operator always requests the default {@link #bufferSize()} amount first, then after
     *  every 75% of that amount delivered, another 75% of this default value. If preserving the request amounts
     *  is to be preferred over potential excess scheduler infrastructure use, consider applying
     *  {@link #delay(long, TimeUnit, Scheduler)} with zero time instead.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param scheduler
     *            the {@code Scheduler} to notify {@link Subscriber}s on
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/observeon.html">ReactiveX operators documentation: ObserveOn</a>
     * @see <a href="http://www.grahamlea.com/2014/07/rxjava-threading-examples/">RxJava Threading Examples</a>
     * @see #subscribeOn
     * @see #observeOn(Scheduler, boolean)
     * @see #observeOn(Scheduler, boolean, int)
     * @see #delay(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> observeOn(@NonNull Scheduler scheduler) {
        return observeOn(scheduler, false, bufferSize());
    }

    /**
     * Signals the items and terminal signals of the current {@code Flowable} on the specified {@link Scheduler},
     * asynchronously with a bounded buffer and optionally delays {@code onError} notifications.
     * <p>
     * <img width="640" height="308" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/observeOn.v3.png" alt="">
     * <p>
     * This operator keeps emitting as many signals as it can on the given {@code Scheduler}'s Worker thread,
     * which may result in a longer than expected occupation of this thread. In other terms,
     * it does not allow per-signal fairness in case the worker runs on a shared underlying thread.
     * If such fairness and signal/work interleaving is preferred, use the delay operator with zero time instead.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator honors backpressure from downstream and expects it from the current {@code Flowable}. Violating this
     *  expectation will lead to {@link MissingBackpressureException}. This is the most common operator where the exception
     *  pops up; look for sources up the chain that don't support backpressure,
     *  such as {@link #interval(long, TimeUnit)}, {@link #timer(long, TimeUnit)},
     *  {@link io.reactivex.rxjava3.processors.PublishProcessor PublishProcessor} or
     *  {@link io.reactivex.rxjava3.processors.BehaviorProcessor BehaviorProcessor} and apply any
     *  of the {@code onBackpressureXXX} operators <strong>before</strong> applying {@code observeOn} itself.
     *  Note also that request amounts are not preserved between the immediate downstream and the
     *  immediate upstream. The operator always requests the default {@link #bufferSize()} amount first, then after
     *  every 75% of that amount delivered, another 75% of this default value. If preserving the request amounts
     *  is to be preferred over potential excess scheduler infrastructure use, consider applying
     *  {@link #delay(long, TimeUnit, Scheduler, boolean)} with zero time instead.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param scheduler
     *            the {@code Scheduler} to notify {@link Subscriber}s on
     * @param delayError
     *            indicates if the {@code onError} notification may not cut ahead of {@code onNext} notification on the other side of the
     *            scheduling boundary. If {@code true}, a sequence ending in {@code onError} will be replayed in the same order as was received
     *            from upstream
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/observeon.html">ReactiveX operators documentation: ObserveOn</a>
     * @see <a href="http://www.grahamlea.com/2014/07/rxjava-threading-examples/">RxJava Threading Examples</a>
     * @see #subscribeOn
     * @see #observeOn(Scheduler)
     * @see #observeOn(Scheduler, boolean, int)
     * @see #delay(long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> observeOn(@NonNull Scheduler scheduler, boolean delayError) {
        return observeOn(scheduler, delayError, bufferSize());
    }

    /**
     * Signals the items and terminal signals of the current {@code Flowable} on the specified {@link Scheduler},
     * asynchronously with a bounded buffer of configurable size and optionally delays {@code onError} notifications.
     * <p>
     * <img width="640" height="308" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/observeOn.v3.png" alt="">
     * <p>
     * This operator keeps emitting as many signals as it can on the given {@code Scheduler}'s Worker thread,
     * which may result in a longer than expected occupation of this thread. In other terms,
     * it does not allow per-signal fairness in case the worker runs on a shared underlying thread.
     * If such fairness and signal/work interleaving is preferred, use the delay operator with zero time instead.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator honors backpressure from downstream and expects it from the current {@code Flowable}. Violating this
     *  expectation will lead to {@link MissingBackpressureException}. This is the most common operator where the exception
     *  pops up; look for sources up the chain that don't support backpressure,
     *  such as {@link #interval(long, TimeUnit)}, {@link #timer(long, TimeUnit)},
     *  {@link io.reactivex.rxjava3.processors.PublishProcessor PublishProcessor} or
     *  {@link io.reactivex.rxjava3.processors.BehaviorProcessor BehaviorProcessor} and apply any
     *  of the {@code onBackpressureXXX} operators <strong>before</strong> applying {@code observeOn} itself.
     *  Note also that request amounts are not preserved between the immediate downstream and the
     *  immediate upstream. The operator always requests the specified {@code bufferSize} amount first, then after
     *  every 75% of that amount delivered, another 75% of this specified value. If preserving the request amounts
     *  is to be preferred over potential excess scheduler infrastructure use, consider applying
     *  {@link #delay(long, TimeUnit, Scheduler, boolean)} with zero time instead.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param scheduler
     *            the {@code Scheduler} to notify {@link Subscriber}s on
     * @param delayError
     *            indicates if the {@code onError} notification may not cut ahead of {@code onNext} notification on the other side of the
     *            scheduling boundary. If {@code true}, a sequence ending in {@code onError} will be replayed in the same order as was received
     *            from upstream
     * @param bufferSize the size of the buffer.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/observeon.html">ReactiveX operators documentation: ObserveOn</a>
     * @see <a href="http://www.grahamlea.com/2014/07/rxjava-threading-examples/">RxJava Threading Examples</a>
     * @see #subscribeOn
     * @see #observeOn(Scheduler)
     * @see #observeOn(Scheduler, boolean)
     * @see #delay(long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> observeOn(@NonNull Scheduler scheduler, boolean delayError, int bufferSize) {
        Objects.requireNonNull(scheduler, "scheduler is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowableObserveOn<>(this, scheduler, delayError, bufferSize));
    }

    /**
     * Filters the items emitted by the current {@code Flowable}, only emitting those of the specified type.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/ofClass.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code ofType} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the output type
     * @param clazz
     *            the class type to filter the items emitted by the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code clazz} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/filter.html">ReactiveX operators documentation: Filter</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<U> ofType(@NonNull Class<U> clazz) {
        Objects.requireNonNull(clazz, "clazz is null");
        return filter(Functions.isInstanceOf(clazz)).cast(clazz);
    }

    /**
     * Buffers an unlimited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <p>
     * An error from the current {@code Flowable} will cut ahead of any unconsumed item. Use {@link #onBackpressureBuffer(boolean)}
     * to have the operator keep the original signal order.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @see #onBackpressureBuffer(boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer() {
        return onBackpressureBuffer(bufferSize(), false, true);
    }

    /**
     * Buffers an unlimited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place, optionally delaying an error until all buffered items have been consumed.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param delayError
     *                if {@code true}, an exception from the current {@code Flowable} is delayed until all buffered elements have been
     *                consumed by the downstream; if {@code false}, an exception is immediately signaled to the downstream, skipping
     *                any buffered element
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer(boolean delayError) {
        return onBackpressureBuffer(bufferSize(), delayError, true);
    }

    /**
     * Buffers an limited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place, however, the resulting {@code Flowable} will signal a
     * {@link MissingBackpressureException} via {@code onError} as soon as the buffer's capacity is exceeded, dropping all undelivered
     * items, and canceling the flow.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <p>
     * An error from the current {@code Flowable} will cut ahead of any unconsumed item. Use {@link #onBackpressureBuffer(int, boolean)}
     * to have the operator keep the original signal order.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     * @see #onBackpressureBuffer(long, Action, BackpressureOverflowStrategy)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer(int capacity) {
        return onBackpressureBuffer(capacity, false, false);
    }

    /**
     * Buffers an limited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place, however, the resulting {@code Flowable} will signal a
     * {@link MissingBackpressureException} via {@code onError} as soon as the buffer's capacity is exceeded, dropping all undelivered
     * items, and canceling the flow.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @param delayError
     *                if {@code true}, an exception from the current {@code Flowable} is delayed until all buffered elements have been
     *                consumed by the downstream; if {@code false}, an exception is immediately signaled to the downstream, skipping
     *                any buffered element
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer(int capacity, boolean delayError) {
        return onBackpressureBuffer(capacity, delayError, false);
    }

    /**
     * Buffers an optionally unlimited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place.
     * If {@code unbounded} is {@code true}, the resulting {@code Flowable} will signal a
     * {@link MissingBackpressureException} via {@code onError} as soon as the buffer's capacity is exceeded, dropping all undelivered
     * items, and canceling the flow.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @param delayError
     *                if {@code true}, an exception from the current {@code Flowable} is delayed until all buffered elements have been
     *                consumed by the downstream; if {@code false}, an exception is immediately signaled to the downstream, skipping
     *                any buffered element
     * @param unbounded
     *                if {@code true}, the capacity value is interpreted as the internal "island" size of the unbounded buffer
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer(int capacity, boolean delayError, boolean unbounded) {
        ObjectHelper.verifyPositive(capacity, "capacity");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureBuffer<>(this, capacity, unbounded, delayError, Functions.EMPTY_ACTION));
    }

    /**
     * Buffers an optionally unlimited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place.
     * If {@code unbounded} is {@code true}, the resulting {@code Flowable} will signal a
     * {@link MissingBackpressureException} via {@code onError} as soon as the buffer's capacity is exceeded, dropping all undelivered
     * items, canceling the flow and calling the {@code onOverflow} action.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @param delayError
     *                if {@code true}, an exception from the current {@code Flowable} is delayed until all buffered elements have been
     *                consumed by the downstream; if {@code false}, an exception is immediately signaled to the downstream, skipping
     *                any buffered element
     * @param unbounded
     *                if {@code true}, the capacity value is interpreted as the internal "island" size of the unbounded buffer
     * @param onOverflow action to execute if an item needs to be buffered, but there are no available slots.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onOverflow} is {@code null}
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onBackpressureBuffer(int capacity, boolean delayError, boolean unbounded,
            @NonNull Action onOverflow) {
        Objects.requireNonNull(onOverflow, "onOverflow is null");
        ObjectHelper.verifyPositive(capacity, "capacity");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureBuffer<>(this, capacity, unbounded, delayError, onOverflow));
    }

    /**
     * Buffers an limited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place, however, the resulting {@code Flowable} will signal a
     * {@link MissingBackpressureException} via {@code onError} as soon as the buffer's capacity is exceeded, dropping all undelivered
     * items, canceling the flow and calling the {@code onOverflow} action.
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @param onOverflow action to execute if an item needs to be buffered, but there are no available slots.  Null is allowed.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onOverflow} is {@code null}
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureBuffer(int capacity, @NonNull Action onOverflow) {
        return onBackpressureBuffer(capacity, false, false, onOverflow);
    }

    /**
     * Buffers an optionally unlimited number of items from the current {@code Flowable} and allows it to emit as fast it can while allowing the
     * downstream to consume the items at its own place.
     * The resulting {@code Flowable} will behave as determined by {@code overflowStrategy} if the buffer capacity is exceeded:
     * <ul>
     *     <li>{@link BackpressureOverflowStrategy#ERROR} (default) will call {@code onError} dropping all undelivered items,
     *     canceling the source, and notifying the producer with {@code onOverflow}. </li>
     *     <li>{@link BackpressureOverflowStrategy#DROP_LATEST} will drop any new items emitted by the producer while
     *     the buffer is full, without generating any {@code onError}.  Each drop will, however, invoke {@code onOverflow}
     *     to signal the overflow to the producer.</li>
     *     <li>{@link BackpressureOverflowStrategy#DROP_OLDEST} will drop the oldest items in the buffer in order to make
     *     room for newly emitted ones. Overflow will not generate an {@code onError}, but each drop will invoke
     *     {@code onOverflow} to signal the overflow to the producer.</li>
     * </ul>
     *
     * <p>
     * <img width="640" height="300" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.buffer.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureBuffer} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param capacity number of slots available in the buffer.
     * @param onOverflow action to execute if an item needs to be buffered, but there are no available slots, {@code null} is allowed.
     * @param overflowStrategy how should the resulting {@code Flowable} react to buffer overflows, {@code null} is not allowed.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onOverflow} or {@code overflowStrategy} is {@code null}
     * @throws IllegalArgumentException if {@code capacity} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 2.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onBackpressureBuffer(long capacity, @Nullable Action onOverflow, @NonNull BackpressureOverflowStrategy overflowStrategy) {
        Objects.requireNonNull(overflowStrategy, "overflowStrategy is null");
        ObjectHelper.verifyPositive(capacity, "capacity");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureBufferStrategy<>(this, capacity, onOverflow, overflowStrategy));
    }

    /**
     * Drops items from the current {@code Flowable} if the downstream is not ready to receive new items (indicated
     * by a lack of {@link Subscription#request(long)} calls from it).
     * <p>
     * <img width="640" height="245" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.drop.v3.png" alt="">
     * <p>
     * If the downstream request count hits 0 then the resulting {@code Flowable} will refrain from calling {@code onNext} until
     * the {@link Subscriber} invokes {@code request(n)} again to increase the request count.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureDrop} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureDrop() {
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureDrop<>(this));
    }

    /**
     * Drops items from the current {@code Flowable} if the downstream is not ready to receive new items (indicated
     * by a lack of {@link Subscription#request(long)} calls from it) and calls the given {@link Consumer} with such
     * dropped items.
     * <p>
     * <img width="640" height="245" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.drop.v3.png" alt="">
     * <p>
     * If the downstream request count hits 0 then the resulting {@code Flowable} will refrain from calling {@code onNext} until
     * the {@link Subscriber} invokes {@code request(n)} again to increase the request count.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureDrop} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onDrop the action to invoke for each item dropped, should be fast and should never block.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code onDrop} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/backpressure.html">ReactiveX operators documentation: backpressure operators</a>
     * @since 1.1.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onBackpressureDrop(@NonNull Consumer<? super T> onDrop) {
        Objects.requireNonNull(onDrop, "onDrop is null");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureDrop<>(this, onDrop));
    }

    /**
     * Drops all but the latest item emitted by the current {@code Flowable} if the downstream is not ready to receive
     * new items (indicated by a lack of {@link Subscription#request(long)} calls from it) and emits this latest
     * item when the downstream becomes ready.
     * <p>
     * <img width="640" height="245" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/bp.obp.latest.v3.png" alt="">
     * <p>
     * Its behavior is logically equivalent to {@code blockingLatest()} with the exception that
     * the downstream is not blocking while requesting more values.
     * <p>
     * Note that if the current {@code Flowable} does support backpressure, this operator ignores that capability
     * and doesn't propagate any backpressure requests from downstream.
     * <p>
     * Note that due to the nature of how backpressure requests are propagated through subscribeOn/observeOn,
     * requesting more than 1 from downstream doesn't guarantee a continuous delivery of {@code onNext} events.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureLatest} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @since 1.1.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureLatest() {
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureLatest<>(this));
    }

    /**
     * Reduces a sequence of two not emitted values via a function into a single value if the downstream is not ready to receive
     * new items (indicated by a lack of {@link Subscription#request(long)} calls from it) and emits this latest
     * item when the downstream becomes ready.
     * <p>
     * <img width="640" height="354" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.onBackpressureReduce.png" alt="">
     * <p>
     * Note that if the current {@code Flowable} does support backpressure, this operator ignores that capability
     * and doesn't propagate any backpressure requests from downstream.
     * <p>
     * Note that due to the nature of how backpressure requests are propagated through subscribeOn/observeOn,
     * requesting more than 1 from downstream doesn't guarantee a continuous delivery of {@code onNext} events.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureReduce} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param reducer the bi-function to call when there is more than one non-emitted value to downstream,
     *                 the first argument of the bi-function is previous item and the second one is currently
     *                 emitting from upstream
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code reducer} is {@code null}
     * @since 3.0.9 - experimental
     * @see #onBackpressureReduce(Supplier, BiFunction)
     */
    @Experimental
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onBackpressureReduce(@NonNull BiFunction<T, T, T> reducer) {
        Objects.requireNonNull(reducer, "reducer is null");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureReduce<>(this, reducer));
    }

    /**
     * Reduces upstream values into an aggregate value, provided by a supplier and combined via a reducer function,
     * while the downstream is not ready to receive items, then emits this aggregate value when the downstream becomes ready.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.onBackpressureReduce.ff.png" alt="">
     * <p>
     * Note that even if the downstream is ready to receive an item, the upstream item will always be aggregated into the output type,
     * calling both the supplier and the reducer to produce the output value.
     * <p>
     * Note that if the current {@code Flowable} does support backpressure, this operator ignores that capability
     * and doesn't propagate any backpressure requests from downstream.
     * <p>
     * Note that due to the nature of how backpressure requests are propagated through subscribeOn/observeOn,
     * requesting more than 1 from downstream doesn't guarantee a continuous delivery of {@code onNext} events.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an unbounded
     *  manner (i.e., not applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onBackpressureReduce} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <R> the aggregate type emitted when the downstream requests more items
     * @param supplier the factory to call to create new item of type R to pass it as the first argument to {@code reducer}.
     *                 It is called when previous returned value by {@code reducer} already sent to
     *                 downstream or the very first update from upstream received.
     * @param reducer the bi-function to call to reduce excessive updates which downstream is not ready to receive.
     *                The first argument of type R is the object returned by {@code supplier} or result of previous
     *                {@code reducer} invocation. The second argument of type T is the current update from upstream.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code supplier} or {@code reducer} is {@code null}
     * @see #onBackpressureReduce(BiFunction)
     * @since 3.0.9 - experimental
     */
    @Experimental
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> onBackpressureReduce(@NonNull Supplier<R> supplier, @NonNull BiFunction<R, ? super T, R> reducer) {
        Objects.requireNonNull(supplier, "supplier is null");
        Objects.requireNonNull(reducer, "reducer is null");
        return RxJavaPlugins.onAssembly(new FlowableOnBackpressureReduceWith<>(this, supplier, reducer));
    }

    /**
     * Returns a {@code Flowable} instance that if the current {@code Flowable} emits an error, it will emit an {@code onComplete}
     * and swallow the throwable.
     * <p>
     * <img width="640" height="372" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.onErrorComplete.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorComplete} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @return the new {@code Flowable} instance
     * @since 3.0.0
     */
    @CheckReturnValue
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @NonNull
    public final Flowable<T> onErrorComplete() {
        return onErrorComplete(Functions.alwaysTrue());
    }

    /**
     * Returns a {@code Flowable} instance that if the current {@code Flowable} emits an error and the predicate returns
     * {@code true}, it will emit an {@code onComplete} and swallow the throwable.
     * <p>
     * <img width="640" height="201" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.onErrorComplete.f.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorComplete} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param predicate the predicate to call when an {@link Throwable} is emitted which should return {@code true}
     * if the {@code Throwable} should be swallowed and replaced with an {@code onComplete}.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onErrorComplete(@NonNull Predicate<? super Throwable> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");

        return RxJavaPlugins.onAssembly(new FlowableOnErrorComplete<>(this, predicate));
    }

    /**
     * Resumes the flow with a {@link Publisher} returned for the failure {@link Throwable} of the current {@code Flowable} by a
     * function instead of signaling the error via {@code onError}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/onErrorResumeNext.v3.png" alt="">
     * <p>
     * By default, when a {@code Publisher} encounters an error that prevents it from emitting the expected item to
     * its {@link Subscriber}, the {@code Publisher} invokes its {@code Subscriber}'s {@code onError} method, and then quits
     * without invoking any more of its {@code Subscriber}'s methods. The {@code onErrorResumeNext} method changes this
     * behavior. If you pass a function that returns a {@code Publisher} ({@code resumeFunction}) to
     * {@code onErrorResumeNext}, if the original {@code Publisher} encounters an error, instead of invoking its
     * {@code Subscriber}'s {@code onError} method, it will instead relinquish control to the {@code Publisher} returned from
     * {@code resumeFunction}, which will invoke the {@code Subscriber}'s {@link Subscriber#onNext onNext} method if it is
     * able to do so. In such a case, because no {@code Publisher} necessarily invokes {@code onError}, the {@code Subscriber}
     * may never know that an error happened.
     * <p>
     * You can use this to prevent errors from propagating or to supply fallback data should errors be
     * encountered.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. This and the resuming {@code Publisher}s
     *  are expected to honor backpressure as well.
     *  If any of them violate this expectation, the operator <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes or
     *  a {@link MissingBackpressureException} is signaled somewhere downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorResumeNext} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param fallbackSupplier
     *            a function that returns a {@code Publisher} that will take over if the current {@code Flowable} encounters
     *            an error
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code fallbackSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/catch.html">ReactiveX operators documentation: Catch</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onErrorResumeNext(@NonNull Function<? super Throwable, ? extends Publisher<@NonNull ? extends T>> fallbackSupplier) {
        Objects.requireNonNull(fallbackSupplier, "fallbackSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableOnErrorNext<>(this, fallbackSupplier));
    }

    /**
     * Resumes the flow with the given {@link Publisher} when the current {@code Flowable} fails instead of
     * signaling the error via {@code onError}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/onErrorResumeWith.v3.png" alt="">
     * <p>
     * By default, when a {@code Publisher} encounters an error that prevents it from emitting the expected item to
     * its {@link Subscriber}, the {@code Publisher} invokes its {@code Subscriber}'s {@code onError} method, and then quits
     * without invoking any more of its {@code Subscriber}'s methods. The {@code onErrorResumeWith} method changes this
     * behavior. If you pass another {@code Publisher} ({@code resumeSequence}) to a {@code Publisher}'s
     * {@code onErrorResumeWith} method, if the original {@code Publisher} encounters an error, instead of invoking its
     * {@code Subscriber}'s {@code onError} method, it will instead relinquish control to {@code resumeSequence} which
     * will invoke the {@code Subscriber}'s {@link Subscriber#onNext onNext} method if it is able to do so. In such a case,
     * because no {@code Publisher} necessarily invokes {@code onError}, the {@code Subscriber} may never know that an error
     * happened.
     * <p>
     * You can use this to prevent errors from propagating or to supply fallback data should errors be
     * encountered.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. This and the resuming {@code Publisher}s
     *  are expected to honor backpressure as well.
     *  If any of them violate this expectation, the operator <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes or
     *  {@link MissingBackpressureException} is signaled somewhere downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorResumeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param fallback
     *            the next {@code Publisher} source that will take over if the current {@code Flowable} encounters
     *            an error
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code fallback} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/catch.html">ReactiveX operators documentation: Catch</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onErrorResumeWith(@NonNull Publisher<@NonNull ? extends T> fallback) {
        Objects.requireNonNull(fallback, "fallback is null");
        return onErrorResumeNext(Functions.justFunction(fallback));
    }

    /**
     * Ends the flow with a last item returned by a function for the {@link Throwable} error signaled by the current
     * {@code Flowable} instead of signaling the error via {@code onError}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/onErrorReturn.v3.png" alt="">
     * <p>
     * By default, when a {@link Publisher} encounters an error that prevents it from emitting the expected item to
     * its {@link Subscriber}, the {@code Publisher} invokes its {@code Subscriber}'s {@code onError} method, and then quits
     * without invoking any more of its {@code Subscriber}'s methods. The {@code onErrorReturn} method changes this
     * behavior. If you pass a function ({@code resumeFunction}) to a {@code Publisher}'s {@code onErrorReturn}
     * method, if the original {@code Publisher} encounters an error, instead of invoking its {@code Subscriber}'s
     * {@code onError} method, it will instead emit the return value of {@code resumeFunction}.
     * <p>
     * You can use this to prevent errors from propagating or to supply fallback data should errors be
     * encountered.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable} is expected to honor
     *  backpressure as well. If it this expectation is violated, the operator <em>may</em> throw
     *  {@link IllegalStateException} when the current {@code Flowable} completes or
     *  {@link MissingBackpressureException} is signaled somewhere downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorReturn} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param itemSupplier
     *            a function that returns a single value that will be emitted along with a regular {@code onComplete} in case
     *            the current {@code Flowable} signals an {@code onError} event
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code itemSupplier} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/catch.html">ReactiveX operators documentation: Catch</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onErrorReturn(@NonNull Function<? super Throwable, ? extends T> itemSupplier) {
        Objects.requireNonNull(itemSupplier, "itemSupplier is null");
        return RxJavaPlugins.onAssembly(new FlowableOnErrorReturn<>(this, itemSupplier));
    }

    /**
     * Ends the flow with the given last item when the current {@code Flowable} fails instead of signaling the error via {@code onError}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/onErrorReturn.v3.png" alt="">
     * <p>
     * By default, when a {@link Publisher} encounters an error that prevents it from emitting the expected item to
     * its {@link Subscriber}, the {@code Publisher} invokes its {@code Subscriber}'s {@code onError} method, and then quits
     * without invoking any more of its {@code Subscriber}'s methods. The {@code onErrorReturn} method changes this
     * behavior. If you pass a function ({@code resumeFunction}) to a {@code Publisher}'s {@code onErrorReturn}
     * method, if the original {@code Publisher} encounters an error, instead of invoking its {@code Subscriber}'s
     * {@code onError} method, it will instead emit the return value of {@code resumeFunction}.
     * <p>
     * You can use this to prevent errors from propagating or to supply fallback data should errors be
     * encountered.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable} is expected to honor
     *  backpressure as well. If it this expectation is violated, the operator <em>may</em> throw
     *  {@link IllegalStateException} when the current {@code Flowable} completes or
     *  {@link MissingBackpressureException} is signaled somewhere downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onErrorReturnItem} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item
     *            the value that is emitted along with a regular {@code onComplete} in case the current
     *            {@code Flowable} signals an exception
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/catch.html">ReactiveX operators documentation: Catch</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> onErrorReturnItem(@NonNull T item) {
        Objects.requireNonNull(item, "item is null");
        return onErrorReturn(Functions.justFunction(item));
    }

    /**
     * Nulls out references to the upstream producer and downstream {@link Subscriber} if
     * the sequence is terminated or downstream cancels.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code onTerminateDetach} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @return the new {@code Flowable} instance
     * the sequence is terminated or downstream cancels
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> onTerminateDetach() {
        return RxJavaPlugins.onAssembly(new FlowableDetach<>(this));
    }

    /**
     * Parallelizes the flow by creating multiple 'rails' (equal to the number of CPUs)
     * and dispatches the upstream items to them in a round-robin fashion.
     * <p>
     * Note that the rails don't execute in parallel on their own and one needs to
     * apply {@link ParallelFlowable#runOn(Scheduler)} to specify the {@link Scheduler} where
     * each rail will execute.
     * <p>
     * To merge the parallel 'rails' back into a single sequence, use {@link ParallelFlowable#sequential()}.
     * <p>
     * <img width="640" height="547" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flowable.parallel.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator requires the upstream to honor backpressure and each 'rail' honors backpressure
     *  as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code parallel} does not operate by default on a particular {@code Scheduler}.</dd>
     * </dl>
     * <p>History: 2.0.5 - experimental; 2.1 - beta
     * @return the new {@link ParallelFlowable} instance
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    public final ParallelFlowable<T> parallel() {
        return ParallelFlowable.from(this);
    }

    /**
     * Parallelizes the flow by creating the specified number of 'rails'
     * and dispatches the upstream items to them in a round-robin fashion.
     * <p>
     * Note that the rails don't execute in parallel on their own and one needs to
     * apply {@link ParallelFlowable#runOn(Scheduler)} to specify the {@link Scheduler} where
     * each rail will execute.
     * <p>
     * To merge the parallel 'rails' back into a single sequence, use {@link ParallelFlowable#sequential()}.
     * <p>
     * <img width="640" height="547" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flowable.parallel.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator requires the upstream to honor backpressure and each 'rail' honors backpressure
     *  as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code parallel} does not operate by default on a particular {@code Scheduler}.</dd>
     * </dl>
     * <p>History: 2.0.5 - experimental; 2.1 - beta
     * @param parallelism the number of 'rails' to use
     * @return the new {@link ParallelFlowable} instance
     * @throws IllegalArgumentException if {@code parallelism} is non-positive
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    public final ParallelFlowable<T> parallel(int parallelism) {
        return ParallelFlowable.from(this, parallelism);
    }

    /**
     * Parallelizes the flow by creating the specified number of 'rails'
     * and dispatches the upstream items to them in a round-robin fashion and
     * uses the defined per-'rail' prefetch amount.
     * <p>
     * Note that the rails don't execute in parallel on their own and one needs to
     * apply {@link ParallelFlowable#runOn(Scheduler)} to specify the {@link Scheduler} where
     * each rail will execute.
     * <p>
     * To merge the parallel 'rails' back into a single sequence, use {@link ParallelFlowable#sequential()}.
     * <p>
     * <img width="640" height="547" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/flowable.parallel.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator requires the upstream to honor backpressure and each 'rail' honors backpressure
     *  as well.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code parallel} does not operate by default on a particular {@code Scheduler}.</dd>
     * </dl>
     * <p>History: 2.0.5 - experimental; 2.1 - beta
     * @param parallelism the number of 'rails' to use
     * @param prefetch the number of items each 'rail' should prefetch
     * @return the new {@link ParallelFlowable} instance
     * @throws IllegalArgumentException if {@code parallelism} or {@code prefetch} is non-positive
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @CheckReturnValue
    @NonNull
    public final ParallelFlowable<T> parallel(int parallelism, int prefetch) {
        return ParallelFlowable.from(this, parallelism, prefetch);
    }

    /**
     * Returns a {@link ConnectableFlowable}, which is a variety of {@link Publisher} that waits until its
     * {@link ConnectableFlowable#connect connect} method is called before it begins emitting items to those
     * {@link Subscriber}s that have subscribed to it.
     * <p>
     * <img width="640" height="510" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/publishConnect.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code ConnectableFlowable} honors backpressure for each of its {@code Subscriber}s
     *  and expects the current {@code Flowable} to honor backpressure as well. If this expectation is violated,
     *  the operator will signal a {@link MissingBackpressureException} to its {@code Subscriber}s and disconnect.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code publish} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code ConnectableFlowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/publish.html">ReactiveX operators documentation: Publish</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final ConnectableFlowable<T> publish() {
        return publish(bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the results of invoking a specified selector on items emitted by a
     * {@link ConnectableFlowable} that shares a single subscription to the underlying sequence.
     * <p>
     * <img width="640" height="510" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/publishConnect.f.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the current {@code Flowable} to honor backpressure and if this expectation is
     *  violated, the operator will signal a {@link MissingBackpressureException} through the {@code Flowable}
     *  provided to the function. Since the {@link Publisher} returned by the {@code selector} may be
     *  independent of the provided {@code Flowable} to the function, the output's backpressure behavior
     *  is determined by this returned {@code Publisher}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code publish} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a function that can use the multicasted source sequence as many times as needed, without
     *            causing multiple subscriptions to the source sequence. {@link Subscriber}s to the given source will
     *            receive all notifications of the source from the time of the subscription forward.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/publish.html">ReactiveX operators documentation: Publish</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> publish(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector) {
        return publish(selector, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the results of invoking a specified selector on items emitted by a
     * {@link ConnectableFlowable} that shares a single subscription to the underlying sequence.
     * <p>
     * <img width="640" height="510" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/publishConnect.f.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects the current {@code Flowable} to honor backpressure and if this expectation is
     *  violated, the operator will signal a {@link MissingBackpressureException} through the {@code Flowable}
     *  provided to the function. Since the {@link Publisher} returned by the {@code selector} may be
     *  independent of the provided {@code Flowable} to the function, the output's backpressure behavior
     *  is determined by this returned {@code Publisher}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code publish} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a function that can use the multicasted source sequence as many times as needed, without
     *            causing multiple subscriptions to the source sequence. {@link Subscriber}s to the given source will
     *            receive all notifications of the source from the time of the subscription forward.
     * @param prefetch
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @throws IllegalArgumentException if {@code prefetch} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/publish.html">ReactiveX operators documentation: Publish</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> publish(@NonNull Function<? super Flowable<T>, ? extends Publisher<@NonNull ? extends R>> selector, int prefetch) {
        Objects.requireNonNull(selector, "selector is null");
        ObjectHelper.verifyPositive(prefetch, "prefetch");
        return RxJavaPlugins.onAssembly(new FlowablePublishMulticast<>(this, selector, prefetch, false));
    }

    /**
     * Returns a {@link ConnectableFlowable}, which is a variety of {@link Publisher} that waits until its
     * {@link ConnectableFlowable#connect connect} method is called before it begins emitting items to those
     * {@link Subscriber}s that have subscribed to it.
     * <p>
     * <img width="640" height="510" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/publishConnect.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code ConnectableFlowable} honors backpressure for each of its {@code Subscriber}s
     *  and expects the current {@code Flowable} to honor backpressure as well. If this expectation is violated,
     *  the operator will signal a {@link MissingBackpressureException} to its {@code Subscriber}s and disconnect.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code publish} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the number of elements to prefetch from the current {@code Flowable}
     * @return the new {@code ConnectableFlowable} instance
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/publish.html">ReactiveX operators documentation: Publish</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final ConnectableFlowable<T> publish(int bufferSize) {
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return RxJavaPlugins.onAssembly(new FlowablePublish<>(this, bufferSize));
    }

    /**
     * Requests {@code n} initially from the upstream and then 75% of {@code n} subsequently
     * after 75% of {@code n} values have been emitted to the downstream.
     *
     * <p>This operator allows preventing the downstream to trigger unbounded mode via {@code request(}{@link Long#MAX_VALUE}{@code )}
     * or compensate for the per-item overhead of small and frequent requests.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator expects backpressure from upstream and honors backpressure from downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code rebatchRequests} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param n the initial request amount, further request will happen after 75% of this value
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code n} is non-positive
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> rebatchRequests(int n) {
        return observeOn(ImmediateThinScheduler.INSTANCE, true, n);
    }

    /**
     * Returns a {@link Maybe} that applies a specified accumulator function to the first item emitted by the current
     * {@code Flowable}, then feeds the result of that function along with the second item emitted by the current
     * {@code Flowable} into the same function, and so on until all items have been emitted by the current and finite {@code Flowable},
     * and emits the final result from the final call to your function as its sole item.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/reduce.v3.png" alt="">
     * <p>
     * This technique, which is called "reduce" here, is sometimes called "aggregate," "fold," "accumulate,"
     * "compress," or "inject" in other programming contexts. Groovy, for instance, has an {@code inject} method
     * that does a similar operation on lists.
     * <p>
     * Note that this operator requires the upstream to signal {@code onComplete} for the accumulator object to
     * be emitted. Sources that are infinite and never complete will never emit anything through this
     * operator and an infinite source may lead to a fatal {@link OutOfMemoryError}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure of its downstream consumer and consumes the
     *  upstream source in unbounded mode.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code reduce} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param reducer
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, whose
     *            result will be used in the next accumulator call
     * @return the new {@code Maybe} instance
     * @throws NullPointerException if {@code reducer} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/reduce.html">ReactiveX operators documentation: Reduce</a>
     * @see <a href="http://en.wikipedia.org/wiki/Fold_(higher-order_function)">Wikipedia: Fold (higher-order function)</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Maybe<T> reduce(@NonNull BiFunction<T, T, T> reducer) {
        Objects.requireNonNull(reducer, "reducer is null");
        return RxJavaPlugins.onAssembly(new FlowableReduceMaybe<>(this, reducer));
    }

    /**
     * Returns a {@link Single} that applies a specified accumulator function to the first item emitted by the current
     * {@code Flowable} and a specified seed value, then feeds the result of that function along with the second item
     * emitted by the current {@code Flowable} into the same function, and so on until all items have been emitted by the
     * current and finite {@code Flowable}, emitting the final result from the final call to your function as its sole item.
     * <p>
     * <img width="640" height="325" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/reduceSeed.v3.png" alt="">
     * <p>
     * This technique, which is called "reduce" here, is sometimes called "aggregate," "fold," "accumulate,"
     * "compress," or "inject" in other programming contexts. Groovy, for instance, has an {@code inject} method
     * that does a similar operation on lists.
     * <p>
     * Note that the {@code seed} is shared among all subscribers to the resulting {@code Flowable}
     * and may cause problems if it is mutable. To make sure each subscriber gets its own value, defer
     * the application of this operator via {@link #defer(Supplier)}:
     * <pre><code>
     * Flowable&lt;T&gt; source = ...
     * Single.defer(() -&gt; source.reduce(new ArrayList&lt;&gt;(), (list, item) -&gt; list.add(item)));
     *
     * // alternatively, by using compose to stay fluent
     *
     * source.compose(o -&gt;
     *     Flowable.defer(() -&gt; o.reduce(new ArrayList&lt;&gt;(), (list, item) -&gt; list.add(item)).toFlowable())
     * ).firstOrError();
     *
     * // or, by using reduceWith instead of reduce
     *
     * source.reduceWith(() -&gt; new ArrayList&lt;&gt;(), (list, item) -&gt; list.add(item)));
     * </code></pre>
     * <p>
     * Note that this operator requires the upstream to signal {@code onComplete} for the accumulator object to
     * be emitted. Sources that are infinite and never complete will never emit anything through this
     * operator and an infinite source may lead to a fatal {@link OutOfMemoryError}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure of its downstream consumer and consumes the
     *  upstream source in unbounded mode.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code reduce} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the accumulator and output value type
     * @param seed
     *            the initial (seed) accumulator value
     * @param reducer
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, the
     *            result of which will be used in the next accumulator call
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code seed} or {@code reducer} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/reduce.html">ReactiveX operators documentation: Reduce</a>
     * @see <a href="http://en.wikipedia.org/wiki/Fold_(higher-order_function)">Wikipedia: Fold (higher-order function)</a>
     * @see #reduceWith(Supplier, BiFunction)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull R> Single<R> reduce(R seed, @NonNull BiFunction<R, ? super T, R> reducer) {
        Objects.requireNonNull(seed, "seed is null");
        Objects.requireNonNull(reducer, "reducer is null");
        return RxJavaPlugins.onAssembly(new FlowableReduceSeedSingle<>(this, seed, reducer));
    }

    /**
     * Returns a {@link Single} that applies a specified accumulator function to the first item emitted by the current
     * {@code Flowable} and a seed value derived from calling a specified {@code seedSupplier}, then feeds the result
     * of that function along with the second item emitted by the current {@code Flowable} into the same function, and so on until
     * all items have been emitted by the current and finite {@code Flowable}, emitting the final result from the final call to your
     * function as its sole item.
     * <p>
     * <img width="640" height="325" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/reduceSeed.v3.png" alt="">
     * <p>
     * This technique, which is called "reduce" here, is sometimes called "aggregate", "fold", "accumulate",
     * "compress", or "inject" in other programming contexts. Groovy, for instance, has an {@code inject} method
     * that does a similar operation on lists.
     * <p>
     * Note that this operator requires the upstream to signal {@code onComplete} for the accumulator object to
     * be emitted. Sources that are infinite and never complete will never emit anything through this
     * operator and an infinite source may lead to a fatal {@link OutOfMemoryError}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure of its downstream consumer and consumes the
     *  upstream source in unbounded mode.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code reduceWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the accumulator and output value type
     * @param seedSupplier
     *            the {@link Supplier} that provides the initial (seed) accumulator value for each individual {@link Subscriber}
     * @param reducer
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, the
     *            result of which will be used in the next accumulator call
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code seedSupplier} or {@code reducer} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/reduce.html">ReactiveX operators documentation: Reduce</a>
     * @see <a href="http://en.wikipedia.org/wiki/Fold_(higher-order_function)">Wikipedia: Fold (higher-order function)</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull R> Single<R> reduceWith(@NonNull Supplier<R> seedSupplier, @NonNull BiFunction<R, ? super T, R> reducer) {
        Objects.requireNonNull(seedSupplier, "seedSupplier is null");
        Objects.requireNonNull(reducer, "reducer is null");
        return RxJavaPlugins.onAssembly(new FlowableReduceWithSingle<>(this, seedSupplier, reducer));
    }

    /**
     * Returns a {@code Flowable} that repeats the sequence of items emitted by the current {@code Flowable} indefinitely.
     * <p>
     * <img width="640" height="309" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/repeat.o.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code repeat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/repeat.html">ReactiveX operators documentation: Repeat</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> repeat() {
        return repeat(Long.MAX_VALUE);
    }

    /**
     * Returns a {@code Flowable} that repeats the sequence of items emitted by the current {@code Flowable} at most
     * {@code count} times.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/repeat.on.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code repeat} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param times
     *            the number of times the current {@code Flowable} items are repeated, a count of 0 will yield an empty
     *            sequence
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException
     *             if {@code times} is less than zero
     * @see <a href="http://reactivex.io/documentation/operators/repeat.html">ReactiveX operators documentation: Repeat</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> repeat(long times) {
        if (times < 0) {
            throw new IllegalArgumentException("times >= 0 required but it was " + times);
        }
        if (times == 0) {
            return empty();
        }
        return RxJavaPlugins.onAssembly(new FlowableRepeat<>(this, times));
    }

    /**
     * Returns a {@code Flowable} that repeats the sequence of items emitted by the current {@code Flowable} until
     * the provided stop function returns {@code true}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/repeat.on.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code repeatUntil} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param stop
     *                a boolean supplier that is called when the current {@code Flowable} completes and unless it returns
     *                {@code false}, the current {@code Flowable} is resubscribed
     * @return the new {@code Flowable} instance
     * @throws NullPointerException
     *             if {@code stop} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/repeat.html">ReactiveX operators documentation: Repeat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> repeatUntil(@NonNull BooleanSupplier stop) {
        Objects.requireNonNull(stop, "stop is null");
        return RxJavaPlugins.onAssembly(new FlowableRepeatUntil<>(this, stop));
    }

    /**
     * Returns a {@code Flowable} that emits the same values as the current {@code Flowable} with the exception of an
     * {@code onComplete}. An {@code onComplete} notification from the source will result in the emission of
     * a {@code void} item to the {@code Flowable} provided as an argument to the {@code notificationHandler}
     * function. If that {@link Publisher} calls {@code onComplete} or {@code onError} then {@code repeatWhen} will
     * call {@code onComplete} or {@code onError} on the child subscription. Otherwise, this {@code Publisher} will
     * resubscribe to the current {@code Flowable}.
     * <p>
     * <img width="640" height="430" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/repeatWhen.f.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code repeatWhen} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param handler
     *            receives a {@code Publisher} of notifications with which a user can complete or error, aborting the repeat.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code handler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/repeat.html">ReactiveX operators documentation: Repeat</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> repeatWhen(@NonNull Function<? super Flowable<Object>, ? extends Publisher<@NonNull ?>> handler) {
        Objects.requireNonNull(handler, "handler is null");
        return RxJavaPlugins.onAssembly(new FlowableRepeatWhen<>(this, handler));
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the underlying {@link Publisher}
     * that will replay all of its items and notifications to any future {@link Subscriber}. A connectable
     * {@code Flowable} resembles an ordinary {@code Flowable}, except that it does not begin emitting items when it is
     * subscribed to, but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code ConnectableFlowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final ConnectableFlowable<T> replay() {
        return FlowableReplay.createFrom(this);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on the items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable}.
     * <p>
     * <img width="640" height="450" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.f.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            the selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector) {
        Objects.requireNonNull(selector, "selector is null");
        return FlowableReplay.multicastSelector(FlowableInternalHelper.replaySupplier(this), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying {@code bufferSize} notifications.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * <p>
     * <img width="640" height="440" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fn.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            the selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param bufferSize
     *            the buffer size that limits the number of items the operator can replay
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(Function, int, boolean)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, int bufferSize) {
        Objects.requireNonNull(selector, "selector is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.multicastSelector(FlowableInternalHelper.replaySupplier(this, bufferSize, false), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying {@code bufferSize} notifications.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * <p>
     * <img width="640" height="440" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fn.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            the selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param bufferSize
     *            the buffer size that limits the number of items the operator can replay
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given bufferSize, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, int bufferSize, boolean eagerTruncate) {
        Objects.requireNonNull(selector, "selector is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.multicastSelector(FlowableInternalHelper.replaySupplier(this, bufferSize, eagerTruncate), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying no more than {@code bufferSize} items that were emitted within a specified time window.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * <p>
     * <img width="640" height="445" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fnt.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param bufferSize
     *            the buffer size that limits the number of items the operator can replay
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} or {@code unit} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, int bufferSize, long time, @NonNull TimeUnit unit) {
        return replay(selector, bufferSize, time, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying no more than {@code bufferSize} items that were emitted within a specified time window.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * <p>
     * <img width="640" height="445" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fnts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param bufferSize
     *            the buffer size that limits the number of items the operator can replay
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that is the time source for the window
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector}, {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(Function, int, long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, int bufferSize, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(selector, "selector is null");
        Objects.requireNonNull(unit, "unit is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.multicastSelector(
                FlowableInternalHelper.replaySupplier(this, bufferSize, time, unit, scheduler, false), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying no more than {@code bufferSize} items that were emitted within a specified time window.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * <p>
     * <img width="640" height="445" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fnts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param bufferSize
     *            the buffer size that limits the number of items the operator can replay
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that is the time source for the window
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given bufferSize/age, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector}, {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code bufferSize} is less than zero
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, int bufferSize, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean eagerTruncate) {
        Objects.requireNonNull(selector, "selector is null");
        Objects.requireNonNull(unit, "unit is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.multicastSelector(
                FlowableInternalHelper.replaySupplier(this, bufferSize, time, unit, scheduler, eagerTruncate), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying all items that were emitted within a specified time window.
     * <p>
     * <img width="640" height="435" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.ft.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector} or {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, long time, @NonNull TimeUnit unit) {
        return replay(selector, time, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying all items that were emitted within a specified time window.
     * <p>
     * <img width="640" height="440" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler that is the time source for the window
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector}, {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(Function, long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(selector, "selector is null");
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.multicastSelector(FlowableInternalHelper.replaySupplier(this, time, unit, scheduler, false), selector);
    }

    /**
     * Returns a {@code Flowable} that emits items that are the results of invoking a specified selector on items
     * emitted by a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable},
     * replaying all items that were emitted within a specified time window.
     * <p>
     * <img width="640" height="440" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.fts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param <R>
     *            the type of items emitted by the resulting {@code Flowable}
     * @param selector
     *            a selector function, which can use the multicasted sequence as many times as needed, without
     *            causing multiple subscriptions to the current {@code Flowable}
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler that is the time source for the window
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given age, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code selector}, {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final <R> Flowable<R> replay(@NonNull Function<? super Flowable<T>, ? extends Publisher<R>> selector, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean eagerTruncate) {
        Objects.requireNonNull(selector, "selector is null");
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.multicastSelector(FlowableInternalHelper.replaySupplier(this, time, unit, scheduler, eagerTruncate), selector);
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays at most {@code bufferSize} items to late {@link Subscriber}s. A Connectable {@code Flowable} resembles
     * an ordinary {@code Flowable}, except that it does not begin emitting items when it is subscribed to, but only
     * when its {@code connect} method is called.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * To ensure no beyond-bufferSize items are referenced,
     * use the {@link #replay(int, boolean)} overload with {@code eagerTruncate = true}.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.n.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the buffer size that limits the number of items that can be replayed
     * @return the new {@code ConnectableFlowable} instance
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(int, boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final ConnectableFlowable<T> replay(int bufferSize) {
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.create(this, bufferSize, false);
    }
    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays at most {@code bufferSize} items to late {@link Subscriber}s. A connectable {@code Flowable} resembles
     * an ordinary {@code Flowable}, except that it does not begin emitting items when it is subscribed to, but only
     * when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.n.v3.png" alt="">
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * To ensure no beyond-bufferSize items are referenced, set {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the buffer size that limits the number of items that can be replayed
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given bufferSize, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code ConnectableFlowable} instance
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final ConnectableFlowable<T> replay(int bufferSize, boolean eagerTruncate) {
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.create(this, bufferSize, eagerTruncate);
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays at most {@code bufferSize} items that were emitted during a specified time window. A connectable
     * {@code Flowable} resembles an ordinary {@code Flowable}, except that it does not begin emitting items when it is
     * subscribed to, but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.nt.v3.png" alt="">
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * To ensure no out-of-date or beyond-bufferSize items are referenced,
     * use the {@link #replay(int, long, TimeUnit, Scheduler, boolean)} overload with {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@link Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the buffer size that limits the number of items that can be replayed
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(int, long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final ConnectableFlowable<T> replay(int bufferSize, long time, @NonNull TimeUnit unit) {
        return replay(bufferSize, time, unit, Schedulers.computation());
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays a maximum of {@code bufferSize} items that are emitted within a specified time window to late {@link Subscriber}s. A
     * connectable {@code Flowable} resembles an ordinary {@code Flowable}, except that it does not begin emitting items
     * when it is subscribed to, but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.nts.v3.png" alt="">
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions.
     * To ensure no out-of-date or beyond-bufferSize items are referenced,
     * use the {@link #replay(int, long, TimeUnit, Scheduler, boolean)} overload with {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the buffer size that limits the number of items that can be replayed
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler that is used as a time source for the window
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(int, long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final ConnectableFlowable<T> replay(int bufferSize, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.create(this, time, unit, scheduler, bufferSize, false);
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays a maximum of {@code bufferSize} items that are emitted within a specified time window to late {@link Subscriber}s. A
     * connectable {@code Flowable} resembles an ordinary {@code Flowable}, except that it does not begin emitting items
     * when it is subscribed to, but only when its {@code connect} method is called.
     * <p>
     * Note that due to concurrency requirements, {@code replay(bufferSize)} may hold strong references to more than
     * {@code bufferSize} source emissions. To ensure no out-of-date or beyond-bufferSize items
     * are referenced, set {@code eagerTruncate = true}.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.nts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param bufferSize
     *            the buffer size that limits the number of items that can be replayed
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler that is used as a time source for the window
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given bufferSize/age, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final ConnectableFlowable<T> replay(int bufferSize, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean eagerTruncate) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        return FlowableReplay.create(this, time, unit, scheduler, bufferSize, eagerTruncate);
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays all items emitted by it within a specified time window to late {@link Subscriber}s. A connectable {@code Flowable}
     * resembles an ordinary {@code Flowable}, except that it does not begin emitting items when it is subscribed to,
     * but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.t.v3.png" alt="">
     * <p>
     * Note that the internal buffer may retain strong references to the oldest item. To ensure no out-of-date items
     * are referenced, use the {@link #replay(long, TimeUnit, Scheduler, boolean)} overload with {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code replay} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final ConnectableFlowable<T> replay(long time, @NonNull TimeUnit unit) {
        return replay(time, unit, Schedulers.computation());
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays all items emitted by it within a specified time window to late {@link Subscriber}s. A connectable {@code Flowable}
     * resembles an ordinary {@code Flowable}, except that it does not begin emitting items when it is subscribed to,
     * but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.ts.v3.png" alt="">
     * <p>
     * Note that the internal buffer may retain strong references to the oldest item. To ensure no out-of-date items
     * are referenced, use the {@link #replay(long, TimeUnit, Scheduler, boolean)} overload with {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that is the time source for the window
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     * @see #replay(long, TimeUnit, Scheduler, boolean)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final ConnectableFlowable<T> replay(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.create(this, time, unit, scheduler, false);
    }

    /**
     * Returns a {@link ConnectableFlowable} that shares a single subscription to the current {@code Flowable} and
     * replays all items emitted by it within a specified time window to late {@link Subscriber}s. A connectable {@code Flowable}
     * resembles an ordinary {@code Flowable}, except that it does not begin emitting items when it is subscribed to,
     * but only when its {@code connect} method is called.
     * <p>
     * <img width="640" height="515" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/replay.ts.v3.png" alt="">
     * <p>
     * Note that the internal buffer may retain strong references to the oldest item. To ensure no out-of-date items
     * are referenced, set {@code eagerTruncate = true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator supports backpressure. Note that the upstream requests are determined by the child
     *  {@code Subscriber} which requests the largest amount: i.e., two child {@code Subscriber}s with requests of 10 and 100 will
     *  request 100 elements from the current {@code Flowable} sequence.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the duration of the window in which the replayed items must have been emitted
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that is the time source for the window
     * @param eagerTruncate
     *            if {@code true}, whenever the internal buffer is truncated to the given bufferSize/age, the
     *            oldest item will be guaranteed dereferenced, thus avoiding unexpected retention
     * @return the new {@code ConnectableFlowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/replay.html">ReactiveX operators documentation: Replay</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final ConnectableFlowable<T> replay(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean eagerTruncate) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return FlowableReplay.create(this, time, unit, scheduler, eagerTruncate);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, resubscribing to it if it calls {@code onError}
     * (infinite retry count).
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/retry.v3.png" alt="">
     * <p>
     * If the current {@code Flowable} calls {@link Subscriber#onError}, this method will resubscribe to the current
     * {@code Flowable} rather than propagating the {@code onError} call.
     * <p>
     * Any and all items emitted by the current {@code Flowable} will be emitted by the resulting {@code Flowable}, even
     * those emitted during failed subscriptions. For example, if the current {@code Flowable} fails at first but emits
     * {@code [1, 2]} then succeeds the second time and emits {@code [1, 2, 3, 4, 5]} then the complete sequence
     * of emissions and notifications would be {@code [1, 2, 1, 2, 3, 4, 5, onComplete]}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retry} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/retry.html">ReactiveX operators documentation: Retry</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> retry() {
        return retry(Long.MAX_VALUE, Functions.alwaysTrue());
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, resubscribing to it if it calls {@code onError}
     * and the predicate returns {@code true} for that specific exception and retry count.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/retry.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retry} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            the predicate that determines if a resubscription may happen in case of a specific exception
     *            and retry count
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see #retry()
     * @see <a href="http://reactivex.io/documentation/operators/retry.html">ReactiveX operators documentation: Retry</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> retry(@NonNull BiPredicate<@NonNull ? super Integer, @NonNull ? super Throwable> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");

        return RxJavaPlugins.onAssembly(new FlowableRetryBiPredicate<>(this, predicate));
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, resubscribing to it if it calls {@code onError}
     * up to a specified number of retries.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/retry.v3.png" alt="">
     * <p>
     * If the current {@code Flowable} calls {@link Subscriber#onError}, this method will resubscribe to the current
     * {@code Flowable} for a maximum of {@code count} resubscriptions rather than propagating the
     * {@code onError} call.
     * <p>
     * Any and all items emitted by the current {@code Flowable} will be emitted by the resulting {@code Flowable}, even
     * those emitted during failed subscriptions. For example, if the current {@code Flowable} fails at first but emits
     * {@code [1, 2]} then succeeds the second time and emits {@code [1, 2, 3, 4, 5]} then the complete sequence
     * of emissions and notifications would be {@code [1, 2, 1, 2, 3, 4, 5, onComplete]}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retry} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param times
     *            the number of times to resubscribe if the current {@code Flowable} fails
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code times} is negative
     * @see <a href="http://reactivex.io/documentation/operators/retry.html">ReactiveX operators documentation: Retry</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> retry(long times) {
        return retry(times, Functions.alwaysTrue());
    }

    /**
     * Retries at most times or until the predicate returns {@code false}, whichever happens first.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retry} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param times the number of times to resubscribe if the current {@code Flowable} fails
     * @param predicate the predicate called with the failure {@link Throwable} and should return {@code true} to trigger a retry.
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @throws IllegalArgumentException if {@code times} is negative
     * @return the new {@code Flowable} instance
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> retry(long times, @NonNull Predicate<@NonNull ? super Throwable> predicate) {
        if (times < 0) {
            throw new IllegalArgumentException("times >= 0 required but it was " + times);
        }
        Objects.requireNonNull(predicate, "predicate is null");

        return RxJavaPlugins.onAssembly(new FlowableRetryPredicate<>(this, times, predicate));
    }

    /**
     * Retries the current {@code Flowable} if the predicate returns {@code true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retry} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate the predicate that receives the failure {@link Throwable} and should return {@code true} to trigger a retry.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> retry(@NonNull Predicate<@NonNull ? super Throwable> predicate) {
        return retry(Long.MAX_VALUE, predicate);
    }

    /**
     * Retries until the given stop function returns {@code true}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retryUntil} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param stop the function that should return {@code true} to stop retrying
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code stop} is {@code null}
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> retryUntil(@NonNull BooleanSupplier stop) {
        Objects.requireNonNull(stop, "stop is null");
        return retry(Long.MAX_VALUE, Functions.predicateReverseFor(stop));
    }

    /**
     * Returns a {@code Flowable} that emits the same values as the current {@code Flowable} with the exception of an
     * {@code onError}. An {@code onError} notification from the source will result in the emission of a
     * {@link Throwable} item to the {@code Flowable} provided as an argument to the {@code notificationHandler}
     * function. If that {@link Publisher} calls {@code onComplete} or {@code onError} then {@code retry} will call
     * {@code onComplete} or {@code onError} on the child subscription. Otherwise, this {@code Publisher} will
     * resubscribe to the current {@code Flowable}.
     * <p>
     * <img width="640" height="430" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/retryWhen.f.v3.png" alt="">
     * <p>
     * Example:
     *
     * This retries 3 times, each time incrementing the number of seconds it waits.
     *
     * <pre><code>
     *  Flowable.create((FlowableEmitter&lt;? super String&gt; s) -&gt; {
     *      System.out.println("subscribing");
     *      s.onError(new RuntimeException("always fails"));
     *  }, BackpressureStrategy.BUFFER).retryWhen(attempts -&gt; {
     *      return attempts.zipWith(Flowable.range(1, 3), (n, i) -&gt; i).flatMap(i -&gt; {
     *          System.out.println("delay retry by " + i + " second(s)");
     *          return Flowable.timer(i, TimeUnit.SECONDS);
     *      });
     *  }).blockingForEach(System.out::println);
     * </code></pre>
     *
     * Output is:
     *
     * <pre> {@code
     * subscribing
     * delay retry by 1 second(s)
     * subscribing
     * delay retry by 2 second(s)
     * subscribing
     * delay retry by 3 second(s)
     * subscribing
     * } </pre>
     * <p>
     * Note that the inner {@code Publisher} returned by the handler function should signal
     * either {@code onNext}, {@code onError} or {@code onComplete} in response to the received
     * {@code Throwable} to indicate the operator should retry or terminate. If the upstream to
     * the operator is asynchronous, signaling {@code onNext} followed by {@code onComplete} immediately may
     * result in the sequence to be completed immediately. Similarly, if this inner
     * {@code Publisher} signals {@code onError} or {@code onComplete} while the upstream is
     * active, the sequence is terminated with the same signal immediately.
     * <p>
     * The following example demonstrates how to retry an asynchronous source with a delay:
     * <pre><code>
     * Flowable.timer(1, TimeUnit.SECONDS)
     *     .doOnSubscribe(s -&gt; System.out.println("subscribing"))
     *     .map(v -&gt; { throw new RuntimeException(); })
     *     .retryWhen(errors -&gt; {
     *         AtomicInteger counter = new AtomicInteger();
     *         return errors
     *                   .takeWhile(e -&gt; counter.getAndIncrement() != 3)
     *                   .flatMap(e -&gt; {
     *                       System.out.println("delay retry by " + counter.get() + " second(s)");
     *                       return Flowable.timer(counter.get(), TimeUnit.SECONDS);
     *                   });
     *     })
     *     .blockingSubscribe(System.out::println, System.out::println);
     * </code></pre>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects both the source
     *  and inner {@code Publisher}s to honor backpressure as well.
     *  If this expectation is violated, the operator <em>may</em> throw an {@link IllegalStateException}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code retryWhen} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param handler
     *            receives a {@code Publisher} of notifications with which a user can complete or error, aborting the
     *            retry
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code handler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/retry.html">ReactiveX operators documentation: Retry</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> retryWhen(
            @NonNull Function<? super Flowable<Throwable>, ? extends Publisher<@NonNull ?>> handler) {
        Objects.requireNonNull(handler, "handler is null");

        return RxJavaPlugins.onAssembly(new FlowableRetryWhen<>(this, handler));
    }

    /**
     * Subscribes to the current {@code Flowable} and wraps the given {@link Subscriber} into a {@link SafeSubscriber}
     * (if not already a {@code SafeSubscriber}) that
     * deals with exceptions thrown by a misbehaving {@code Subscriber} (that doesn't follow the
     * <em>Reactive Streams</em> specification).
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator leaves the reactive world and the backpressure behavior depends on the {@code Subscriber}'s behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code safeSubscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param subscriber the incoming {@code Subscriber} instance
     * @throws NullPointerException if {@code subscriber} is {@code null}
     */
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void safeSubscribe(@NonNull Subscriber<@NonNull ? super T> subscriber) {
        Objects.requireNonNull(subscriber, "subscriber is null");
        if (subscriber instanceof SafeSubscriber) {
            subscribe((SafeSubscriber<@NonNull ? super T>)subscriber);
        } else {
            subscribe(new SafeSubscriber<>(subscriber));
        }
    }

    /**
     * Returns a {@code Flowable} that emits the most recently emitted item (if any) emitted by the current {@code Flowable}
     * within periodic time intervals.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sample} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param period
     *            the sampling rate
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleLast(long, TimeUnit)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> sample(long period, @NonNull TimeUnit unit) {
        return sample(period, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits the most recently emitted item (if any) emitted by the current {@code Flowable}
     * within periodic time intervals and optionally emit the very last upstream item when the upstream completes.
     * <p>
     * <img width="640" height="277" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.emitlast.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sample} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * <p>History: 2.0.5 - experimental
     * @param period
     *            the sampling rate
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @param emitLast
     *            if {@code true}, and the upstream completes while there is still an unsampled item available,
     *            that item is emitted to downstream before completion
     *            if {@code false}, an unsampled last item is ignored.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleLast(long, TimeUnit)
     * @since 2.1
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> sample(long period, @NonNull TimeUnit unit, boolean emitLast) {
        return sample(period, unit, Schedulers.computation(), emitLast);
    }

    /**
     * Returns a {@code Flowable} that emits the most recently emitted item (if any) emitted by the current {@code Flowable}
     * within periodic time intervals, where the intervals are defined on a particular {@link Scheduler}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param period
     *            the sampling rate
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @param scheduler
     *            the {@code Scheduler} to use when sampling
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleLast(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> sample(long period, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableSampleTimed<>(this, period, unit, scheduler, false));
    }

    /**
     * Returns a {@code Flowable} that emits the most recently emitted item (if any) emitted by the current {@code Flowable}
     * within periodic time intervals, where the intervals are defined on a particular {@link Scheduler}
     * and optionally emit the very last upstream item when the upstream completes.
     * <p>
     * <img width="640" height="277" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.s.emitlast.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * <p>History: 2.0.5 - experimental
     * @param period
     *            the sampling rate
     * @param unit
     *            the {@link TimeUnit} in which {@code period} is defined
     * @param scheduler
     *            the {@code Scheduler} to use when sampling
     * @param emitLast
     *            if {@code true} and the upstream completes while there is still an unsampled item available,
     *            that item is emitted to downstream before completion
     *            if {@code false}, an unsampled last item is ignored.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #throttleLast(long, TimeUnit, Scheduler)
     * @since 2.1
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> sample(long period, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean emitLast) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableSampleTimed<>(this, period, unit, scheduler, emitLast));
    }

    /**
     * Returns a {@code Flowable} that, when the specified {@code sampler} {@link Publisher} emits an item or completes,
     * emits the most recently emitted item (if any) emitted by the current {@code Flowable} since the previous
     * emission from the {@code sampler} {@code Publisher}.
     * <p>
     * <img width="640" height="290" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.o.nolast.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses the emissions of the {@code sampler}
     *      {@code Publisher} to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code sample} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the element type of the sampler {@code Publisher}
     * @param sampler
     *            the {@code Publisher} to use for sampling the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sampler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> sample(@NonNull Publisher<U> sampler) {
        Objects.requireNonNull(sampler, "sampler is null");
        return RxJavaPlugins.onAssembly(new FlowableSamplePublisher<>(this, sampler, false));
    }

    /**
     * Returns a {@code Flowable} that, when the specified {@code sampler} {@link Publisher} emits an item or completes,
     * emits the most recently emitted item (if any) emitted by the current {@code Flowable} since the previous
     * emission from the {@code sampler} {@code Publisher}
     * and optionally emit the very last upstream item when the upstream or other {@code Publisher} complete.
     * <p>
     * <img width="640" height="290" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/sample.o.emitlast.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses the emissions of the {@code sampler}
     *      {@code Publisher} to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code sample} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * <p>History: 2.0.5 - experimental
     * @param <U> the element type of the sampler {@code Publisher}
     * @param sampler
     *            the {@code Publisher} to use for sampling the current {@code Flowable}
     * @param emitLast
     *            if {@code true} and the upstream completes while there is still an unsampled item available,
     *            that item is emitted to downstream before completion
     *            if {@code false}, an unsampled last item is ignored.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code sampler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @since 2.1
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> sample(@NonNull Publisher<U> sampler, boolean emitLast) {
        Objects.requireNonNull(sampler, "sampler is null");
        return RxJavaPlugins.onAssembly(new FlowableSamplePublisher<>(this, sampler, emitLast));
    }

    /**
     * Returns a {@code Flowable} that emits the first value emitted by the current {@code Flowable}, then emits one value
     * for each subsequent value emitted by the current {@code Flowable}. Each emission after the first is the result of
     * applying the specified accumulator function to the previous emission and the corresponding value from the current {@code Flowable}.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/scan.v3.png" alt="">
     * <p>
     * This sort of function is sometimes called an accumulator.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  Violating this expectation, a {@link MissingBackpressureException} <em>may</em> get signaled somewhere downstream.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code scan} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param accumulator
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, whose
     *            result will be emitted to {@link Subscriber}s via {@link Subscriber#onNext onNext} and used in the
     *            next accumulator call
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code accumulator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/scan.html">ReactiveX operators documentation: Scan</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> scan(@NonNull BiFunction<T, T, T> accumulator) {
        Objects.requireNonNull(accumulator, "accumulator is null");
        return RxJavaPlugins.onAssembly(new FlowableScan<>(this, accumulator));
    }

    /**
     * Returns a {@code Flowable} that emits the provided initial (seed) value, then emits one value for each value emitted
     * by the current {@code Flowable}. Each emission after the first is the result of applying the specified accumulator
     * function to the previous emission and the corresponding value from the current {@code Flowable}.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/scanSeed.v3.png" alt="">
     * <p>
     * This sort of function is sometimes called an accumulator.
     * <p>
     * Note that the {@code Flowable} that results from this method will emit {@code initialValue} as its first
     * emitted item.
     * <p>
     * Note that the {@code initialValue} is shared among all subscribers to the resulting {@code Flowable}
     * and may cause problems if it is mutable. To make sure each subscriber gets its own value, defer
     * the application of this operator via {@link #defer(Supplier)}:
     * <pre><code>
     * Publisher&lt;T&gt; source = ...
     * Flowable.defer(() -&gt; source.scan(new ArrayList&lt;&gt;(), (list, item) -&gt; list.add(item)));
     *
     * // alternatively, by using compose to stay fluent
     *
     * source.compose(o -&gt;
     *     Flowable.defer(() -&gt; o.scan(new ArrayList&lt;&gt;(), (list, item) -&gt; list.add(item)))
     * );
     * </code></pre>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  Violating this expectation, a {@link MissingBackpressureException} <em>may</em> get signaled somewhere downstream.
     *  The downstream request pattern is not preserved across this operator.
     *  The upstream is requested {@link #bufferSize()} - 1 upfront and 75% of {@link #bufferSize()} thereafter.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code scan} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the initial, accumulator and result type
     * @param initialValue
     *            the initial (seed) accumulator item
     * @param accumulator
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, whose
     *            result will be emitted to {@link Subscriber}s via {@link Subscriber#onNext onNext} and used in the
     *            next accumulator call
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code initialValue} or {@code accumulator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/scan.html">ReactiveX operators documentation: Scan</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull R> Flowable<R> scan(R initialValue, @NonNull BiFunction<R, ? super T, R> accumulator) {
        Objects.requireNonNull(initialValue, "initialValue is null");
        return scanWith(Functions.justSupplier(initialValue), accumulator);
    }

    /**
     * Returns a {@code Flowable} that emits the provided initial (seed) value, then emits one value for each value emitted
     * by the current {@code Flowable}. Each emission after the first is the result of applying the specified accumulator
     * function to the previous emission and the corresponding value from the current {@code Flowable}.
     * <p>
     * <img width="640" height="320" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/scanSeed.v3.png" alt="">
     * <p>
     * This sort of function is sometimes called an accumulator.
     * <p>
     * Note that the {@code Flowable} that results from this method will emit the value returned by
     * the {@code seedSupplier} as its first item.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors downstream backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  Violating this expectation, a {@link MissingBackpressureException} <em>may</em> get signaled somewhere downstream.
     *  The downstream request pattern is not preserved across this operator.
     *  The upstream is requested {@link #bufferSize()} - 1 upfront and 75% of {@link #bufferSize()} thereafter.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code scanWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the initial, accumulator and result type
     * @param seedSupplier
     *            a {@link Supplier} that returns the initial (seed) accumulator item for each individual {@link Subscriber}
     * @param accumulator
     *            an accumulator function to be invoked on each item emitted by the current {@code Flowable}, whose
     *            result will be emitted to {@code Subscriber}s via {@link Subscriber#onNext onNext} and used in the
     *            next accumulator call
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code seedSupplier} or {@code accumulator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/scan.html">ReactiveX operators documentation: Scan</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <@NonNull R> Flowable<R> scanWith(@NonNull Supplier<R> seedSupplier, @NonNull BiFunction<R, ? super T, R> accumulator) {
        Objects.requireNonNull(seedSupplier, "seedSupplier is null");
        Objects.requireNonNull(accumulator, "accumulator is null");
        return RxJavaPlugins.onAssembly(new FlowableScanSeed<>(this, seedSupplier, accumulator));
    }

    /**
     * Forces the current {@code Flowable}'s emissions and notifications to be serialized and for it to obey
     * <a href="http://reactivex.io/documentation/contract.html">the {@code Publisher} contract</a> in other ways.
     * <p>
     * It is possible for a {@link Publisher} to invoke its {@link Subscriber}s' methods asynchronously, perhaps from
     * different threads. This could make such a {@code Publisher} poorly-behaved, in that it might try to invoke
     * {@code onComplete} or {@code onError} before one of its {@code onNext} invocations, or it might call
     * {@code onNext} from two different threads concurrently. You can force such a {@code Publisher} to be
     * well-behaved and sequential by applying the {@code serialize} method to it.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/synchronize.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code serialize} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/serialize.html">ReactiveX operators documentation: Serialize</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> serialize() {
        return RxJavaPlugins.onAssembly(new FlowableSerialized<>(this));
    }

    /**
     * Returns a new {@code Flowable} that multicasts (and shares a single subscription to) the current {@code Flowable}. As long as
     * there is at least one {@link Subscriber}, the current {@code Flowable} will be subscribed and emitting data.
     * When all subscribers have canceled it will cancel the current {@code Flowable}.
     * <p>
     * This is an alias for {@link #publish()}.{@link ConnectableFlowable#refCount() refCount()}.
     * <p>
     * <img width="640" height="510" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/publishRefCount.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure and expects the current {@code Flowable} to honor backpressure as well.
     *  If this expectation is violated, the operator will signal a {@link MissingBackpressureException} to
     *  its {@code Subscriber}s.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code share} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/refcount.html">ReactiveX operators documentation: RefCount</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> share() {
        return publish().refCount();
    }

    /**
     * Returns a {@link Maybe} that completes if this {@code Flowable} is empty, signals one item if this {@code Flowable}
     * signals exactly one item or signals an {@link IllegalArgumentException} if this {@code Flowable} signals
     * more than one item.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/single.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code singleElement} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Maybe} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Maybe<T> singleElement() {
        return RxJavaPlugins.onAssembly(new FlowableSingleMaybe<>(this));
    }

    /**
     * Returns a {@link Single} that emits the single item emitted by the current {@code Flowable} if it
     * emits only a single item, or a default item if the current {@code Flowable} emits no items. If the current
     * {@code Flowable} emits more than one item, an {@link IllegalArgumentException} is signaled instead.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/singleOrDefault.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code single} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param defaultItem
     *            a default value to emit if the current {@code Flowable} emits no item
     * @return the new {@code Single} instance
     * @throws NullPointerException if {@code defaultItem} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Single<T> single(@NonNull T defaultItem) {
        Objects.requireNonNull(defaultItem, "defaultItem is null");
        return RxJavaPlugins.onAssembly(new FlowableSingleSingle<>(this, defaultItem));
    }

    /**
     * Returns a {@link Single} that emits the single item emitted by this {@code Flowable}, if this {@code Flowable}
     * emits only a single item, otherwise
     * if this {@code Flowable} completes without emitting any items a {@link NoSuchElementException} will be signaled and
     * if this {@code Flowable} emits more than one item, an {@link IllegalArgumentException} will be signaled.
     * <p>
     * <img width="640" height="206" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/singleOrError.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code singleOrError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Single} instance
     * @see <a href="http://reactivex.io/documentation/operators/first.html">ReactiveX operators documentation: First</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Single<T> singleOrError() {
        return RxJavaPlugins.onAssembly(new FlowableSingleSingle<>(this, null));
    }

    /**
     * Returns a {@code Flowable} that skips the first {@code count} items emitted by the current {@code Flowable} and emits
     * the remainder.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skip.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code skip} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the number of items to skip
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code count} is negative
     * @see <a href="http://reactivex.io/documentation/operators/skip.html">ReactiveX operators documentation: Skip</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> skip(long count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 expected but it was " + count);
        }
        if (count == 0) {
            return RxJavaPlugins.onAssembly(this);
        }
        return RxJavaPlugins.onAssembly(new FlowableSkip<>(this, count));
    }

    /**
     * Returns a {@code Flowable} that skips values emitted by the current {@code Flowable} before a specified time window
     * elapses.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skip.t.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code skip} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window to skip
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skip.html">ReactiveX operators documentation: Skip</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> skip(long time, @NonNull TimeUnit unit) {
        return skipUntil(timer(time, unit));
    }

    /**
     * Returns a {@code Flowable} that skips values emitted by the current {@code Flowable} before a specified time window
     * on a specified {@link Scheduler} elapses.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skip.ts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use for the timed skipping</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window to skip
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} on which the timed wait happens
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skip.html">ReactiveX operators documentation: Skip</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> skip(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return skipUntil(timer(time, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that drops a specified number of items from the end of the sequence emitted by the
     * current {@code Flowable}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.v3.png" alt="">
     * <p>
     * This {@link Subscriber} accumulates a queue long enough to store the first {@code count} items. As more items are
     * received, items are taken from the front of the queue and emitted by the resulting {@code Flowable}. This causes
     * such items to be delayed.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code skipLast} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            number of items to drop from the end of the source sequence
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> skipLast(int count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        }
        if (count == 0) {
            return RxJavaPlugins.onAssembly(this);
        }
        return RxJavaPlugins.onAssembly(new FlowableSkipLast<>(this, count));
    }

    /**
     * Returns a {@code Flowable} that drops items emitted by the current {@code Flowable} during a specified time window
     * before the source completes.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.t.v3.png" alt="">
     * <p>
     * Note: this action will cache the latest items arriving in the specified time window.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code skipLast} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> skipLast(long time, @NonNull TimeUnit unit) {
        return skipLast(time, unit, Schedulers.computation(), false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that drops items emitted by the current {@code Flowable} during a specified time window
     * before the source completes.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.t.v3.png" alt="">
     * <p>
     * Note: this action will cache the latest items arriving in the specified time window.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code skipLast} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> skipLast(long time, @NonNull TimeUnit unit, boolean delayError) {
        return skipLast(time, unit, Schedulers.computation(), delayError, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that drops items emitted by the current {@code Flowable} during a specified time window
     * (defined on a specified scheduler) before the source completes.
     * <p>
     * <img width="640" height="340" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.ts.v3.png" alt="">
     * <p>
     * Note: this action will cache the latest items arriving in the specified time window.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use for tracking the current time</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler used as the time source
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> skipLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return skipLast(time, unit, scheduler, false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that drops items emitted by the current {@code Flowable} during a specified time window
     * (defined on a specified scheduler) before the source completes.
     * <p>
     * <img width="640" height="340" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.ts.v3.png" alt="">
     * <p>
     * Note: this action will cache the latest items arriving in the specified time window.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use to track the current time</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler used as the time source
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> skipLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError) {
        return skipLast(time, unit, scheduler, delayError, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that drops items emitted by the current {@code Flowable} during a specified time window
     * (defined on a specified scheduler) before the source completes.
     * <p>
     * <img width="640" height="340" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipLast.ts.v3.png" alt="">
     * <p>
     * Note: this action will cache the latest items arriving in the specified time window.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't support backpressure as it uses time to skip an arbitrary number of elements and
     *  thus has to consume the current {@code Flowable} in an unbounded manner (i.e., no backpressure applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the scheduler used as the time source
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @param bufferSize
     *            the hint about how many elements to expect to be skipped
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/skiplast.html">ReactiveX operators documentation: SkipLast</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> skipLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError, int bufferSize) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        // the internal buffer holds pairs of (timestamp, value) so double the default buffer size
        int s = bufferSize << 1;
        return RxJavaPlugins.onAssembly(new FlowableSkipLastTimed<>(this, time, unit, scheduler, s, delayError));
    }

    /**
     * Returns a {@code Flowable} that skips items emitted by the current {@code Flowable} until a second {@link Publisher} emits
     * an item.
     * <p>
     * <img width="640" height="375" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipUntil.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code skipUntil} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <U> the element type of the other {@code Publisher}
     * @param other
     *            the second {@code Publisher} that has to emit an item before the current {@code Flowable}'s elements begin
     *            to be mirrored by the resulting {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skipuntil.html">ReactiveX operators documentation: SkipUntil</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> skipUntil(@NonNull Publisher<U> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableSkipUntil<>(this, other));
    }

    /**
     * Returns a {@code Flowable} that skips all items emitted by the current {@code Flowable} as long as a specified
     * condition holds {@code true}, but emits all further source items as soon as the condition becomes {@code false}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/skipWhile.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code skipWhile} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            a function to test each item emitted from the current {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/skipwhile.html">ReactiveX operators documentation: SkipWhile</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> skipWhile(@NonNull Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");
        return RxJavaPlugins.onAssembly(new FlowableSkipWhile<>(this, predicate));
    }
    /**
     * Returns a {@code Flowable} that emits the events emitted by source {@link Publisher}, in a
     * sorted order. Each item emitted by the {@code Publisher} must implement {@link Comparable} with respect to all
     * other items in the sequence.
     *
     * <p>If any item emitted by this {@code Flowable} does not implement {@code Comparable} with respect to
     *             all other items emitted by this {@code Flowable}, no items will be emitted and the
     *             sequence is terminated with a {@link ClassCastException}.
     * <p>Note that calling {@code sorted} with long, non-terminating or infinite sources
     * might cause {@link OutOfMemoryError}
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sorted} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> sorted() {
        return toList().toFlowable().map(Functions.listSorter(Functions.naturalComparator())).flatMapIterable(Functions.identity());
    }

    /**
     * Returns a {@code Flowable} that emits the events emitted by source {@link Publisher}, in a
     * sorted order based on a specified comparison function.
     *
     * <p>Note that calling {@code sorted} with long, non-terminating or infinite sources
     * might cause {@link OutOfMemoryError}
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., without applying backpressure to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code sorted} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param comparator
     *            a function that compares two items emitted by the current {@code Flowable} and returns an {@link Integer}
     *            that indicates their sort order
     * @throws NullPointerException if {@code comparator} is {@code null}
     * @return the new {@code Flowable} instance
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> sorted(@NonNull Comparator<@NonNull ? super T> comparator) {
        Objects.requireNonNull(comparator, "comparator is null");
        return toList().toFlowable().map(Functions.listSorter(comparator)).flatMapIterable(Functions.identity());
    }

    /**
     * Returns a {@code Flowable} that emits the items in a specified {@link Iterable} before it begins to emit items
     * emitted by the current {@code Flowable}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/startWith.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The Current {@code Flowable}
     *  is expected to honor backpressure as well. If it violates this rule, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWithIterable} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param items
     *            an {@code Iterable} that contains the items you want the resulting {@code Flowable} to emit first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code items} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/startwith.html">ReactiveX operators documentation: StartWith</a>
     * @see #startWithArray(Object...)
     * @see #startWithItem(Object)
     * @since 3.0.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> startWithIterable(@NonNull Iterable<@NonNull ? extends T> items) {
        return concatArray(fromIterable(items), this);
    }

    /**
     * Returns a {@code Flowable} which first runs the other {@link CompletableSource}
     * then the current {@code Flowable} if the other completed normally.
     * <p>
     * <img width="640" height="268" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.startWith.c.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Flowable} honors the backpressure of the downstream consumer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param other the other {@code CompletableSource} to run first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public final Flowable<T> startWith(@NonNull CompletableSource other) {
        Objects.requireNonNull(other, "other is null");
        return Flowable.concat(Completable.wrap(other).<T>toFlowable(), this);
    }

    /**
     * Returns a {@code Flowable} which first runs the other {@link SingleSource}
     * then the current {@code Flowable} if the other succeeded normally.
     * <p>
     * <img width="640" height="248" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.startWith.s.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Flowable} honors the backpressure of the downstream consumer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param other the other {@code SingleSource} to run first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public final Flowable<T> startWith(@NonNull SingleSource<T> other) {
        Objects.requireNonNull(other, "other is null");
        return Flowable.concat(Single.wrap(other).toFlowable(), this);
    }

    /**
     * Returns a {@code Flowable} which first runs the other {@link MaybeSource}
     * then the current {@code Flowable} if the other succeeded or completed normally.
     * <p>
     * <img width="640" height="168" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/Flowable.startWith.m.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The returned {@code Flowable} honors the backpressure of the downstream consumer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param other the other {@code MaybeSource} to run first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @SchedulerSupport(SchedulerSupport.NONE)
    @BackpressureSupport(BackpressureKind.FULL)
    public final Flowable<T> startWith(@NonNull MaybeSource<T> other) {
        Objects.requireNonNull(other, "other is null");
        return Flowable.concat(Maybe.wrap(other).toFlowable(), this);
    }

    /**
     * Returns a {@code Flowable} that emits the items in a specified {@link Publisher} before it begins to emit
     * items emitted by the current {@code Flowable}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/startWith.o.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the {@code other} {@code Publisher}s
     *  are expected to honor backpressure as well. If any of then violates this rule, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param other
     *            a {@code Publisher} that contains the items you want the modified {@code Publisher} to emit first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/startwith.html">ReactiveX operators documentation: StartWith</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> startWith(@NonNull Publisher<@NonNull ? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return concatArray(other, this);
    }

    /**
     * Returns a {@code Flowable} that emits a specified item before it begins to emit items emitted by the current
     * {@code Flowable}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/startWith.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}
     *  is expected to honor backpressure as well. If it violates this rule, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWithItem} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param item
     *            the item to emit first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code item} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/startwith.html">ReactiveX operators documentation: StartWith</a>
     * @see #startWithArray(Object...)
     * @see #startWithIterable(Iterable)
     * @since 3.0.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> startWithItem(@NonNull T item) {
        Objects.requireNonNull(item, "item is null");
        return concatArray(just(item), this);
    }

    /**
     * Returns a {@code Flowable} that emits the specified items before it begins to emit items emitted by the current
     * {@code Flowable}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/startWith.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The current {@code Flowable}
     *  is expected to honor backpressure as well. If it violates this rule, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code startWithArray} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param items
     *            the array of values to emit first
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code items} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/startwith.html">ReactiveX operators documentation: StartWith</a>
     * @see #startWithItem(Object)
     * @see #startWithIterable(Iterable)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @SafeVarargs
    @NonNull
    public final Flowable<T> startWithArray(@NonNull T... items) {
        Flowable<T> fromArray = fromArray(items);
        if (fromArray == empty()) {
            return RxJavaPlugins.onAssembly(this);
        }
        return concatArray(fromArray, this);
    }

    /**
     * Subscribes to the current {@code Flowable} and ignores {@code onNext} and {@code onComplete} emissions.
     * <p>
     * If the {@code Flowable} emits an error, it is wrapped into an
     * {@link io.reactivex.rxjava3.exceptions.OnErrorNotImplementedException OnErrorNotImplementedException}
     * and routed to the {@link RxJavaPlugins#onError(Throwable)} handler.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@link Disposable} instance that allows cancelling the flow
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable subscribe() {
        return subscribe(Functions.emptyConsumer(), Functions.ON_ERROR_MISSING, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the current {@code Flowable} and provides a callback to handle the items it emits.
     * <p>
     * If the {@code Flowable} emits an error, it is wrapped into an
     * {@link io.reactivex.rxjava3.exceptions.OnErrorNotImplementedException OnErrorNotImplementedException}
     * and routed to the {@link RxJavaPlugins#onError(Throwable)} handler.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *             the {@code Consumer<T>} you have designed to accept emissions from the current {@code Flowable}
     * @return the new {@link Disposable} instance that allows cancelling the flow
     * @throws NullPointerException
     *             if {@code onNext} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable subscribe(@NonNull Consumer<? super T> onNext) {
        return subscribe(onNext, Functions.ON_ERROR_MISSING, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the current {@code Flowable} and provides callbacks to handle the items it emits and any error
     * notification it issues.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *             the {@code Consumer<T>} you have designed to accept emissions from the current {@code Flowable}
     * @param onError
     *             the {@code Consumer<Throwable>} you have designed to accept any error notification from the
     *             current {@code Flowable}
     * @return the new {@link Disposable} instance that allows cancelling the flow
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     * @throws NullPointerException
     *             if {@code onNext} or {@code onError} is {@code null}
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable subscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError) {
        return subscribe(onNext, onError, Functions.EMPTY_ACTION);
    }

    /**
     * Subscribes to the current {@code Flowable} and provides callbacks to handle the items it emits and any error or
     * completion notification it issues.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner (i.e., no
     *  backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param onNext
     *             the {@code Consumer<T>} you have designed to accept emissions from the current {@code Flowable}
     * @param onError
     *             the {@code Consumer<Throwable>} you have designed to accept any error notification from the
     *             current {@code Flowable}
     * @param onComplete
     *             the {@link Action} you have designed to accept a completion notification from the
     *             the current {@code Flowable}
     * @return the new {@link Disposable} instance that allows cancelling the flow
     * @throws NullPointerException
     *             if {@code onNext}, {@code onError} or {@code onComplete} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribe.html">ReactiveX operators documentation: Subscribe</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Disposable subscribe(@NonNull Consumer<? super T> onNext, @NonNull Consumer<? super Throwable> onError,
            @NonNull Action onComplete) {
        Objects.requireNonNull(onNext, "onNext is null");
        Objects.requireNonNull(onError, "onError is null");
        Objects.requireNonNull(onComplete, "onComplete is null");

        LambdaSubscriber<T> ls = new LambdaSubscriber<>(onNext, onError, onComplete, FlowableInternalHelper.RequestMax.INSTANCE);

        subscribe(ls);

        return ls;
    }

    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @Override
    public final void subscribe(@NonNull Subscriber<@NonNull ? super T> subscriber) {
        if (subscriber instanceof FlowableSubscriber) {
            subscribe((FlowableSubscriber<@NonNull ? super T>)subscriber);
        } else {
            Objects.requireNonNull(subscriber, "subscriber is null");
            subscribe(new StrictSubscriber<>(subscriber));
        }
    }

    /**
     * Establish a connection between this {@code Flowable} and the given {@link FlowableSubscriber} and
     * start streaming events based on the demand of the {@code FlowableSubscriber}.
     * <p>
     * This is a "factory method" and can be called multiple times, each time starting a new {@link Subscription}.
     * <p>
     * Each {@code Subscription} will work for only a single {@code FlowableSubscriber}.
     * <p>
     * If the same {@code FlowableSubscriber} instance is subscribed to multiple {@code Flowable}s and/or the
     * same {@code Flowable} multiple times, it must ensure the serialization over its {@code onXXX}
     * methods manually.
     * <p>
     * If the {@code Flowable} rejects the subscription attempt or otherwise fails it will signal
     * the error via {@link FlowableSubscriber#onError(Throwable)}.
     * <p>
     * This subscribe method relaxes the following <em>Reactive Streams</em> rules:
     * <ul>
     * <li>§1.3: {@code onNext} should not be called concurrently until {@code onSubscribe} returns.
     *     <b>{@link FlowableSubscriber#onSubscribe(Subscription)} should make sure a sync or async call triggered by request() is safe.</b></li>
     * <li>§2.3: {@code onError} or {@code onComplete} must not call cancel.
     *     <b>Calling request() or cancel() is NOP at this point.</b></li>
     * <li>§2.12: {@code onSubscribe} must be called at most once on the same instance.
     *     <b>{@code FlowableSubscriber} reuse is not checked and if happens, it is the responsibility of
     *     the {@code FlowableSubscriber} to ensure proper serialization of its onXXX methods.</b></li>
     * <li>§3.9: negative requests should emit an {@code onError(IllegalArgumentException)}.
     *     <b>Non-positive requests signal via {@link RxJavaPlugins#onError(Throwable)} and the stream is not affected.</b></li>
     * </ul>
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The backpressure behavior/expectation is determined by the supplied {@code FlowableSubscriber}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribe} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.0.7 - experimental; 2.1 - beta
     * @param subscriber the {@code FlowableSubscriber} that will consume signals from this {@code Flowable}
     * @throws NullPointerException if {@code subscriber} is {@code null}
     * @since 2.2
     */
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final void subscribe(@NonNull FlowableSubscriber<@NonNull ? super T> subscriber) {
        Objects.requireNonNull(subscriber, "subscriber is null");
        try {
            Subscriber<@NonNull ? super T> flowableSubscriber = RxJavaPlugins.onSubscribe(this, subscriber);

            Objects.requireNonNull(flowableSubscriber, "The RxJavaPlugins.onSubscribe hook returned a null FlowableSubscriber. Please check the handler provided to RxJavaPlugins.setOnFlowableSubscribe for invalid null returns. Further reading: https://github.com/ReactiveX/RxJava/wiki/Plugins");

            subscribeActual(flowableSubscriber);
        } catch (NullPointerException e) { // NOPMD
            throw e;
        } catch (Throwable e) {
            Exceptions.throwIfFatal(e);
            // can't call onError because no way to know if a Subscription has been set or not
            // can't call onSubscribe because the call might have set a Subscription already
            RxJavaPlugins.onError(e);

            NullPointerException npe = new NullPointerException("Actually not, but can't throw other exceptions due to RS");
            npe.initCause(e);
            throw npe;
        }
    }

    /**
     * Operator implementations (both source and intermediate) should implement this method that
     * performs the necessary business logic and handles the incoming {@link Subscriber}s.
     * <p>There is no need to call any of the plugin hooks on the current {@code Flowable} instance or
     * the {@code Subscriber}; all hooks and basic safeguards have been
     * applied by {@link #subscribe(Subscriber)} before this method gets called.
     * @param subscriber the incoming {@code Subscriber}, never {@code null}
     */
    protected abstract void subscribeActual(@NonNull Subscriber<@NonNull ? super T> subscriber);

    /**
     * Subscribes a given {@link Subscriber} (subclass) to this {@code Flowable} and returns the given
     * {@code Subscriber} as is.
     * <p>Usage example:
     * <pre><code>
     * Flowable&lt;Integer&gt; source = Flowable.range(1, 10);
     * CompositeDisposable composite = new CompositeDisposable();
     *
     * ResourceSubscriber&lt;Integer&gt; rs = new ResourceSubscriber&lt;&gt;() {
     *     // ...
     * };
     *
     * composite.add(source.subscribeWith(rs));
     * </code></pre>
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The backpressure behavior/expectation is determined by the supplied {@code Subscriber}.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code subscribeWith} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * @param <E> the type of the {@code Subscriber} to use and return
     * @param subscriber the {@code Subscriber} (subclass) to use and return, not {@code null}
     * @return the input {@code subscriber}
     * @throws NullPointerException if {@code subscriber} is {@code null}
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <@NonNull E extends Subscriber<@NonNull ? super T>> E subscribeWith(E subscriber) {
        subscribe(subscriber);
        return subscriber;
    }

    /**
     * Asynchronously subscribes {@link Subscriber}s to the current {@code Flowable} on the specified {@link Scheduler}.
     * <p>
     * If there is a {@link #create(FlowableOnSubscribe, BackpressureStrategy)} type source up in the
     * chain, it is recommended to use {@code subscribeOn(scheduler, false)} instead
     * to avoid same-pool deadlock because requests may pile up behind an eager/blocking emitter.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/subscribeOn.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param scheduler
     *            the {@code Scheduler} to perform subscription actions on
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribeon.html">ReactiveX operators documentation: SubscribeOn</a>
     * @see <a href="http://www.grahamlea.com/2014/07/rxjava-threading-examples/">RxJava Threading Examples</a>
     * @see #observeOn
     * @see #subscribeOn(Scheduler, boolean)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> subscribeOn(@NonNull Scheduler scheduler) {
        Objects.requireNonNull(scheduler, "scheduler is null");
        return subscribeOn(scheduler, !(this instanceof FlowableCreate));
    }

    /**
     * Asynchronously subscribes {@link Subscriber}s to the current {@code Flowable} on the specified {@link Scheduler}
     * optionally reroutes requests from other threads to the same {@code Scheduler} thread.
     * <p>
     * If there is a {@link #create(FlowableOnSubscribe, BackpressureStrategy)} type source up in the
     * chain, it is recommended to have {@code requestOn} {@code false} to avoid same-pool deadlock
     * because requests may pile up behind an eager/blocking emitter.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/subscribeOn.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     * <p>History: 2.1.1 - experimental
     * @param scheduler
     *            the {@code Scheduler} to perform subscription actions on
     * @param requestOn if {@code true}, requests are rerouted to the given {@code Scheduler} as well (strong pipelining)
     *                  if {@code false}, requests coming from any thread are simply forwarded to
     *                  the upstream on the same thread (weak pipelining)
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/subscribeon.html">ReactiveX operators documentation: SubscribeOn</a>
     * @see <a href="http://www.grahamlea.com/2014/07/rxjava-threading-examples/">RxJava Threading Examples</a>
     * @see #observeOn
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> subscribeOn(@NonNull Scheduler scheduler, boolean requestOn) {
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableSubscribeOn<>(this, scheduler, requestOn));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} or the items of an alternate
     * {@link Publisher} if the current {@code Flowable} is empty.
     * <p>
     * <img width="640" height="256" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchifempty.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>If the current {@code Flowable} is empty, the alternate {@code Publisher} is expected to honor backpressure.
     *  If the current {@code Flowable} is non-empty, it is expected to honor backpressure as instead.
     *  In either case, if violated, a {@link MissingBackpressureException} <em>may</em> get
     *  signaled somewhere downstream.
     *  </dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchIfEmpty} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param other
     *              the alternate {@code Publisher} to subscribe to if the source does not emit any items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 1.1.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> switchIfEmpty(@NonNull Publisher<@NonNull ? extends T> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchIfEmpty<>(this, other));
    }

    /**
     * Returns a new {@code Flowable} by applying a function that you supply to each item emitted by the current
     * {@code Flowable} that returns a {@link Publisher}, and then emitting the items emitted by the most recently emitted
     * of these {@code Publisher}s.
     * <p>
     * The resulting {@code Flowable} completes if both the current {@code Flowable} and the last inner {@code Publisher}, if any, complete.
     * If the current {@code Flowable} signals an {@code onError}, the inner {@code Publisher} is canceled and the error delivered in-sequence.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the element type of the inner {@code Publisher}s and the output
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @see #switchMapDelayError(Function)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> switchMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return switchMap(mapper, bufferSize());
    }

    /**
     * Returns a new {@code Flowable} by applying a function that you supply to each item emitted by the current
     * {@code Flowable} that returns a {@link Publisher}, and then emitting the items emitted by the most recently emitted
     * of these {@code Publisher}s.
     * <p>
     * The resulting {@code Flowable} completes if both the current {@code Flowable} and the last inner {@code Publisher}, if any, complete.
     * If the current {@code Flowable} signals an {@code onError}, the inner {@code Publisher} is canceled and the error delivered in-sequence.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMap} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the element type of the inner {@code Publisher}s and the output
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param bufferSize
     *            the number of elements to prefetch from the current active inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @see #switchMapDelayError(Function, int)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> switchMap(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int bufferSize) {
        return switchMap0(mapper, bufferSize, false);
    }

    /**
     * Maps the upstream values into {@link CompletableSource}s, subscribes to the newer one while
     * disposing the subscription to the previous {@code CompletableSource}, thus keeping at most one
     * active {@code CompletableSource} running.
     * <p>
     * <img width="640" height="522" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMapCompletable.f.png" alt="">
     * <p>
     * Since a {@code CompletableSource} doesn't produce any items, the resulting reactive type of
     * this operator is a {@link Completable} that can only indicate successful completion or
     * a failure in any of the inner {@code CompletableSource}s or the failure of the current
     * {@code Flowable}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner and otherwise
     *  does not have backpressure in its return type because no items are ever produced.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapCompletable} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>If either this {@code Flowable} or the active {@code CompletableSource} signals an {@code onError},
     *  the resulting {@code Completable} is terminated immediately with that {@link Throwable}.
     *  Use the {@link #switchMapCompletableDelayError(Function)} to delay such inner failures until
     *  every inner {@code CompletableSource}s and the main {@code Flowable} terminates in some fashion.
     *  If they fail concurrently, the operator may combine the {@code Throwable}s into a
     *  {@link io.reactivex.rxjava3.exceptions.CompositeException CompositeException}
     *  and signal it to the downstream instead. If any inactivated (switched out) {@code CompletableSource}
     *  signals an {@code onError} late, the {@code Throwable}s will be signaled to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors.
     *  </dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with each upstream item and should return a
     *               {@code CompletableSource} to be subscribed to and awaited for
     *               (non blockingly) for its terminal event
     * @return the new {@code Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapCompletableDelayError(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Completable switchMapCompletable(@NonNull Function<? super T, ? extends CompletableSource> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapCompletable<>(this, mapper, false));
    }

    /**
     * Maps the upstream values into {@link CompletableSource}s, subscribes to the newer one while
     * disposing the subscription to the previous {@code CompletableSource}, thus keeping at most one
     * active {@code CompletableSource} running and delaying any main or inner errors until all
     * of them terminate.
     * <p>
     * <img width="640" height="453" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMapCompletableDelayError.f.png" alt="">
     * <p>
     * Since a {@code CompletableSource} doesn't produce any items, the resulting reactive type of
     * this operator is a {@link Completable} that can only indicate successful completion or
     * a failure in any of the inner {@code CompletableSource}s or the failure of the current
     * {@code Flowable}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator consumes the current {@code Flowable} in an unbounded manner and otherwise
     *  does not have backpressure in its return type because no items are ever produced.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapCompletableDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>The errors of this {@code Flowable} and all the {@code CompletableSource}s, who had the chance
     *  to run to their completion, are delayed until
     *  all of them terminate in some fashion. At this point, if there was only one failure, the respective
     *  {@link Throwable} is emitted to the downstream. If there was more than one failure, the
     *  operator combines all {@code Throwable}s into a {@link io.reactivex.rxjava3.exceptions.CompositeException CompositeException}
     *  and signals that to the downstream.
     *  If any inactivated (switched out) {@code CompletableSource}
     *  signals an {@code onError} late, the {@code Throwable}s will be signaled to the global error handler via
     *  {@link RxJavaPlugins#onError(Throwable)} method as {@link UndeliverableException} errors.
     *  </dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param mapper the function called with each upstream item and should return a
     *               {@code CompletableSource} to be subscribed to and awaited for
     *               (non blockingly) for its terminal event
     * @return the new {@code Completable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapCompletable(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Completable switchMapCompletableDelayError(@NonNull Function<? super T, ? extends CompletableSource> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapCompletable<>(this, mapper, true));
    }

    /**
     * Returns a new {@code Flowable} by applying a function that you supply to each item emitted by the current
     * {@code Flowable} that returns a {@link Publisher}, and then emitting the items emitted by the most recently emitted
     * of these {@code Publisher}s and delays any error until all {@code Publisher}s terminate.
     * <p>
     * The resulting {@code Flowable} completes if both the current {@code Flowable} and the last inner {@code Publisher}, if any, complete.
     * If the current {@code Flowable} signals an {@code onError}, the termination of the last inner {@code Publisher} will emit that error as is
     * or wrapped into a {@link CompositeException} along with the other possible errors the former inner {@code Publisher}s signaled.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the element type of the inner {@code Publisher}s and the output
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @see #switchMap(Function)
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> switchMapDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper) {
        return switchMapDelayError(mapper, bufferSize());
    }

    /**
     * Returns a new {@code Flowable} by applying a function that you supply to each item emitted by the current
     * {@code Flowable} that returns a {@link Publisher}, and then emitting the items emitted by the most recently emitted
     * of these {@code Publisher}s and delays any error until all {@code Publisher}s terminate.
     * <p>
     * The resulting {@code Flowable} completes if both the current {@code Flowable} and the last inner {@code Publisher}, if any, complete.
     * If the current {@code Flowable} signals an {@code onError}, the termination of the last inner {@code Publisher} will emit that error as is
     * or wrapped into a {@link CompositeException} along with the other possible errors the former inner {@code Publisher}s signaled.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The outer {@code Publisher} is consumed in an
     *  unbounded manner (i.e., without backpressure) and the inner {@code Publisher}s are expected to honor
     *  backpressure but it is not enforced; the operator won't signal a {@link MissingBackpressureException}
     *  but the violation <em>may</em> lead to {@link OutOfMemoryError} due to internal buffer bloat.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <R> the element type of the inner {@code Publisher}s and the output
     * @param mapper
     *            a function that, when applied to an item emitted by the current {@code Flowable}, returns a
     *            {@code Publisher}
     * @param bufferSize
     *            the number of elements to prefetch from the current active inner {@code Publisher}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/flatmap.html">ReactiveX operators documentation: FlatMap</a>
     * @see #switchMap(Function, int)
     * @since 2.0
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.SPECIAL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <R> Flowable<R> switchMapDelayError(@NonNull Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int bufferSize) {
        return switchMap0(mapper, bufferSize, true);
    }

    <R> Flowable<R> switchMap0(Function<? super T, ? extends Publisher<@NonNull ? extends R>> mapper, int bufferSize, boolean delayError) {
        Objects.requireNonNull(mapper, "mapper is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        if (this instanceof ScalarSupplier) {
            @SuppressWarnings("unchecked")
            T v = ((ScalarSupplier<T>)this).get();
            if (v == null) {
                return empty();
            }
            return FlowableScalarXMap.scalarXMap(v, mapper);
        }
        return RxJavaPlugins.onAssembly(new FlowableSwitchMap<>(this, mapper, bufferSize, delayError));
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and switches (subscribes) to the newer ones
     * while disposing the older ones (and ignoring their signals) and emits the latest success value of the current one if
     * available while failing immediately if this {@code Flowable} or any of the
     * active inner {@code MaybeSource}s fail.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The main {@code Flowable} is consumed in an
     *  unbounded manner (i.e., without backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapMaybe} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>This operator terminates with an {@code onError} if this {@code Flowable} or any of
     *  the inner {@code MaybeSource}s fail while they are active. When this happens concurrently, their
     *  individual {@link Throwable} errors may get combined and emitted as a single
     *  {@link CompositeException}. Otherwise, a late
     *  (i.e., inactive or switched out) {@code onError} from this {@code Flowable} or from any of
     *  the inner {@code MaybeSource}s will be forwarded to the global error handler via
     *  {@link io.reactivex.rxjava3.plugins.RxJavaPlugins#onError(Throwable)} as
     *  {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the output value type
     * @param mapper the function called with the current upstream event and should
     *               return a {@code MaybeSource} to replace the current active inner source
     *               and get subscribed to.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapMaybeDelayError(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> switchMapMaybe(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapMaybe<>(this, mapper, false));
    }

    /**
     * Maps the upstream items into {@link MaybeSource}s and switches (subscribes) to the newer ones
     * while disposing the older ones  (and ignoring their signals) and emits the latest success value of the current one if
     * available, delaying errors from this {@code Flowable} or the inner {@code MaybeSource}s until all terminate.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The main {@code Flowable} is consumed in an
     *  unbounded manner (i.e., without backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapMaybeDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the output value type
     * @param mapper the function called with the current upstream event and should
     *               return a {@code MaybeSource} to replace the current active inner source
     *               and get subscribed to.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapMaybe(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> switchMapMaybeDelayError(@NonNull Function<? super T, ? extends MaybeSource<? extends R>> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapMaybe<>(this, mapper, true));
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and switches (subscribes) to the newer ones
     * while disposing the older ones (and ignoring their signals) and emits the latest success value of the current one
     * while failing immediately if this {@code Flowable} or any of the
     * active inner {@code SingleSource}s fail.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The main {@code Flowable} is consumed in an
     *  unbounded manner (i.e., without backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapSingle} does not operate by default on a particular {@link Scheduler}.</dd>
     *  <dt><b>Error handling:</b></dt>
     *  <dd>This operator terminates with an {@code onError} if this {@code Flowable} or any of
     *  the inner {@code SingleSource}s fail while they are active. When this happens concurrently, their
     *  individual {@link Throwable} errors may get combined and emitted as a single
     *  {@link CompositeException}. Otherwise, a late
     *  (i.e., inactive or switched out) {@code onError} from this {@code Flowable} or from any of
     *  the inner {@code SingleSource}s will be forwarded to the global error handler via
     *  {@link io.reactivex.rxjava3.plugins.RxJavaPlugins#onError(Throwable)} as
     *  {@link io.reactivex.rxjava3.exceptions.UndeliverableException UndeliverableException}</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the output value type
     * @param mapper the function called with the current upstream event and should
     *               return a {@code SingleSource} to replace the current active inner source
     *               and get subscribed to.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapSingleDelayError(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> switchMapSingle(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapSingle<>(this, mapper, false));
    }

    /**
     * Maps the upstream items into {@link SingleSource}s and switches (subscribes) to the newer ones
     * while disposing the older ones  (and ignoring their signals) and emits the latest success value of the current one,
     * delaying errors from this {@code Flowable} or the inner {@code SingleSource}s until all terminate.
     * <p>
     * <img width="640" height="350" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/switchMap.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The main {@code Flowable} is consumed in an
     *  unbounded manner (i.e., without backpressure).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code switchMapSingleDelayError} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.11 - experimental
     * @param <R> the output value type
     * @param mapper the function called with the current upstream event and should
     *               return a {@code SingleSource} to replace the current active inner source
     *               and get subscribed to.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code mapper} is {@code null}
     * @see #switchMapSingle(Function)
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.UNBOUNDED_IN)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <R> Flowable<R> switchMapSingleDelayError(@NonNull Function<? super T, ? extends SingleSource<? extends R>> mapper) {
        Objects.requireNonNull(mapper, "mapper is null");
        return RxJavaPlugins.onAssembly(new FlowableSwitchMapSingle<>(this, mapper, true));
    }

    /**
     * Returns a {@code Flowable} that emits only the first {@code count} items emitted by the current {@code Flowable}.
     * If the source emits fewer than {@code count} items then all of its items are emitted.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/take.v3.png" alt="">
     * <p>
     * This method returns a {@code Flowable} that will invoke a subscribing {@link Subscriber}'s
     * {@link Subscriber#onNext onNext} function a maximum of {@code count} times before invoking
     * {@link Subscriber#onComplete onComplete}.
     * <p>
     * Limits both the number of upstream items (after which the sequence completes)
     * and the total downstream request amount requested from the upstream to
     * possibly prevent the creation of excess items by the upstream.
     * <p>
     * The operator requests at most the given {@code count} of items from upstream even
     * if the downstream requests more than that. For example, given a {@code take(5)},
     * if the downstream requests 1, a request of 1 is submitted to the upstream
     * and the operator remembers that only 4 items can be requested now on. A request
     * of 5 at this point will request 4 from the upstream and any subsequent requests will
     * be ignored.
     * <p>
     * Note that requests are negotiated on an operator boundary and {@code take}'s amount
     * may not be preserved further upstream. For example,
     * {@code source.observeOn(Schedulers.computation()).take(5)} will still request the
     * default (128) elements from the given {@code source}.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The current {@code Flowable} is consumed in a bounded manner.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code take} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items and the total request amount, non-negative.
     *            Zero will immediately cancel the upstream on subscription and complete
     *            the downstream.
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException if {@code count} is negative
     * @see <a href="http://reactivex.io/documentation/operators/take.html">ReactiveX operators documentation: Take</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> take(long count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        }
        return RxJavaPlugins.onAssembly(new FlowableTake<>(this, count));
    }

    /**
     * Returns a {@code Flowable} that emits those items emitted by source {@link Publisher} before a specified time runs
     * out.
     * <p>
     * If time runs out before the {@code Flowable} completes normally, the {@code onComplete} event will be
     * signaled on the default {@code computation} {@link Scheduler}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/take.t.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code take} operates by default on the {@code computation} {@code Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/take.html">ReactiveX operators documentation: Take</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> take(long time, @NonNull TimeUnit unit) {
        return takeUntil(timer(time, unit));
    }

    /**
     * Returns a {@code Flowable} that emits those items emitted by source {@link Publisher} before a specified time (on a
     * specified {@link Scheduler}) runs out.
     * <p>
     * If time runs out before the {@code Flowable} completes normally, the {@code onComplete} event will be
     * signaled on the provided {@code Scheduler}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/take.ts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} used for time source
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/take.html">ReactiveX operators documentation: Take</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> take(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return takeUntil(timer(time, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that emits at most the last {@code count} items emitted by the current {@code Flowable}. If the source emits fewer than
     * {@code count} items then all of its items are emitted.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.n.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream if the {@code count} is non-zero; ignores
     *  backpressure if the {@code count} is zero as it doesn't signal any values.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code takeLast} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items to emit from the end of the sequence of items emitted by the current
     *            {@code Flowable}
     * @return the new {@code Flowable} instance
     * @throws IllegalArgumentException
     *             if {@code count} is negative
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> takeLast(int count) {
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        } else
        if (count == 0) {
            return RxJavaPlugins.onAssembly(new FlowableIgnoreElements<>(this));
        } else
        if (count == 1) {
            return RxJavaPlugins.onAssembly(new FlowableTakeLastOne<>(this));
        }
        return RxJavaPlugins.onAssembly(new FlowableTakeLast<>(this, count));
    }

    /**
     * Returns a {@code Flowable} that emits at most a specified number of items from the current {@code Flowable} that were
     * emitted in a specified window of time before the current {@code Flowable} completed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.tn.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code takeLast} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items to emit
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @throws IllegalArgumentException if {@code count} is negative
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<T> takeLast(long count, long time, @NonNull TimeUnit unit) {
        return takeLast(count, time, unit, Schedulers.computation(), false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits at most a specified number of items from the current {@code Flowable} that were
     * emitted in a specified window of time before the current {@code Flowable} completed, where the timing information is
     * provided by a given {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.tns.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use for tracking the current time</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items to emit
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that provides the timestamps for the observed items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code count} is less than zero
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> takeLast(long count, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return takeLast(count, time, unit, scheduler, false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits at most a specified number of items from the current {@code Flowable} that were
     * emitted in a specified window of time before the current {@code Flowable} completed, where the timing information is
     * provided by a given {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.tns.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it).</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use for tracking the current time</dd>
     * </dl>
     *
     * @param count
     *            the maximum number of items to emit
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that provides the timestamps for the observed items
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @param bufferSize
     *            the hint about how many elements to expect to be last
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException
     *             if {@code count} is negative or {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> takeLast(long count, long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError, int bufferSize) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        ObjectHelper.verifyPositive(bufferSize, "bufferSize");
        if (count < 0) {
            throw new IllegalArgumentException("count >= 0 required but it was " + count);
        }
        return RxJavaPlugins.onAssembly(new FlowableTakeLastTimed<>(this, count, time, unit, scheduler, bufferSize, delayError));
    }

    /**
     * Returns a {@code Flowable} that emits the items from the current {@code Flowable} that were emitted in a specified
     * window of time before the current {@code Flowable} completed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.t.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it) but note that this <em>may</em>
     *  lead to {@link OutOfMemoryError} due to internal buffer bloat.
     *  Consider using {@link #takeLast(long, long, TimeUnit)} in this case.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code takeLast} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> takeLast(long time, @NonNull TimeUnit unit) {
        return takeLast(time, unit, Schedulers.computation(), false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the items from the current {@code Flowable} that were emitted in a specified
     * window of time before the current {@code Flowable} completed.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.t.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it) but note that this <em>may</em>
     *  lead to {@link OutOfMemoryError} due to internal buffer bloat.
     *  Consider using {@link #takeLast(long, long, TimeUnit)} in this case.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code takeLast} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> takeLast(long time, @NonNull TimeUnit unit, boolean delayError) {
        return takeLast(time, unit, Schedulers.computation(), delayError, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the items from the current {@code Flowable} that were emitted in a specified
     * window of time before the current {@code Flowable} completed, where the timing information is provided by a specified
     * {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.ts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it) but note that this <em>may</em>
     *  lead to {@link OutOfMemoryError} due to internal buffer bloat.
     *  Consider using {@link #takeLast(long, long, TimeUnit, Scheduler)} in this case.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that provides the timestamps for the observed items
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> takeLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return takeLast(time, unit, scheduler, false, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the items from the current {@code Flowable} that were emitted in a specified
     * window of time before the current {@code Flowable} completed, where the timing information is provided by a specified
     * {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.ts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it) but note that this <em>may</em>
     *  lead to {@link OutOfMemoryError} due to internal buffer bloat.
     *  Consider using {@link #takeLast(long, long, TimeUnit, Scheduler)} in this case.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that provides the timestamps for the observed items
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> takeLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError) {
        return takeLast(time, unit, scheduler, delayError, bufferSize());
    }

    /**
     * Returns a {@code Flowable} that emits the items from the current {@code Flowable} that were emitted in a specified
     * window of time before the current {@code Flowable} completed, where the timing information is provided by a specified
     * {@link Scheduler}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeLast.ts.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream and consumes the current {@code Flowable} in an
     *  unbounded manner (i.e., no backpressure is applied to it) but note that this <em>may</em>
     *  lead to {@link OutOfMemoryError} due to internal buffer bloat.
     *  Consider using {@link #takeLast(long, long, TimeUnit, Scheduler)} in this case.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param time
     *            the length of the time window
     * @param unit
     *            the time unit of {@code time}
     * @param scheduler
     *            the {@code Scheduler} that provides the timestamps for the observed items
     * @param delayError
     *            if {@code true}, an exception signaled by the current {@code Flowable} is delayed until the regular elements are consumed
     *            by the downstream; if {@code false}, an exception is immediately signaled and all regular elements dropped
     * @param bufferSize
     *            the hint about how many elements to expect to be last
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @throws IllegalArgumentException if {@code bufferSize} is non-positive
     * @see <a href="http://reactivex.io/documentation/operators/takelast.html">ReactiveX operators documentation: TakeLast</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> takeLast(long time, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean delayError, int bufferSize) {
        return takeLast(Long.MAX_VALUE, time, unit, scheduler, delayError, bufferSize);
    }

    /**
     * Returns a {@code Flowable} that emits items emitted by the current {@code Flowable}, checks the specified predicate
     * for each item, and then completes when the condition is satisfied.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeUntil.p.v3.png" alt="">
     * <p>
     * The difference between this operator and {@link #takeWhile(Predicate)} is that here, the condition is
     * evaluated <em>after</em> the item is emitted.
     *
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator is a pass-through for backpressure; the backpressure behavior is determined by the upstream
     *  source and the downstream consumer.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code takeUntil} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param stopPredicate
     *            a function that evaluates an item emitted by the current {@code Flowable} and returns a {@link Boolean}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code stopPredicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takeuntil.html">ReactiveX operators documentation: TakeUntil</a>
     * @see Flowable#takeWhile(Predicate)
     * @since 1.1.0
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> takeUntil(@NonNull Predicate<? super T> stopPredicate) {
        Objects.requireNonNull(stopPredicate, "stopPredicate is null");
        return RxJavaPlugins.onAssembly(new FlowableTakeUntilPredicate<>(this, stopPredicate));
    }

    /**
     * Returns a {@code Flowable} that emits the items emitted by the current {@code Flowable} until a second {@link Publisher}
     * emits an item.
     * <p>
     * <img width="640" height="380" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeUntil.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code takeUntil} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param other
     *            the {@code Publisher} whose first emitted item will cause {@code takeUntil} to stop emitting items
     *            from the current {@code Flowable}
     * @param <U>
     *            the type of items emitted by {@code other}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code other} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takeuntil.html">ReactiveX operators documentation: TakeUntil</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <U> Flowable<T> takeUntil(@NonNull Publisher<U> other) {
        Objects.requireNonNull(other, "other is null");
        return RxJavaPlugins.onAssembly(new FlowableTakeUntil<>(this, other));
    }

    /**
     * Returns a {@code Flowable} that emits items emitted by the current {@code Flowable} so long as each item satisfied a
     * specified condition, and then completes as soon as this condition is not satisfied.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/takeWhile.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code takeWhile} does not operate by default on a particular {@link Scheduler}.</dd>
     * </dl>
     *
     * @param predicate
     *            a function that evaluates an item emitted by the current {@code Flowable} and returns a {@link Boolean}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code predicate} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/takewhile.html">ReactiveX operators documentation: TakeWhile</a>
     * @see Flowable#takeUntil(Predicate)
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final Flowable<T> takeWhile(@NonNull Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate, "predicate is null");
        return RxJavaPlugins.onAssembly(new FlowableTakeWhile<>(this, predicate));
    }

    /**
     * Returns a {@code Flowable} that emits only the first item emitted by the current {@code Flowable} during sequential
     * time windows of a specified duration.
     * <p>
     * This differs from {@link #throttleLast} in that this only tracks the passage of time whereas
     * {@link #throttleLast} ticks at scheduled intervals.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleFirst.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code throttleFirst} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param windowDuration
     *            time to wait before emitting another item after emitting the last item
     * @param unit
     *            the unit of time of {@code windowDuration}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> throttleFirst(long windowDuration, @NonNull TimeUnit unit) {
        return throttleFirst(windowDuration, unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits only the first item emitted by the current {@code Flowable} during sequential
     * time windows of a specified duration, where the windows are managed by a specified {@link Scheduler}.
     * <p>
     * This differs from {@link #throttleLast} in that this only tracks the passage of time whereas
     * {@link #throttleLast} ticks at scheduled intervals.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleFirst.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param skipDuration
     *            time to wait before emitting another item after emitting the last item
     * @param unit
     *            the unit of time of {@code skipDuration}
     * @param scheduler
     *            the {@code Scheduler} to use internally to manage the timers that handle timeout for each
     *            event
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> throttleFirst(long skipDuration, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableThrottleFirstTimed<>(this, skipDuration, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that emits only the last item emitted by the current {@code Flowable} during sequential
     * time windows of a specified duration.
     * <p>
     * This differs from {@link #throttleFirst} in that this ticks along at a scheduled interval whereas
     * {@link #throttleFirst} does not tick, it just tracks the passage of time.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLast.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code throttleLast} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param intervalDuration
     *            duration of windows within which the last item emitted by the current {@code Flowable} will be
     *            emitted
     * @param unit
     *            the unit of time of {@code intervalDuration}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #sample(long, TimeUnit)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> throttleLast(long intervalDuration, @NonNull TimeUnit unit) {
        return sample(intervalDuration, unit);
    }

    /**
     * Returns a {@code Flowable} that emits only the last item emitted by the current {@code Flowable} during sequential
     * time windows of a specified duration, where the duration is governed by a specified {@link Scheduler}.
     * <p>
     * This differs from {@link #throttleFirst(long, TimeUnit, Scheduler)} in that this ticks along at a scheduled interval whereas
     * {@code throttleFirst} does not tick, it just tracks the passage of time.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLast.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param intervalDuration
     *            duration of windows within which the last item emitted by the current {@code Flowable} will be
     *            emitted
     * @param unit
     *            the unit of time of {@code intervalDuration}
     * @param scheduler
     *            the {@code Scheduler} to use internally to manage the timers that handle timeout for each
     *            event
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/sample.html">ReactiveX operators documentation: Sample</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #sample(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> throttleLast(long intervalDuration, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return sample(intervalDuration, unit, scheduler);
    }

    /**
     * Throttles items from the upstream {@code Flowable} by first emitting the next
     * item from upstream, then periodically emitting the latest item (if any) when
     * the specified timeout elapses between them.
     * <p>
     * <img width="640" height="326" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLatest.png" alt="">
     * <p>
     * Unlike the option with {@link #throttleLatest(long, TimeUnit, boolean)}, the very last item being held back
     * (if any) is not emitted when the upstream completes.
     * <p>
     * If no items were emitted from the upstream during this timeout phase, the next
     * upstream item is emitted immediately and the timeout window starts from then.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.
     *  If the downstream is not ready to receive items, a
     *  {@link io.reactivex.rxjava3.exceptions.MissingBackpressureException MissingBackpressureException}
     *  will be signaled.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code throttleLatest} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.14 - experimental
     * @param timeout the time to wait after an item emission towards the downstream
     *                before trying to emit the latest item from upstream again
     * @param unit    the time unit
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @since 2.2
     * @see #throttleLatest(long, TimeUnit, boolean)
     * @see #throttleLatest(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> throttleLatest(long timeout, @NonNull TimeUnit unit) {
        return throttleLatest(timeout, unit, Schedulers.computation(), false);
    }

    /**
     * Throttles items from the upstream {@code Flowable} by first emitting the next
     * item from upstream, then periodically emitting the latest item (if any) when
     * the specified timeout elapses between them.
     * <p>
     * <img width="640" height="326" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLatest.e.png" alt="">
     * <p>
     * If no items were emitted from the upstream during this timeout phase, the next
     * upstream item is emitted immediately and the timeout window starts from then.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.
     *  If the downstream is not ready to receive items, a
     *  {@link io.reactivex.rxjava3.exceptions.MissingBackpressureException MissingBackpressureException}
     *  will be signaled.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code throttleLatest} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     * <p>History: 2.1.14 - experimental
     * @param timeout the time to wait after an item emission towards the downstream
     *                before trying to emit the latest item from upstream again
     * @param unit    the time unit
     * @param emitLast If {@code true}, the very last item from the upstream will be emitted
     *                 immediately when the upstream completes, regardless if there is
     *                 a timeout window active or not. If {@code false}, the very last
     *                 upstream item is ignored and the flow terminates.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see #throttleLatest(long, TimeUnit, Scheduler, boolean)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> throttleLatest(long timeout, @NonNull TimeUnit unit, boolean emitLast) {
        return throttleLatest(timeout, unit, Schedulers.computation(), emitLast);
    }

    /**
     * Throttles items from the upstream {@code Flowable} by first emitting the next
     * item from upstream, then periodically emitting the latest item (if any) when
     * the specified timeout elapses between them.
     * <p>
     * <img width="640" height="326" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLatest.s.png" alt="">
     * <p>
     * Unlike the option with {@link #throttleLatest(long, TimeUnit, Scheduler, boolean)}, the very last item being held back
     * (if any) is not emitted when the upstream completes.
     * <p>
     * If no items were emitted from the upstream during this timeout phase, the next
     * upstream item is emitted immediately and the timeout window starts from then.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.
     *  If the downstream is not ready to receive items, a
     *  {@link io.reactivex.rxjava3.exceptions.MissingBackpressureException MissingBackpressureException}
     *  will be signaled.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     * <p>History: 2.1.14 - experimental
     * @param timeout the time to wait after an item emission towards the downstream
     *                before trying to emit the latest item from upstream again
     * @param unit    the time unit
     * @param scheduler the {@code Scheduler} where the timed wait and latest item
     *                  emission will be performed
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see #throttleLatest(long, TimeUnit, Scheduler, boolean)
     * @since 2.2
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> throttleLatest(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return throttleLatest(timeout, unit, scheduler, false);
    }

    /**
     * Throttles items from the upstream {@code Flowable} by first emitting the next
     * item from upstream, then periodically emitting the latest item (if any) when
     * the specified timeout elapses between them.
     * <p>
     * <img width="640" height="326" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleLatest.se.png" alt="">
     * <p>
     * If no items were emitted from the upstream during this timeout phase, the next
     * upstream item is emitted immediately and the timeout window starts from then.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.
     *  If the downstream is not ready to receive items, a
     *  {@link io.reactivex.rxjava3.exceptions.MissingBackpressureException MissingBackpressureException}
     *  will be signaled.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@link Scheduler} this operator will use.</dd>
     * </dl>
     * <p>History: 2.1.14 - experimental
     * @param timeout the time to wait after an item emission towards the downstream
     *                before trying to emit the latest item from upstream again
     * @param unit    the time unit
     * @param scheduler the {@code Scheduler} where the timed wait and latest item
     *                  emission will be performed
     * @param emitLast If {@code true}, the very last item from the upstream will be emitted
     *                 immediately when the upstream completes, regardless if there is
     *                 a timeout window active or not. If {@code false}, the very last
     *                 upstream item is ignored and the flow terminates.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @since 2.2
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> throttleLatest(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, boolean emitLast) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableThrottleLatest<>(this, timeout, unit, scheduler, emitLast));
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, except that it drops items emitted by the
     * current {@code Flowable} that are followed by newer items before a timeout value expires. The timer resets on
     * each emission (alias to {@link #debounce(long, TimeUnit)}).
     * <p>
     * <em>Note:</em> If items keep being emitted by the current {@code Flowable} faster than the timeout then no items
     * will be emitted by the resulting {@code Flowable}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleWithTimeout.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code throttleWithTimeout} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timeout
     *            the length of the window of time that must pass after the emission of an item from the current
     *            {@code Flowable} in which it emits no items in order for the item to be emitted by the
     *            resulting {@code Flowable}
     * @param unit
     *            the unit of time for the specified {@code timeout}
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/debounce.html">ReactiveX operators documentation: Debounce</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #debounce(long, TimeUnit)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> throttleWithTimeout(long timeout, @NonNull TimeUnit unit) {
        return debounce(timeout, unit);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, except that it drops items emitted by the
     * current {@code Flowable} that are followed by newer items before a timeout value expires on a specified
     * {@link Scheduler}. The timer resets on each emission (alias to {@link #debounce(long, TimeUnit, Scheduler)}).
     * <p>
     * <em>Note:</em> If items keep being emitted by the current {@code Flowable} faster than the timeout then no items
     * will be emitted by the resulting {@code Flowable}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/throttleWithTimeout.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>This operator does not support backpressure as it uses time to control data flow.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timeout
     *            the length of the window of time that must pass after the emission of an item from the current
     *            {@code Flowable} in which it emits no items in order for the item to be emitted by the
     *            resulting {@code Flowable}
     * @param unit
     *            the unit of time for the specified {@code timeout}
     * @param scheduler
     *            the {@code Scheduler} to use internally to manage the timers that handle the timeout for each
     *            item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/debounce.html">ReactiveX operators documentation: Debounce</a>
     * @see <a href="https://github.com/ReactiveX/RxJava/wiki/Backpressure">RxJava wiki: Backpressure</a>
     * @see #debounce(long, TimeUnit, Scheduler)
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.ERROR)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> throttleWithTimeout(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return debounce(timeout, unit, scheduler);
    }

    /**
     * Returns a {@code Flowable} that emits records of the time interval between consecutive items emitted by the
     * current {@code Flowable}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeInterval.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code timeInterval} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @return the new {@code Flowable} instance
     * @see <a href="http://reactivex.io/documentation/operators/timeinterval.html">ReactiveX operators documentation: TimeInterval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<Timed<T>> timeInterval() {
        return timeInterval(TimeUnit.MILLISECONDS, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits records of the time interval between consecutive items emitted by the
     * current {@code Flowable}, where this interval is computed on a specified {@link Scheduler}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeInterval.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code timeInterval} does not operate on any particular scheduler but uses the current time
     *  from the specified {@code Scheduler}.</dd>
     * </dl>
     *
     * @param scheduler
     *            the {@code Scheduler} used to compute time intervals
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeinterval.html">ReactiveX operators documentation: TimeInterval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE) // Supplied scheduler is only used for creating timestamps.
    @NonNull
    public final Flowable<Timed<T>> timeInterval(@NonNull Scheduler scheduler) {
        return timeInterval(TimeUnit.MILLISECONDS, scheduler);
    }

    /**
     * Returns a {@code Flowable} that emits records of the time interval between consecutive items emitted by the
     * current {@code Flowable}.
     * <p>
     * <img width="640" height="310" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeInterval.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code timeInterval} does not operate on any particular scheduler but uses the current time
     *  from the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param unit the time unit for the current time
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeinterval.html">ReactiveX operators documentation: TimeInterval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final Flowable<Timed<T>> timeInterval(@NonNull TimeUnit unit) {
        return timeInterval(unit, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that emits records of the time interval between consecutive items emitted by the
     * current {@code Flowable}, where this interval is computed on a specified {@link Scheduler}.
     * <p>
     * <img width="640" height="315" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeInterval.s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>{@code timeInterval} does not operate on any particular scheduler but uses the current time
     *  from the specified {@code Scheduler}.</dd>
     * </dl>
     *
     * @param unit the time unit for the current time
     * @param scheduler
     *            the {@code Scheduler} used to compute time intervals
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeinterval.html">ReactiveX operators documentation: TimeInterval</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE) // Supplied scheduler is only used for creating timestamps.
    @NonNull
    public final Flowable<Timed<T>> timeInterval(@NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        Objects.requireNonNull(unit, "unit is null");
        Objects.requireNonNull(scheduler, "scheduler is null");
        return RxJavaPlugins.onAssembly(new FlowableTimeInterval<>(this, unit, scheduler));
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, but notifies {@link Subscriber}s of a
     * {@link TimeoutException} if an item emitted by the current {@code Flowable} doesn't arrive within a window of
     * time after the emission of the previous item, where that period of time is measured by a {@link Publisher} that
     * is a function of the previous item.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout3.v3.png" alt="">
     * <p>
     * Note: The arrival of the first source item is never timed out.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the current {@code Flowable}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code timeout} operates by default on the {@code immediate} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <V>
     *            the timeout value type (ignored)
     * @param itemTimeoutIndicator
     *            a function that returns a {@code Publisher} for each item emitted by the current
     *            {@code Flowable} and that determines the timeout window for the subsequent item
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code itemTimeoutIndicator} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.NONE)
    @NonNull
    public final <V> Flowable<T> timeout(@NonNull Function<? super T, ? extends Publisher<V>> itemTimeoutIndicator) {
        return timeout0(null, itemTimeoutIndicator, null);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, but that switches to a fallback {@link Publisher} if
     * an item emitted by the current {@code Flowable} doesn't arrive within a window of time after the emission of the
     * previous item, where that period of time is measured by a {@code Publisher} that is a function of the previous
     * item.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout4.v3.png" alt="">
     * <p>
     * Note: The arrival of the first source item is never timed out.
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the current {@code Flowable}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code timeout} operates by default on the {@code immediate} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param <V>
     *            the timeout value type (ignored)
     * @param itemTimeoutIndicator
     *            a function that returns a {@code Publisher}, for each item emitted by the current {@code Flowable}, that
     *            determines the timeout window for the subsequent item
     * @param fallback
     *            the fallback {@code Publisher} to switch to if the current {@code Flowable} times out
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code itemTimeoutIndicator} or {@code fallback} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.NONE)
    public final <V> Flowable<T> timeout(@NonNull Function<? super T, ? extends Publisher<@NonNull V>> itemTimeoutIndicator, @NonNull Publisher<@NonNull ? extends T> fallback) {
        Objects.requireNonNull(fallback, "fallback is null");
        return timeout0(null, itemTimeoutIndicator, fallback);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable} but applies a timeout policy for each emitted
     * item. If the next item isn't emitted within the specified timeout duration starting from its predecessor,
     * the resulting {@code Flowable} terminates and notifies {@link Subscriber}s of a {@link TimeoutException}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout.1.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code timeout} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timeout
     *            maximum duration between emitted items before a timeout occurs
     * @param unit
     *            the unit of time that applies to the {@code timeout} argument.
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    @NonNull
    public final Flowable<T> timeout(long timeout, @NonNull TimeUnit unit) {
        return timeout0(timeout, unit, null, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable} but applies a timeout policy for each emitted
     * item. If the next item isn't emitted within the specified timeout duration starting from its predecessor,
     * the current {@code Flowable} is disposed and the resulting {@code Flowable} begins instead to mirror a fallback {@link Publisher}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout.2.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the current {@code Flowable}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>This version of {@code timeout} operates by default on the {@code computation} {@link Scheduler}.</dd>
     * </dl>
     *
     * @param timeout
     *            maximum duration between items before a timeout occurs
     * @param unit
     *            the unit of time that applies to the {@code timeout} argument
     * @param fallback
     *            the fallback {@code Publisher} to use in case of a timeout
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code fallback} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.COMPUTATION)
    public final Flowable<T> timeout(long timeout, @NonNull TimeUnit unit, @NonNull Publisher<@NonNull ? extends T> fallback) {
        Objects.requireNonNull(fallback, "fallback is null");
        return timeout0(timeout, unit, fallback, Schedulers.computation());
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable} but applies a timeout policy for each emitted
     * item using a specified {@link Scheduler}. If the next item isn't emitted within the specified timeout duration
     * starting from its predecessor, the current {@code Flowable} is disposed and the resulting {@code Flowable} begins
     * instead to mirror a fallback {@link Publisher}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout.2s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. The {@code Publisher}
     *  sources are expected to honor backpressure as well.
     *  If any of the current {@code Flowable}s violate this, it <em>may</em> throw an
     *  {@link IllegalStateException} when the current {@code Flowable} completes.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timeout
     *            maximum duration between items before a timeout occurs
     * @param unit
     *            the unit of time that applies to the {@code timeout} argument
     * @param scheduler
     *            the {@code Scheduler} to run the timeout timers on
     * @param fallback
     *            the {@code Publisher} to use as the fallback in case of a timeout
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit}, {@code scheduler} or {@code fallback} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @NonNull
    @BackpressureSupport(BackpressureKind.FULL)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    public final Flowable<T> timeout(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler, @NonNull Publisher<@NonNull ? extends T> fallback) {
        Objects.requireNonNull(fallback, "fallback is null");
        return timeout0(timeout, unit, fallback, scheduler);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable} but applies a timeout policy for each emitted
     * item, where this policy is governed by a specified {@link Scheduler}. If the next item isn't emitted within the
     * specified timeout duration starting from its predecessor, the resulting {@code Flowable} terminates and
     * notifies {@link Subscriber}s of a {@link TimeoutException}.
     * <p>
     * <img width="640" height="305" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout.1s.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator doesn't interfere with backpressure which is determined by the current {@code Flowable}'s backpressure
     *  behavior.</dd>
     *  <dt><b>Scheduler:</b></dt>
     *  <dd>You specify which {@code Scheduler} this operator will use.</dd>
     * </dl>
     *
     * @param timeout
     *            maximum duration between items before a timeout occurs
     * @param unit
     *            the unit of time that applies to the {@code timeout} argument
     * @param scheduler
     *            the {@code Scheduler} to run the timeout timers on
     * @return the new {@code Flowable} instance
     * @throws NullPointerException if {@code unit} or {@code scheduler} is {@code null}
     * @see <a href="http://reactivex.io/documentation/operators/timeout.html">ReactiveX operators documentation: Timeout</a>
     */
    @CheckReturnValue
    @BackpressureSupport(BackpressureKind.PASS_THROUGH)
    @SchedulerSupport(SchedulerSupport.CUSTOM)
    @NonNull
    public final Flowable<T> timeout(long timeout, @NonNull TimeUnit unit, @NonNull Scheduler scheduler) {
        return timeout0(timeout, unit, null, scheduler);
    }

    /**
     * Returns a {@code Flowable} that mirrors the current {@code Flowable}, but notifies {@link Subscriber}s of a
     * {@link TimeoutException} if either the first item emitted by the current {@code Flowable} or any subsequent item
     * doesn't arrive within time windows defined by other {@link Publisher}s.
     * <p>
     * <img width="640" height="400" src="https://raw.github.com/wiki/ReactiveX/RxJava/images/rx-operators/timeout5.v3.png" alt="">
     * <dl>
     *  <dt><b>Backpressure:</b></dt>
     *  <dd>The operator honors backpressure from downstream. Both this and the returned {@code Publisher}s
    