package io.intercom.android.sdk;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.intercom.twig.Twig;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.intercom.android.sdk.logger.LumberMill;
import io.intercom.android.sdk.utilities.CustomAttributeValidator;

/**
 * <p>The UserAttributes object is used for updating a user in Intercom.</p>
 * <p>All of the default attributes you can modify are available as methods on the UserAttributes.Builder.</p>
 * <p>This is an example of how to create a UserAttributes object to update default attributes.</p>
 * <pre>
 * {@code
 * UserAttributes userAttributes = new UserAttributes.Builder()
 *         .withUserId("12345")
 *         .withEmail("test@email.com")
 *         .withName("Andy").build();
 * }
 * </pre>
 *
 * <p>You can also add custom attributes to your user.</p>
 *
 * <pre>
 * {@code
 * UserAttributes userAttributes = new UserAttributes.Builder()
 *         .withUserId("12345")
 *         .withEmail("test@email.com")
 *         .withCustomAttribute("items_in_cart", 8);
 * }
 * </pre>
 *
 * <p>If you want to add multiple custom attributes at once you can pass a Map to the withCustomAttributes method. </p>
 */
public class UserAttributes {
    private static final Twig TWIG = LumberMill.getLogger();

    private static final String EMAIL = "email";
    private static final String USER_ID = "user_id";
    private static final String NAME = "name";
    private static final String PHONE = "phone";
    private static final String CUSTOM_ATTRIBUTES = "custom_attributes";
    private static final String COMPANIES = "companies";
    private static final String SIGNED_UP_AT = "signed_up_at";
    private static final String UNSUBSCRIBED_FROM_EMAILS = "unsubscribed_from_emails";
    private static final String LANGUAGE_OVERRIDE = "language_override";

    private final Map<String, Object> attributes;
    private final Map<String, Object> customAttributes;
    private final List<Map<String, Object>> companies;

    UserAttributes(Builder builder) {
        this.attributes = builder.attributes;
        this.customAttributes = builder.customAttributes;
        this.companies = builder.companies;
    }

    public Map<String, Object> toMap() {
        if (!customAttributes.isEmpty()) {
            attributes.put(CUSTOM_ATTRIBUTES, customAttributes);
        }
        if (!companies.isEmpty()) {
            attributes.put(COMPANIES, companies);
        }
        return attributes;
    }

    public boolean isEmpty() {
        return attributes.isEmpty() && customAttributes.isEmpty() && companies.isEmpty();
    }

    public static final class Builder {
        final Map<String, Object> attributes = new HashMap<>();
        final Map<String, Object> customAttributes = new HashMap<>();
        final List<Map<String, Object>> companies = new ArrayList<>();

        /**
         * Build the userAttributes object to update a userAttributes in Intercom.
         *
         * @return the UserAttributes object
         */
        public UserAttributes build() {
            return new UserAttributes(this);
        }

        /**
         * Set the email address for this user.
         *
         * @param email the user's email address
         * @return the UserAttributes.Builder object
         */
        public Builder withEmail(@Nullable String email) {
            attributes.put(EMAIL, email);
            return this;
        }

        /**
         * Set the user ID on this user.
         *
         * @param userId a unique user ID
         * @return the UserAttributes.Builder object
         */
        public Builder withUserId(@Nullable String userId) {
            attributes.put(USER_ID, userId);
            return this;
        }

        /**
         * Set the name of this user.
         *
         * @param name the user's name
         * @return the UserAttributes.Builder object
         */
        public Builder withName(@Nullable String name) {
            attributes.put(NAME, name);
            return this;
        }

        /**
         * Set the phone number of this user.
         *
         * @param phone the user's phone number
         * @return the UserAttributes.Builder object
         */
        public Builder withPhone(@Nullable String phone) {
            attributes.put(PHONE, phone);
            return this;
        }

        /**
         * Set the time when the user signed up.
         *
         * @param signedUpAt a Unix timestamp representing when the user signed up
         * @return the UserAttributes.Builder object
         */
        public Builder withSignedUpAt(@Nullable Long signedUpAt) {
            attributes.put(SIGNED_UP_AT, signedUpAt);
            return this;
        }

        /**
         * Set the time when the user signed up.
         *
         * @param signedUpAt a Date representing when the user signed up
         * @return the UserAttributes.Builder object
         */
        public Builder withSignedUpAt(@Nullable Date signedUpAt) {
            return withSignedUpAt(signedUpAt == null ? null : TimeUnit.MILLISECONDS.toSeconds(signedUpAt.getTime()));
        }

        /**
         * Set if the user has unsubscribed to emails.
         *
         * @param unsubscribedFromEmails a boolean to modify the unsubscribed from emails status
         * @return the UserAttributes.Builder object
         */
        public Builder withUnsubscribedFromEmails(@Nullable Boolean unsubscribedFromEmails) {
            attributes.put(UNSUBSCRIBED_FROM_EMAILS, unsubscribedFromEmails);
            return this;
        }

        /**
         * Set the language override code for this user.
         *
         * <p>languageOverride must be a valid language code.</p>
         * <p>For more information see [here](https://docs.intercom.com/configure-intercom-for-your-product-or-site/
         * customize-the-intercom-messenger/localize-intercom-to-work-with-multiple-languages ).</p>
         *
         * @param languageOverride a string for the language
         * @return the UserAttributes.Builder object
         */
        public Builder withLanguageOverride(@Nullable String languageOverride) {
            attributes.put(LANGUAGE_OVERRIDE, languageOverride);
            return this;
        }

        /**
         * Set a company on the user.
         *
         * @param company a Company object to add to the user
         * @return the UserAttributes.Builder object
         */
        public Builder withCompany(@NonNull Company company) {
            if (company == null) {
                TWIG.w("The company you provided was null");
            } else {
                companies.add(company.getAttributes());
            }
            return this;
        }

        /**
         * Set a custom attribute on the user.
         *
         * <p>The key provided must not be null. This method accepts the following types: </p>
         * <p>String, Long, Float, Double, Boolean, Character, Byte, Short and Integer. </p>
         * <p>Any other types will not be added to the update.</p>
         *
         * @param key the name of the attribute
         * @param attribute the value of the attribute
         * @return the UserAttributes.Builder object
         */
        public Builder withCustomAttribute(@NonNull String key, @Nullable Object attribute) {
            if (key == null) {
                TWIG.w("The key you provided was null for the attribute " + attribute);
            } else if (CustomAttributeValidator.isValid(attribute)) {
                customAttributes.put(key, attribute);
            } else {
                TWIG.w("The custom user attribute " + key + " was of type "
                        + attribute.getClass().getSimpleName() + " We only accept the following types: "
                        + CustomAttributeValidator.getAcceptedTypes());
            }
            return this;
        }

        /**
         * Set several custom attitudes on the user at once.
         *
         * <p>This method accepts the following types: </p>
         * <p>String, Long, Float, Double, Boolean, Character, Byte, Short and Integer. </p>
         * <p>Any other types will not be added to the update.</p>
         *
         * @param attributes the map of attributes to update
         * @return the UserAttributes.Builder object
         */
        public Builder withCustomAttributes(@NonNull Map<String, ?> attributes) {
            if (attributes == null) {
                TWIG.w("The map of attributes you provided was null.");
            } else {
                for (Map.Entry<String, ?> entry : attributes.entrySet()) {
                    Object attribute = entry.getValue();
                    if (CustomAttributeValidator.isValid(attribute)) {
                        customAttributes.put(entry.getKey(), attribute);
                    } else {
                        TWIG.w("The custom user attribute " + entry.getKey() + " was of type "
                                + attribute.getClass().getSimpleName() + " We only accept the following types: "
                                + CustomAttributeValidator.getAcceptedTypes());
                    }
                }
            }
            return this;
        }
    }
}
