package io.intercom.android.sdk;

import android.annotation.SuppressLint;
import android.app.Application;
import android.app.TaskStackBuilder;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.intercom.twig.Twig;

import java.util.Map;

import io.intercom.android.sdk.identity.Registration;
import io.intercom.android.sdk.logger.TwigFactory;
import io.intercom.android.sdk.utilities.ValidatorUtil;

/**
 * <p>Intercom is your direct line of communication to every user, right inside your app. Intercom's in-app messages
 * are up to 10 times more effective than email too!
 * Send the right messages, to the right users, at exactly the right time.</p>
 */
public abstract class Intercom {
    public enum Visibility {
        GONE, VISIBLE
    }

    public static final Visibility VISIBLE = Visibility.VISIBLE;
    public static final Visibility GONE = Visibility.GONE;

    public static final String GCM_RECEIVER = "intercom_sdk";

    private static final Twig TWIG = TwigFactory.getLogger();

    @SuppressLint("StaticFieldLeak") // the reference to the Activity is unset if it stops
    private static final LateInitializationPreparer lateInitializationPreparer = new LateInitializationPreparer();

    private static @Nullable Intercom instance;

    /**
     * <p>Initializes the Intercom singleton.</p>
     *
     * <p>Must be called before trying to access the object with Intercom.client()</p>
     *
     * @param application Your app's Application object
     * @param apiKey The Android SDK API key found on the Intercom for Android settings page.
     * @param appId The app ID of your Intercom app.
     * @since 1.0.0
     */
    public static synchronized void initialize(Application application, String apiKey, String appId) {
        if (instance != null) {
            TWIG.i("Intercom has already been initialized");
            return;
        }
        if (ValidatorUtil.isValidConstructorParams(application, apiKey, appId)) {
            TWIG.i("Intercom has already been initialized");
            instance = RealIntercom.create(application, apiKey, appId);
            lateInitializationPreparer.handlePastLifecycleEvents(application, Injector.get());
        } else {
            instance = new InvalidIntercom();
        }
    }

    /**
     * <p>Begin tracking Activity lifecycle events so that Intercom can be initialized later.</p>
     *
     * <p>This must be called in Application.onCreate()</p>
     *
     * @param application Your app's Application object
     * @since 3.0.19
     */
    public static synchronized void registerForLaterInitialisation(@NonNull Application application) {
        if (instance != null) {
            TWIG.i("Intercom has already been initialized");
            return;
        }
        if (application == null) {
            throw new NullPointerException("Cannot call registerForLaterInitialisation() with a null Application");
        }
        lateInitializationPreparer.register(application);
    }

    /**
     * <p>Stop tracking Activity lifecycle events. This will prevent you from initializing Intercom correctly later.</p>
     *
     * @param application Your app's Application object
     * @since 3.0.19
     */
    public static void unregisterForLateInitialisation(@NonNull Application application) {
        if (application == null) {
            throw new NullPointerException("Cannot call registerForLaterInitialisation() with a null Application");
        }
        lateInitializationPreparer.unregister(application);
    }

    /**
     * <p>Provides access to the Intercom client.</p>
     *
     * @return the Intercom client
     * @since 1.0.0
     */
    public static synchronized Intercom client() {
        if (instance == null) {
            throw new IllegalStateException("Please call Intercom.initialize() before requesting the client.");
        }
        return instance;
    }

    /**
     * <p>Registers an unidentified user with Intercom.</p>
     *
     * <p>If you call registerUnidentifiedUser, all activity will be tracked anonymously. If you choose to subsequently
     * identify that user, all that anonymous activity will be merged into the identified user. This means that you
     * will no longer see the anonymous user in Intercom, but rather the identified one.</p>
     *
     * @since 1.0.0
     */
    public abstract void registerUnidentifiedUser();

    /**
     * <p>Registers an identified user with Intercom.</p>
     *
     * <p>In order to keep track of a specific user, you must identify it with a unique user identifier, an email
     * address, or both. By supplying information like this Intercom provides richer user profiles for your users.
     * This is a user ID, supplied by you (e.g. from an existing web service for your product) to represent your
     * user in Intercom, once set it cannot be changed.</p>
     *
     * @param userRegistration the {@link Registration} for a user
     * @since 1.0.0
     */
    public abstract void registerIdentifiedUser(Registration userRegistration);

    /**
     * <p>Sets the secure hash and data to be used for Secure Mode.</p>
     *
     * <p>Secure Mode helps to make sure that conversations between you and your users are kept private and that one
     * user can't impersonate another.
     * In Secure Mode the SDK will sign all requests going to the Intercom servers with tokens.
     * It requires your mobile application to have its own server which authenticates the app's users
     * and which can store a secret. More information on secure mode can be found <a href="https://docs.intercom.com/configure-intercom-for-your-product-or-site/staying-secure/enabling-secure-mode-on-intercom-for-android">here</a>.</p>
     *
     * <p>This should be called before any user registration takes place.</p>
     *
     * @param secureHash a HMAC digest of secureData
     * @param secureData a piece of user data
     * @since 1.0.0
     */
    public abstract void setSecureMode(String secureHash, String secureData);

    /**
     * <p>Updates a user in Intercom.</p>
     *
     * <p>You can send any data you like to Intercom. Typically our customers see a lot of value in sending data that
     * relates to customer development, such as price plan, value of purchases, etc. Once these have been sent to
     * Intercom you can then apply filters based on these attributes.</p>
     *
     * <p>A detailed list of the fields you can use to update a user is available <a href="https://developers.intercom.com/reference#create-or-update-user">here</a></p>
     *
     * <p>Attributes such as the user email or name can be updated by calling</p>
     *
     * <pre>
     * {@code
     * Map<String, Object> userAttributes = new HashMap<>();
     * userAttributes.put("email", "admin@exaple.com");
     * userAttributes.put("name", "Admin name");
     * Intercom.client().updateUser(userAttributes);
     * }
     * </pre>
     *
     * <p>Custom user attributes can be created and modified by passing a custom_attributes map.
     * You do not have to create attributes in Intercom beforehand. If one hasn't been seen before it will be
     * created for you automatically.</p>
     *
     * <pre>
     * {@code
     * Map<String, Object> userAttributes = new HashMap<>();
     * Map<String, Object> customAttributes = new HashMap<>();
     * customAttributes.put("paid_subscriber", true);
     * customAttributes.put("monthly_spend", 155.5);
     * customAttributes.put("team_mates", 3);
     * userAttributes.put("custom_attributes", customAttributes);
     * Intercom.client().updateUser(userAttributes);
     * }
     * </pre>
     *
     * <p>You can also set company data via this call by submitting an attribute map like</p>
     *
     * <pre>
     * {@code
     * Map<String, Object> userMap = new HashMap<>();
     * Map<String, Object> companyData = new HashMap<>();
     * List<Map<String, Object>> companies = new ArrayList<>();
     * Map<String, Object> customCompanyData = new HashMap<>();
     * companyData.put("name", "My Company");
     * companyData.put("id", "abcd1234");
     * customCompanyData.put("team_mates", 8);
     * customCompanyData.put("monthly_spend", 155.98);
     * companyData.put("custom_attributes", customCompanyData);
     * companies.add(companyData);
     * userMap.put("companies", companies);
     * Intercom.client().updateUser(userMap);
     * }
     * </pre>
     *
     * <p>`id` is a required field for adding or modifying a company.</p>
     *
     * <p>Attributes may be a `string`, `int`, `double`, `unix timestamp` or `bool`.</p>
     *
     * @param attributes This is a map containing key/value pairs for multiple attributes.</p>
     * @since 1.0.0
     */
    @Deprecated public abstract void updateUser(Map<String, ?> attributes);

    /**
     * <p>Updates a user in Intercom.</p>
     *
     * <p>You can send any data you like to Intercom. Typically our customers see a lot of value in sending data that
     * relates to customer development, such as price plan, value of purchases, etc. Once these have been sent to
     * Intercom you can then apply filters based on these attributes.</p>
     *
     * <p>A detailed list of the fields you can use to update a user is available <a href="https://developers.intercom.com/reference#create-or-update-user">here</a></p>
     *
     * <p>Attributes such as the user email or name can be updated by calling</p>
     *
     * <pre>
     * {@code
     *
     * User user = new User.Builder()
     *                        .withUserId("1234")
     *                        .withEmail("test@test.com")
     *                        .build();
     * Intercom.client().updateUser(user);
     * }
     * </pre>
     *
     * <p>Custom user attributes can be created and modified by passing a custom_attributes map.
     * You do not have to create attributes in Intercom beforehand. If one hasn't been seen before it will be
     * created for you automatically.</p>
     *
     * <pre>
     * {@code
     * User user = new User.Builder()
     *                       .withUserId("1234")
     *                       .withEmail("test@test.com")
     *                       .withCustomAttribute("team_mates", 3)
     *                       .build();
     * Intercom.client().updateUser(user);
     * }
     * </pre>
     *
     * <p>For multiple user attributes you can chain withCustomAttribute several times
     * or pass a Map to withCustomAttributes</p>
     *
     * <p>You can also set company data via this call by submitting an attribute map like</p>
     *
     * <pre>
     * {@code
     * Company company = new Company.Builder()
     *                                .withCompanyId("1234")
     *                                .withName("TestCorp")
     *                                .build();
     * User user = new User.Builder()
     *                       .withCompany(company)
     *                       .build();
     * Intercom.client().updateUser(user);
     * }
     * </pre>
     *
     * <p>`id` is a required field for adding or modifying a company. A detailed description of the company model
     * is available <a href="https://developers.intercom.com/docs#companies-and--users">here</a></p>
     *
     * <p>Attributes may be a `string`, `int`, `double`, `unix timestamp` or `bool`.</p>
     *
     * @param user This is a user object to update this user in Intercom.</p>
     * @since 3.0.19
     */
    public abstract void updateUser(User user);

    /**
     * <p>Logs an event with a given name.</p>
     *
     * <p>You can log events in Intercom based on user actions in your app. Events are different
     * to custom user attributes in that events are information on what Users did and when they
     * did it, whereas custom user attributes represent the User's current state as seen in their
     * profile. See details about Events <a href="https://developers.intercom.com/docs#events">here</a>.</p>
     *
     * @param name The name of the event that it is going to be logged.
     * @since 1.0.0
     */
    public abstract void logEvent(String name);


    /**
     * <p>Logs an event with a given name and some metadata.</p>
     *
     * <p>Metadata Objects support a few simple types that Intercom can present on your behalf, see the
     * <a href="https://developers.intercom.com/docs#event-metadata-types">Intercom API docs</a></p>
     *
     * <pre>
     * {@code
     * Map<String, Object> metadata = new HashMap<>();
     * Map<String, Object> orderDetails = new HashMap<>();
     * orderDetails.put("price", 123);
     * orderDetails.put("reference_value", "3434-3434");
     * orderDetails.put("url", "https://example.org/orders/3434-3434");
     * metadata.put("order", orderDetails);
     * metadata.put("order_date", 1392036272);
     * Intercom.client().logEvent("ordered_item", metadata);
     * }
     * </pre>
     *
     * @param name The name of the event you wish to track.
     * @param metaData a map of simple types to present to Intercom
     * @since 1.0.0
     */
    public abstract void logEvent(String name, Map<String, ?> metaData);

    /**
     * <p>Displays the Messenger.</p>
     *
     * <p>Opens the Intercom Messenger automatically to the best place for your users.</p>
     *
     * @since 3.0.0
     */
    public abstract void displayMessenger();

    /**
     * <p>Displays the message composer.</p>
     *
     * @since 1.0.0
     */
    public abstract void displayMessageComposer();

    /**
     * <p>Displays the message composer.</p>
     *
     * <p>Open the conversation screen with the composer pre-populated for example:</p>
     * <p>Intercom.client().displayMessageComposer("Message sent from FAQ screen: ");</p>
     * <p>Note:</p>
     * <p> - This only applies when the user is starting a new conversation.</p>
     * <p> - The text input cursor will start at the end of text passed in.</p>
     * <p> - This text is editable so the user can delete or modify it.</p>
     *
     * @param initialMessage Text that will be pre-populated in the composer
     * @since 3.0.9
     */
    public abstract void displayMessageComposer(String initialMessage);

    /**
     * <p>Displays the conversations list.</p>
     *
     * @since 1.0.0
     */
    public abstract void displayConversationsList();

    /**
     * <p>Toggles visibility of in-app messages.</p>
     *
     * <p>Set to Intercom.Visibility.GONE to hide all incoming Intercom messages and message previews in the parts of
     * your app where you do not wish to interrupt users: camera views, parts of a game etc.</p>
     *
     * @param visibility Intercom.GONE for hidden, Intercom.VISIBLE to show
     * @since 3.0.0
     */
    public abstract void setInAppMessageVisibility(Visibility visibility);

    /**
     * <p>Toggles visibility of the launcher view.</p>
     *
     * <p>Set as Intercom.Visibility.GONE to hide the launcher when you don't want it to be visible.</p>
     *
     * @param visibility Intercom.GONE for hidden, Intercom.VISIBLE to show
     * @since 3.0.0
     */
    public abstract void setLauncherVisibility(Visibility visibility);

    /**
     * <p>Dismisses the Intercom Messenger if it is on screen.</p>
     *
     * @since 3.0.0
     */
    public abstract void hideMessenger();

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.0
     * @deprecated replaced with {@link #handlePushMessage()}
     */
    @Deprecated public abstract void openGcmMessage();

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @param customStack provide a TaskStackBuilder to allow for a custom back stack
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.0
     * @deprecated replaced with {@link #handlePushMessage(TaskStackBuilder)}
     */
    @Deprecated public abstract void openGcmMessage(TaskStackBuilder customStack);

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.3
     */
    public abstract void handlePushMessage();

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @param customStack provide a TaskStackBuilder to allow for a custom back stack
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.3
     */
    public abstract void handlePushMessage(TaskStackBuilder customStack);

    /**
     * <p>Clears all data from the Intercom SDK.</p>
     *
     * <p>Reset is used to reset all local caches and user data the Intercom SDK has created.
     * Use this at a time when you wish to log a user out of your app or change a user.
     * Once called, the SDK will no longer communicate with Intercom until a further registration is made.</p>
     *
     * @since 1.0.0
     */
    public abstract void reset();

    /**
     * <p>Gets the number of unread conversations for a user.</p>
     *
     * @return the current count of unread conversations for the registered user
     * @since 3.0.0
     */
    public abstract int getUnreadConversationCount();

    /**
     * <p>Sets a listener that will be notified when the unread conversation count for the registered user changes.</p>
     *
     * <p>This listener will immediately receive an update with the latest unread count, and it will be notified of all
     * future updates. If you want to know the unread count right now you can use the
     * {@link #getUnreadConversationCount() getUnreadConversationCount()} method</p>
     *
     * <p>Multiple listeners may be set. A strong reference is kept to each listener.</p>
     *
     * @param listener the listener to be notified of unread count updates
     * @since 3.0.0
     */
    public abstract void addUnreadConversationCountListener(@NonNull UnreadConversationCountListener listener);

    /**
     * <p>Removes a listener from the collection that receive updates when the unread conversation count changes.</p>
     *
     * @param listener the listener to be removed
     * @since 3.0.0
     */
    public abstract void removeUnreadConversationCountListener(UnreadConversationCountListener listener);

    /**
     * <p>Set the level of the logger</p>
     *
     * <p>Set the level of logs to display in the console.
     * Pass in an Intercom log level eg. `setLogLevel(IntercomLogger.ERROR);`
     * Passing in IntercomLogger.DISABLED results in no logging being logged.
     * Default log level is set to IntercomLogger.WARN
     *
     * @param logLevel the level of logs to display
     * @since 1.1.4
     */
    public static void setLogLevel(int logLevel) {
        TwigFactory.setLogLevel(logLevel);
    }

}
