package io.intercom.android.sdk;

import android.app.Application;
import android.app.TaskStackBuilder;
import android.util.Log;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.intercom.twig.Twig;

import java.util.List;
import java.util.Map;

import io.intercom.android.sdk.helpcenter.api.CollectionContentRequestCallback;
import io.intercom.android.sdk.helpcenter.api.CollectionRequestCallback;
import io.intercom.android.sdk.helpcenter.api.SearchRequestCallback;
import io.intercom.android.sdk.identity.Registration;
import io.intercom.android.sdk.logger.LumberMill;
import io.intercom.android.sdk.utilities.ValidatorUtil;

/**
 * <p>Intercom is your direct line of communication to every user, right inside your app. Intercom's in-app messages
 * are up to 10 times more effective than email too!
 * Send the right messages, to the right users, at exactly the right time.</p>
 */
public abstract class Intercom {
    public enum Visibility {
        GONE, VISIBLE
    }

    public static final Visibility VISIBLE = Visibility.VISIBLE;
    public static final Visibility GONE = Visibility.GONE;

    @IntDef({LogLevel.VERBOSE, LogLevel.DEBUG, LogLevel.INFO, LogLevel.WARN,
            LogLevel.ERROR, LogLevel.ASSERT, LogLevel.DISABLED})
    //incorrect indentation warning for annotations see: https://github.com/checkstyle/checkstyle/issues/3302
    @SuppressWarnings("Indentation")
    public @interface LogLevel {
        int VERBOSE = Log.VERBOSE;
        int DEBUG = Log.DEBUG;
        int INFO = Log.INFO;
        int WARN = Log.WARN;
        int ERROR = Log.ERROR;
        int ASSERT = Log.ASSERT;
        int DISABLED = Twig.DISABLED;
    }

    public static final String PUSH_RECEIVER = "intercom_sdk";

    private static final Twig TWIG = LumberMill.getLogger();

    private static @Nullable Intercom instance;

    /**
     * <p>Initializes the Intercom singleton.</p>
     *
     * <p>Must be called before trying to access the object with Intercom.client()</p>
     *
     * @param application Your app's Application object
     * @param apiKey The Android SDK API key found on the Intercom for Android settings page.
     * @param appId The app ID of your Intercom app.
     * @since 1.0.0
     */
    public static synchronized void initialize(Application application, String apiKey, String appId) {
        if (instance != null) {
            TWIG.i("Intercom has already been initialized");
            return;
        }
        if (ValidatorUtil.isValidConstructorParams(application, apiKey, appId)) {
            TWIG.i("Intercom has already been initialized");
            instance = RealIntercom.create(application, apiKey, appId);
            LateInitializationPreparer.getInstance().handlePastLifecycleEvents(application, Injector.get());
        } else {
            instance = new InvalidIntercom();
        }
    }

    /**
     * <p>Begin tracking Activity lifecycle events so that Intercom can be initialized later.</p>
     *
     * <p>This must be called in Application.onCreate()</p>
     *
     * @param application Your app's Application object
     * @since 3.0.19
     */
    public static synchronized void registerForLaterInitialisation(@NonNull Application application) {
        if (instance != null) {
            TWIG.i("Intercom has already been initialized");
            return;
        }
        if (application == null) {
            throw new NullPointerException("Cannot call registerForLaterInitialisation() with a null Application");
        }
        LateInitializationPreparer.getInstance().register(application);
    }

    /**
     * <p>Stop tracking Activity lifecycle events. This will prevent you from initializing Intercom correctly later.</p>
     *
     * @param application Your app's Application object
     * @since 3.0.19
     */
    public static void unregisterForLateInitialisation(@NonNull Application application) {
        if (application == null) {
            throw new NullPointerException("Cannot call unregisterForLateInitialisation() with a null Application");
        }
        LateInitializationPreparer.getInstance().unregister(application);
    }

    /**
     * <p>Provides access to the Intercom client.</p>
     *
     * @return the Intercom client
     * @since 1.0.0
     */
    public static synchronized Intercom client() {
        if (instance == null) {
            throw new IllegalStateException("Please call Intercom.initialize() before requesting the client.");
        }
        return instance;
    }

    /**
     * <p>Registers an unidentified user with Intercom.</p>
     *
     * <p>If you call registerUnidentifiedUser, all activity will be tracked anonymously. If you choose to subsequently
     * identify that user, all that anonymous activity will be merged into the identified user. This means that you
     * will no longer see the anonymous user in Intercom, but rather the identified one.</p>
     *
     * @since 1.0.0
     */
    public abstract void registerUnidentifiedUser();

    /**
     * <p>Registers an identified user with Intercom.</p>
     *
     * <p>In order to keep track of a specific user, you must identify it with a unique user identifier, an email
     * address, or both. By supplying information like this Intercom provides richer user profiles for your users.
     * This is a user ID, supplied by you (e.g. from an existing web service for your product) to represent your
     * user in Intercom, once set it cannot be changed.</p>
     *
     * @param userRegistration the {@link Registration} for a user
     * @since 1.0.0
     */
    public abstract void registerIdentifiedUser(Registration userRegistration);

    /**
     * <p>Sets the user hash necessary for validation when Identity Verification is enabled.</p>
     *
     * <p>Identity Verification helps to make sure that conversations between you and your users are
     * kept private and that one user can't impersonate another.
     * If Identity Verification is enabled for your app the messenger will sign all requests going to the Intercom
     * servers with tokens.
     * It requires your mobile application to have its own server which authenticates the app's users
     * and which can store a secret. More information on Identity Verification can be found <a href="https://docs.intercom.com/configure-intercom-for-your-product-or-site/staying-secure/enable-identity-verification-on-intercom-for-android">here</a>.</p>
     *
     * <p>This should be called before any user registration takes place.</p>
     *
     * @param userHash A HMAC digest of the user ID or email
     * @since 3.1.6
     */
    public abstract void setUserHash(String userHash);

    /**
     * <p>Updates a user in Intercom.</p>
     *
     * <p>You can send any data you like to Intercom. Typically our customers see a lot of value in sending data that
     * relates to customer development, such as price plan, value of purchases, etc. Once these have been sent to
     * Intercom you can then apply filters based on these attributes.</p>
     *
     * <p>A detailed list of the fields you can use to update a user is available <a href="https://developers.intercom.com/reference#create-or-update-user">here</a></p>
     *
     * <p>Attributes such as the user email or name can be updated by calling</p>
     *
     * <pre>
     * {@code
     *
     * UserAttributes userAttributes = new UserAttributes.Builder()
     *             .withUserId("1234")
     *             .withEmail("test@test.com")
     *             .build();
     * Intercom.client().updateUser(userAttributes);
     * }
     * </pre>
     *
     * <p>Custom user attributes can be created and modified by passing a custom_attributes map.
     * You do not have to create attributes in Intercom beforehand. If one hasn't been seen before it will be
     * created for you automatically.</p>
     *
     * <pre>
     * {@code
     * UserAttributes userAttributes = new UserAttributes.Builder()
     *             .withUserId("1234")
     *             .withEmail("test@test.com")
     *             .withCustomAttribute("team_mates", 3)
     *             .build();
     * Intercom.client().updateUser(userAttributes);
     * }
     * </pre>
     *
     * <p>For multiple user attributes you can chain withCustomAttribute several times
     * or pass a Map to withCustomAttributes</p>
     *
     * <p>All of the updatable User attributes are described here: {@link UserAttributes ()}</p>
     *
     * <p>You can also set company data via this call by submitting an attribute map like</p>
     *
     * <pre>
     * {@code
     * Company company = new Company.Builder()
     *             .withCompanyId("1234")
     *             .withName("TestCorp")
     *             .build();
     * UserAttributes userAttributes = new UserAttributes.Builder()
     *             .withCompany(company)
     *             .build();
     * Intercom.client().updateUser(userAttributes);
     * }
     * </pre>
     *
     * <p>All of the updatable Company attributes are described here: {@link Company ()}</p>
     *
     * <p>Custom attributes may be of type: String, Long, Float, Double, Boolean, Character, Byte, Short, Integer.</p>
     *
     * @param userAttributes The userAttributes object with the attributes to be set on the user in Intercom.
     * @since 3.1.0
     */
    public abstract void updateUser(UserAttributes userAttributes);

    /**
     * <p>Logs an event with a given name.</p>
     *
     * <p>You can log events in Intercom based on user actions in your app. Events are different
     * to custom user attributes in that events are information on what Users did and when they
     * did it, whereas custom user attributes represent the User's current state as seen in their
     * profile. See details about Events <a href="https://developers.intercom.com/docs#events">here</a>.</p>
     *
     * @param name The name of the event that it is going to be logged.
     * @since 1.0.0
     */
    public abstract void logEvent(String name);


    /**
     * <p>Logs an event with a given name and some metadata.</p>
     *
     * <p>Metadata Objects support a few simple types that Intercom can present on your behalf, see the
     * <a href="https://developers.intercom.com/docs#event-metadata-types">Intercom API docs</a></p>
     *
     * <pre>
     * {@code
     * Map<String, Object> metadata = new HashMap<>();
     * Map<String, Object> orderDetails = new HashMap<>();
     * orderDetails.put("price", 123);
     * orderDetails.put("reference_value", "3434-3434");
     * orderDetails.put("url", "https://example.org/orders/3434-3434");
     * metadata.put("order", orderDetails);
     * metadata.put("order_date", 1392036272);
     * Intercom.client().logEvent("ordered_item", metadata);
     * }
     * </pre>
     *
     * @param name The name of the event you wish to track.
     * @param metaData a map of simple types to present to Intercom
     * @since 1.0.0
     */
    public abstract void logEvent(String name, Map<String, ?> metaData);

    /**
     * <p>Displays the Messenger.</p>
     *
     * <p>Opens the Intercom Messenger automatically to the best place for your users.</p>
     *
     * @since 3.0.0
     */
    public abstract void displayMessenger();

    /**
     * <p>Displays the message composer.</p>
     *
     * @since 1.0.0
     */
    public abstract void displayMessageComposer();

    /**
     * <p>Displays the message composer.</p>
     *
     * <p>Open the conversation screen with the composer pre-populated for example:</p>
     * <p>Intercom.client().displayMessageComposer("Message sent from FAQ screen: ");</p>
     * <p>Note:</p>
     * <p> - This only applies when the user is starting a new conversation.</p>
     * <p> - The text input cursor will start at the end of text passed in.</p>
     * <p> - This text is editable so the user can delete or modify it.</p>
     *
     * @param initialMessage Text that will be pre-populated in the composer
     * @since 3.0.9
     */
    public abstract void displayMessageComposer(String initialMessage);

    /**
     * <p>Displays the conversations list.</p>
     *
     * @since 1.0.0
     * @deprecated call {@link #displayMessenger()} instead
     */
    @Deprecated public abstract void displayConversationsList();

    /**
     * <p>Open up your apps help center.</p>
     *
     * @since 4.1.7
     */
    public abstract void displayHelpCenter();

    /**
     * <p>Open help center with a filtered list of collections.</p>
     *
     * @since 10.0.0
     */
    public abstract void displayHelpCenterCollections(List<String> collectionIds);

    /**
     * <p>Displays carousel.</p>
     *
     * @param carouselId Id of carousel that will be shown.
     *
     * @since 8.3.0
     */
    public abstract void displayCarousel(String carouselId);

    /**
     * <p>Set the bottom padding of in app messages and the launcher.</p>
     *
     * <p>Setting the bottom padding will increase how far from the bottom of the screen the default launcher
     * and in app messages will appear.</p>
     * <p>Previously you could add `intercom_bottom_padding` to your `dimens.xml` to set the padding at startup.
     * This should be removed and your should call setBottomPadding instead.
     * `intercom_bottom_padding` will still function for the time being.</p>
     * <p>You can call setBottomPadding at any time to animate the position of your in-apps and launcher.</p>
     *
     * @param bottomPadding The amount of pixels to set the bottom padding
     * @since 3.2.2
     */
    public abstract void setBottomPadding(int bottomPadding);

    /**
     * <p>Toggles visibility of in-app messages.</p>
     *
     * <p>Set to Intercom.Visibility.GONE to hide all incoming Intercom messages and message previews in the parts of
     * your app where you do not wish to interrupt users: camera views, parts of a game etc.</p>
     *
     * @param visibility Intercom.GONE for hidden, Intercom.VISIBLE to show
     * @since 3.0.0
     */
    public abstract void setInAppMessageVisibility(Visibility visibility);

    /**
     * <p>Toggles visibility of the launcher view.</p>
     *
     * <p>Set as Intercom.Visibility.GONE to hide the launcher when you don't want it to be visible.</p>
     *
     * @param visibility Intercom.GONE for hidden, Intercom.VISIBLE to show
     * @since 3.0.0
     */
    public abstract void setLauncherVisibility(Visibility visibility);

    /**
     * <p>Hide all Intercom UI that are currently displayed.
     *  This will hide the Messenger, Help Center, Articles, and in-product messages (eg. Mobile Carousels, chats, and posts).</p>
     *
     * @since 10.0.0
     */
    public abstract void hideIntercom();

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.3
     */
    public abstract void handlePushMessage();

    /**
     * <p>Handles the opening of an Intercom push message.</p>
     *
     * <p>This will retrieve the URI from the last Intercom push message.</p>
     * <p>If the message is an in-app message or reply this will open up the Messenger (Push Notifications).</p>
     * <p>If the message was an Intercom push message this will follow the URI you provided (Push Messages).</p>
     *
     * @param customStack provide a TaskStackBuilder to allow for a custom back stack
     * @see <a href="https://docs.intercom.com/intercom-s-key-features-explained/sending-messages/how-mobile-push-notifications-and-messages-work">How do mobile push notifications work with Intercom?</a>
     * @since 3.0.3
     */
    public abstract void handlePushMessage(TaskStackBuilder customStack);

    /**
     * <p>Clears all data from the Intercom SDK.</p>
     *
     * <p>Reset is used to reset all local caches and user data the Intercom SDK has created.
     * Use this at a time when you wish to log a user out of your app or change a user.
     * Once called, the SDK will no longer communicate with Intercom until a further registration is made.</p>
     *
     * @since 1.0.0
     * @deprecated call {@link #logout()} instead
     */
    @Deprecated public abstract void reset();

    /**
     * <p>Clears all data from the Intercom SDK.</p>
     *
     * <p>Logout is used to clear all local caches and user data the Intercom SDK has created.
     * Use this at a time when you wish to log a user out of your app or change a user.
     * Once called, the SDK will no longer communicate with Intercom until a further registration is made.</p>
     *
     * @since 4.1.0
     */
    public abstract void logout();

    /**
     * <p>Gets the number of unread conversations for a user.</p>
     *
     * @return the current count of unread conversations for the registered user
     * @since 3.0.0
     */
    public abstract int getUnreadConversationCount();

    /**
     * <p>Sets a listener that will be notified when the unread conversation count for the registered user changes.</p>
     *
     * <p>This listener will immediately receive an update with the latest unread count, and it will be notified of all
     * future updates. If you want to know the unread count right now you can use the
     * {@link #getUnreadConversationCount() getUnreadConversationCount()} method</p>
     *
     * <p>Multiple listeners may be set. A strong reference is kept to each listener.</p>
     *
     * @param listener the listener to be notified of unread count updates
     * @since 3.0.0
     */
    public abstract void addUnreadConversationCountListener(@NonNull UnreadConversationCountListener listener);

    /**
     * <p>Removes a listener from the collection that receive updates when the unread conversation count changes.</p>
     *
     * @param listener the listener to be removed
     * @since 3.0.0
     */
    public abstract void removeUnreadConversationCountListener(UnreadConversationCountListener listener);

    /**
     * <p>Set the level of the logger</p>
     *
     * <p>Set the level of logs to display in the console by passing in an
     * Intercom.LogLevel eg. `setLogLevel(Intercom.LogLevel.ERROR);`
     * Passing in Intercom.LogLevel.DISABLED results in no logs being displayed.
     * Default log level is set to Intercom.LogLevel.WARN
     *
     * @param logLevel the level of logs to display
     * @since 1.1.4
     */
    public static void setLogLevel(@LogLevel int logLevel) {
        LumberMill.setLogLevel(logLevel);
    }

    /**
     * <p>Opens an article.</p>
     *
     * <p>Opens an article in a new activity
     *
     * @param articleId ID of the article
     * @since 8.3.0
     */
    public abstract void displayArticle(@NonNull String articleId);

    /**
     * <p>Fetches a list of collections from Intercom.</p>
     * @param collectionRequestCallback A callback that will be triggered after fetching data
     * @since 10.0.0
     */
    public abstract void fetchHelpCenterCollections(CollectionRequestCallback collectionRequestCallback);

    /**
     * <p>Fetches a list of articles and sections in a given collection.</p>
     * @param collectionId The collection Id for which a list of articles and sections should be loaded
     * @param collectionContentRequestCallback A callback that will be triggered after fetching data
     * @since 10.0.0
     */
    public abstract void fetchHelpCenterCollection(String collectionId,
                                                   CollectionContentRequestCallback collectionContentRequestCallback);

    /**
     * <p>Fetches a list of articles for a search term.</p>
     * @param searchTerm The text to search the help center with.
     * @param searchRequestCallback The collection Id for which a list of articles and sections should be loaded
     * @since 10.0.0
     */
    public abstract void searchHelpCenter(String searchTerm, SearchRequestCallback searchRequestCallback);
}
