/*
 * Decompiled with CFR 0.152.
 */
package jadx.plugins.tools;

import jadx.api.plugins.JadxPlugin;
import jadx.api.plugins.JadxPluginInfo;
import jadx.core.Jadx;
import jadx.core.plugins.versions.VerifyRequiredVersion;
import jadx.core.utils.GsonUtils;
import jadx.core.utils.StringUtils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.plugins.tools.JadxExternalPluginsLoader;
import jadx.plugins.tools.data.JadxInstalledPlugins;
import jadx.plugins.tools.data.JadxPluginMetadata;
import jadx.plugins.tools.data.JadxPluginUpdate;
import jadx.plugins.tools.resolvers.IJadxPluginResolver;
import jadx.plugins.tools.resolvers.ResolversRegistry;
import jadx.plugins.tools.utils.PluginFiles;
import jadx.plugins.tools.utils.PluginUtils;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JadxPluginsTools {
    private static final Logger LOG = LoggerFactory.getLogger(JadxPluginsTools.class);
    private static final JadxPluginsTools INSTANCE = new JadxPluginsTools();

    public static JadxPluginsTools getInstance() {
        return INSTANCE;
    }

    private JadxPluginsTools() {
    }

    public JadxPluginMetadata install(String locationId) {
        IJadxPluginResolver resolver = ResolversRegistry.getResolver(locationId);
        boolean hasVersion = resolver.hasVersion(locationId);
        if (hasVersion) {
            JadxPluginMetadata pluginMetadata = resolver.resolve(locationId).orElseThrow(() -> new JadxRuntimeException("Failed to resolve plugin location: " + locationId));
            this.fillMetadata(pluginMetadata);
            this.install(pluginMetadata);
            return pluginMetadata;
        }
        VerifyRequiredVersion verifyRequiredVersion = new VerifyRequiredVersion();
        for (int i = 1; i <= 5; ++i) {
            try {
                for (JadxPluginMetadata pluginMetadata : resolver.resolveVersions(locationId, i, 1)) {
                    this.fillMetadata(pluginMetadata);
                    if (!verifyRequiredVersion.isCompatible(pluginMetadata.getRequiredJadxVersion())) continue;
                    this.install(pluginMetadata);
                    return pluginMetadata;
                }
                continue;
            }
            catch (Exception e) {
                LOG.warn("Failed to fetch plugin ({} version before latest)", (Object)i, (Object)e);
            }
        }
        throw new JadxRuntimeException("Can't find compatible version to install");
    }

    public JadxPluginMetadata resolveMetadata(String locationId) {
        IJadxPluginResolver resolver = ResolversRegistry.getResolver(locationId);
        JadxPluginMetadata pluginMetadata = resolver.resolve(locationId).orElseThrow(() -> new RuntimeException("Failed to resolve locationId: " + locationId));
        this.fillMetadata(pluginMetadata);
        return pluginMetadata;
    }

    public List<JadxPluginMetadata> getVersionsByLocation(String locationId, int page, int perPage) {
        IJadxPluginResolver resolver = ResolversRegistry.getResolver(locationId);
        List<JadxPluginMetadata> list = resolver.resolveVersions(locationId, page, perPage);
        for (JadxPluginMetadata pluginMetadata : list) {
            this.fillMetadata(pluginMetadata);
        }
        return list;
    }

    public List<JadxPluginUpdate> updateAll() {
        JadxInstalledPlugins plugins = this.loadPluginsJson();
        int size = plugins.getInstalled().size();
        ArrayList<JadxPluginUpdate> updates = new ArrayList<JadxPluginUpdate>(size);
        ArrayList<JadxPluginMetadata> newList = new ArrayList<JadxPluginMetadata>(size);
        for (JadxPluginMetadata plugin : plugins.getInstalled()) {
            JadxPluginMetadata newVersion = null;
            try {
                newVersion = this.update(plugin);
            }
            catch (Exception e) {
                LOG.warn("Failed to update plugin: {}", (Object)plugin.getPluginId(), (Object)e);
            }
            if (newVersion != null) {
                updates.add(new JadxPluginUpdate(plugin, newVersion));
                newList.add(newVersion);
                continue;
            }
            newList.add(plugin);
        }
        if (!updates.isEmpty()) {
            plugins.setUpdated(System.currentTimeMillis());
            plugins.setInstalled(newList);
            this.savePluginsJson(plugins);
        }
        return updates;
    }

    public Optional<JadxPluginUpdate> update(String pluginId) {
        JadxInstalledPlugins plugins = this.loadPluginsJson();
        JadxPluginMetadata plugin = plugins.getInstalled().stream().filter(p -> p.getPluginId().equals(pluginId)).findFirst().orElseThrow(() -> new RuntimeException("Plugin not found: " + pluginId));
        JadxPluginMetadata newVersion = this.update(plugin);
        if (newVersion == null) {
            return Optional.empty();
        }
        plugins.setUpdated(System.currentTimeMillis());
        plugins.getInstalled().remove(plugin);
        plugins.getInstalled().add(newVersion);
        this.savePluginsJson(plugins);
        return Optional.of(new JadxPluginUpdate(plugin, newVersion));
    }

    public boolean uninstall(String pluginId) {
        JadxInstalledPlugins plugins = this.loadPluginsJson();
        Optional<JadxPluginMetadata> found = plugins.getInstalled().stream().filter(p -> p.getPluginId().equals(pluginId)).findFirst();
        if (found.isEmpty()) {
            return false;
        }
        JadxPluginMetadata plugin = found.get();
        this.deletePluginJar(plugin);
        plugins.getInstalled().remove(plugin);
        this.savePluginsJson(plugins);
        return true;
    }

    public List<JadxPluginMetadata> getInstalled() {
        return this.loadPluginsJson().getInstalled();
    }

    public List<JadxPluginInfo> getAllPluginsInfo() {
        try (JadxExternalPluginsLoader pluginsLoader = new JadxExternalPluginsLoader();){
            List<JadxPluginInfo> list = pluginsLoader.load().stream().map(JadxPlugin::getPluginInfo).collect(Collectors.toList());
            return list;
        }
    }

    public List<Path> getAllPluginJars() {
        ArrayList<Path> list = new ArrayList<Path>();
        for (JadxPluginMetadata pluginMetadata : this.loadPluginsJson().getInstalled()) {
            list.add(PluginFiles.INSTALLED_DIR.resolve(pluginMetadata.getJar()));
        }
        JadxPluginsTools.collectJarsFromDir(list, PluginFiles.DROPINS_DIR);
        return list;
    }

    public List<Path> getEnabledPluginJars() {
        ArrayList<Path> list = new ArrayList<Path>();
        for (JadxPluginMetadata pluginMetadata : this.loadPluginsJson().getInstalled()) {
            if (pluginMetadata.isDisabled()) continue;
            list.add(PluginFiles.INSTALLED_DIR.resolve(pluginMetadata.getJar()));
        }
        JadxPluginsTools.collectJarsFromDir(list, PluginFiles.DROPINS_DIR);
        return list;
    }

    public boolean changeDisabledStatus(String pluginId, boolean disabled) {
        JadxInstalledPlugins data = this.loadPluginsJson();
        JadxPluginMetadata plugin = data.getInstalled().stream().filter(p -> p.getPluginId().equals(pluginId)).findFirst().orElseThrow(() -> new RuntimeException("Plugin not found: " + pluginId));
        if (plugin.isDisabled() == disabled) {
            return false;
        }
        plugin.setDisabled(disabled);
        data.setUpdated(System.currentTimeMillis());
        this.savePluginsJson(data);
        return true;
    }

    @Nullable
    private JadxPluginMetadata update(JadxPluginMetadata plugin) {
        IJadxPluginResolver resolver = ResolversRegistry.getResolver(plugin.getLocationId());
        if (!resolver.isUpdateSupported()) {
            return null;
        }
        Optional<JadxPluginMetadata> updateOpt = resolver.resolve(plugin.getLocationId());
        if (updateOpt.isEmpty()) {
            return null;
        }
        JadxPluginMetadata update = updateOpt.get();
        if (Objects.equals(update.getVersion(), plugin.getVersion())) {
            return null;
        }
        this.fillMetadata(update);
        this.install(update);
        return update;
    }

    private void install(JadxPluginMetadata metadata) {
        String reqVersionStr = metadata.getRequiredJadxVersion();
        if (!VerifyRequiredVersion.isJadxCompatible((String)reqVersionStr)) {
            throw new JadxRuntimeException("Can't install plugin, required version: \"" + reqVersionStr + "\" is not compatible with current jadx version: " + Jadx.getVersion());
        }
        String version = metadata.getVersion();
        String fileName = metadata.getPluginId() + (String)(StringUtils.notBlank((String)version) ? "-" + version : "") + ".jar";
        Path pluginJar = PluginFiles.INSTALLED_DIR.resolve(fileName);
        this.copyJar(Paths.get(metadata.getJar(), new String[0]), pluginJar);
        metadata.setJar(PluginFiles.INSTALLED_DIR.relativize(pluginJar).toString());
        JadxInstalledPlugins plugins = this.loadPluginsJson();
        plugins.getInstalled().removeIf(p -> {
            if (p.getPluginId().equals(metadata.getPluginId())) {
                this.deletePluginJar((JadxPluginMetadata)p);
                return true;
            }
            return false;
        });
        plugins.getInstalled().add(metadata);
        plugins.setUpdated(System.currentTimeMillis());
        this.savePluginsJson(plugins);
    }

    private void fillMetadata(JadxPluginMetadata metadata) {
        try {
            Path tmpJar;
            if (JadxPluginsTools.needDownload(metadata.getJar())) {
                tmpJar = Files.createTempFile(metadata.getName(), "plugin.jar", new FileAttribute[0]);
                PluginUtils.downloadFile(metadata.getJar(), tmpJar);
                metadata.setJar(tmpJar.toAbsolutePath().toString());
            } else {
                tmpJar = Paths.get(metadata.getJar(), new String[0]);
            }
            this.fillMetadataFromJar(metadata, tmpJar);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to fill plugin metadata, plugin: " + metadata.getPluginId(), e);
        }
    }

    private void fillMetadataFromJar(JadxPluginMetadata metadata, Path jar) {
        try (JadxExternalPluginsLoader loader = new JadxExternalPluginsLoader();){
            JadxPlugin jadxPlugin = loader.loadFromJar(jar);
            JadxPluginInfo pluginInfo = jadxPlugin.getPluginInfo();
            metadata.setPluginId(pluginInfo.getPluginId());
            metadata.setName(pluginInfo.getName());
            metadata.setDescription(pluginInfo.getDescription());
            metadata.setHomepage(pluginInfo.getHomepage());
            metadata.setRequiredJadxVersion(pluginInfo.getRequiredJadxVersion());
        }
        catch (NoSuchMethodError e) {
            throw new RuntimeException("Looks like plugin uses unknown API, try to update jadx version", e);
        }
    }

    private static boolean needDownload(String jar) {
        return jar.startsWith("https://") || jar.startsWith("http://");
    }

    private void copyJar(Path sourceJar, Path destJar) {
        try {
            Files.copy(sourceJar, destJar, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to copy plugin jar: " + String.valueOf(sourceJar) + " to: " + String.valueOf(destJar), e);
        }
    }

    private void deletePluginJar(JadxPluginMetadata plugin) {
        try {
            Files.deleteIfExists(PluginFiles.INSTALLED_DIR.resolve(plugin.getJar()));
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private JadxInstalledPlugins loadPluginsJson() {
        JadxInstalledPlugins jadxInstalledPlugins;
        block9: {
            if (!Files.isRegularFile(PluginFiles.PLUGINS_JSON, new LinkOption[0])) {
                JadxInstalledPlugins plugins = new JadxInstalledPlugins();
                plugins.setVersion(1);
                return plugins;
            }
            BufferedReader reader = Files.newBufferedReader(PluginFiles.PLUGINS_JSON, StandardCharsets.UTF_8);
            try {
                JadxInstalledPlugins data = (JadxInstalledPlugins)GsonUtils.buildGson().fromJson((Reader)reader, JadxInstalledPlugins.class);
                this.upgradePluginsData(data);
                jadxInstalledPlugins = data;
                if (reader == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            ((Reader)reader).close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    throw new RuntimeException("Failed to read file: " + String.valueOf(PluginFiles.PLUGINS_JSON));
                }
            }
            ((Reader)reader).close();
        }
        return jadxInstalledPlugins;
    }

    private void savePluginsJson(JadxInstalledPlugins data) {
        if (data.getInstalled().isEmpty()) {
            try {
                Files.deleteIfExists(PluginFiles.PLUGINS_JSON);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to remove file: " + String.valueOf(PluginFiles.PLUGINS_JSON), e);
            }
            return;
        }
        data.getInstalled().sort(null);
        try (BufferedWriter writer = Files.newBufferedWriter(PluginFiles.PLUGINS_JSON, StandardCharsets.UTF_8, new OpenOption[0]);){
            GsonUtils.buildGson().toJson((Object)data, (Appendable)writer);
        }
        catch (Exception e) {
            throw new RuntimeException("Error saving file: " + String.valueOf(PluginFiles.PLUGINS_JSON), e);
        }
    }

    private void upgradePluginsData(JadxInstalledPlugins data) {
        if (data.getVersion() == 0) {
            data.setVersion(1);
        }
    }

    private static void collectJarsFromDir(List<Path> list, Path dir) {
        try (Stream<Path> files = Files.list(dir);){
            files.filter(p -> p.getFileName().toString().endsWith(".jar")).forEach(list::add);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}

