/*
 * Decompiled with CFR 0.152.
 */
package jadx.plugins.input.javaconvert;

import jadx.api.plugins.utils.CommonFileUtils;
import jadx.api.plugins.utils.ZipSecurity;
import jadx.plugins.input.javaconvert.AsmUtils;
import jadx.plugins.input.javaconvert.ConvertResult;
import jadx.plugins.input.javaconvert.D8Converter;
import jadx.plugins.input.javaconvert.DxConverter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.List;
import java.util.Objects;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JavaConvertLoader {
    private static final Logger LOG = LoggerFactory.getLogger(JavaConvertLoader.class);

    public static ConvertResult process(List<Path> input) {
        ConvertResult result = new ConvertResult();
        JavaConvertLoader.processJars(input, result);
        JavaConvertLoader.processAars(input, result);
        JavaConvertLoader.processClassFiles(input, result);
        return result;
    }

    private static void processJars(List<Path> input, ConvertResult result) {
        PathMatcher jarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.jar");
        input.stream().filter(jarMatcher::matches).forEach(path -> {
            try {
                JavaConvertLoader.convertJar(result, path);
            }
            catch (Exception e) {
                LOG.error("Failed to convert file: {}", (Object)path.toAbsolutePath(), (Object)e);
            }
        });
    }

    private static void processClassFiles(List<Path> input, ConvertResult result) {
        PathMatcher jarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.class");
        List clsFiles = input.stream().filter(jarMatcher::matches).collect(Collectors.toList());
        if (clsFiles.isEmpty()) {
            return;
        }
        try {
            Path jarFile = Files.createTempFile("jadx-", ".jar", new FileAttribute[0]);
            try (JarOutputStream jo = new JarOutputStream(Files.newOutputStream(jarFile, new OpenOption[0]));){
                for (Path file : clsFiles) {
                    String clsName = AsmUtils.getNameFromClassFile(file);
                    if (clsName == null || !ZipSecurity.isValidZipEntryName((String)clsName)) {
                        throw new IOException("Can't read class name from file: " + file);
                    }
                    JavaConvertLoader.addFileToJar(jo, file, clsName + ".class");
                }
            }
            result.addTempPath(jarFile);
            LOG.debug("Packed class files {} into jar {}", clsFiles, (Object)jarFile);
            JavaConvertLoader.convertJar(result, jarFile);
        }
        catch (Exception e) {
            LOG.error("Error process class files", (Throwable)e);
        }
    }

    private static void processAars(List<Path> input, ConvertResult result) {
        PathMatcher aarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.aar");
        input.stream().filter(aarMatcher::matches).forEach(path -> ZipSecurity.readZipEntries((File)path.toFile(), (entry, in) -> {
            try {
                String entryName = entry.getName();
                if (entryName.endsWith(".jar")) {
                    Path tempJar = CommonFileUtils.saveToTempFile((InputStream)in, (String)".jar");
                    result.addTempPath(tempJar);
                    LOG.debug("Loading jar: {} ...", (Object)entryName);
                    JavaConvertLoader.convertJar(result, tempJar);
                }
            }
            catch (Exception e) {
                LOG.error("Failed to process zip entry: {}", entry, (Object)e);
            }
        }));
    }

    private static void convertJar(ConvertResult result, Path path) throws Exception {
        if (JavaConvertLoader.repackAndConvertJar(result, path)) {
            return;
        }
        JavaConvertLoader.convertSimpleJar(result, path);
    }

    private static boolean repackAndConvertJar(ConvertResult result, Path path) throws Exception {
        Boolean repackNeeded = (Boolean)ZipSecurity.visitZipEntries((File)path.toFile(), (zipFile, zipEntry) -> {
            String entryName = zipEntry.getName();
            if (zipEntry.isDirectory()) {
                if (entryName.equals("BOOT-INF/")) {
                    return true;
                }
                if (entryName.equals("META-INF/versions/")) {
                    return true;
                }
            }
            if (entryName.endsWith(".jar")) {
                return true;
            }
            if (entryName.endsWith("module-info.class")) {
                return true;
            }
            return null;
        });
        if (!Objects.equals(repackNeeded, Boolean.TRUE)) {
            return false;
        }
        Path jarFile = Files.createTempFile("jadx-classes-", ".jar", new FileAttribute[0]);
        result.addTempPath(jarFile);
        try (JarOutputStream jo = new JarOutputStream(Files.newOutputStream(jarFile, new OpenOption[0]));){
            ZipSecurity.readZipEntries((File)path.toFile(), (entry, in) -> {
                try {
                    String entryName = entry.getName();
                    if (entryName.endsWith(".class")) {
                        if (entryName.endsWith("module-info.class") || entryName.startsWith("META-INF/versions/")) {
                            return;
                        }
                        byte[] clsFileContent = CommonFileUtils.loadBytes((InputStream)in);
                        String clsName = AsmUtils.getNameFromClassFile(clsFileContent);
                        if (clsName == null || !ZipSecurity.isValidZipEntryName((String)clsName)) {
                            throw new IOException("Can't read class name from file: " + entryName);
                        }
                        JavaConvertLoader.addJarEntry(jo, clsName + ".class", clsFileContent, entry.getLastModifiedTime());
                    } else if (entryName.endsWith(".jar")) {
                        Path tempJar = CommonFileUtils.saveToTempFile((InputStream)in, (String)".jar");
                        result.addTempPath(tempJar);
                        JavaConvertLoader.convertJar(result, tempJar);
                    }
                }
                catch (Exception e) {
                    LOG.error("Failed to process jar entry: {} in {}", new Object[]{entry, path, e});
                }
            });
        }
        JavaConvertLoader.convertSimpleJar(result, jarFile);
        return true;
    }

    private static void convertSimpleJar(ConvertResult result, Path path) throws Exception {
        Path tempDirectory = Files.createTempDirectory("jadx-", new FileAttribute[0]);
        result.addTempPath(tempDirectory);
        try {
            DxConverter.run(path, tempDirectory);
        }
        catch (Throwable e) {
            LOG.warn("DX convert failed, trying D8, path: {}", (Object)path);
            try {
                D8Converter.run(path, tempDirectory);
            }
            catch (Throwable ex) {
                LOG.error("D8 convert failed: {}", (Object)ex.getMessage());
            }
        }
        LOG.debug("Converted to dex: {}", (Object)path.toAbsolutePath());
        result.addConvertedFiles(JavaConvertLoader.collectFilesInDir(tempDirectory));
    }

    private static List<Path> collectFilesInDir(Path tempDirectory) throws IOException {
        PathMatcher dexMatcher = FileSystems.getDefault().getPathMatcher("glob:**.dex");
        try (Stream<Path> pathStream = Files.walk(tempDirectory, 1, new FileVisitOption[0]);){
            List<Path> list = pathStream.filter(p -> Files.isRegularFile(p, LinkOption.NOFOLLOW_LINKS)).filter(dexMatcher::matches).collect(Collectors.toList());
            return list;
        }
    }

    private static void addFileToJar(JarOutputStream jar, Path source, String entryName) throws IOException {
        byte[] fileContent = Files.readAllBytes(source);
        FileTime lastModifiedTime = Files.getLastModifiedTime(source, LinkOption.NOFOLLOW_LINKS);
        JavaConvertLoader.addJarEntry(jar, entryName, fileContent, lastModifiedTime);
    }

    private static void addJarEntry(JarOutputStream jar, String entryName, byte[] content, FileTime modTime) throws IOException {
        JarEntry entry = new JarEntry(entryName);
        if (modTime != null) {
            entry.setTime(modTime.toMillis());
        }
        jar.putNextEntry(entry);
        jar.write(content);
        jar.closeEntry();
    }
}

