/*
 * Decompiled with CFR 0.152.
 */
package nl.altindag.ssl.util;

import java.io.IOException;
import java.io.InputStream;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.net.Proxy;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509ExtendedTrustManager;
import nl.altindag.ssl.SSLFactory;
import nl.altindag.ssl.exception.GenericCertificateException;
import nl.altindag.ssl.exception.GenericIOException;
import nl.altindag.ssl.util.CertificateUtils;
import nl.altindag.ssl.util.SSLSessionUtils;
import nl.altindag.ssl.util.TrustManagerUtils;
import nl.altindag.ssl.util.internal.CollectorsUtils;

public class CertificateExtractingClient {
    private static final Pattern CA_ISSUERS_AUTHORITY_INFO_ACCESS = Pattern.compile("(?s)^AuthorityInfoAccess\\h+\\[\\R\\s*\\[\\R.*?accessMethod:\\h+caIssuers\\R\\h*accessLocation: URIName:\\h+(https?://\\S+)", 8);
    private static CertificateExtractingClient instance;
    private final boolean shouldResolveRootCa;
    private final Proxy proxy;
    private final SSLFactory sslFactoryForCertificateCapturing;
    private final SSLFactory unsafeSslFactory;
    private final SSLSocketFactory unsafeSslSocketFactory;
    private final SSLSocketFactory certificateCapturingSslSocketFactory;
    private final List<X509Certificate> certificatesCollector;

    private CertificateExtractingClient(boolean shouldResolveRootCa, Proxy proxy, PasswordAuthentication passwordAuthentication) {
        this.shouldResolveRootCa = shouldResolveRootCa;
        this.proxy = proxy;
        if (passwordAuthentication != null) {
            FelixAuthenticator authenticator = new FelixAuthenticator(passwordAuthentication);
            Authenticator.setDefault(authenticator);
        }
        this.certificatesCollector = new CopyOnWriteArrayList<X509Certificate>();
        X509ExtendedTrustManager certificateCapturingTrustManager = TrustManagerUtils.createCertificateCapturingTrustManager(this.certificatesCollector);
        this.sslFactoryForCertificateCapturing = SSLFactory.builder().withTrustMaterial(certificateCapturingTrustManager).build();
        this.unsafeSslFactory = SSLFactory.builder().withUnsafeTrustMaterial().build();
        this.certificateCapturingSslSocketFactory = this.sslFactoryForCertificateCapturing.getSslSocketFactory();
        this.unsafeSslSocketFactory = this.unsafeSslFactory.getSslSocketFactory();
    }

    static CertificateExtractingClient getInstance() {
        if (instance == null) {
            instance = new CertificateExtractingClient(true, null, null);
        } else {
            CertificateExtractingClient.instance.certificatesCollector.clear();
            SSLSessionUtils.invalidateCaches(CertificateExtractingClient.instance.sslFactoryForCertificateCapturing);
        }
        return instance;
    }

    public List<X509Certificate> get(String url) {
        try {
            URL parsedUrl = new URL(url);
            if ("https".equalsIgnoreCase(parsedUrl.getProtocol())) {
                HttpsURLConnection connection = (HttpsURLConnection)this.createConnection(parsedUrl);
                connection.setSSLSocketFactory(this.certificateCapturingSslSocketFactory);
                connection.connect();
                connection.disconnect();
                if (this.shouldResolveRootCa) {
                    List<X509Certificate> resolvedRootCa = this.getRootCaFromChainIfPossible(this.certificatesCollector);
                    List<X509Certificate> list = Stream.of(this.certificatesCollector, resolvedRootCa).flatMap(Collection::stream).collect(CollectorsUtils.toUnmodifiableList());
                    return list;
                }
                List<X509Certificate> list = Collections.unmodifiableList(this.certificatesCollector);
                return list;
            }
            List<X509Certificate> list = Collections.emptyList();
            return list;
        }
        catch (IOException e) {
            throw new GenericIOException(String.format("Failed getting certificate from: [%s]", url), e);
        }
        finally {
            SSLSessionUtils.invalidateCaches(this.sslFactoryForCertificateCapturing);
        }
    }

    private URLConnection createConnection(URL url) throws IOException {
        return this.proxy != null ? url.openConnection(this.proxy) : url.openConnection();
    }

    List<X509Certificate> getRootCaFromChainIfPossible(List<X509Certificate> certificates) {
        String subject;
        X509Certificate certificate;
        String issuer;
        boolean isSelfSignedCertificate;
        if (!certificates.isEmpty() && !(isSelfSignedCertificate = (issuer = (certificate = certificates.get(certificates.size() - 1)).getIssuerX500Principal().getName()).equals(subject = certificate.getSubjectX500Principal().getName()))) {
            return this.getRootCaIfPossible(certificate);
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getRootCaIfPossible(X509Certificate x509Certificate) {
        List<X509Certificate> rootCaFromAuthorityInfoAccessExtension = this.getRootCaFromAuthorityInfoAccessExtensionIfPresent(x509Certificate);
        if (!rootCaFromAuthorityInfoAccessExtension.isEmpty()) {
            return rootCaFromAuthorityInfoAccessExtension;
        }
        List<X509Certificate> rootCaFromJdkTrustedCertificates = this.getRootCaFromJdkTrustedCertificates(x509Certificate);
        if (!rootCaFromJdkTrustedCertificates.isEmpty()) {
            return rootCaFromJdkTrustedCertificates;
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getRootCaFromAuthorityInfoAccessExtensionIfPresent(X509Certificate certificate) {
        String certificateContent = certificate.toString();
        Matcher caIssuersMatcher = CA_ISSUERS_AUTHORITY_INFO_ACCESS.matcher(certificateContent);
        if (caIssuersMatcher.find()) {
            String issuerLocation = caIssuersMatcher.group(1);
            return this.getCertificatesFromRemoteFile(URI.create(issuerLocation), certificate);
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getCertificatesFromRemoteFile(URI uri, X509Certificate intermediateCertificate) {
        try {
            URL url = uri.toURL();
            URLConnection connection = this.createConnection(url);
            if (connection instanceof HttpsURLConnection) {
                ((HttpsURLConnection)connection).setSSLSocketFactory(this.unsafeSslSocketFactory);
            }
            InputStream inputStream = connection.getInputStream();
            List<X509Certificate> certificates = CertificateUtils.parseDerCertificate(inputStream).stream().filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).filter(issuer -> this.isIssuerOfIntermediateCertificate(intermediateCertificate, (X509Certificate)issuer)).collect(CollectorsUtils.toUnmodifiableList());
            inputStream.close();
            List<X509Certificate> list = certificates;
            return list;
        }
        catch (IOException e) {
            throw new GenericCertificateException(e);
        }
        finally {
            SSLSessionUtils.invalidateCaches(this.unsafeSslFactory);
        }
    }

    List<X509Certificate> getRootCaFromJdkTrustedCertificates(X509Certificate intermediateCertificate) {
        List<X509Certificate> jdkTrustedCertificates = CertificateUtils.getJdkTrustedCertificates();
        return jdkTrustedCertificates.stream().filter(issuer -> this.isIssuerOfIntermediateCertificate(intermediateCertificate, (X509Certificate)issuer)).collect(CollectorsUtils.toUnmodifiableList());
    }

    boolean isIssuerOfIntermediateCertificate(X509Certificate intermediateCertificate, X509Certificate issuer) {
        try {
            intermediateCertificate.verify(issuer.getPublicKey());
            return true;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | CertificateException e) {
            return false;
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Proxy proxy = null;
        private PasswordAuthentication passwordAuthentication = null;
        private boolean shouldResolveRootCa = true;

        public Builder withProxy(Proxy proxy) {
            this.proxy = proxy;
            return this;
        }

        public Builder withPasswordAuthentication(PasswordAuthentication passwordAuthentication) {
            this.passwordAuthentication = passwordAuthentication;
            return this;
        }

        public Builder withResolvedRootCa(boolean shouldResolveRootCa) {
            this.shouldResolveRootCa = shouldResolveRootCa;
            return this;
        }

        public CertificateExtractingClient build() {
            return new CertificateExtractingClient(this.shouldResolveRootCa, this.proxy, this.passwordAuthentication);
        }
    }

    private static class FelixAuthenticator
    extends Authenticator {
        private final PasswordAuthentication passwordAuthentication;

        private FelixAuthenticator(PasswordAuthentication passwordAuthentication) {
            this.passwordAuthentication = passwordAuthentication;
        }

        @Override
        protected PasswordAuthentication getPasswordAuthentication() {
            return this.passwordAuthentication;
        }
    }
}

