/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.github.glytching.junit.extension.folder;

import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ExtensionContext.Namespace;
import org.junit.jupiter.api.extension.ParameterContext;
import org.junit.jupiter.api.extension.ParameterResolutionException;
import org.junit.jupiter.api.extension.ParameterResolver;

/**
 * The temporary folder extension provides a test with access to temporary files and directories.
 * The temporary folder extension provides a {@link TemporaryFolder} which you can use to create a
 * temporary file or directory for use by your test. The {@link TemporaryFolder} can be injected
 * into your test or test case with any of the following approaches:
 *
 * <ul>
 *   <li>Instance variable injection into a {@code @BeforeEach} method. The {@link TemporaryFolder}
 *       will be destroyed during {@code @AfterEach} and no exception will be thrown in cases where
 *       the deletion fails. For example:
 *       <pre>
 *  private TemporaryFolder temporaryFolder;
 *
 *  &#064;BeforeEach
 *  public void setUp(TemporaryFolder temporaryFolder) {
 *      this.temporaryFolder = temporaryFolder
 *      // ...
 *  }
 * </pre>
 *   <li>Parameter injection into a {@code @Test} method. The {@link TemporaryFolder} will be
 *       destroyed during {@code @AfterEach} and no exception will be thrown in cases where the
 *       deletion fails. For example:
 *       <pre>
 *  &#064;Test
 *  public void testUsingTemporaryFolder(TemporaryFolder temporaryFolder) {
 *      // ...
 *  }
 * </pre>
 *   <li>Class variable injection using a {@code @BeforeAll} method. Note: in this case <b>all</b>
 *       tests in the test case will share the same instance of the {@code TemporaryFolder}. The
 *       {@link TemporaryFolder} will be destroyed after any {@code @AfterAll} method completes and
 *       no exception will be thrown in cases where the deletion fails. For example:
 *       <pre>
 *  private static TemporaryFolder TEMPORARY_FOLDER;
 *
 *  &#064;BeforeAll
 *  public static void setUp(TemporaryFolder givenTemporaryFolder) {
 *      TEMPORARY_FOLDER = givenTemporaryFolder
 *      // ...
 *  }
 * </pre>
 * </ul>
 *
 * <p>Usage examples:
 *
 * <p>Injecting a {@code TemporaryFolder} in a {@code @BeforeEach} method:
 *
 * <pre>
 * &#064;ExtendWith(TemporaryFolderExtension.class)
 * public class MyTest {
 *
 *     private TemporaryFolder temporaryFolder;
 *
 *     &#064;BeforeEach
 *     public void setUp(TemporaryFolder temporaryFolder) {
 *         this.temporaryFolder = temporaryFolder
 *         // ...
 *     }
 *
 *     &#064;Test
 *     public void testUsingTemporaryFile() {
 *         File file = temporaryFolder.createFile("foo.txt");
 *         // ...
 *     }
 *
 *     &#064;Test
 *     public void testUsingTemporaryDirectory() {
 *         // use the temporary folder itself
 *         File root = temporaryFolder.getRoot();
 *
 *         // create a sub directory within the temporary folder
 *         File file = temporaryFolder.createDirectory("foo");
 *         // ...
 *     }
 * }
 * </pre>
 *
 * <p>Injecting a {@code TemporaryFolder} in a {@code @Test} method:
 *
 * <pre>
 * public class MyTest {
 *
 *     &#064;Test
 *     &#064;ExtendWith(TemporaryFolderExtension.class)
 *     public void testUsingTemporaryFile(TemporaryFolder temporaryFolder) {
 *         File file = temporaryFolder.createFile("foo.txt");
 *         // ...
 *     }
 *
 *     &#064;Test
 *     &#064;ExtendWith(TemporaryFolderExtension.class)
 *     public void testUsingTemporaryDirectory(TemporaryFolder temporaryFolder) {
 *         // use the temporary folder itself
 *         File root = temporaryFolder.getRoot();
 *
 *         // create a sub directory within the temporary folder
 *         File file = temporaryFolder.createDirectory("foo");
 *         // ...
 *     }
 * }
 * </pre>
 *
 * @see <a href="https://github.com/junit-team/junit4/wiki/Rules#temporaryfolder-rule">JUnit 4
 *     TemporaryFolder Rule</a>
 * @since 1.0.0
 */
public class TemporaryFolderExtension implements ParameterResolver {

  private static final Namespace NAMESPACE = Namespace.create(TemporaryFolderExtension.class);

  /**
   * Does this extension support injection for parameters of the type described by the given {@code
   * parameterContext}?
   *
   * @param parameterContext the context for the parameter for which an argument should be resolved
   * @param extensionContext the <em>context</em> in which the current test or container is being
   *     executed
   * @return true if the given {@code parameterContext} describes a parameter of type: {@link
   *     TemporaryFolder}, false otherwise
   * @throws ParameterResolutionException
   */
  @Override
  public boolean supportsParameter(
      ParameterContext parameterContext, ExtensionContext extensionContext)
      throws ParameterResolutionException {
    return appliesTo(parameterContext.getParameter().getType());
  }

  /**
   * Provides a value for any parameter context which has passed the {@link
   * #supportsParameter(ParameterContext, ExtensionContext)} gate.
   *
   * @param parameterContext the context for the parameter for which an argument should be resolved
   * @param extensionContext the <em>context</em> in which the current test or container is being
   *     executed
   * @return a new {@link TemporaryFolder}
   * @throws ParameterResolutionException
   */
  @Override
  public Object resolveParameter(
      ParameterContext parameterContext, ExtensionContext extensionContext)
      throws ParameterResolutionException {
    return extensionContext
        .getStore(NAMESPACE)
        .getOrComputeIfAbsent(
            parameterContext, key -> new TemporaryFolder(), TemporaryFolder.class);
  }

  private boolean appliesTo(Class<?> clazz) {
    return clazz == TemporaryFolder.class;
  }
}
